console.log('🚀 ZenCode AI loading...');

window.addEventListener('load', () => {
    console.log('📍 Window loaded');
    const rootElement = document.getElementById('ai_app_root');
    if (!rootElement) {
        console.error('❌ ai_app_root element not found!');
        return;
    }

    if (typeof zencode_ai_app_vars === 'undefined') {
        console.error('❌ zencode_ai_app_vars not defined - WordPress localization may have failed');
        rootElement.innerHTML = '<div style="padding: 2rem; text-align: center; color: #f56565;"><h3>Configuration Error</h3><p>WordPress variables not loaded. Please check your plugin configuration and ensure scripts are enqueued correctly.</p></div>';
        return;
    }

    console.log('📍 Starting app initialization...');

    const App = {
       // REPLACE your current state object with this cleaned-up version.
state: {
    isBatchUpdating: false,
    _postUploadSave: 0,
    isSaving: false, 
	reflectionMemory: [],
	lastAPICallTime: null,           // NEW: Track last API call
    rateLimitHit: false, 
 modificationQueue: [], // <<< ADD THIS LINE	// NEW: Flag when rate limited
    reflectionCount: 0,              // NEW: Count reflections per session
    maxReflectionsPerPlan: 25,        // NEW: Limit reflections to avoid rate limits
    masterLearnings: [],
    pendingChatImage: null, // Only appears once now
    isExecutingProcess: false,
    pendingPlanPrompt: null,
    currentVisionPrompt: null,
	 codeMirrorInstance: null,
    hasCodeMirror: false,
    projectFiles: {},
    fileOrder: [],
	
    activeFile: null,
    chatHistory: [],
    polishSessionActive: false,
    polishResults: null,
    isAILoading: false,
    _lastResponseWasImagePlan: false,
    isTyping: false,
    isSidebarOpen: window.innerWidth > 768,
    currentAIMode: 'normal',
    pendingModification: null,
    hasJszip: false,
    hasJsDiff: false,
    collapsedFolders: new Set(),
    currentModel: zencode_ai_app_vars.default_model || 'google/gemini-2.5-pro',
    generatedWebsiteURL: null,
    contextFiles: new Set(),
    isOneShotBuild: false,
    oneShotImage: null,
    saveState: 'saved',
	  planContextCache: null,           // ADD THIS - stores cached file selections for current plan
    planContextCacheTimestamp: null,  // ADD THIS - tracks when cache was created
    autoSaveTimer: null,
    currentRequestController: null,
    allProjects: [],
    visibleProjectCount: 5,
    projectContext: {
        originalVision: '',
        designRequirements: '',
        technicalNotes: '',
        targetAudience: '',
        createdAt: null
    },
    longRequestTimer: null,
    refinementToken: null,
    lastGatedHash: null,
    lastGatedAt: 0,
    generatedWebsiteResources: null,
    tempVisionImage: null,
    tempVisionImagePath: null,
    isExecutingPlan: false,
    currentPlan: [],
    versionControlOpen: false,
    projectVersions: [],
    isVersionLoading: false,
    visibleVersionCount: 5,
    currentStepIndex: 0,    // ✨ NEW: Parallel step preparation
    preparedSteps: new Map(),           // Cache of prepared step contexts
    preparationQueue: null,             // Current preparation promise
    isPreparingNextStep: false,         // Flag for background preparation

    // PERFORMANCE OPTIMIZATIONS
    planExecutionMode: false,        // Batch operations during plan execution
    enableReflection: true,          // Can be disabled during plans
    pendingSaveOperation: null,      // Track pending saves during plan
    // NEW: Step selection and guardrails
    selectedStepCount: 5,            // Default to balanced 5 steps
    minimumRecommendedSteps: 3,      // Will be calculated based on complexity
    stepSelectionMode: 'balanced',   // 'quick', 'balanced', 'detailed', 'custom'
    skipPolishUnify: false,          // Whether to skip optional polish step
},

config: {
    TYPING_DELAY_MS: 1,      // Set this back to 1ms
    TYPING_CHUNK_SIZE: 5,    // NEW: This controls how many letters are typed per cycle.
    AUTOSAVE_DELAY_MS: 2000,
    modelProviderMap: {
        'google/gemini-2.5-pro': 'google',
		 'google/gemini-2.5-pro': 'google', // <<< ADD THIS LINE
    },
    EDITABLE_FILE_EXTENSIONS: new Set([
        'txt', 'html', 'css', 'js', 'json', 'md', 'php', 'xml', 'svg', 'jsx', 'ts', 'py', 'rb', 'sh'
    ]),
},

        getUserPlan() {
            return (zencode_ai_app_vars?.user_plan || '').toString().trim().toLowerCase();
        },

        async init() {
            try {
                console.log('🏗️ Initializing app...');
                this.injectCSS();
                this.injectHTML();
                this.cacheDOMElements();
				  if (this.state.hasCodeMirror) { this.initializeCodeMirror(); } // ADD THIS LINE
                this.bindEvents();
                this.state.currentModel = 'google/gemini-2.5-pro';
                this.updateModeToggleVisuals();

                await this.loadProject();

                if (zencode_ai_app_vars.user_plan === 'spark' && this.state.currentAIMode === 'vector') {
                    this.state.currentAIMode = 'normal';
                    this.updateModeToggleVisuals();
                }

                if (Object.keys(this.state.projectFiles).length === 0) {
                    this.startNewProject('Default Project', false);
                }
				


                this.updateContextWithAllFiles();
                this.handleResponsiveLayout();
                if (this.state.chatHistory.length === 0) {
                    this.initializeWelcome();
                }
                this.updateChatPlaceholder();
                this.activateFile(this.state.fileOrder[0] || null);
                this.updatePreviewButtonVisibility();

                window.addEventListener('beforeunload', () => {
                    this.cleanupPreviewResources();
                });
                
                console.log('🎉 App initialization complete!');
				 this.initializeTour();
            } catch (error) {
                console.error('💥 App initialization failed:', error);
                if (rootElement) {
                    rootElement.innerHTML = `<div style="padding: 2rem; text-align: center; color: #f56565; font-family: -apple-system, BlinkMacSystemFont, sans-serif;"><h3>🚨 Initialization Error</h3><p>The AI Assistant failed to load properly.</p><details style="margin-top: 1rem; text-align: left; background: #1a1a1a; padding: 1rem; border-radius: 8px;"><summary style="cursor: pointer; color: #fff;">Show Error Details</summary><pre style="margin-top: 0.5rem; font-size: 12px; color: #ff6b6b;">${error.stack || error.message}</pre></details><button onclick="location.reload()" style="margin-top: 1rem; padding: 0.5rem 1rem; background: #667eea; color: white; border: none; border-radius: 6px; cursor: pointer;">🔄 Reload Page</button></div>`;
                }
            }
        },


initializeTour() {
    // Skip tour setup on mobile devices
    if (window.innerWidth <= 768) {
        return;
    }
    
    // Replace footer brand text with tour button on desktop
    const footerBrand = document.querySelector('.footer-brand');
    if (footerBrand) {
        footerBrand.innerHTML = `
            <button class="footer-tour-button" title="Take Tour">
                <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="10"></circle>
                    <path d="M9.09 9a3 3 0 0 1 5.83 1c0 2-3 3-3 3"></path>
                    <line x1="12" y1="17" x2="12.01" y2="17"></line>
                </svg>
                Tour
            </button>
        `;
        
        const tourButton = footerBrand.querySelector('.footer-tour-button');
        tourButton.addEventListener('click', () => this.startTour());
    }
    
    // Auto-start tour for new users
    if (!localStorage.getItem('zc_tour_completed')) {
        setTimeout(() => {
            this.showWelcomeModal();
        }, 1000);
    }
},

showWelcomeModal() {
    const modal = document.createElement('div');
    modal.className = 'tour-welcome-modal';
    modal.innerHTML = `
        <div class="tour-welcome-overlay"></div>
        <div class="tour-welcome-content">
            <div class="tour-welcome-header">
                <h2>Welcome to ZenCode AI!</h2>
                <button class="tour-close-btn">&times;</button>
            </div>
            <div class="tour-welcome-body">
                <p>This is your AI-powered development environment. Would you like a quick tour to get started?</p>
                <div class="tour-welcome-buttons">
                    <button class="tour-btn-secondary" onclick="App.skipTour()">Skip Tour</button>
                    <button class="tour-btn-primary" onclick="App.startTour()">Start Tour</button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modal);
    
    // Close handlers
    modal.querySelector('.tour-close-btn').addEventListener('click', () => this.skipTour());
    modal.querySelector('.tour-welcome-overlay').addEventListener('click', () => this.skipTour());
},

skipTour() {
  const modal = document.querySelector('.tour-welcome-modal');
  if (modal) modal.remove();

  // also remove step bubble/backdrop/highlights if tour already started
  this.removeTourStep?.();

  localStorage.setItem('zc_tour_completed', 'skipped');
},
// [REPLACE] your entire executeScaffoldingStep function with this definitive, corrected version.
async executeScaffoldingStep(originalStepText) {
    this.renderChatMessage('ai', "🚀 Starting intelligent project scaffolding...");
    
    const filesBefore = {};

    // Create a much smarter scaffolding prompt that is less ambiguous.
    const fullPlanText = this.state.currentPlan.map((s, i) => `${i + 1}. ${s.text}`).join('\n');
    
    // THE FIX: Escape all literal backticks (`) inside the template literal with a backslash (\).
    const scaffoldingPrompt = `INITIAL PROJECT SCAFFOLDING REQUEST:

PROJECT VISION: "${this.state.projectContext.originalVision || originalStepText}"

FULL DEVELOPMENT PLAN:
${fullPlanText}

**CRITICAL SCAFFOLDING INSTRUCTIONS:**

1.  **ANALYZE, THEN CREATE:** First, analyze the **FULL DEVELOPMENT PLAN** to determine the project type (e.g., a WordPress Plugin, a static website, a React app).
2.  **PRIORITIZE FUNCTIONAL FILES:** Your primary goal is to create the **real, functional files** mentioned in the first few steps of the plan. For a WordPress plugin, this means creating the main \`plugin-name.php\` file, NOT \`index.html\`.
3.  **IMPLEMENT THE FIRST STEP:** After creating the necessary files and folders, you must write the complete, working code for the **first feature** as described in the plan.
4.  **DO NOT CREATE ONLY PLACEHOLDERS:** You are strictly forbidden from creating a project structure that consists only of empty \`index.php\` or \`index.html\` files. You must create the functional code.
5.  **INCLUDE PROFESSIONAL EXTRAS:** In addition to the functional files, always include a \`README.md\` and a \`.gitignore\` file.

Execute this scaffolding task now, ensuring you create a working, functional foundation based on the project type you identified from the plan.`;

    await this._executeSend(scaffoldingPrompt, true, null, "Intelligent Project Scaffolding", { 
        isStepExecution: true,
        isScaffoldingStep: true 
    });

    // Update state like a normal step
    const filesAfter = this.state.projectFiles;
    this.updatePlanContextAfterStep(0, originalStepText, filesBefore, filesAfter);
    this.generateStepSummary(1, "Project Foundation", filesBefore, filesAfter);

    // Mark the original first step as 'done' since scaffolding has replaced it
    this.state.currentPlan[0].status = 'done';
    this.updateStepUI(0, 'done');
    
    return true;
},


startTour() {
    // Remove welcome modal if present
    const modal = document.querySelector('.tour-welcome-modal');
    if (modal) modal.remove();
    
    this.currentTourStep = 0;
    this.tourSteps = [
    {
        element: '.sidebar',
        title: 'Project Panel',
        text: 'This is your project file manager. Create folders, upload files, and organize your project structure here.',
        position: 'right'
    },
    {
        element: '.context-toggle',
        title: 'Context Control (Critical!)',
        text: 'These checkboxes control which files the AI can see. Only checked files are included in the AI\'s context. This is essential for managing costs and focusing the AI\'s attention.',
        position: 'right',
        highlight: true
    },
    {
        element: '.code-editor-panel',
        title: 'Code Editor',
        text: 'Write and edit your code here. The AI can modify files directly in this editor, and you can accept or reject changes.',
        position: 'left'
    },
    {
        element: '.chat-panel',
        title: 'AI Assistant',
        text: 'Chat with the AI here. Ask it to create files, modify code, explain concepts, or help with debugging.',
        position: 'left'
    },
    {
        element: '#mode-toggle-switch',
        title: 'AI Modes',
        text: 'Switch between different AI modes: Normal (direct coding), Plan (step-by-step project planning), and Vector (query your knowledge base).',
        position: 'bottom',
        highlight: true
    },
    {
        element: '.build-website-button',
        title: 'One-Shot Builder',
        text: 'Generate a complete website instantly from a description. Perfect for rapid prototyping and getting started quickly.',
        position: 'bottom'
    },
{
    element: '.file-tree .vectorize-btn',
    title: 'Knowledge Base',
    text: 'Save important files to the AI\'s long-term memory. This creates a personal knowledge base for more context-aware assistance.',
    position: 'right',
    fallback: '.file-tree'
},
    {
        element: '#btn-view-generated-website',
        title: 'Preview Your Work',
        text: 'Click here to preview your HTML files in a new tab. See your websites come to life!',
        position: 'bottom',
        fallback: '.editor-actions'
    }
];
    
    this.showTourStep();
},
// REPLACE your entire showPlanConfigModal function with this.
async showPlanConfigModal(prompt, specData = null, displayMessage = null) {
    // Store the pending plan data, using the new displayMessage if provided.
    this.state._pendingPlan = { 
        prompt: prompt,
        specData: specData,
        // If a displayMessage is provided, use it. Otherwise, fall back to the main prompt.
        displayMessage: displayMessage || prompt 
    };
    
    if (!this.state._pendingPlan || !this.state._pendingPlan.prompt) {
        console.error("Plan generation triggered without a valid prompt.");
        return;
    }
    
    // Show step selection modal as before
    this.showStepSelectionModal(prompt);
},

// NEW: This function is called after user confirms step selection
async proceedWithPlanGeneration() {
    const staged = this.state._pendingPlan || null;
    
    if (!staged || !staged.prompt) {
        console.error("No pending plan to generate.");
        return;
    }

    this.state.isExecutingProcess = true;
    this.showPlanGenerationLoading();

    let finalPrompt;
    const histMsg = staged?.displayMessage || staged.prompt;

    // Include step count in the instruction
    const jsonInstruction = `
**RESPONSE FORMAT:**
Your entire response MUST be a single, valid JSON object.
Do not write any text or markdown before or after the JSON object.
The JSON object must have the following structure:
{
  "introduction": "A brief, one-paragraph conversational introduction. For image plans, describe the image and its influence. For text plans, paraphrase the user's request.",
  "plan": [
    "A concise, actionable, self-contained coding step. DO NOT start this string with a number.",
    "Another concise, actionable, self-contained coding step. DO NOT start this string with a number.",
    "..."
  ]
}

IMPORTANT: The 'plan' array should contain EXACTLY ${this.state.selectedStepCount} steps. Each step should be substantial enough to avoid timeouts.
The "plan" array must also follow the PLANNING CONSTITUTION.`;

    if (staged?.imageData) {
        finalPrompt = `${staged.prompt}\n\n${jsonInstruction}`;
    } else {
        const userPrompt = staged.prompt;
        finalPrompt = `You are an expert AI project architect creating a plan for: "${userPrompt}"

**PLANNING CONSTITUTION:**
1.  **CODE-ONLY TASKS:** Every step must be a direct coding or file creation task.
2.  **NO FLUFF:** Do not include conceptual steps like "Planning" or "Design."
3.  **BE CONCISE:** Each step must be a single, self-contained sentence.
4.  **AVOID TIMEOUTS:** Each step should be substantial but not overly complex to prevent execution timeouts.

${jsonInstruction}`;
    }

    this.renderChatMessage('user', histMsg);
    this.elements.chatInput.value = '';

    const executionOptions = { 
        specData: staged?.specData,
        numSteps: this.state.selectedStepCount  // Pass step count to backend
    };

    try {
        if (staged?.imageData) {
            await this._executeSend(finalPrompt, true, staged.imageData, histMsg, executionOptions);
        } else {
            await this._executeSend(finalPrompt, true, null, histMsg, executionOptions);
        }
    } finally {
        this.hidePlanGenerationLoading();
        this.state.isExecutingProcess = false;
        this.state._pendingPlan = null;
        this.state.pendingPlanPrompt = null;
    }
},
showTourStep() {
    const step = this.tourSteps[this.currentTourStep];
    if (!step) {
        this.completeTour();
        return;
    }
    
    // Remove existing tour step
    this.removeTourStep();
    
    // Find target element
    let targetElement = document.querySelector(step.element);
    if (!targetElement && step.fallback) {
        targetElement = document.querySelector(step.fallback);
    }
    
    if (!targetElement) {
        this.nextTourStep();
        return;
    }
    
    // Create tour step
    const tourStep = document.createElement('div');
    tourStep.className = 'tour-step';
    tourStep.setAttribute('data-step', this.currentTourStep);
    tourStep.innerHTML = `
        <div class="tour-step-content">
            <div class="tour-step-header">
                <h3>${step.title}</h3>
                <span class="tour-step-counter">${this.currentTourStep + 1} of ${this.tourSteps.length}</span>
            </div>
            <p>${step.text}</p>
            <div class="tour-step-buttons">
                <button class="tour-btn-secondary" onclick="App.skipTour()">Skip Tour</button>
                <div class="tour-nav-buttons">
                    ${this.currentTourStep > 0 ? '<button class="tour-btn-secondary" onclick="App.prevTourStep()">Previous</button>' : ''}
                    <button class="tour-btn-primary" onclick="App.nextTourStep()">
                        ${this.currentTourStep === this.tourSteps.length - 1 ? 'Finish' : 'Next'}
                    </button>
                </div>
            </div>
        </div>
        <div class="tour-step-arrow"></div>
    `;
    
    document.body.appendChild(tourStep);
    
    // Position the tour step
    this.positionTourStep(tourStep, targetElement, step.position);
    
    // Highlight target element
    if (step.highlight) {
        targetElement.classList.add('tour-highlight');
    }
    
    // Scroll into view
    targetElement.scrollIntoView({ 
        behavior: 'smooth', 
        block: 'center' 
    });
    
    // Add backdrop
    const backdrop = document.createElement('div');
    backdrop.className = 'tour-backdrop';
    document.body.appendChild(backdrop);
},

// [MAKE SURE your initializeCodeMirror function looks like this]
initializeCodeMirror() {
    this.state.codeMirrorInstance = CodeMirror.fromTextArea(this.elements.codeArea, {
        lineNumbers: true,
        theme: 'dracula',
        mode: 'javascript',
        autoCloseBrackets: true,
        matchBrackets: true,
        matchTags: {bothTags: true},
        tabSize: 2,
        lineWrapping: true,
        readOnly: true,
        
        // CRITICAL: Add these to limit memory usage
        undoDepth: 100, // Only keep last 100 changes in history (default is 200)
        historyEventDelay: 500, // Batch changes to reduce history entries
        cursorScrollMargin: 5, // Reduce viewport tracking
    });

    this.state.codeMirrorInstance.on('change', (instance, changeObj) => {
        if (this.state.codeMirrorInstance.state.focused) {
            // Don't auto-save for very large files
            const content = instance.getValue();
            if (content.length < 100000) { // Only auto-save files under 100KB
                this.scheduleAutoSave();
            }
        }
    });

    // Clear history periodically to prevent memory bloat
    setInterval(() => {
        if (this.state.codeMirrorInstance && this.state.codeMirrorInstance.historySize().undo > 50) {
            this.state.codeMirrorInstance.clearHistory();
        }
    }, 60000); // Clear every minute

    const resizeObserver = new ResizeObserver(() => {
        if (this.state.codeMirrorInstance) {
            this.state.codeMirrorInstance.refresh();
        }
    });
    resizeObserver.observe(this.elements.codePanel);
},

handleOversizedProject(size) {
    const sizeMB = (size / 1024 / 1024).toFixed(2);
    console.error(`🚨 Project too large: ${sizeMB}MB`);
    
    // Show user-friendly options
    if (confirm(`Project size (${sizeMB}MB) exceeds server limits. Clear chat history to reduce size?`)) {
        this.state.chatHistory = [];
        this.updateChatDisplay();
        this.saveProject(); // Retry without chat history
    } else {
        this.showNotification('Project too large to save. Please remove large files or clear chat history.', 'error', 10000);
    }
},

getCodeMirrorMode(path) {
    const extension = path.split('.').pop().toLowerCase();
    switch (extension) {
        case 'js': return 'javascript';
        case 'json': return { name: 'javascript', json: true };
        case 'css': return 'css';
        case 'html': return 'htmlmixed';
        case 'php': return 'php';
        case 'xml': return 'xml';
        case 'md': return 'markdown';
        default: return 'text/plain';
    }
},
// ============================================
// 🧠 REFLECTION SYSTEM - Makes AI Learn From Mistakes
// ============================================

// Function #1: Reflect on a Single Step
// REPLACE your entire reflectOnStep function with this:
// REPLACE your entire reflectOnStep function with this COMPLETE version
// REPLACE your entire reflectOnStep function with this COMPLETE version
// THIS IS THE DEFINITIVE, FINAL, AND COMPLETE VERSION. PLEASE USE THIS.
// THIS IS THE DEFINITIVE, FINAL, AND COMPLETE VERSION. PLEASE USE THIS.
async reflectOnStep(stepIndex, stepText, success, validation, filesBefore, filesAfter) {
    console.log(`🤔 REFLECTION SYSTEM: Starting step ${stepIndex + 1} reflection`);
    
    const newFiles = Object.keys(filesAfter).filter(f => !filesBefore[f]);
    const modifiedFiles = Object.keys(filesAfter).filter(f => 
        filesBefore[f] && filesBefore[f].content !== filesAfter[f].content
    );
    const hasMeaningfulOutput = newFiles.length > 0 || modifiedFiles.length > 0;

    if (success && !validation.hasIssues && !hasMeaningfulOutput) {
        console.log(`⚠️ Step ${stepIndex + 1} had no output - skipping reflection`);
        return {
            stepNumber: stepIndex + 1,
            timestamp: Date.now(),
            stepText: stepText,
            success: success,
            validationIssues: false,
            self_rating: 7,
            what_went_wrong: "No issues detected",
            why_it_happened: "Step completed successfully",
            lesson_learned: "Continue current approach",
            better_file_selection: [],
            is_recurring_mistake: false,
            confidence: 8,
            skipped: true
        };
    }

    let betterFiles = [];
    if (validation.hasIssues) {
        validation.missingReferences.forEach(issue => {
            if (issue.type === 'CSS' && !betterFiles.includes('style.css')) {
                betterFiles.push('style.css');
            }
            if (issue.type === 'JavaScript' && !betterFiles.includes('script.js')) {
                betterFiles.push('script.js');
            }
        });
    }

    const reflectionPrompt = `CRITICAL SELF-REVIEW - Analyze your performance on this coding step.

STEP EXECUTED: "${stepText}"

RESULTS ANALYSIS:
- Success Status: ${success ? 'SUCCESS ✅' : 'HAD PROBLEMS ❌'}
- Validation Issues: ${validation.hasIssues ? 'YES - ' + validation.missingReferences.length + ' missing references found' : 'NO issues'}
- Better files needed: ${betterFiles.length > 0 ? betterFiles.join(', ') : 'None identified'}
- Files in context: ${Array.from(this.state.contextFiles).join(', ') || 'None'}
- New files created: ${newFiles.join(', ') || 'None'}
- Files modified: ${modifiedFiles.join(', ') || 'None'}

CRITICAL SELF-ASSESSMENT QUESTIONS:
1. FILE SELECTION FAILURE: Did I miss critical files in context? What specific files should have been included?
2. ROOT CAUSE: Why did I forget those files? Was it rushing, poor analysis, or token limits?
3. PATTERN RECOGNITION: Is this a recurring mistake I make? Do I consistently forget CSS/JS files?
4. CONCRETE PREVENTION: What specific rule should I follow next time to avoid this exact failure?

RESPOND WITH THIS EXACT JSON FORMAT ONLY:
{
    "self_rating": ${success && !validation.hasIssues ? '7' : '3'},
    "what_went_wrong": "Brief description of the file selection failure",
    "why_it_happened": "Root cause analysis", 
    "lesson_learned": "Specific, actionable rule for file selection",
    "better_file_selection": ${JSON.stringify(betterFiles)},
    "is_recurring_mistake": false,
    "confidence": 8
}`;

    try {
        const response = await this._makeAPICall([
            { role: 'user', content: reflectionPrompt }
        ], null, { isReflection: true });
        
        let reflectionData;
        try {
            const jsonMatch = response.match(/\{[\s\S]*\}/);
            if (jsonMatch) {
                reflectionData = JSON.parse(jsonMatch[0]);
                
                if (!reflectionData.self_rating || !reflectionData.lesson_learned) {
                    console.warn('Invalid reflection data structure, using fallbacks');
                    reflectionData = {
                        self_rating: success ? 7 : 3,
                        what_went_wrong: "File selection mismatch",
                        why_it_happened: "Context analysis incomplete",
                        lesson_learned: "Always include related CSS/JS files",
                        better_file_selection: betterFiles,
                        is_recurring_mistake: false,
                        confidence: 6
                    };
                }
                
                console.log(`🧠 Reflection data parsed:`, reflectionData);
            } else {
                throw new Error('No JSON found in reflection response');
            }
        } catch (e) {
            console.log('⚠️ Reflection response parsing failed:', e);
            reflectionData = {
                self_rating: success ? 7 : 3,
                what_went_wrong: "Failed to parse reflection response",
                why_it_happened: "AI response format issue",
                lesson_learned: "Always validate file references before proceeding",
                better_file_selection: betterFiles,
                is_recurring_mistake: false,
                confidence: 5
            };
        }
        
        const reflection = {
            stepNumber: stepIndex + 1,
            timestamp: Date.now(),
            stepText: stepText,
            success: success,
            validationIssues: validation.hasIssues,
            ...reflectionData
        };
        
        if (!this.state.reflectionMemory) {
            this.state.reflectionMemory = [];
        }
        this.state.reflectionMemory.push(reflection);
        
        console.log(`📚 Reflection saved to memory. Total: ${this.state.reflectionMemory.length}`);
        
        await this.saveReflectionToDatabase(reflection, stepText, filesBefore, filesAfter);
        
        const truncatedStepTextForDisplay = stepText.length > 200 ? 
            stepText.substring(0, 200) + "..." : stepText;

        const reflectionMessage = 
            `🧠 **AI Self-Reflection for Step ${stepIndex + 1}:**\n\n` +
            `**Step:** "${truncatedStepTextForDisplay}"\n` +
            `**Self-Rating:** ${reflectionData.self_rating}/10\n` +
            `**Critical Mistake:** ${reflectionData.what_went_wrong || 'Not specified'}\n` +
            `**Root Cause:** ${reflectionData.why_it_happened || 'Not specified'}\n` +
            `**🎯 NEW RULE:** ${reflectionData.lesson_learned || 'Not specified'}`;
        
        this.renderChatMessage('ai', reflectionMessage, false);
        
        return reflection;
        
    } catch (error) {
        console.error('❌ Reflection failed:', error);
        
        // Check if it's a rate limit error
        const isRateLimitError = error.message && error.message.includes('rate limit');
        
        const emergencyReflection = {
            stepNumber: stepIndex + 1,
            timestamp: Date.now(),
            stepText: stepText,
            success: success,
            validationIssues: validation.hasIssues,
            self_rating: success ? 6 : 2,
            what_went_wrong: isRateLimitError ? "Rate limit reached" : "Reflection system encountered an error",
            why_it_happened: isRateLimitError ? "Too many API calls in short time" : "Network or API failure",
            lesson_learned: isRateLimitError ? "Reduce reflection frequency" : "Proceed with caution and manual validation",
            better_file_selection: betterFiles,
            is_recurring_mistake: false,
            confidence: 3,
            error_occurred: true,
            rate_limited: isRateLimitError
        };
        
        if (!this.state.reflectionMemory) {
            this.state.reflectionMemory = [];
        }
        this.state.reflectionMemory.push(emergencyReflection);
        
        if (isRateLimitError) {
            this.state.rateLimitHit = true;
            this.renderChatMessage('system', 
                `⚠️ Rate limit reached. Continuing without reflection for remaining steps.`, false);
        } else {
            this.renderChatMessage('system', 
                `⚠️ Reflection system encountered an error for step ${stepIndex + 1}. Proceeding with basic validation.`, false);
        }
        
        // Return the emergency reflection instead of throwing
        return emergencyReflection;
    }
},
getFileType(path) {
        const extension = path.split('.').pop().toLowerCase();
        switch (extension) {
            case 'html': return 'html';
            case 'css': return 'css';
            case 'js': return 'javascript';
            case 'json': return 'json';
            case 'md': return 'markdown';
            default: return 'text';
        }
    },
async safeReflectOnStep(stepIndex, stepText, success, validation, filesBefore, filesAfter) {
    try {
        console.log(`🛡️ Starting safe reflection for step ${stepIndex + 1}`);
        
        // Check if we've hit rate limit or exceeded max reflections
        if (this.state.rateLimitHit) {
            console.log(`⏭️ Skipping reflection - rate limit hit previously`);
            return {
                stepNumber: stepIndex + 1,
                timestamp: Date.now(),
                stepText: stepText,
                success: success,
                skipped: true,
                reason: 'Rate limit protection'
            };
        }
        
        if (this.state.reflectionCount >= this.state.maxReflectionsPerPlan) {
            console.log(`⏭️ Skipping reflection - max reflections (${this.state.maxReflectionsPerPlan}) reached`);
            return {
                stepNumber: stepIndex + 1,
                timestamp: Date.now(),
                stepText: stepText,
                success: success,
                skipped: true,
                reason: 'Max reflections reached'
            };
        }
        
        // Add rate limiting delay before reflection
        await this.rateLimitDelay(2000);
        
        const reflection = await this.reflectOnStep(stepIndex, stepText, success, validation, filesBefore, filesAfter);
        
        // Increment counter if reflection actually ran
        if (!reflection.skipped) {
            this.state.reflectionCount++;
        }
        
        // Check if reflection actually succeeded
        if (reflection?.error_occurred) {
            console.log(`⚠️ Reflection completed with fallback data for step ${stepIndex + 1}`);
        } else if (reflection?.skipped) {
            console.log(`⏭️ Reflection skipped for step ${stepIndex + 1}`);
        } else {
            console.log(`✅ Reflection completed successfully for step ${stepIndex + 1}`);
        }
        
        return reflection;
        
    } catch (error) {
        console.error(`❌ Reflection system failed for step ${stepIndex + 1}:`, error);
        console.error('Reflection error details:', {
            stepIndex,
            stepText: stepText?.substring(0, 100),
            errorMessage: error.message
        });
        
        // Create fallback reflection
        const fallbackReflection = {
            stepNumber: stepIndex + 1,
            timestamp: Date.now(),
            stepText: stepText,
            success: success,
            validationIssues: validation?.hasIssues || false,
            self_rating: success ? 7 : 3,
            what_went_wrong: "Reflection system error",
            why_it_happened: `System error: ${error.message}`,
            lesson_learned: "Continue with standard validation",
            better_file_selection: [],
            is_recurring_mistake: false,
            confidence: 0,
            error_occurred: true
        };
        
        if (!this.state.reflectionMemory) {
            this.state.reflectionMemory = [];
        }
        this.state.reflectionMemory.push(fallbackReflection);
        
        // Show user-friendly message
        this.renderChatMessage('system', 
            `⚠️ Advanced analysis skipped for step ${stepIndex + 1}. Continuing with standard validation.`
        );
        
        return fallbackReflection;
    }
},
getConservativeContextForStep(stepText, currentContext) {
    const stepLower = stepText.toLowerCase();
    const conservativeContext = new Set(currentContext);
    
    // Always include CSS for any visual/styling task
    if (stepLower.includes('style') || stepLower.includes('css') || stepLower.includes('design') || 
        stepLower.includes('layout') || stepLower.includes('color') || stepText.includes('BMI')) {
        const cssFiles = this.state.fileOrder.filter(f => f.endsWith('.css'));
        if (cssFiles.length > 0) {
            cssFiles.forEach(file => conservativeContext.add(file));
        }
    }
    
    // Always include JS for any interactive task
    if (stepLower.includes('javascript') || stepLower.includes('js') || stepLower.includes('function') ||
        stepLower.includes('interactive') || stepLower.includes('calculate') || stepText.includes('BMI')) {
        const jsFiles = this.state.fileOrder.filter(f => f.endsWith('.js'));
        if (jsFiles.length > 0) {
            jsFiles.forEach(file => conservativeContext.add(file));
        }
    }
    
    // Always include HTML for any page structure task
    if (stepLower.includes('html') || stepLower.includes('page') || stepLower.includes('structure')) {
        const htmlFiles = this.state.fileOrder.filter(f => f.endsWith('.html'));
        if (htmlFiles.length > 0) {
            htmlFiles.forEach(file => conservativeContext.add(file));
        }
    }
    
    return conservativeContext;
},

async rateLimitDelay(ms = 2000) {
    console.log(`⏳ Rate limit protection: waiting ${ms}ms`);
    return new Promise(resolve => setTimeout(resolve, ms));
},
// 🆕 NEW FUNCTION: Save reflection to database
async saveReflectionToDatabase(reflection, stepText, filesBefore, filesAfter) {
    try {
        const patternData = {
            reflection: reflection,
            stepText: stepText, // CRITICAL: Store full step text for learning
            context: {
                filesInContext: Array.from(this.state.contextFiles),
                projectType: this.state.projectContext?.originalVision || 'unknown',
                fileChanges: {
                    newFiles: Object.keys(filesAfter).filter(f => !filesBefore[f]),
                    modifiedFiles: Object.keys(filesAfter).filter(f => 
                        filesBefore[f] && filesBefore[f].content !== filesAfter[f].content
                    )
                }
            },
            timestamp: new Date().toISOString()
        };
        
        const formData = new FormData();
        formData.append('action', 'zencode_ai_save_learning');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('learning_type', 'step_reflection');
        formData.append('pattern_data', JSON.stringify(patternData));
        formData.append('success_rate', reflection.self_rating / 10);
        formData.append('confidence_score', (reflection.confidence || 5) / 10);
        
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            console.log('✅ Learning saved to database (ID:', data.data.learning_id, ')');
        } else {
            console.warn('⚠️ Failed to save learning:', data.data?.message);
        }
    } catch (error) {
        console.error('❌ Failed to save learning to database:', error);
    }
},

// Function #2: Use Past Learnings When Picking Files
// REPLACE your entire applyReflectionLearnings function with this:
// REPLACE your entire applyReflectionLearnings function with this:
// REPLACE your entire applyReflectionLearnings function with this COMPLETE version
// REPLACE your entire applyReflectionLearnings function with this:
// REPLACE your entire applyReflectionLearnings function with this FIXED version:
async applyReflectionLearnings(stepText, currentContext) {
    console.log(`🎓 REFLECTION: Checking for past learnings for: "${stepText.substring(0, 100)}..."`);
    
    // Show thinking message
    const thinkingMessage = this.renderChatMessage('ai', '🎓 Consulting past experience...', true, true);
    this.elements.chatMessages.appendChild(thinkingMessage);
    this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;

    try {
        // Get relevant learnings with better matching
        const dbLearnings = await this.getRelevantLearningsFromDatabase(stepText, 8);
        
        if (dbLearnings.length === 0) {
            thinkingMessage.querySelector('.message-bubble').innerHTML = '🎓 No relevant past experience found. Using standard approach.';
            return currentContext;
        }

        // Filter for high-impact learnings (low scores or specific failures)
        const impactfulLearnings = dbLearnings.filter(learning => {
            const pattern = learning.pattern_data?.reflection || learning.pattern_data;
            const successRate = learning.success_rate || (pattern.self_rating / 10);
            
            // Include learnings with low success OR specific file-related failures
            const hadFileIssues = pattern.what_went_wrong?.toLowerCase().includes('file') || 
                                 pattern.better_file_selection?.length > 0;
            
            return successRate < 0.7 || hadFileIssues;
        });

        if (impactfulLearnings.length === 0) {
            thinkingMessage.querySelector('.message-bubble').innerHTML = '🎓 Past attempts were successful. Using standard approach.';
            return currentContext;
        }

        thinkingMessage.querySelector('.message-bubble').innerHTML = `🎓 Found ${impactfulLearnings.length} critical learnings. Applying hard-won experience...`;

        // Build a STRONGER prompt that emphasizes not repeating mistakes
        const learningsText = impactfulLearnings.map((learning, index) => {
            const pattern = learning.pattern_data?.reflection || learning.pattern_data;
            const successRate = Math.round((learning.success_rate || (pattern.self_rating / 10)) * 100);
            
            // SAFE TRUNCATION: Limit text length and remove special characters
            const failure = (pattern.what_went_wrong || 'Unknown failure')
                .substring(0, 150)
                .replace(/[`"'{}[\]()]/g, ''); // Remove problematic characters
            
            const lesson = (pattern.lesson_learned || 'No lesson')
                .substring(0, 150)
                .replace(/[`"'{}[\]()]/g, '');
                
            const betterFiles = pattern.better_file_selection || 'Not specified';
            
            return `LEARNING ${index + 1} (${successRate}% success):\n- Failure: ${failure}\n- Lesson: ${lesson}\n- Better Files: ${betterFiles}`;
        }).join('\n\n');

        const enhancedPrompt = `CRITICAL: You MUST learn from past failures and NOT repeat the same mistakes.

**PAST FAILURES AND LESSONS (MUST OBEY):**
${learningsText}

**CURRENT TASK:**
"${stepText}"

**ALL AVAILABLE FILES:**
${this.state.fileOrder.join(', ')}

**CURRENT SELECTION (may be flawed):**
${Array.from(currentContext).join(', ')}

**MANDATORY REQUIREMENTS:**
1. NEVER repeat the specific file selection mistakes listed above
2. ALWAYS include files mentioned in "Better Files" from past lessons
3. If past failures mention missing CSS/JS files, ensure they're included now
4. Be more conservative - include potentially relevant files rather than risk missing critical ones

Respond with ONLY valid JSON in this exact format:
{
    "optimal_files": ["file1.js", "file2.css", "etc..."],
    "reasoning": "Why this selection avoids past mistakes"
}

IMPORTANT: Your response must be ONLY the JSON object, no other text, no code fences, no explanations.`;

       
       const response = await this._makeAPICall([
    { role: 'user', content: enhancedPrompt }
], null);  // Changed null to 'sonnet' to use Haiku

        let improvements;
        try {
            // First, clean the response by removing markdown code fences
            let cleanResponse = response.trim();
            cleanResponse = cleanResponse.replace(/```json\s*/g, '');
            cleanResponse = cleanResponse.replace(/```\s*/g, '');
            
            // Try to parse the cleaned response as JSON
            improvements = JSON.parse(cleanResponse);
        } catch (e1) {
            // If that fails, try to extract JSON from the response using regex
            try {
                const jsonMatch = response.match(/\{[\s\S]*\}/);
                if (jsonMatch) {
                    improvements = JSON.parse(jsonMatch[0]);
                } else {
                    throw new Error('No JSON found in response');
                }
            } catch (e2) {
                console.error('Failed to parse learning response:', e2);
                console.log('Raw response was:', response);
                thinkingMessage.querySelector('.message-bubble').innerHTML = '🎓 Could not parse AI learning response. Using standard context selection.';
                return currentContext;
            }
        }

        // Validate the response structure
        if (!improvements.optimal_files || !Array.isArray(improvements.optimal_files)) {
            console.error('Invalid improvements structure:', improvements);
            thinkingMessage.querySelector('.message-bubble').innerHTML = '🎓 AI response had invalid structure. Using standard context selection.';
            return currentContext;
        }

        // Apply the improved context
        const newContext = new Set(improvements.optimal_files.filter(file => this.state.projectFiles[file]));
        const originalContext = new Set(currentContext);

        const addedFiles = [...newContext].filter(file => !originalContext.has(file));
        const removedFiles = [...originalContext].filter(file => !newContext.has(file));

        // Update learning usage stats
        await this.updateLearningUsageStats(impactfulLearnings, true);

        if (addedFiles.length > 0 || removedFiles.length > 0) {
// --- START OF THE FIX ---
// This new version shows which specific past failure is being corrected.

// Find the most relevant learning that triggered the change.
// We'll prioritize the one with the lowest score (the biggest past failure).
const mostRelevantLearning = impactfulLearnings.sort((a, b) => a.success_rate - b.success_rate)[0];
const pastFailureSummary = mostRelevantLearning.pattern_data?.reflection?.what_went_wrong || "a past mistake";

let changeMessage = `🎓 <strong>Learning Applied!</strong> Based on a past failure related to "<em>${this.escapeHtml(pastFailureSummary)}</em>", I've adjusted the context.`;

if (addedFiles.length > 0) {
    changeMessage += `<br>+ Added: <code>${addedFiles.join(', ')}</code>`;
}
if (removedFiles.length > 0) {
    changeMessage += `<br>- Removed: <code>${removedFiles.join(', ')}</code>`;
}

// We'll hide the AI's detailed reasoning in a collapsible details tag to keep the UI clean.
if (improvements.reasoning) {
    const safeReasoning = improvements.reasoning.replace(/[`"'{}[\]()]/g, '');
    changeMessage += `
        <details style="margin-top: 0.5rem; font-size: 0.9em; opacity: 0.8;">
            <summary style="cursor: pointer;">Show AI's reasoning</summary>
            <p style="margin-top: 0.5rem;">${safeReasoning}</p>
        </details>
    `;
}

thinkingMessage.querySelector('.message-bubble').innerHTML = changeMessage;
            
            console.log(`🎓 Learning applied: +${addedFiles.length}, -${removedFiles.length} files`);
            return newContext;
        } else {
            thinkingMessage.querySelector('.message-bubble').innerHTML = `🎓 AI confirmed current context is optimal based on past experience.`;
        }
        
    } catch (error) {
        console.error('Error applying reflections:', error);
        // SAFE ERROR MESSAGE - don't include raw content
        const errorMsg = error.message.includes('special characters') 
            ? 'Learning response contained complex formatting. Using fallback selection.'
            : `Learning system encountered an issue. Using fallback.`;
        
        thinkingMessage.querySelector('.message-bubble').innerHTML = `🎓 ${errorMsg}`;
    }
    
    return currentContext;
},
// 🆕 NEW FUNCTION: Get relevant learnings from database
async getRelevantLearningsFromDatabase(stepText, limit = 8) {
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_get_learnings');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('step_text', stepText);
        formData.append('limit', limit);
        formData.append('learning_type', 'step_reflection');
        formData.append('min_relevance_score', '0.3'); // NEW: Filter for more relevant learnings
        
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success && data.data.learnings) {
            console.log(`✅ Retrieved ${data.data.learnings.length} learnings from database`);
            
            // Enhanced filtering: prioritize learnings with similar tasks and low scores
            const filteredLearnings = data.data.learnings.filter(learning => {
                const pattern = learning.pattern_data?.reflection || learning.pattern_data;
                
                // Give higher priority to learnings with low success rates
                const successRate = learning.success_rate || (pattern.self_rating / 10);
                const isLowScore = successRate < 0.7;
                
                // Check for task similarity
                const stepSimilarity = this.calculateStepSimilarity(stepText, pattern.stepText || '');
                
                return stepSimilarity > 0.3 || isLowScore;
            });
            
            // Sort by relevance (low scores first, then by similarity)
            filteredLearnings.sort((a, b) => {
                const aScore = a.success_rate || (a.pattern_data?.reflection?.self_rating / 10);
                const bScore = b.success_rate || (b.pattern_data?.reflection?.self_rating / 10);
                
                // Prioritize low scores
                if (aScore !== bScore) return aScore - bScore;
                
                // Then by similarity
                const aSimilarity = this.calculateStepSimilarity(stepText, a.pattern_data?.reflection?.stepText || '');
                const bSimilarity = this.calculateStepSimilarity(stepText, b.pattern_data?.reflection?.stepText || '');
                
                return bSimilarity - aSimilarity;
            });
            
            return filteredLearnings.slice(0, limit);
        }
        return [];
    } catch (error) {
        console.error('❌ Failed to retrieve learnings:', error);
        return [];
    }
},

calculateStepSimilarity(currentStep, pastStep) {
    if (!pastStep) return 0;
    
    const current = currentStep.toLowerCase();
    const past = pastStep.toLowerCase();
    
    // Keyword matching with weights
    const keywords = [
        'bmi', 'calculator', 'convert', 'web', 'browser', 'html', 'css', 'javascript', 
        'style', 'design', 'layout', 'function', 'calculate', 'weight', 'height'
    ];
    
    let score = 0;
    keywords.forEach(keyword => {
        if (current.includes(keyword) && past.includes(keyword)) {
            score += 2; // Both have the keyword
        } else if (current.includes(keyword) || past.includes(keyword)) {
            score += 0.5; // Only one has the keyword
        }
    });
    
    // File type mentions
    const fileTypes = ['css', 'javascript', 'html', 'style'];
    fileTypes.forEach(type => {
        if (current.includes(type) && past.includes(type)) {
            score += 1.5;
        }
    });
    
    // Normalize score to 0-1 range
    return Math.min(score / 10, 1);
}
,

// 🆕 NEW FUNCTION: Update learning usage statistics
async updateLearningUsageStats(learnings, wasSuccessful) {
    const dbLearnings = learnings.filter(l => l.id);
    
    if (dbLearnings.length === 0) return;
    
    console.log(`📊 Updating usage stats for ${dbLearnings.length} learnings`);
    
    // Update each learning's success rate
    for (const learning of dbLearnings) {
        try {
            const formData = new FormData();
            formData.append('action', 'zencode_ai_update_learning_usage');
            formData.append('nonce', zencode_ai_app_vars.nonce);
            formData.append('learning_id', learning.id);
            formData.append('was_successful', wasSuccessful ? 'true' : 'false');
            
            const response = await fetch(zencode_ai_app_vars.ajaxurl, {
                method: 'POST',
                body: formData
            });
            
            const data = await response.json();
            if (data.success) {
                console.log(`✅ Updated learning ${learning.id}: ${Math.round(data.data.new_success_rate * 100)}% success rate`);
            }
        } catch (error) {
            console.error(`❌ Failed to update learning ${learning.id}:`, error);
        }
    }
},

// Function #3: Big Picture Reflection at End of Plan
// REPLACE your entire reflectOnEntirePlan function with this:
// REPLACE your entire reflectOnEntirePlan function with this:
// REPLACE your entire reflectOnEntirePlan function with this COMPLETE version
// In app.js, replace your entire reflectOnEntirePlan function with this:

// In app.js, replace your old function with this corrected version:

async reflectOnEntirePlan() {
    console.log(`🔍 REFLECTION SYSTEM: Starting final plan reflection`);
    
    if (!this.state.reflectionMemory || this.state.reflectionMemory.length === 0) {
        this.renderChatMessage('ai', '📊 **Plan Complete!** No reflection data available for final analysis.');
        return;
    }
    
    // Helper function to sanitize content - removes problematic special characters
    const sanitizeContent = (text) => {
        if (!text) return '';
        return text
            .replace(/[{}[\]<>]/g, '') // Remove brackets and braces
            .replace(/[`'"]/g, '') // Remove quotes
            .replace(/\\/g, '/') // Replace backslashes with forward slashes
            .replace(/\s+/g, ' ') // Collapse multiple spaces
            .trim();
    };
    
    // Build step summaries with sanitization and length limits
    const stepSummaries = this.state.currentPlan.map((step, i) => {
        const reflection = this.state.reflectionMemory.find(r => r.stepNumber === i + 1);
        const stepText = sanitizeContent(step.text.substring(0, 40)); // Reduced from 50 to 40
        return `Step ${i + 1}: ${step.status} | Rating: ${reflection?.self_rating || 'N/A'}/10 | ${stepText}`;
    }).join('\n');
    
    // Build lessons with sanitization and truncation
    const allLessons = this.state.reflectionMemory
        .map(r => {
            const lesson = sanitizeContent(r.lesson_learned || '');
            // Truncate each lesson to 100 chars to prevent overflow
            return lesson.length > 100 ? lesson.substring(0, 100) + '...' : lesson;
        })
        .join('\n- ');
    
    // Build the prompt with sanitized content
    const metaPrompt = `You just finished a complete coding plan. Time for a BIG PICTURE reflection.

ALL STEPS:
${stepSummaries}

ALL LESSONS YOU LEARNED:
- ${allLessons}

FINAL QUESTIONS:
1. What was your BIGGEST mistake overall?
2. What pattern do you see in your mistakes?
3. If you had to do this plan again what is the number 1 thing you would change?
4. Overall grade for yourself out of 10

Respond with JSON:
{
    "overall_grade": 7,
    "biggest_mistake": "short answer",
    "mistake_pattern": "short answer",
    "key_improvement": "short answer",
    "will_remember": "One sentence rule for next time"
}`;

    // Additional safety check on total prompt length
    if (metaPrompt.length > 8000) {
        console.warn(`⚠️ Meta prompt too long (${metaPrompt.length} chars), truncating`);
        const truncatedPrompt = metaPrompt.substring(0, 8000) + "\n\nNote: Content was truncated. Respond with best effort JSON.";
        
        try {
            const response = await this._makeAPICall([{ role: 'user', content: truncatedPrompt }], 'sonnet', { isReflection: true });
            await this.processReflectionResponse(response);
        } catch (error) {
            console.error('❌ Final reflection failed:', error);
            this.renderChatMessage('ai', '📊 **Plan Complete!** Final reflection encountered an error but your work is saved.');
        }
        return;
    }

    try {
        const response = await this._makeAPICall([{ role: 'user', content: metaPrompt }], 'sonnet', { isReflection: true });
        await this.processReflectionResponse(response);
    } catch (error) {
        console.error('❌ Final reflection failed:', error);
        this.renderChatMessage('ai', '📊 **Plan Complete!** Final reflection encountered an error but your work is saved.');
    }
},

async processReflectionResponse(response) {
    const jsonMatch = response.match(/\{[\s\S]*\}/);
    
    if (jsonMatch) {
        const meta = JSON.parse(jsonMatch[0]);
        
        const finalReflectionMessage = 
            `\n` +
            `╔════════════════════════════════════════╗\n` +
            `║  🎓 FINAL REFLECTION - PLAN COMPLETE  ║\n` +
            `╚════════════════════════════════════════╝\n\n` +
            `**Overall Grade:** ${this.escapeHtml(String(meta.overall_grade))}/10\n\n` +
            `**Biggest Mistake:** ${this.escapeHtml(meta.biggest_mistake)}\n\n` +
            `**Pattern I Noticed:** ${this.escapeHtml(meta.mistake_pattern)}\n\n` +
            `**Key Improvement:** ${this.escapeHtml(meta.key_improvement)}\n\n` +
            `**Rule for Next Time:** "${this.escapeHtml(meta.will_remember)}"\n`;
        
        this.renderChatMessage('ai', finalReflectionMessage, false);
        
        if (!this.state.masterLearnings) {
            this.state.masterLearnings = [];
        }
        this.state.masterLearnings.push({
            timestamp: Date.now(),
            planSteps: this.state.currentPlan.length,
            ...meta
        });
        
        await this.saveMasterLearningToDatabase(meta);
        
    } else {
        this.renderChatMessage('ai', '📊 **Plan Complete!** Final reflection analysis completed.');
    }
},

// 🆕 NEW FUNCTION: Save master learning to database
async saveMasterLearningToDatabase(masterLearning) {
    try {
        const patternData = {
            masterLearning: masterLearning,
            planContext: {
                totalSteps: this.state.currentPlan.length,
                projectType: this.state.projectContext?.originalVision || 'unknown',
                allReflections: this.state.reflectionMemory,
                projectFiles: Object.keys(this.state.projectFiles)
            },
            timestamp: new Date().toISOString()
        };
        
        const formData = new FormData();
        formData.append('action', 'zencode_ai_save_learning');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('learning_type', 'master_plan_reflection');
        formData.append('pattern_data', JSON.stringify(patternData));
        formData.append('success_rate', masterLearning.overall_grade / 10);
        formData.append('confidence_score', 0.9); // High confidence for plan-level learnings
        
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            console.log('✅ Master learning saved to database (ID:', data.data.learning_id, ')');
            this.renderChatMessage('ai', '💾 Master learning saved to your permanent knowledge base.', true);
        }
    } catch (error) {
        console.error('❌ Failed to save master learning:', error);
    }
},

// 🆕 NEW FUNCTION: Show learning statistics dashboard
async showLearningStats() {
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_get_learning_stats');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            const stats = data.data;
            
            // --- THIS IS THE FIX ---
            // The PHP backend already multiplies by 100, so we just need to format the number.
            const avgSuccessRate = stats.avg_success_rate !== undefined ? 
                parseFloat(stats.avg_success_rate).toFixed(1) : '0.0';
            const bestSuccessRate = stats.best_success_rate !== undefined ?
                parseFloat(stats.best_success_rate).toFixed(1) : '0.0';
            // --- END OF FIX ---
            
            const statsMessage = `
╔════════════════════════════════════════╗
║     🧠 AI LEARNING STATISTICS          ║
╚════════════════════════════════════════╝

**Knowledge Base Overview:**
- Total Learnings Stored: **${stats.total_learnings || 0}**
- Total Times Applied: **${stats.total_applications || 0}**
- Average Success Rate: **${avgSuccessRate}%**

**Best Performing Pattern:**
- Type: **${stats.best_pattern || 'None yet'}**
- Success Rate: **${bestSuccessRate}%**

**Learning Growth:**
${(stats.total_learnings || 0) > 0 ? '✅ Your AI is actively learning!' : '🌱 Start building projects to train your AI!'}

The more you use ZenCode AI, the smarter it becomes at understanding your coding patterns and preferences!
            `;
            
            this.renderChatMessage('ai', statsMessage, false);
        } else {
            throw new Error(data.data?.message || 'Failed to get learning stats');
        }
    } catch (error) {
        console.error('❌ Failed to get learning stats:', error);
        this.renderChatMessage('ai', '❌ Could not retrieve learning statistics. ' + error.message);
    }
},

// 🆕 NEW FUNCTION: Clean up low-performing learnings
async cleanupLowPerformingLearnings() {
    const confirmed = await this.showConfirmation(
        "This will remove learnings with low success rates (below 30%) that have been applied multiple times. This helps keep your AI knowledge base optimized. Continue?"
    );
    
    if (!confirmed) return;
    
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_cleanup_learnings');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            this.renderChatMessage('ai', `✅ ${data.data.message}`);
        }
    } catch (error) {
        console.error('❌ Failed to cleanup learnings:', error);
        this.renderChatMessage('ai', '❌ Cleanup failed.');
    }
},

// REPLACE your _makeAPICall function with this FINAL WORKING version:
// REPLACE your _makeAPICall function with this
// REPLACE your entire _makeAP-ICall function with this final, corrected version.
// REPLACE your _makeAPICall function with this CLEAN version
// ENSURE your _makeAPICall function includes this content cleaning:
async _makeAPICall(messages, model, options = {}) {
    // ===== RATE LIMITING LOGIC =====
    const now = Date.now();
    const minDelay = 1500; // Minimum 1.5 seconds between API calls
    
    if (this.state.lastAPICallTime) {
        const elapsed = now - this.state.lastAPICallTime;
        if (elapsed < minDelay) {
            const waitTime = minDelay - elapsed;
            console.log(`⏳ Rate limit protection: waiting ${waitTime}ms before API call`);
            await new Promise(resolve => setTimeout(resolve, waitTime));
        }
    }
    
    this.state.lastAPICallTime = Date.now();
    // ===== END RATE LIMITING =====
    
    const formData = new FormData();
    formData.append('action', 'zencode_ai_call_ai_api');
    formData.append('nonce', zencode_ai_app_vars.nonce);
    
    const lastMessage = messages[messages.length - 1];
    
    const isReflection = options.isReflection || false;
    const contentLimit = isReflection ? 8000 : 4000;
    const historyLimit = isReflection ? 4000 : 2000;
    
    let cleanContent = lastMessage.content;
    if (cleanContent.length > contentLimit) {
        console.warn(`⚠️ Truncating content from ${cleanContent.length} to ${contentLimit} characters`);
        cleanContent = cleanContent.substring(0, contentLimit) + "... [truncated for length]";
    }
    
    formData.append('prompt', cleanContent);
    
    let fullModelName;
    if (model === 'sonnet') {
        fullModelName = 'google/gemini-2.5-pro';
    } else {
        fullModelName = model || zencode_ai_app_vars.default_model || 'google/gemini-2.5-pro';
    }
	 console.log('Using model for this API call:', fullModelName, '| Is Reflection:', options.isReflection || false);
    formData.append('model', fullModelName);
    
    const chatHistory = messages
        .filter(m => m.role !== 'system')
        .map(m => ({ 
            role: m.role, 
            content: m.content.length > historyLimit ? m.content.substring(0, historyLimit) + "..." : m.content 
        }));
    formData.append('chat_history', JSON.stringify(chatHistory));
    
    formData.append('project_files', '{}');
    formData.append('project_structure', 'Reflection mode - no project context needed');
    formData.append('is_automated', 'true');
    formData.append('is_plan_mode', 'false');
    formData.append('is_step_execution', 'false');
    formData.append('use_knowledge_base', 'false');
    formData.append('is_one_shot_build', 'false');
    formData.append('project_context', '{}');
    
    try {
        const response = await fetch(zencode_ai_app_vars.ajaxurl, { 
            method: 'POST', 
            body: formData 
        });
        const data = await response.json();
        
        if (data.success) {
            return data.data.ai_response;
        } else {
            // Check if it's a rate limit error
            if (data.data?.message && data.data.message.includes('rate limit')) {
                this.state.rateLimitHit = true;
                console.error('🚫 Rate limit detected - marking for future avoidance');
            }
            throw new Error(data.data?.message || 'Internal API call failed');
        }
    } catch (error) {
        console.error('❌ _makeAPICall network error:', error);
        throw error;
    }
},
// REPLACE your entire getAdaptiveContextForStep function with this (includes the learning integration):


positionTourStep(tourStep, targetElement, position) {
    const targetRect = targetElement.getBoundingClientRect();
    const arrow = tourStep.querySelector('.tour-step-arrow');
    
    // Reset positioning
    tourStep.style.position = 'fixed';
    tourStep.style.zIndex = '1000001';
    
    // Special positioning for specific steps
    const isStep5 = this.currentTourStep === 4;
    const isStep7 = this.currentTourStep === 6; // Knowledge Base
    
    switch (position) {
        case 'right':
            let rightOffset = 20;
            if (isStep7) {
                rightOffset = 40; // Move Knowledge Base tooltip further right
            }
            tourStep.style.left = (targetRect.right + rightOffset) + 'px';
            tourStep.style.top = (targetRect.top - 20) + 'px';
            arrow.className = 'tour-step-arrow arrow-left';
            break;
        case 'left':
            tourStep.style.right = (window.innerWidth - targetRect.left + 20) + 'px';
            tourStep.style.top = (targetRect.top - 20) + 'px';
            arrow.className = 'tour-step-arrow arrow-right';
            break;
        case 'bottom':
            let leftOffset = -50; // default offset
            if (isStep5) {
                leftOffset = -150; // Move tooltip further left so arrow points to right side of button
            }
            tourStep.style.left = (targetRect.left + leftOffset) + 'px';
            tourStep.style.top = (targetRect.bottom + 20) + 'px';
            arrow.className = 'tour-step-arrow arrow-top';
            break;
        default:
            tourStep.style.left = (targetRect.left - 50) + 'px';
            tourStep.style.top = (targetRect.top - 100) + 'px';
            arrow.className = 'tour-step-arrow arrow-bottom';
            break;
    }
    
    // Ensure tour step stays within viewport
    const stepRect = tourStep.getBoundingClientRect();
    if (stepRect.right > window.innerWidth) {
        tourStep.style.left = (window.innerWidth - stepRect.width - 20) + 'px';
    }
    if (stepRect.left < 0) {
        tourStep.style.left = '20px';
    }
    if (stepRect.bottom > window.innerHeight) {
        tourStep.style.top = (window.innerHeight - stepRect.height - 20) + 'px';
    }
},

nextTourStep() {
    this.currentTourStep++;
    this.showTourStep();
},

prevTourStep() {
    this.currentTourStep--;
    this.showTourStep();
},

removeTourStep() {
    const existingStep = document.querySelector('.tour-step');
    if (existingStep) existingStep.remove();
    
    const backdrop = document.querySelector('.tour-backdrop');
    if (backdrop) backdrop.remove();
    
    // Remove highlights
    document.querySelectorAll('.tour-highlight').forEach(el => {
        el.classList.remove('tour-highlight');
    });
},

completeTour() {
    this.removeTourStep();
    localStorage.setItem('zc_tour_completed', 'completed');
    
    // Show completion message
    const completionModal = document.createElement('div');
    completionModal.className = 'tour-completion-modal';
    completionModal.innerHTML = `
        <div class="tour-completion-overlay"></div>
        <div class="tour-completion-content">
            <h2>Tour Complete!</h2>
            <p>You're ready to start building with ZenCode AI. Remember:</p>
            <ul style="text-align:center">
                <li>Use checkboxes to control AI context</li>
                <li>Try different AI modes for different tasks</li>
                <li>The One-Shot Builder is great for quick prototypes</li>
            </ul>
            <button class="tour-btn-primary" onclick="this.parentElement.parentElement.remove()">
                Start Building
            </button>
        </div>
    `;
    
    document.body.appendChild(completionModal);
    
    // Auto-remove after 5 seconds
    setTimeout(() => {
        if (completionModal.parentNode) {
            completionModal.remove();
        }
    }, 5000);
}
,
				
				
injectCSS() {
    // --- Load Font Awesome for UI Icons ---
    if (!document.querySelector('link[href*="font-awesome"]')) {
        const faLink = document.createElement('link');
        faLink.rel = 'stylesheet';
        faLink.href = 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css';
        document.head.appendChild(faLink);
    }
// --- Load CodeMirror CSS ---
if (!document.querySelector('link[href*="codemirror"]')) {
    const cmLink = document.createElement('link');
    cmLink.rel = 'stylesheet';
    cmLink.href = 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/codemirror.min.css';
    document.head.appendChild(cmLink);
    
    const cmThemeLink = document.createElement('link');
    cmThemeLink.rel = 'stylesheet';
    cmThemeLink.href = 'https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/theme/dracula.min.css';
    document.head.appendChild(cmThemeLink);
}

    const css = `
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&family=JetBrains+Mono:wght@400;500;600&display=swap');
        
        /* CSS Variables */
        :root {
            --font-sans: 'Inter', -apple-system, BlinkMacSystemFont, sans-serif;
            --font-mono: 'JetBrains Mono', 'Fira Code', monospace;
            --bg-primary: #0a0c10;
            --bg-secondary: #111318;
            --bg-tertiary: #151517;
            --bg-accent: #1b1b1e;
            --bg-glass: rgba(12,12,14,0.85);
            --border-primary: #26272b;
            --border-secondary: #32343a;
            --border-accent: #8b5cf6;
            --text-primary: #f7fafc;
            --text-secondary: #cccccc;
            --text-tertiary: #999999;
            --accent-primary: #8b5cf6;
            --accent-secondary: #b692ff;
            --accent-gradient: linear-gradient(45deg, #8b5cf6, #b692ff);
            --success: #48bb78;
            --warning: #ed8936;
            --error: #f56565;
            --shadow-md: 0 4px 6px -1px rgba(0,0,0,0.1), 0 2px 4px -1px rgba(0,0,0,0.06);
            --shadow-xl: 0 20px 25px -5px rgba(0,0,0,0.35);
            --shadow-glow: 0 0 20px rgba(139,92,246,0.25);
            --radius-md: 8px;
            --radius-lg: 12px;
            --radius-xl: 16px;
            --transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
		
		
// [REPLACE THIS ENTIRE CSS BLOCK in app.js]

// [REPLACE THIS ENTIRE CSS BLOCK in app.js for the final theme]

#ai_app_root .cm-s-dracula .CodeMirror-line span {
    color: inherit;
}
/* Keywords & Built-ins -> Purple (from your 'Build a Website' button) */
#ai_app_root .cm-s-dracula .cm-keyword,
#ai_app_root .cm-s-dracula .cm-builtin { color: #bd93f9 !important; }

/* Atoms (true, false, null) -> Purple */
#ai_app_root .cm-s-dracula .cm-atom { color: #bd93f9 !important; }

/* Numbers & CSS Properties -> Cyan */
#ai_app_root .cm-s-dracula .cm-number,
#ai_app_root .cm-s-dracula .cm-property { color: #8be9fd !important; }

/* Variable definitions -> Green (from your 'Saved' status) */
#ai_app_root .cm-s-dracula .cm-def { color: #50fa7b !important; }

/* Variables, Brackets -> White/Default Text */
#ai_app_root .cm-s-dracula .cm-variable,
#ai_app_root .cm-s-dracula .cm-bracket { color: #f8f8f2 !important; }

/* Operators & HTML Tags -> Pink */
#ai_app_root .cm-s-dracula .cm-operator,
#ai_app_root .cm-s-dracula .cm-tag { color: #ff79c6 !important; }

/* Strings -> Yellow */
#ai_app_root .cm-s-dracula .cm-string,
#ai_app_root .cm-s-dracula .cm-string-2 { color: #f1fa8c !important; }

#ai_app_root .cm-s-dracula .cm-meta { color: #f8f8f2 !important; }

/* Errors -> Red Wavy Underline */
#ai_app_root .cm-s-dracula .cm-error { 
    background: transparent !important;
    text-decoration: underline wavy #ff5555 !important; /* A common Dracula red */
    text-decoration-skip-ink: none;
}

#ai_app_root .cm-s-dracula .cm-qualifier { color: #50fa7b !important; } /* Green */

/* HTML Attributes -> Green */
#ai_app_root .cm-s-dracula .cm-attribute { color: #50fa7b !important; }

#ai_app_root .cm-s-dracula .cm-hr { color: #999; }

/* Links -> Purple */
#ai_app_root .cm-s-dracula .cm-link { color: #bd93f9 !important; }

/* Comments -> Muted Purple/Grey */
#ai_app_root .cm-s-dracula .cm-comment { color: #6272a4 !important; font-style: italic; }
/* --- END: CODEMIRROR CUSTOM THEME FIX --- */

        /* Reset */
        *, *::before, *::after { 
            box-sizing: border-box; 
            margin: 0; 
            padding: 0; 
        }

        /* Layout Structure */
        body #ai_app_root {
            height: 100vh !important; 
            min-height: 100vh !important;
            background: var(--bg-primary) !important; 
            color: var(--text-primary) !important;
            font-family: var(--font-sans) !important; 
            overflow: auto !important; 
            display: grid !important; 
            grid-template-rows: auto 1fr !important; 
        }

        .zencode-main-wrapper {
            display: grid;
            grid-template-rows: auto 1fr;
            height: 100%;
            min-height: 0;
        }

        body .main-layout { 
            display: grid !important; 
            grid-template-columns: auto 1fr !important; 
            min-height: 0 !important; 
            height: 100% !important; 
        }

        body .content-area { 
            display: grid !important; 
            grid-template-rows: auto 1fr !important; 
            height: 100% !important; 
            min-height: 0 !important; 
            overflow: hidden !important;
            position: relative !important; /* Added for absolute positioning context */
        }

        /* Header */
        body .header { 
            display: flex !important; 
            justify-content: space-between !important; 
            align-items: center !important; 
            padding: 0.75rem 1.5rem !important; 
            background: var(--bg-glass) !important; 
            border-bottom: 1px solid var(--border-primary) !important; 
            backdrop-filter: blur(20px) !important; 
            flex-wrap: nowrap !important; 
            gap: 1rem !important;
            min-height: 50px !important; 
        }

        .header-left, .header-right { 
            display: flex; 
            align-items: center; 
            gap: 1rem; 
            flex-shrink: 0; 
        }

        .header-title { 
            font-weight: 500; 
            color: var(--text-secondary); 
            text-align: center; 
            font-size: 1rem; 
            white-space: nowrap; 
            overflow: hidden; 
            text-overflow: ellipsis; 
            flex: 1 1 0; 
            min-width: 0; 
            max-width: 350px; 
        }

        .header > h1 {
            margin-top: -14px !important;
        }

        /* Feature Bar */
        body .feature-bar { 
            display: flex !important; 
            flex-direction: column !important; 
            gap: 0.75rem !important; 
            padding: 0.75rem 1.25rem !important; 
            background: var(--bg-secondary) !important; 
            border-bottom: 1px solid rgba(38, 39, 43, 0.3) !important; 
            grid-row: 1 !important; 
        }

        .feature-bar-main { 
            display: flex; 
            justify-content: center; 
            align-items: center; 
            gap: 20px; 
            width: 100%; 
            flex-wrap: wrap; 
        }

        .feature-bar-text { 
            font-size: 0.9rem; 
            color: var(--text-secondary); 
            text-align: center; 
            flex-shrink: 1; 
            margin-top: 25px !important; 
        }

        /* Panels Layout */
        body .panels-container { 
            display: flex !important; 
            flex-direction: column !important; 
            height: 100% !important; 
            min-height: 0 !important; 
            grid-row: 2 !important; 
            overflow: hidden !important; 
        }

        body .code-editor-panel { 
            flex-basis: 30% !important; 
            flex-grow: 1 !important; 
            flex-shrink: 1 !important; 
            display: flex !important; 
            flex-direction: column !important; 
            background: var(--bg-primary) !important; 
            min-height: 0 !important; 
            height: auto !important; 
        }

        body .chat-panel {
            flex-basis: 70% !important;
            flex-grow: 1 !important;
            flex-shrink: 1 !important;
			
            display: flex !important;
            flex-direction: column !important;
            background: var(--bg-secondary) !important;
            border-left: none !important;
            border-top: 1px solid rgba(38, 39, 43, 0.3) !important;
            min-height: 0 !important;
            height: auto !important;
            position: relative !important; 
        }

       .resizer { 
    height: 8px; 
    width: 100%; 
    cursor: row-resize; 
    background: var(--border-primary); 
    flex-shrink: 0; 
    padding: 5px 0; 
    background-clip: content-box; 
    display: block !important;
    position: relative;
    transition: background-color 0.2s ease;
}

.resizer::before {
    content: '';
    position: absolute;
    left: 50%;
    top: 50%;
    transform: translate(-50%, -50%);
    width: 30px;
    height: 4px;
    background: var(--text-tertiary);
    border-radius: 2px;
    transition: all 0.2s ease;
}

.resizer:hover {
    background: var(--border-secondary);
}

.resizer:hover::before {
    background: var(--accent-primary);
    width: 50px;
}

.resizer:active::before {
    background: var(--accent-secondary);
}

/* Desktop Layout */
@media (min-width: 1200px) {
    .resizer {
        width: 8px !important;
        height: 100% !important;
        cursor: col-resize !important;
        padding: 0 5px;
        background-clip: content-box;
    }
    
    .resizer::before {
        width: 4px;
        height: 30px;
        border-radius: 2px;
    }
    
    .resizer:hover::before {
        height: 50px;
        width: 4px;
    }
}

        .panel-header { 
            padding: 0.75rem 1.25rem; 
            border-bottom: 1px solid rgba(38, 39, 43, 0.2); 
            font-weight: 500; 
            color: var(--text-secondary); 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            gap: 1rem; 
            flex-wrap: wrap; 
        }

        .chat-panel-header-left { 
            display: flex; 
            align-items: center; 
            gap: 1rem; 
        }

        .chat-panel-header-right { 
            display: flex; 
            align-items: center; 
            gap: 0.5rem; 
        }

        #editor-filename { 
            flex-grow: 1; 
            min-width: 0; 
            word-break: break-all; 
        }

        .editor-actions { 
            display: flex; 
            gap: 0.5rem; 
            flex-shrink: 0; 
        }

        .editor-content { 
            flex-grow: 1; 
            position: relative; 
            flex: 1; 
            min-height: 0; 
        }

        /* Code Area */
        .code-area, .code-preview-area { 
            position: absolute; 
            inset: 0; 
            background: transparent; 
            border: none; 
            color: var(--text-primary); 
            font-family: var(--font-mono); 
            font-size: 14px; 
            resize: none; 
            padding: 1.5rem; 
            outline: none; 
            white-space: pre-wrap; 
            word-wrap: break-word; 
            overflow: auto; 
        }

        .code-area:focus {
            background: transparent !important;
            border: none !important;
            outline: none !important;
            box-shadow: none !important;
        }

        .code-preview-area .line-highlight { 
            background-color: rgba(72,187,120,0.15); 
            display: block; 
            margin: 0 -1.5rem; 
            padding: 0 1.5rem; 
        }

        /* Chat Messages */
        .chat-messages { 
            flex-grow: 1; 
            overflow-y: auto; 
            padding: 1.5rem; 
            flex: 1; 
            min-height: 0;
            margin-bottom: 80px !important;
			scroll-snap-type: none; /* Disable snapping if present */
			 scroll-behavior: smooth;
  -webkit-overflow-scrolling: touch; /* Better mobile scrolling */
        }

        .chat-messages::after {
            display: none !important;
        }

        .chat-message { 
            display: flex; 
            gap: 1rem; 
            margin-bottom: 1.5rem; 
            max-width: 100%; 
            position: relative; 
        }

        .chat-message.user { 
            flex-direction: row-reverse; 
        }

        .chat-avatar { 
            width: 36px; 
            height: 36px; 
            border-radius: 50%; 
            display: grid; 
            place-items: center; 
            font-weight: 600; 
            flex-shrink: 0; 
        }

        .chat-message.ai .chat-avatar { 
            background: var(--accent-gradient); 
            color: white; 
        }

        .chat-message.user .chat-avatar { 
            background: var(--accent-primary); 
            color: white !important; 
        }

        .message-bubble { 
            max-width: 85% !important; 
            line-height: 1.6; 
            font-size: 0.95rem; 
            padding: 1rem 1.25rem; 
            border-radius: var(--radius-lg); 
            position: relative; 
            box-sizing: border-box !important; 
            word-wrap: break-word !important; 
            overflow-wrap: break-word !important; 
            word-break: break-word !important; 
            hyphens: auto !important; 
            min-width: 0 !important; 
            flex-shrink: 1 !important; 
        }

        .message-bubble pre, .message-bubble code { 
            white-space: pre-wrap !important; 
            overflow-wrap: break-word !important; 
            max-width: 100% !important; 
        }

        .message-bubble img, .message-bubble video, .message-bubble iframe { 
            max-width: 100% !important; 
            height: auto !important; 
        }

        .message-bubble * { 
            max-width: 100% !important; 
            box-sizing: border-box !important; 
        }

        .message-bubble strong { 
            font-weight: 600; 
            color: var(--text-primary); 
        }

        .message-bubble code { 
            background: rgba(0,0,0,0.2); 
            padding: 0.15rem 0.3rem; 
            border-radius: 4px; 
            font-size: 0.9em; 
        }

        .message-bubble pre { 
            background: var(--bg-primary); 
            padding: 1rem; 
            border-radius: var(--radius-md); 
            margin: 0.5rem 0; 
            font-size: 0.9em; 
            white-space: pre-wrap; 
            word-wrap: break-word; 
            overflow-x: auto; 
            word-break: break-all; 
        }

        .chat-message.user .message-bubble { 
            background: var(--accent-primary); 
            color: white; 
            border-bottom-right-radius: 4px; 
        }

        .chat-message.ai .message-bubble { 
            background: linear-gradient(135deg, #2a2a2d 0%, #3a3a3e 100%); 
            color: var(--text-primary); 
            border-bottom-left-radius: 4px; 
        }

        .chat-message.assistant .message-bubble {
            background-color: #222222 !important;
            border-bottom-left-radius: 3px !important;
        }

        /* Chat Input Container */
        body .chat-input-container {
            position: absolute !important;
            bottom: 0 !important;
            left: 0 !important;
            right: 0 !important;
            background: var(--bg-secondary) !important;
            border-top: 1px solid var(--border-primary) !important;
            box-shadow: 0 -2px 8px rgba(0,0,0,0.25) !important;
            z-index: 10 !important;
            padding: 1rem !important;
            min-height: 80px !important;
            display: flex !important; 
            align-items: center !important; 
            gap: 1rem !important;
        }

        .chat-input { 
            flex-grow: 1; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            border-radius: var(--radius-lg); 
            color: var(--text-primary); 
            padding: 0.875rem; 
            resize: none; 
            outline: none; 
            min-height: 48px; 
            max-height: 150px;
            transition: all 0.2s ease-in-out !important;
        }

        .chat-input:focus::placeholder {
            color: var(--text-secondary) !important;
        }

        /* Buttons */
        .voice-button { 
            height: 48px; 
            width: 38px !important; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            border-radius: 29px; 
            display: grid; 
            place-items: center; 
            cursor: pointer; 
            flex-shrink: 0; 
            transition: all 0.3s ease; 
            padding: 0; 
        }

        .voice-button:hover { 
            background: var(--bg-accent); 
        }

        .voice-button.active { 
            background: var(--error); 
            animation: pulse-voice 1.5s infinite; 
        }

        body .send-button {
            height: 48px;
            width: auto;
            min-width: 48px;
            padding: 0 1rem;
            background-color: #181f24 !important;
            border: 1px solid var(--border-primary) !important;
            border-radius: var(--radius-lg);
            display: grid;
            place-items: center;
            cursor: pointer;
            flex-shrink: 0;
            transition: all 0.2s ease-in-out !important;
            transform: none !important;
            box-shadow: none !important;
            filter: none !important;
            opacity: 1 !important;
        }

        .send-button:hover {
            background: var(--accent-gradient) !important;
            border-color: transparent !important;
            transform: translateY(-2px) !important;
            box-shadow: var(--shadow-glow) !important;
        }

        .send-button:active {
            background: var(--accent-gradient) !important;
            filter: brightness(0.9) !important;
            transform: translateY(1px) !important;
            box-shadow: inset 0 2px 4px rgba(0,0,0,0.3) !important;
        }

        .send-button.stop-button-active {
            background: #2d3748 !important;
            border: 1px solid var(--border-secondary) !important;
            box-shadow: none !important;
            transform: none !important;
            filter: none !important;
            width: auto !important;
            padding: 0 1rem !important;
        }

        .send-button.stop-button-active:hover {
            background: #4a5568 !important;
            border-color: var(--border-secondary) !important;
        }

        .send-button-content, .stop-button-content {
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 0.5rem;
            width: 100%;
            height: 100%;
			color:#fff !important
        }

        .send-button-content {
            gap: 0.6rem;
            font-weight: 500;
            font-size: 0.9rem;
        }

        .shortcut-keys {
            display: flex;
            gap: 0.25rem;
            opacity: 0.8;
        }

        .send-button kbd {
            background-color: rgba(0, 0, 0, 0.2);
            border: 1px solid rgba(0, 0, 0, 0.3);
            border-radius: 4px;
            padding: 6px 6px 2px 6px !important;
            font-size: 15px;
            font-family: var(--font-sans);
            line-height: 1;
        }

        .icon-button { 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            color: var(--text-secondary); 
            width: 32px; 
            height: 32px; 
            padding: 0; 
            border-radius: var(--radius-md); 
            cursor: pointer; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            gap: 0.5rem; 
            font-weight: 500; 
        }

        .build-website-button { 
            background: var(--accent-gradient); 
            color: white; 
            padding: 0.5rem 1rem; 
            font-size: 0.9rem; 
            height: auto; 
            border: none; 
            border-radius: var(--radius-md); 
            cursor: pointer; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            gap: 0.5rem; 
            font-weight: 500; 
            flex-shrink: 0; 
        }

        .icon-button:hover, .sidebar-toggle-header:hover, .voice-button:hover, #image-upload-area:hover { 
            background: var(--bg-accent); 
            color: var(--text-primary); 
            border-color: var(--border-secondary); 
        }

        /* Sidebar */
        .sidebar { 
            background: var(--bg-secondary); 
            border-right: 1px solid var(--border-primary); 
            width: 300px; 
            flex-shrink: 0; 
            transition: margin-left 0.3s ease, transform 0.3s ease; 
            position: relative; 
            display: flex; 
            flex-direction: column; 
            height: 100%; 
            overflow: hidden; 
        }

        .sidebar.hidden { 
            margin-left: -300px; 
        }

        .sidebar-toggle-header { 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            color: var(--text-secondary); 
            width: 38px; 
            height: 38px; 
            padding: 0; 
            justify-content: center; 
            align-items: center; 
            display: flex; 
            border-radius: var(--radius-md); 
            cursor: pointer; 
        }

        .project-header { 
            padding: 1rem 1rem 0.75rem 1rem; 
            border-bottom: 1px solid rgba(38, 39, 43, 0.2); 
        }

        .project-title-bar { 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            margin-bottom: 0.75rem; 
        }

        .project-title-bar span { 
            font-weight: 500; 
            color: var(--text-secondary); 
        }

        .project-actions { 
            display: flex; 
            gap: 0.5rem; 
        }

        .project-actions .icon-button { 
            background: transparent; 
            border: 1px solid var(--border-primary); 
            color: var(--text-secondary); 
            width: 32px; 
            height: 32px; 
            padding: 0; 
            justify-content: center; 
        }

        .project-actions .icon-button:hover { 
            background: var(--bg-accent); 
            color: var(--text-primary); 
            border-color: var(--border-secondary); 
        }

        .project-name-input { 
            width: 100%; 
            padding: 0.5rem; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            color: white; 
            border-radius: var(--radius-md); 
        }

        .file-tree { 
            padding: 0.75rem; 
            overflow-y: auto; 
            flex: 1; 
        }

        .file-name { 
            display: flex; 
            align-items: center; 
            gap: 0.5rem; 
        }

        div.file-name { 
            padding-right: 20px !important; 
        }

        .context-toggle { 
            margin-right: 5px; 
            cursor: pointer; 
            transform: scale(0.9); 
        }

        .folder-item, .file-tree-item { 
            display: flex; 
            align-items: center; 
            padding: 0.5rem 0.75rem; 
            border-radius: var(--radius-md); 
            cursor: pointer; 
            transition: background 0.2s ease; 
            user-select: none; 
            position: relative; 
        }

        div.file-tree-item { 
            margin-bottom: 7px !important; 
        }

        .folder-item:hover, .file-tree-item:hover { 
            background: var(--bg-accent); 
        }

        .folder-item.drag-over { 
            background: var(--accent-primary); 
            box-shadow: inset 0 0 0 2px var(--border-accent); 
        }

        .sidebar.drag-over { 
            box-shadow: inset 0 0 0 2px var(--accent-primary); 
        }

        .file-tree-item.active { 
            background: linear-gradient(135deg, rgba(255,255,255,0.05) 0%, rgba(255,255,255,0.03) 100%); 
            border-left: 2px solid var(--accent-primary); 
        }

        .folder-name, .file-name { 
            display: flex; 
            align-items: center; 
            gap: 0.5rem; 
            flex-grow: 1; 
            white-space: normal; 
            word-break: break-all; 
        }

        .folder-item { 
            color: var(--text-secondary); 
            font-weight: 500; 
        }

        .folder-item.open > .folder-name > .folder-arrow { 
            transform: rotate(90deg); 
        }

        .folder-arrow { 
            transition: transform 0.2s; 
        }

        .file-tree-item { 
            padding-left: 2rem; 
        }

        .vectorize-btn, .delete-btn, .preview-btn { 
            visibility: hidden; 
            opacity: 0.7; 
            cursor: pointer; 
            transition: opacity 0.2s ease, color 0.2s ease; 
            padding: 2px; 
            flex-shrink: 0; 
        }

        .file-tree-item:hover .delete-btn, .folder-item:hover .delete-btn, .file-tree-item:hover .preview-btn, .file-tree-item:hover .vectorize-btn { 
            visibility: visible; 
        }

        .preview-btn { 
            margin-right: 8px; 
        }

        .preview-btn:hover { 
            opacity: 1; 
            color: var(--accent-primary); 
        }

        .vectorize-btn { 
            color: #FF1493 !important; 
            margin-left: auto; 
            margin-right: 5px; 
        }

        .delete-btn { 
            margin-left: 5px; 
        }

        .vectorize-btn:hover { 
            opacity: 1; 
            filter: drop-shadow(0 0 3px #FF1493); 
        }

        .delete-btn:hover { 
            opacity: 1; 
            color: var(--error); 
        }

        .file-actions-container { 
            margin-left: auto; 
            display: flex; 
            align-items: center; 
        }

          /* Status & Navigation */
        .zen-app-subnav {
            background-color: #14161a !important;
            border-bottom: 1px solid var(--border-primary);
            padding: 0px 0px 0px 0px !important;
            flex-shrink: 0;
			margin-top:30px !important;
			height:1px !important
        }

        .zen-app-subnav ul {
            list-style: none;
            margin: 0;
            padding: 0;
            display: flex;
            align-items: center;
            width: 100%;
            position: relative;
            height: 24px;
        }

        .zen-app-subnav li { 
            font-size: 14px; 
            font-weight: 500; 
        }

        .zen-app-subnav a {
            color: #8c8a8a;
            text-decoration: none;
            transition: color 0.2s ease;
            display: flex;
            align-items: center;
            gap: 0.4rem;
        }

        .zen-app-subnav a:hover { 
            color: var(--text-primary); 
        }

        .zen-app-subnav .nav-left { 
            position: absolute; 
            left: 0; 
        }

        .zen-app-subnav .nav-center { 
            position: absolute; 
            left: 50%; 
            transform: translateX(-50%); 
        }

        .zen-app-subnav .nav-right { 
            position: absolute; 
            right: 0; 
        }

        .zen-app-subnav li.active a { 
            color: var(--text-primary); 
            font-weight: 600; 
        }

        .status-indicator { 
            width: 10px; 
            height: 10px; 
            border-radius: 50%; 
            background-color: var(--text-tertiary); 
            transition: background-color 0.3s ease; 
        }

        .status-indicator.online { 
            background-color: var(--success); 
            box-shadow: 0 0 10px rgba(72,187,120,0.5); 
        }

        .status-indicator.pending { 
            background-color: var(--success); 
            animation: pulse 2s ease-in-out infinite; 
        }

        /* AI Mode Toggle */
        .mode-toggle-container { 
            display: flex; 
            align-items: center; 
            gap: 0.5rem; 
            color: var(--text-secondary); 
        }

        .mode-toggle-container label { 
            font-size: 0.85rem; 
            cursor: default; 
            color: var(--warning); 
        }
  input.disabled, input:disabled, select.disabled, select:disabled, textarea.disabled, textarea:disabled
{background:#000 !important}
        body .panel-tab.active {
            color: var(--accent-primary) !important;
            background: var(--bg-secondary) !important;
            border-bottom: 2px solid var(--accent-primary) !important;
        }
		
        .mode-toggle-button { 
            padding: 0.4rem 0.8rem; 
            border: 1px solid var(--border-primary); 
            border-radius: var(--radius-md); 
            cursor: pointer; 
            font-weight: 600; 
            font-size: 13px; 
            width: 80px; 
            text-align: center; 
            transition: var(--transition); 
        }

        .mode-toggle-button.mode-normal { 
            background-color: var(--bg-accent); 
            color: var(--text-primary); 
        }

        .mode-toggle-button.mode-plan { 
            background-color: var(--warning); 
            color: var(--bg-primary); 
            border-color: var(--warning); 
        }

        .mode-toggle-button.mode-vector { 
            background: var(--accent-gradient); 
            color: white; 
            border: none; 
            padding: calc(0.4rem + 1px) calc(0.8rem + 1px); 
        }

        /* Model Selector */
        #model-selector-container { 
            width: 100%; 
        }

        #model-selector { 
            width: 100% !important; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            color: var(--text-primary); 
            border-radius: var(--radius-md); 
            padding: 0.5rem; 
            font-size: 0.9rem;
            margin-bottom: 10px !important;
            max-width: 100% !important;
        }

        /* Save Status */
        .save-status { 
            font-size: 0.8rem; 
            color: var(--text-tertiary); 
            transition: all 0.3s ease; 
            margin-right: auto; 
            padding-left: 1rem; 
        }

        .save-status.saving { 
            color: var(--warning); 
        }

        .save-status.saved { 
            color: var(--success); 
        }

        /* Modals */
        .modal-overlay, .confirm-modal-overlay { 
            position: fixed; 
            inset: 0; 
            background: rgba(10,14,26,0.8); 
            z-index: 1000000; 
            padding: 1rem; 
            transition: opacity 0.3s ease; 
            opacity: 0; 
            pointer-events: none; 
            display: flex; 
            overflow-y: auto; 
        }

        .modal-overlay.visible, .confirm-modal-overlay.visible { 
            opacity: 1; 
            pointer-events: auto; 
        }
		
		/* 2. ADD THIS NEW, SPECIFIC RULE: This is the key to the fix. */
.confirm-modal-overlay.visible {
    z-index: 1000001; /* This applies a higher z-index ONLY to the visible confirmation modal */
}

        .modal-content, .confirm-modal-content { 
            margin: auto; 
            background: var(--bg-secondary); 
            padding: 2rem; 
            border-radius: var(--radius-lg); 
            border: 1px solid var(--border-primary); 
            width: 100%; 
            text-align: center; 
            box-shadow: var(--shadow-xl); 
            transform: scale(0.95); 
            transition: transform 0.3s ease; 
        }

        .modal-overlay.visible .modal-content, .confirm-modal-overlay.visible .confirm-modal-content { 
            transform: scale(1); 
        }

        .modal-content { 
            max-width: 600px; 
        }

        .confirm-modal-content { 
            max-width: 400px; 
        }

        .modal-content h2, .confirm-modal-content h2 { 
            font-size: 1.5rem; 
            margin-bottom: 0.5rem; 
            background: var(--accent-gradient); 
            -webkit-background-clip: text; 
            -webkit-text-fill-color: transparent; 
            background-clip: text; 
        }

        .modal-content h3, .confirm-modal-content h3 { 
            font-size: 1.5rem; 
            margin-bottom: 0.5rem;
            background: var(--accent-gradient);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .modal-content p, .confirm-modal-content p { 
            color: var(--text-secondary); 
            margin-bottom: 1.5rem; 
            line-height: 1.6; 
        }

        .modal-buttons, .confirm-modal-buttons { 
            display: flex; 
            gap: 1rem; 
            justify-content: center; 
        }

        .modal-button, .modal-button-secondary { 
            padding: 0.75rem 1.5rem; 
            border-radius: var(--radius-md); 
            font-weight: 600; 
            cursor: pointer; 
            border: none; 
            font-size: 1rem; 
            text-decoration: none; 
        }

        .modal-button { 
            background: var(--accent-gradient); 
            color: white; 
            display: inline-flex; 
            align-items: center; 
            justify-content: center; 
        }

        .modal-button-secondary { 
            background: var(--bg-accent); 
            color: var(--text-primary); 
        }
		
		@media (min-width: 0px) and (max-width: 602px) {
		
		body .zen-app-subnav {margin-top:0px !important;}
		
		
		}
		
		
			@media (min-width: 603px) and (max-width: 778px) {
		
		body .zen-app-subnav {margin-top:45px !important;}
		
		
		
		
		}
		
		@media (min-width: 0px) and (max-width: 1100px) {
		.feature-bar-text{padding-bottom:15px !important}
		
		}
#wpbody-content {padding-bottom:0px !important}
#wpfooter{display:none !important}
        /* One-shot Builder */
        #one-shot-input { 
            width: 100%; 
            height: 120px; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            color: var(--text-primary); 
            border-radius: var(--radius-md); 
            padding: 1rem; 
            resize: vertical; 
            font-family: var(--font-sans); 
            font-size: 1rem; 
        }

        #one-shot-input:focus { 
            border-color: var(--border-accent); 
            outline: none; 
        }

        .textarea-wrapper { 
            position: relative; 
            width: 100%; 
            margin-bottom: 1.5rem; 
        }

        #one-shot-input { 
            padding-right: 50px; 
            margin-bottom: 0; 
        }

        #btn-refine-prompt { 
            position: absolute; 
            top: 12px; 
            right: 12px; 
            width: 36px; 
            height: 36px; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-primary); 
            color: var(--text-secondary); 
            border-radius: var(--radius-md); 
            z-index: 10; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
        }

        #btn-refine-prompt:hover { 
            color: var(--accent-primary); 
            border-color: var(--accent-primary); 
        }

        #btn-refine-prompt.loading .refine-icon { 
            display: none; 
        }

        #btn-refine-prompt.loading .loading-spinner { 
            display: block !important; 
        }

        #image-upload-area { 
            margin-bottom: 1.5rem; 
            border: 2px dashed var(--border-primary); 
            border-radius: var(--radius-md); 
            padding: 1rem; 
            text-align: center; 
            cursor: pointer; 
            transition: var(--transition); 
        }

        #image-upload-area:hover { 
            border-color: var(--border-accent); 
            background: var(--bg-tertiary); 
        }

        #image-upload-area.has-image { 
            padding: 0; 
            border-style: solid; 
        }

        #image-preview-container { 
            position: relative; 
        }

        #image-preview { 
            max-width: 100%; 
            max-height: 200px; 
            border-radius: var(--radius-md); 
            display: block; 
        }

        #remove-image-btn { 
            position: absolute; 
            top: 10px; 
            right: 10px; 
            background: rgba(10, 14, 26, 0.8); 
            color: white; 
            border: none; 
            border-radius: 50%; 
            width: 28px; 
            height: 28px; 
            cursor: pointer; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            font-weight: bold; 
            line-height: 1; 
            font-size: 18px; 
        }

        #one-shot-loading-overlay { 
            position: fixed; 
            inset: 0; 
            z-index: 99999; 
            background: rgba(10,14,26,0.9); 
            backdrop-filter: blur(5px); 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            flex-direction: column; 
        }

        .loading-content { 
            text-align: center; 
            color: var(--text-primary); 
        }

        #loading-text-anim { 
            font-size: 1.25rem; 
            font-weight: 500; 
            margin-left: 15px; 
            margin-right: 15px; 
            margin-top: 1.5rem; 
            letter-spacing: 0.5px; 
            transition: opacity 0.3s ease; 
        }

        .magic-spinner path { 
            animation-timing-function: linear !important; 
        }

        #cancel-generation-btn { 
            margin-top: 2rem; 
            background: var(--bg-tertiary); 
            border: 1px solid var(--border-secondary); 
            padding: 0.75rem 1.5rem; 
        }

        #cancel-generation-btn:hover { 
            background: var(--bg-accent); 
            color: var(--text-primary); 
        }

        /* Image Modal */
        #image-preview-modal { 
            backdrop-filter: blur(8px); 
            -webkit-backdrop-filter: blur(8px); 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            padding: 30px; 
			
        }

        .image-modal-content { 
            position: relative; 
            max-width: 20vw; 
            max-height: 85vh; 
            line-height: 0; 
            box-shadow: var(--shadow-xl); 
            border-radius: var(--radius-lg); 
			
		}
		body .vision-badge img{ margin-bottom:0px !important}
		
		#image-modal-controls{bottom:60px !important}
		
		
		.image-modal-content img { margin-bottom:0px !important}
		
		
		
		@media (max-width: 1600px) and (min-width: 1128px) {.image-modal-content { max-width: 43vw;  } .image-modal-content img { margin-bottom:0px !important}}
		
		
		
		
		@media (max-width: 1127px) and (min-width: 611px) {
  .image-modal-content { 
    max-width: 55vw; 
    top: 30px;

	
	
  }
  
  .image-modal-content img { margin-bottom:0px !important}
}
		
		
		@media (max-width: 610px) {
          .image-modal-content { max-width: 100vw;  }
		  
		  #image-modal-controls{bottom:-250px !important}
		
		}
		
		
		.image-modal-content img { 
            width: 100%; 
            height: 100%; 
            max-height:100vh; 
			
            object-fit: contain; 
            border-radius: var(--radius-lg); 
        }

        .close-image-modal-btn { 
            position: absolute; 
            top: -18px; 
            right: -18px; 
            z-index: 10; 
            color: white; 
            background: rgba(10,14,26,0.8); 
            width: 36px; 
            height: 36px; 
            border-radius: 50%; 
            display: flex; 
            align-items: center; 
            justify-content: center; 
            font-size: 28px; 
            font-weight: bold; 
            line-height: 1; 
            cursor: pointer; 
            transition: transform 0.2s ease, background 0.2s ease; 
            border: 2px solid white; 
        }

        .close-image-modal-btn:hover { 
            transform: scale(1.1); 
            background: rgba(0,0,0,1); 
        }

        #image-modal-controls { 
            position: absolute; 
            bottom: 20px; 
            left: 50%; 
            transform: translateX(-50%); 
            background: rgba(10, 14, 26, 0.8); 
            backdrop-filter: blur(10px); 
            border: 1px solid var(--border-primary); 
            border-radius: 12px; 
            padding: 12px; 
            display: flex; 
            flex-direction: column; 
            gap: 8px; 
            box-shadow: var(--shadow-md); 
            transition: all 0.3s ease; 
            min-width: 300px; 
            max-width: 90vw; 
        }

        #image-plan-prompt {
            transition: all 0.3s ease;
        }

        #image-plan-prompt:focus {
            border-color: var(--accent-primary);
            box-shadow: 0 0 0 2px rgba(139, 92, 246, 0.2);
            outline: none;
        }

        /* Polish Options */
        .polish-option {
            padding: 1rem;
            border: 2px solid var(--border-primary);
            border-radius: var(--radius-md);
            cursor: pointer;
            transition: all 0.3s ease;
            text-align: left;
        }

        .polish-option:hover {
            border-color: var(--accent-primary);
            background: rgba(139, 92, 246, 0.05);
        }

        .polish-option.selected {
            border-color: var(--accent-primary);
            background: rgba(139, 92, 246, 0.1);
        }

        .polish-option h4 {
            margin: 0 0 0.5rem 0;
            color: var(--text-primary);
        }

        .polish-option p {
            margin: 0;
            font-size: 0.9rem;
            color: var(--text-secondary);
        }

        .polish-summary h4 {
            color: #fff !important;
        }

        .polish-intensity label {
            color: #fff !important;
        }

        .polish-spinner {
            display: block;
            margin: 0 auto;
            transform-origin: center;
        }

        .polish-spinner circle {
            transform-origin: 50px 50px;
        }

        .polish-spinner animateTransform {
            transform-origin: 50px 50px;
        }


        /* Step Selection Modal Styles */
        .step-option {
            transition: all 0.2s ease;
        }
        
        .step-option:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
            border-color: var(--border-secondary) !important;
        }
        
        .step-option.selected {
            border-color: var(--accent-primary) !important;
            background: rgba(102, 126, 234, 0.1);
        }
        
        .step-option.recommended {
            box-shadow: 0 0 20px rgba(102, 126, 234, 0.2);
        }
        
        #step-count-slider {
            -webkit-appearance: none;
            appearance: none;
            height: 6px;
            border-radius: 3px;
            background: var(--bg-secondary);
            outline: none;
        }
        
        #step-count-slider::-webkit-slider-thumb {
            -webkit-appearance: none;
            appearance: none;
            width: 18px;
            height: 18px;
            border-radius: 50%;
            background: var(--accent-gradient);
            cursor: pointer;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
        }
        
        #step-count-slider::-moz-range-thumb {
            width: 18px;
            height: 18px;
            border-radius: 50%;
            background: var(--accent-gradient);
            cursor: pointer;
            border: none;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.3);
        }

        #polish-trigger-btn {
            background: linear-gradient(45deg, #FF6B6B, #4ECDC4, #45B7D1, #96CEB4, #FFEAA7, #DDA0DD);
            background-size: 300% 300%;
            animation: polish-gradient-shift 3s ease infinite;
            color: white;
            border: none;
            padding: 0.5rem 1rem;
            border-radius: var(--radius-md);
            font-weight: 600;
            cursor: pointer;
            position: relative;
            overflow: hidden;
        }

        #polish-trigger-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(139, 92, 246, 0.4);
        }

        /* Projects Modal */
        .projects-modal-list { 
            list-style: none; 
            margin-top: 1.5rem; 
        }

        .projects-modal-list li { 
            display: flex; 
            justify-content: space-between; 
            align-items: center; 
            padding: 0.75rem 1rem; 
            border-radius: var(--radius-md); 
            background: var(--bg-tertiary); 
            margin-bottom: 0.5rem; 
        }

        .projects-modal-list .project-info { 
            text-align: left; 
        }

        .projects-modal-list .project-name { 
            font-weight: 500; 
            color: var(--text-primary); 
        }

        .projects-modal-list .project-date { 
            font-size: 0.8rem; 
            color: var(--text-secondary); 
        }

        .projects-modal-list .project-actions-modal button { 
            margin-left: 0.5rem; 
        }

        #btn-close-projects-modal { 
            margin-top: 20px !important; 
        }

        /* Settings */
        .settings-grid { 
            display: grid; 
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); 
            gap: 1rem; 
            margin-top: 1.5rem; 
        }

        .settings-item { 
            display: flex; 
            flex-direction: column; 
            align-items: center; 
            justify-content: center; 
            gap: 0.5rem; 
            padding: 1.5rem 1rem; 
            background: var(--bg-tertiary); 
            border-radius: var(--radius-md); 
            cursor: pointer; 
            transition: var(--transition); 
            color: var(--text-secondary); 
        }

        .settings-item:hover { 
            background: var(--bg-accent); 
            color: var(--text-primary); 
            transform: translateY(-2px); 
        }

        .settings-item svg { 
            width: 24px; 
            height: 24px; 
        }

        .settings-actions-destructive { 
            margin-top: 1.5rem; 
            padding-top: 1.5rem; 
            border-top: 1px solid var(--border-primary); 
            display: flex; 
            justify-content: center; 
        }

        .settings-item-destructive { 
            width: 60%; 
            border: 1px solid var(--bg-accent); 
        }

        .settings-item-destructive:hover { 
            color: var(--error); 
            border-color: var(--error); 
        }

        #btn-clear-kb-modal:hover { 
            color: var(--error); 
        }

        .modal-feedback { 
            margin-top: 1rem; 
            padding: 0.75rem 1rem; 
            border-radius: var(--radius-md); 
            font-size: 0.9rem; 
            display: none; 
        }

        .modal-feedback.success { 
            background-color: rgba(72,187,120,0.1); 
            color: var(--success); 
            display: block; 
        }

        .modal-feedback.error { 
            background-color: rgba(245,101,101,0.1); 
            color: var(--error); 
            display: block; 
			
        }
.modal-feedback{margin-bottom:12px !important}
        /* Diff Highlighting */
        .line-highlight-add { 
            background-color: rgba(72, 187, 120, 0.15); 
            display: block; 
            margin: 0 -1.5rem; 
            padding: 0 1.5rem; 
        }

        .line-highlight-remove { 
            background-color: rgba(245, 101, 101, 0.15); 
            display: block; 
            margin: 0 -1.5rem; 
            padding: 0 1.5rem; 
            text-decoration: line-through; 
        }

        .diff-btn { 
            padding: 0.4rem 0.8rem; 
            border: none; 
            border-radius: var(--radius-md); 
            cursor: pointer; 
            color: white; 
            font-weight: 500; 
            height: 28px; 
            font-size: 13px; 
        }

        .diff-btn.accept { 
            background: var(--success); 
        }

        .diff-btn.reject { 
            background: var(--error); 
        }

        .heal-btn { 
            margin-right: 10px !important; 
        }

        /* Mobile Overlay */
        #mobile-overlay { 
            position: fixed; 
            inset: 0; 
            background: rgba(0,0,0,0.5); 
            z-index: 99; 
            display: none; 
        }

        #mobile-overlay.visible { 
            display: block; 
        }

        #mobile-status-container { 
            display: none; 
        }

        /* Drag & Drop */
        .drop-indicator { 
            position: absolute; 
            left: 0; 
            right: 0; 
            height: 2px; 
            background-color: var(--accent-secondary); 
            z-index: 10; 
            pointer-events: none; 
        }

        .drop-indicator.top { 
            top: -1px; 
        }

        .drop-indicator.bottom { 
            bottom: -1px; 
        }

        /* Preview Button */
        #btn-view-generated-website { 
            border-color: var(--success) !important; 
            color: var(--success) !important; 
            animation: gentle-glow 2.5s infinite; 
        }

        /* Scrollbars */
        #one-shot-input::-webkit-scrollbar, 
        .file-tree::-webkit-scrollbar, 
        .chat-messages::-webkit-scrollbar, 
        .code-area::-webkit-scrollbar, 
        #chat-input::-webkit-scrollbar,
        #image-plan-prompt::-webkit-scrollbar,
        body::-webkit-scrollbar { 
            width: 8px; 
            height: 8px;
        }

        #one-shot-input::-webkit-scrollbar-track, 
        .file-tree::-webkit-scrollbar-track, 
        .chat-messages::-webkit-scrollbar-track, 
        .code-area::-webkit-scrollbar-track, 
        #chat-input::-webkit-scrollbar-track,
        #image-plan-prompt::-webkit-scrollbar-track,
        body::-webkit-scrollbar-track { 
            background: rgba(26, 27, 43, 0.3);
            border-radius: 4px;
        }

        #one-shot-input::-webkit-scrollbar-thumb, 
        .file-tree::-webkit-scrollbar-thumb, 
        .chat-messages::-webkit-scrollbar-thumb, 
        .code-area::-webkit-scrollbar-thumb, 
        #chat-input::-webkit-scrollbar-thumb,
        #image-plan-prompt::-webkit-scrollbar-thumb,
        body::-webkit-scrollbar-thumb { 
            background: linear-gradient(135deg, #4a4b57, #36374a);
            border-radius: 4px; 
            border: 1px solid rgba(139, 92, 246, 0.2);
            box-shadow: inset 0 1px 0 rgba(255,255,255,0.1);
        }

        #one-shot-input::-webkit-scrollbar-thumb:hover, 
        .file-tree::-webkit-scrollbar-thumb:hover, 
        .chat-messages::-webkit-scrollbar-thumb:hover, 
        .code-area::-webkit-scrollbar-thumb:hover, 
        #chat-input::-webkit-scrollbar-thumb:hover,
        #image-plan-prompt::-webkit-scrollbar-thumb:hover,
        body::-webkit-scrollbar-thumb:hover { 
            background: linear-gradient(135deg, #5a5b67, #46475a);
            border-color: rgba(139, 92, 246, 0.4);
            box-shadow: inset 0 1px 0 rgba(255,255,255,0.2);
        }

        #one-shot-input::-webkit-scrollbar-thumb:active, 
        .file-tree::-webkit-scrollbar-thumb:active, 
        .chat-messages::-webkit-scrollbar-thumb:active, 
        .code-area::-webkit-scrollbar-thumb:active, 
        #chat-input::-webkit-scrollbar-thumb:active,
        #image-plan-prompt::-webkit-scrollbar-thumb:active,
        body::-webkit-scrollbar-thumb:active { 
            background: linear-gradient(135deg, #6a6b77, #56576a);
            border-color: var(--accent-primary);
        }

        #one-shot-input::-webkit-scrollbar-corner, 
        .file-tree::-webkit-scrollbar-corner, 
        .chat-messages::-webkit-scrollbar-corner, 
        .code-area::-webkit-scrollbar-corner, 
        #chat-input::-webkit-scrollbar-corner,
        #image-plan-prompt::-webkit-scrollbar-corner,
        body::-webkit-scrollbar-corner { 
            background: rgba(26, 27, 43, 0.3);
        }

        #one-shot-input, 
        .file-tree, 
        .chat-messages, 
        .code-area, 
        #chat-input,
        #image-plan-prompt,
        body { 
            scrollbar-width: thin; 
            scrollbar-color: #4a4b57 rgba(26, 27, 43, 0.3); 
        }

        /* Animations */
        @keyframes spin { 
            to { transform: rotate(360deg); } 
        }

        @keyframes spin-perpetual {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        @keyframes pulse-perpetual {
            0%, 100% { stroke-dashoffset: 100; }
            50% { stroke-dashoffset: 40; }
        }

        @keyframes pulse { 
            0%, 100% { opacity: 1; } 
            50% { opacity: 0.5; } 
        }

        @keyframes gentle-glow { 
            0% { box-shadow: 0 0 0 0 rgba(72,187,120,0.6); } 
            70% { box-shadow: 0 0 0 8px rgba(72,187,120,0); } 
            100% { box-shadow: 0 0 0 0 rgba(72,187,120,0); } 
        }

        @keyframes polish-gradient-shift {
            0% { background-position: 0% 50%; }
            50% { background-position: 100% 50%; }
            100% { background-position: 0% 50%; }
        }

        @keyframes countdown-animation {
            from { stroke-dashoffset: 62.83; }
            to { stroke-dashoffset: 0; }
        }

        .spinner-group {
            animation: spin-perpetual 1s linear infinite;
            transform-origin: center;
        }

        .countdown-arc {
            stroke-dasharray: 40 100;
            stroke-dashoffset: 100;
            animation: pulse-perpetual 1.5s ease-in-out infinite;
            transform-origin: center;
        }

        /* Special Icons & Elements */
        .tarrow {
            font-size: 23px !important; 
            position: relative; 
            top: -2px !important;
        }

        .runs {
            position: relative; 
            bottom: -2.5px !important;
        }

        #send-button #send-icon { 
            position: relative !important; 
            left: -10px !important; 
        }

        #loading-icon { 
            position: relative !important; 
            left: -10px !important; 
        }

        /* Desktop Layout */
        @media (min-width: 1200px) {
            body .panels-container {
                flex-direction: row !important; 
            }
            
            body .code-editor-panel {
                flex-basis: 45% !important;
                border-top: none !important;
                border-right: 1px solid rgba(38, 39, 43, 0.3) !important;
            }
            
            body .chat-panel {
                flex-basis: 55% !important;
                border-top: none !important;
                border-left: 1px solid rgba(38, 39, 43, 0.3) !important;
            }
            
            body .resizer {
                width: 8px !important;
                height: 100% !important;
                cursor: col-resize !important; 
            }
        }

        /* Tablet Responsive */
        @media (min-width: 501px) and (max-width: 768px) {
            #ai_app_root {
                height: auto !important;
                min-height: 100vh !important;
                overflow-y: auto !important;
            }
        }

        @media (min-width: 501px) and (max-width: 1100px) {
            body .feature-bar {
                padding: 0.5rem 1rem !important; 
            }
            
            body .feature-bar-text {
                font-size: 0.8rem !important; 
                margin-top: 0.5rem !important;
            }
        }

        /* Desktop Input Bar Styling */
        @media (min-width: 501px) and (max-width: 2000px) {
            .chat-input-container {
                border-radius: 999px !important;
                border: 1px solid var(--border-primary) !important;
                background: var(--bg-tertiary) !important;
                padding: 0.5rem 0.75rem !important;
                transition: all 0.2s ease-in-out !important;
                display: flex !important;
                align-items: center !important;
                gap: 0.5rem !important; 
                margin-right: 20px; 
                margin-left: 20px;
                margin-bottom: 25px !important;
            }

            .chat-input,
            .voice-button,
            .send-button {
                background: transparent !important;
                border: none !important;
                box-shadow: none !important;
            }

            .chat-input-container:has(.chat-input:focus) {
                border-color: var(--border-accent) !important;
                box-shadow: var(--shadow-glow) !important;
            }

            .chat-input:focus {
                background: transparent !important;
                border-color: transparent !important;
                box-shadow: none !important;
                color: var(--text-primary) !important;
            }

            .send-button:hover {
                background: var(--accent-gradient) !important;
                transform: none !important;
            }
            
            .chat-input {
                flex-grow: 1 !important;
                padding-left: 0.5rem !important;
            }

            .voice-button,
            .send-button {
                flex-shrink: 0 !important;
            }
        }

        /* Mobile Specific */
        @media (max-width: 768px) {
            .header { 
                display: flex; 
                align-items: center; 
                justify-content: space-between; 
                padding: 0.75rem 1rem 0.75rem calc(1rem + 10px);
                gap: 0.75rem; 
            }

            .header-left { 
                display: flex; 
                align-items: center; 
            }

            .header > h1 { 
                margin-left: auto; 
                flex: 0 0 auto; 
                text-align: right; 
                line-height: 1; 
                margin-right: 10px; 
                margin-top: -14px !important; 
                display: flex; 
                align-items: center; 
                justify-content: flex-end; 
            }

            .header-right { 
                display: none; 
            }

            .header-title { 
                text-align: right; 
                max-width: 150px; 
            }

            .main-layout { 
                grid-template-columns: 1fr; 
            }

            .sidebar { 
                position: fixed; 
                top: 0; 
                left: 0; 
                height: 100vh; 
                transform: translateX(-100%); 
                margin-left: 0; 
                z-index: 999999 !important; 
            }

            .sidebar.open { 
                transform: translateX(0); 
            }

            .sidebar-toggle-header { 
                display: flex; 
            }

            .desktop-status-indicator { 
                display: none !important; 
            }

            #mobile-status-container { 
                display: flex; 
                align-items: center; 
                gap: 0.5rem; 
            }

            body .panels-container {
                flex-direction: column !important;
            }
            
            body .code-editor-panel { 
                flex-basis: 30% !important; 
            }
            
            body .chat-panel { 
                flex-basis: 70% !important; 
            }

            .chat-panel .panel-header { 
                flex-direction: row !important; 
                align-items: center !important; 
                justify-content: space-between !important; 
                padding: 0.5rem 1.25rem !important; 
                min-height: auto !important; 
                width: 100% !important; 
            }

            .chat-panel-header-left { 
                justify-content: flex-start !important; 
                width: auto !important; 
                max-width: 60% !important; 
                flex-shrink: 0 !important; 
            }

            .chat-panel-header-left > span:first-of-type { 
                display: none !important; 
            }

            .chat-panel-header-right { 
                justify-content: flex-end !important; 
                width: auto !important; 
                max-width: 40% !important; 
                flex-shrink: 0 !important; 
                margin-left: auto !important; 
            }

            .mode-toggle-container { 
                display: flex !important; 
                align-items: center !important; 
                gap: 0.3rem !important; 
                flex-shrink: 0 !important; 
            }

            .mode-toggle-button { 
                padding: 0.25rem 0.5rem !important; 
                font-size: 11px !important; 
                width: 60px !important; 
            }

            .feature-bar-main { 
                flex-direction: column; 
            }

            body .chat-messages {
                margin-bottom: 100px !important;
            }
            
            body .chat-input-container {
                min-height: 100px !important;
            }
        }

        /* Very Small Mobile */
        @media (max-width: 500px) {
            .chat-panel {
                position: relative !important;
            }

            .chat-messages {
                padding-bottom: 150px !important;
            }

            .chat-input-container {
                position: absolute !important;
                bottom: 0 !important;
                left: 0 !important;
                right: 0 !important;
                background: var(--bg-secondary) !important;
                border-top: 1px solid var(--border-primary) !important;
                flex-wrap: wrap !important;
                gap: 0.75rem !important;
            }
            
            .chat-input {
                flex-basis: 100% !important;
                min-height: 70px !important;
                resize: vertical !important;
            }

            .voice-button,
            .send-button {
                flex-grow: 1 !important;
            }

            .send-button-content {
                justify-content: center !important;
            }

            .chat-input-container .chat-input:focus {
                background: var(--bg-primary) !important;
                color: var(--text-primary) !important;
                border-color: var(--border-accent) !important;
                box-shadow: var(--shadow-glow) !important;
            }

            #image-modal-controls {
                width: 80%;
                bottom: 30px;
                min-width: unset;
                flex-direction: column;
            }
            
            #image-modal-controls .modal-button,
            #image-modal-controls .modal-button-secondary {
                width: 100%;
                font-size: 1rem;
            }

            .projects-modal-list .project-actions-modal {
                display: flex;
                flex-direction: column;
            }

            .projects-modal-list .project-actions-modal button {
                margin-left: 0;
            }

            .projects-modal-list .project-actions-modal button:first-child {
                margin-bottom: 10px;
            }
        }

        /* Very Small Screens */
        @media (max-width: 570px) {
            #image-preview-modal { 
                display: block; 
                overflow-y: auto; 
                padding: 0; 
            }

            .image-modal-content { 
                margin: 40px 30px 20px 30px; 
                max-width: none; 
                max-height: none; 
            }

            @media screen and (max-width: 1920px) and (min-width: 571px) { 
                .image-modal-content img { 
                    max-height: 500px; 
                    overflow-y: scroll; 
                }
            }
        }

        /* Medium Screens */
        @media (max-width: 1100px) {
            .build-website-button { 
                line-height: 1.4 !important; 
                margin-top: -30px !important; 
            }

            .feature-bar-text { 
                width: 100% !important; 
                float: left !important; 
                margin-top: 20px !important; 
                padding-left: 10px !important; 
                padding-right: 10px !important; 
            }

            .feature-bar { 
                display: inline-block !important; 
                text-align: center !important; 
                padding-bottom: 0px !important; 
                padding-top: 0px !important; 
				
            }
			
			
			.feature-bar-main {padding-bottom:20px }
        }

        /* Special Overrides */
        #ai_app_root::before { 
            content: ''; 
            position: absolute; 
            top: 0; 
            left: 0; 
            right: 0; 
            height: 0; 
            background: transparent; 
        }

        .vectorize-btn, .delete-btn, .preview-btn { 
            visibility: visible !important; 
            opacity: 0.7 !important; 
        }

        .vectorize-btn:hover, .delete-btn:hover, .preview-btn:hover { 
            opacity: 1 !important; 
        }

        .file-tree-item:hover .delete-btn, 
        .folder-item:hover .delete-btn, 
        .file-tree-item:hover .preview-btn, 
        .file-tree-item:hover .vectorize-btn { 
            visibility: visible !important; 
        }

        /* Panel Focus Modes */
        body .panel-toggle {
            position: absolute !important;
            top: 0.5rem !important;
            right: 0.5rem !important;
            z-index: 10 !important;
            background: var(--bg-tertiary) !important;
            border: 1px solid var(--border-primary) !important;
            color: var(--text-secondary) !important;
            width: 24px !important;
            height: 24px !important;
            border-radius: 4px !important;
            cursor: pointer !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            font-size: 12px !important;
            opacity: 0.7 !important;
            transition: all 0.2s ease !important;
        }

        body .panel-toggle:hover {
            opacity: 1 !important;
            color: var(--accent-primary) !important;
        }

        body .code-editor-panel.collapsed {
            flex-basis: 0 !important;
            min-height: 40px !important;
            overflow: hidden !important;
        }

        body .chat-panel.collapsed {
            flex-basis: 0 !important;
            min-height: 40px !important;
            overflow: hidden !important;
        }

        body .code-editor-panel.expanded {
            flex-basis: 95% !important;
        }

        body .chat-panel.expanded {
            flex-basis: 95% !important;
        }

        body .panel-tabs {
            display: flex !important;
            background: var(--bg-tertiary) !important;
            border-bottom: 1px solid var(--border-primary) !important;
        }
        
        body .panel-tab {
            flex: 1 !important;
            padding: 0.75rem !important;
            text-align: center !important;
            cursor: pointer !important;
            background: transparent !important;
            border: none !important;
            color: var(--text-secondary) !important;
            transition: all 0.2s ease !important;
        }
        
        body .panel-tab.active {
            color: var(--accent-primary) !important;
            background: var(--bg-secondary) !important;
            border-bottom: 2px solid var(--accent-primary) !important;
        }
		
		
			.runsa{top:0px !important;position:relative !important; color:#fff !important}
		.shortcut-keys{color:#fff !important}
		
		@media (max-width: 500px) {body .runsa{top:1px !important;position:relative !important}}
.app-footer {
    background: var(--bg-tertiary);
    border-top: 1px solid var(--border-primary);
    padding: 0.5rem 1.5rem;
    display: flex;
    align-items: center;
    justify-content: space-between;
    min-height: 36px;
    flex-shrink: 0;
}

.footer-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    width: 100%;
    font-size: 0.8rem;
    color: var(--text-tertiary);
}

.footer-brand {
    font-weight: 500;
    color: var(--text-secondary);
}

.footer-status {
    display: flex;
    align-items: center;
    gap: 0.4rem;
}

.footer-idle {
    color: var(--text-tertiary);
    opacity: 0.4;
    animation: pulse-gentle 3s ease-in-out infinite;
}

@keyframes pulse-gentle {
    0%, 100% { opacity: 0.2; }
    50% { opacity: 0.6; }
}

.footer-status.processing .footer-idle {
    display: none;
}

.footer-status.processing::after {
    content: '⋅ ⋅ ⋅';
    animation: dots-pulse 1.5s ease-in-out infinite;
    font-size: 1.2em;
    color: var(--accent-primary);
    letter-spacing: 2px;
}

@keyframes cursor-breathe {
    0%, 100% { opacity: 0.2; }
    50% { opacity: 0.5; }
}

@keyframes dots-pulse {
    0%, 20% { opacity: 0.3; }
    40% { opacity: 1; }
    100% { opacity: 0.3; }
}

/* Footer Progress Bar Styles - Beautiful Overlay */
.footer-progress-bar {
    position: absolute;
    bottom: 1.5rem;
    left: 1.5rem;
    right: 1.5rem;
    width: auto;
    background: linear-gradient(135deg, rgba(139, 92, 246, 0.05) 0%, rgba(167, 139, 250, 0.05) 100%);
    border: 1px solid var(--border-primary);
    border-radius: 12px;
    padding: 1.25rem 1.5rem;
    backdrop-filter: blur(10px);
    box-shadow: 
        0 4px 6px rgba(0, 0, 0, 0.05),
        0 1px 3px rgba(0, 0, 0, 0.1),
        inset 0 1px 0 rgba(255, 255, 255, 0.05);
    transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
    z-index: 1000;
    pointer-events: auto;
}

.footer-progress-bar:hover {
    border-color: var(--accent-primary);
    box-shadow: 
        0 8px 12px rgba(139, 92, 246, 0.1),
        0 2px 4px rgba(0, 0, 0, 0.1),
        inset 0 1px 0 rgba(255, 255, 255, 0.05);
}

.footer-progress-content {
    width: 100%;
    max-width: 100%;
}

.footer-progress-info {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.875rem;
}

.footer-progress-title {
    font-size: 0.9375rem;
    font-weight: 600;
    color: var(--text-primary);
    letter-spacing: -0.01em;
}

.footer-progress-step {
    font-size: 0.8125rem;
    color: var(--accent-primary);
    font-weight: 600;
    padding: 0.25rem 0.625rem;
    background: rgba(139, 92, 246, 0.1);
    border-radius: 6px;
    margin-left: 0.75rem;
}

.footer-progress-close {
    background: rgba(255, 255, 255, 0.05);
    border: 1px solid var(--border-primary);
    color: var(--text-secondary);
    cursor: pointer;
    padding: 0.375rem 0.625rem;
    font-size: 1.125rem;
    line-height: 1;
    border-radius: 6px;
    transition: all 0.2s cubic-bezier(0.4, 0, 0.2, 1);
    margin-left: auto;
}

.footer-progress-close:hover {
    color: var(--text-primary);
    background: rgba(239, 68, 68, 0.1);
    border-color: rgba(239, 68, 68, 0.3);
    transform: translateY(-1px);
}

.footer-progress-close:active {
    transform: translateY(0);
}

.footer-progress-details {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 0.75rem;
    gap: 1rem;
}

.footer-progress-text {
    font-size: 0.875rem;
    color: var(--text-secondary);
    flex: 1;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.footer-progress-percentage {
    font-size: 1rem;
    font-weight: 700;
    color: var(--accent-primary);
    font-variant-numeric: tabular-nums;
    min-width: 48px;
    text-align: right;
}

.footer-progress-bar-track {
    position: relative;
    width: 100%;
    height: 10px;
    background: rgba(0, 0, 0, 0.15);
    border-radius: 6px;
    overflow: hidden;
    box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.15);
}

.footer-progress-bar-fill {
    height: 100%;
    background: linear-gradient(90deg, #8b5cf6 0%, #a78bfa 50%, #c4b5fd 100%);
    background-size: 200% 100%;
    animation: shimmer 2.5s ease-in-out infinite;
    transition: width 0.5s cubic-bezier(0.4, 0, 0.2, 1);
    border-radius: 6px;
    box-shadow: 
        0 0 10px rgba(139, 92, 246, 0.4),
        0 0 20px rgba(139, 92, 246, 0.2);
    position: relative;
}

.footer-progress-bar-fill::after {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 50%;
    background: linear-gradient(to bottom, rgba(255, 255, 255, 0.3), transparent);
    border-radius: 6px 6px 0 0;
}

@keyframes shimmer {
    0% { background-position: 0% 0%; }
    50% { background-position: 100% 0%; }
    100% { background-position: 0% 0%; }
}

/* Hide on tablet and mobile */
@media (max-width: 768px) {
    .footer-progress-bar {
        display: none;
    }
}

/* Responsive adjustments for smaller desktop screens */
@media (min-width: 769px) and (max-width: 1100px) {
    .footer-progress-bar {
        bottom: 1rem;
        left: 1rem;
        right: 1rem;
        padding: 1rem 1.25rem;
    }
    
    .footer-progress-title {
        font-size: 0.875rem;
    }
    
    .footer-progress-step {
        font-size: 0.75rem;
        padding: 0.2rem 0.5rem;
    }
    
    .footer-progress-percentage {
        font-size: 0.9375rem;
    }
    
    .footer-progress-text {
        font-size: 0.8125rem;
    }
    
    .footer-progress-bar-track {
        height: 8px;
    }
}

@media (max-width: 768px) {
    .app-footer {
        padding: 0.4rem 1rem;
        min-height: 32px;
    }
    
    .footer-content {
        font-size: 0.75rem;
    }
}




@media (min-width: 550px) and (max-width: 1100px) {
    
        #image-modal-controls{bottom:-280px !important
				  
    }
}


 

@media (min-width: 771px) and (max-width: 3000px) {

body .chat-panel {
	
	
	min-width:550px !important
	
}}
/* 1) Ensure the chat column is the scrollport */
.chat-messages {
  position: relative !important;
  min-height: 0 !important;
  flex: 1 1 auto !important;
  overflow-y: auto !important;
}

/* 2) Don't let big blocks inside messages create their own scroll areas */
.chat-messages .message-bubble * {
  max-height: none !important;
}

.chat-messages .message-bubble [style*="overflow:"],
.chat-messages .message-bubble .scroll,
.chat-messages .message-bubble .scrollable,
.chat-messages .message-bubble .plan,
.chat-messages .message-bubble .plan * {
  overflow: visible !important;
}

/* 3) Kill overlays that can intercept wheel/touch */
.chat-messages .message-bubble [style*="position: fixed"],
.chat-messages .message-bubble [style*="position: absolute"],
.chat-messages .message-bubble [style*="position: sticky"],
.chat-messages .message-bubble .sticky,
.chat-messages .message-bubble .overlay {
  position: static !important;
}

/* 4) Re-enable touch/trackpad scrolling if plan content disabled it */
.chat-messages .message-bubble *,
.chat-messages .message-bubble .plan *,
.chat-messages .message-bubble .plan {
  touch-action: auto !important;
  -ms-touch-action: auto !important;
}

/* 5) Safety: make sure nothing sits on top of the scrollport */
.chat-messages::before,
.chat-messages::after {
  pointer-events: none !important;
}

/* Make the chat column the scrollport */
.chat-messages {
  flex: 1 1 auto !important;
  min-height: 0 !important;
  overflow-y: auto !important;
}

/* STRICT fix: neutralize scroll/overlay traps only inside plan blocks */
.chat-messages .message-bubble .plan-step,
.chat-messages .message-bubble .plan-step * {
  overflow: visible !important;      /* kill nested scroll areas */
  position: static !important;       /* disable sticky/fixed toolbars inside plan */
  touch-action: auto !important;     /* re-enable trackpad/touch scrolling */
  max-height: none !important;       /* avoid accidental clipping */
  -webkit-overflow-scrolling: auto !important;
}

/* If the plan inserts any “sticky”/overlay helpers by class or inline style */
.chat-messages .message-bubble .plan-step .sticky,
.chat-messages .message-bubble .plan-step [style*="position:sticky"],
.chat-messages .message-bubble .plan-step [style*="position:fixed"],
.chat-messages .message-bubble .plan-step [style*="position:absolute"] {
  position: static !important;
}

/* Safety: keep invisible decorations from intercepting scroll */
.chat-messages::before,
.chat-messages::after {
  pointer-events: none !important;
}

/* Ensure content isn't hidden behind the absolute input bar */
.chat-messages { padding-bottom: 120px !important; }


/* Enhanced plan mode scroll fixes */
.chat-messages .plan-container,
.chat-messages .plan-step,
.chat-messages .plan-controls,
.chat-messages .plan-step * {
    position: static !important;
    overflow: visible !important;
    max-height: none !important;
    height: auto !important;
    touch-action: auto !important;
    -webkit-overflow-scrolling: auto !important;
    pointer-events: auto !important;
}

/* Ensure plan buttons don't interfere with scrolling */
.chat-messages .plan-step-controls button {
    position: static !important;
    z-index: auto !important;
}

/* Force chat messages to remain the scroll container */
body.plan-mode .chat-messages {
    overflow-y: auto !important;
    -webkit-overflow-scrolling: touch !important;
    scroll-behavior: smooth !important;
}




#polish-summary h4{color:#fff !important}
#polish-summary h5{color:#fff !important}
/* Limit Notification Styles */
        .limit-notification {
            position: fixed;
            top: 20px;
            right: 20px;
            max-width: 420px;
            z-index: 999999;
            border-radius: 16px;
            box-shadow: 0 20px 25px -5px rgba(0,0,0,0.4);
            transform: translateX(100%);
            transition: all 0.5s cubic-bezier(0.4, 0, 0.2, 1);
            backdrop-filter: blur(12px);
            border: 1px solid rgba(255,255,255,0.1);
            overflow: hidden;
        }
        .limit-notification.show { transform: translateX(0); }
        .limit-notification.warning {
            background: linear-gradient(135deg, rgba(237, 137, 54, 0.95) 0%, rgba(245, 101, 101, 0.95) 100%);
            border-left: 4px solid #f59e0b;
        }
        .limit-notification.critical {
            background: linear-gradient(135deg, rgba(239, 68, 68, 0.95) 0%, rgba(220, 38, 127, 0.95) 100%);
            border-left: 4px solid #ef4444;
        }
        .limit-notification.reached {
            background: linear-gradient(135deg, rgba(139, 92, 246, 0.95) 0%, rgba(79, 70, 229, 0.95) 100%);
            border-left: 4px solid #8b5cf6;
        }
        .notification-header { display: flex; align-items: center; justify-content: space-between; padding: 16px 20px 12px 20px; }
        .notification-icon { width: 32px; height: 32px; border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 16px; font-weight: bold; color: white; animation: pulse-icon 2s ease-in-out infinite; }
        .notification-icon.warning { background: rgba(251, 191, 36, 0.2); border: 2px solid #fbbf24; }
        .notification-icon.critical { background: rgba(248, 113, 113, 0.2); border: 2px solid #f87171; }
        .notification-icon.reached { background: rgba(167, 139, 250, 0.2); border: 2px solid #a78bfa; }
        .notification-close { background: rgba(255,255,255,0.1); border: none; width: 28px; height: 28px; border-radius: 50%; color: white; cursor: pointer; display: flex; align-items: center; justify-content: center; transition: all 0.2s ease; font-size: 14px; font-weight: bold; }
        .notification-close:hover { background: rgba(255,255,255,0.2); transform: scale(1.1); }
        .notification-body { padding: 0 20px 16px 20px; color: white; }
        .notification-title { font-size: 16px; font-weight: 600; margin: 0 0 8px 0; }
        .notification-message { font-size: 14px; line-height: 1.5; margin: 0 0 12px 0; opacity: 0.95; }
        .notification-stats { display: flex; align-items: center; justify-content: space-between; background: rgba(0,0,0,0.2); border-radius: 8px; padding: 8px 12px; margin-bottom: 12px; font-size: 13px; font-weight: 500; }
        .notification-actions { display: flex; gap: 8px; }
        .notification-btn { padding: 8px 16px; border: none; border-radius: 8px; font-size: 13px; font-weight: 600; cursor: pointer; transition: all 0.2s ease; }
        .notification-btn.primary { background: rgba(255,255,255,0.2); color: white; border: 1px solid rgba(255,255,255,0.3); }
        .notification-btn.primary:hover { background: rgba(255,255,255,0.3); transform: translateY(-1px); }
        .notification-btn.secondary { background: transparent; color: rgba(255,255,255,0.8); border: 1px solid rgba(255,255,255,0.2); }
        .notification-btn.secondary:hover { background: rgba(255,255,255,0.1); color: white; }
        @keyframes pulse-icon { 0%, 100% { transform: scale(1); } 50% { transform: scale(1.1); } }
        @media (max-width: 480px) { .limit-notification { top: 10px; right: 10px; left: 10px; max-width: none; } }
    
.tour-welcome-modal {
    position: fixed;
    inset: 0;
    z-index: 1000000;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 2rem;
}

.tour-welcome-overlay {
    position: absolute;
    inset: 0;
    background: rgba(0, 0, 0, 0.8);
    backdrop-filter: blur(5px);
}

.tour-welcome-content {
    position: relative;
    background: var(--bg-secondary);
    border-radius: var(--radius-xl);
    border: 1px solid var(--border-primary);
    box-shadow: var(--shadow-xl);
    max-width: 500px;
    width: 100%;
}

.tour-welcome-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 2rem 2rem 1rem 2rem;
    border-bottom: 1px solid var(--border-primary);
}

.tour-welcome-header h2 {
    margin: 0;
    background: var(--accent-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.tour-close-btn {
    background: none;
    border: none;
    color: var(--text-secondary);
    font-size: 1.5rem;
    cursor: pointer;
    padding: 0;
    width: 24px;
    height: 24px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.tour-welcome-body {
    padding: 1rem 2rem 2rem 2rem;
}

.tour-welcome-buttons {
    display: flex;
    gap: 1rem;
    margin-top: 1.5rem;
    justify-content: flex-end;
    flex-wrap: wrap; /* prevent overflow */
}

.tour-btn-primary {
    background: var(--accent-gradient);
    color: white;
    border: none;
    padding: 0.75rem 1.5rem;
    border-radius: var(--radius-md);
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition);
    max-width: 100%; /* never overflow container */
}

.tour-btn-primary:hover {
    transform: translateY(-2px);
    box-shadow: var(--shadow-glow);
}

.tour-btn-secondary {
    background: var(--bg-tertiary);
    color: var(--text-primary);
    border: 1px solid var(--border-primary);
    padding: 0.75rem 1.5rem;
    border-radius: var(--radius-md);
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition);
    max-width: 100%;
}

.tour-btn-secondary:hover {
    background: var(--bg-accent);
}

.tour-step {
    position: fixed;
    z-index: 1000001;
    background: var(--bg-secondary);
    border: 1px solid var(--border-primary);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-xl);
    padding: 0;

    /* fixes for off-screen */
    max-width: min(360px, calc(100vw - 32px)) !important;
    width: max-content;
    overflow-wrap: anywhere;
    word-break: break-word;
}

.tour-step-content {
    padding: 1.5rem;
}

.tour-step-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.tour-step-header h3 {
    margin: 0;
    font-size: 1.1rem;
    color: var(--accent-primary);
}

.tour-step-counter {
    font-size: 0.8rem;
    color: var(--text-tertiary);
    background: var(--bg-tertiary);
    padding: 0.25rem 0.5rem;
    border-radius: 12px;
}

.tour-step p {
    margin: 0 0 1.5rem 0;
    line-height: 1.6;
    color: var(--text-secondary);
}

.tour-step-buttons {
    display: flex;
    justify-content: space-between;
    align-items: center;
    flex-wrap: wrap; /* keep buttons inside */
    gap: 0.5rem;
}

.tour-nav-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.tour-step-arrow {
    position: absolute;
    width: 0;
    height: 0;
    border: 8px solid transparent;
}

.arrow-left {
    left: -16px;
    top: 20px;
    border-right-color: var(--bg-secondary);
}

.arrow-right {
    right: -16px;
    top: 20px;
    border-left-color: var(--bg-secondary);
}

.arrow-top {
    top: -16px;
    left: 50px;
    border-bottom-color: var(--bg-secondary);
}

.arrow-bottom {
    bottom: -16px;
    left: 50px;
    border-top-color: var(--bg-secondary);
}

.tour-backdrop {
    position: fixed;
    inset: 0;
    background: rgba(0, 0, 0, 0.6);
    z-index: 1000000;
    backdrop-filter: blur(2px);
}

.tour-highlight {
    position: relative;
    z-index: 1000001;
    box-shadow: 0 0 0 4px rgba(139, 92, 246, 0.5) !important;
    border-radius: 4px;
    animation: tour-pulse 2s ease-in-out infinite;
}

@keyframes tour-pulse {
    0%, 100% { box-shadow: 0 0 0 4px rgba(139, 92, 246, 0.5); }
    50% { box-shadow: 0 0 0 8px rgba(139, 92, 246, 0.3); }
}

.tour-completion-modal {
    position: fixed;
    inset: 0;
    z-index: 1000000;
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 2rem;
}
.tour-completion-modal  p{color:#fff !important}


.tour-completion-overlay {
    position: absolute;
    inset: 0;
    background: rgba(0, 0, 0, 0.8);
    backdrop-filter: blur(5px);
}

.tour-completion-content {
    position: relative;
    background: var(--bg-secondary);
    border-radius: var(--radius-xl);
    border: 1px solid var(--border-primary);
    box-shadow: var(--shadow-xl);
    max-width: 400px;
    width: 100%;
    padding: 2rem;
    text-align: center;
}

.tour-completion-content h2 {
    margin: 0 0 1rem 0;
    background: var(--accent-gradient);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
}

.tour-completion-content ul {
    text-align: left;
    margin: 1rem 0;
    color: var(--text-secondary);
}

.tour-button {
    margin-right: 0.5rem;
}

/* Mobile adjustments */
@media (max-width: 768px) {
    .tour-step {
        max-width: min(340px, calc(100vw - 24px)) !important;
    }
    .tour-step-arrow {
        display: none; /* prevent clipping off screen */
    }
}
/* Animation keyframes */
@keyframes tour-fade-slide {
  from {
    opacity: 0;
    transform: translateY(10px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
}

/* Apply to every tour step */
.tour-step {
  animation: tour-fade-slide 0.25s ease-out;
  will-change: opacity, transform;
}
.tour-welcome-content p{color:#fff !important}

/* Hide tour on mobile */
@media (max-width: 768px) {
    .tour-button,
    .tour-welcome-modal,
    .tour-step,
    .tour-backdrop,
    .tour-completion-modal {
        display: none !important;
    }
}
/* Footer tour button (desktop only) */
.footer-tour-button {
    background: none;
    border: none;
    color: var(--text-secondary);
    font-size: 0.8rem;
    font-weight: 500;
    cursor: pointer;
    display: flex;
    align-items: center;
    gap: 0.4rem;
    padding: 0.25rem 0.5rem;
    border-radius: 4px;
    transition: all 0.2s ease;
}

.footer-tour-button:hover {
    color: var(--accent-primary);
    background: rgba(139, 92, 246, 0.1);
}

.footer-tour-button svg {
    width: 16px;
    height: 16px;
}

/* Hide tour elements on mobile, but keep footer brand */
@media (max-width: 768px) {
    .footer-tour-button,
    .tour-welcome-modal,
    .tour-step,
    .tour-backdrop,
    .tour-completion-modal,
    .tour-highlight {
        display: none !important;
    }
    
    /* Restore original footer brand text on mobile */
    .footer-brand {
        font-weight: 500;
        color: var(--text-secondary);
    }
    
    
}

/* Special arrow positioning for step 5 (mode toggle) */
.tour-step[data-step="4"] .tour-step-arrow.arrow-top {
    left: auto !important;
    right: 50px !important;
}


@media (max-width: 1100px) {
    .resizer {
        display: none !important;
    }
    
    body .code-editor-panel { 
        flex-basis: 40% !important; /* Even taller */
        min-height: 250px !important;
		  max-height: 300px !important;
    }
    
    body .chat-panel { 
        flex-basis: 60% !important; /* Adjusted to balance */
        min-height: 350px !important;
		max-height: 400px !important;
    }
}

@media (min-width: 0px) and (max-width: 1100px) {
    body #ai_app_root {
        height: auto !important;
        min-height: 120vh !important; /* Taller than viewport */
        max-height: none !important; /* Remove height restriction */
    }
}

/* NEW: Add these styles to the end of your existing CSS */

/* Version Control Styles */
.version-item {
    transition: transform 0.2s ease, box-shadow 0.2s ease;
}

.version-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
}

.version-item .icon-button {
    width: 32px;
    height: 32px;
    border-radius: 6px;
    transition: all 0.2s ease;
}

.version-item .icon-button:hover {
    transform: scale(1.1);
}

#versions-list {
    /* max-height: 400px;  <-- REMOVE THIS LINE */
    /* overflow-y: auto;   <-- REMOVE THIS LINE */
    border: 1px solid var(--border-primary);
    border-radius: var(--radius-md);
    padding: 1rem;
    background: var(--bg-secondary);
}

#versions-list:empty::after {
    content: 'Loading versions...';
    display: block;
    text-align: center;
    color: var(--text-secondary);
    font-style: italic;
    padding: 2rem;
}

/* Version badges */
.version-type-badge {
    font-size: 0.75rem;
    padding: 0.2rem 0.5rem;
    border-radius: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.version-type-auto {
    background: #fbbf24;
    color: #000;
}

.version-type-manual {
    background: var(--accent-primary);
    color: #fff;
}

/* Loading state */
.version-loading {
    display: flex;
    align-items: center;
    justify-content: center;
    padding: 3rem;
    color: var(--text-secondary);
}

.version-loading::before {
    content: '';
    width: 20px;
    height: 20px;
    border: 2px solid var(--border-primary);
    border-top: 2px solid var(--accent-primary);
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-right: 0.5rem;
}

.file-tree-updating {
    /* This class will trigger our keyframe animation */
    animation: file-tree-blink 0.4s ease-in-out;
}

@keyframes file-tree-blink {
    /* The animation will be a quick pulse of an inset shadow */
    0% {
        box-shadow: inset 0 0 0 0 rgba(139, 92, 246, 0);
    }

/* Streaming cursor animation */
.stream-cursor {
    display: inline-block;
    animation: cursor-blink 1s step-end infinite;
    margin-left: 2px;
}

@keyframes cursor-blink {
    0%, 50% { opacity: 1; }
    51%, 100% { opacity: 0; }
}

.chat-message.streaming .stream-cursor {
    display: inline-block;
}

.chat-message:not(.streaming) .stream-cursor {
    display: none;
}

    50% {
        /* Glow with your app's accent color */
        box-shadow: inset 0 0 15px 2px rgba(139, 92, 246, 0.5);
    }
    100% {
        box-shadow: inset 0 0 0 0 rgba(139, 92, 246, 0);
    }
}
/* ADD THIS NEW CSS to your injectCSS function */

.editor-content {
    /* This is critical for making the absolute positioning work correctly */
    position: relative;
}

#debug-button {
    position: absolute;
    bottom: 20px;
    right: 20px;
    z-index: 100; /* Ensures it floats above the textarea */
    
    /* Some nice visual flair */
    box-shadow: var(--shadow-md);
    transition: all 0.2s ease-in-out;
}

#debug-button:hover {
    transform: scale(1.1);
    box-shadow: var(--shadow-glow);
}

#debug-button.active {
    background: var(--accent-gradient);
    color: white;
    border-color: var(--accent-primary);
}


/* --- Debug Modal Enhancements --- */

/* Ensure all text inside the modal is fully visible */
#debug-modal .modal-content p,
#debug-modal .modal-content h3 {
    color: var(--text-primary) !important;
}

/* Style the labels to make them stand out as clear titles */
#debug-modal label {
    color: var(--accent-primary) !important;
    font-weight: 600 !important;
    font-size: 1rem !important;
    padding-top: 1rem;
    border-top: 1px solid var(--border-primary);
    display: block;
}

/* Add a bit more space to the first label to separate it from the intro paragraph */
#debug-modal div:first-of-type > label {
    margin-top: 1.5rem;
}

/* Remove the top border from the very first label */
#debug-modal div:first-of-type > label {
    border-top: none;
    padding-top: 0;
}

/* Ensure the textarea placeholders are also clearly visible */
#debug-modal textarea::placeholder {
    color: var(--text-tertiary);
    opacity: 1;
}
/* --- CLEAN Step Execution Summary Card --- */
body .step-summary {
    background: var(--bg-secondary, #f8f9fa) !important;
    border: 1px solid var(--border-primary, #e1e5e9) !important;
    border-left: 4px solid var(--accent-primary, #007acc) !important;
    border-radius: 8px !important;
    margin: 16px 0 !important;
    padding: 20px !important;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05) !important;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
}

body .step-summary h5 {
    margin: 0 0 12px 0 !important;
    padding: 0 !important;
    font-size: 1.1em !important;
    font-weight: 600 !important;
    color: var(--text-primary, #1a1a1a) !important;
    display: flex !important;
    align-items: center !important;
    gap: 8px !important;
}

body .step-summary h5::before {
    content: "✓" !important;
    color: var(--success-color, #28a745) !important;
    font-weight: bold !important;
}

.step-summary .ai-summary-quote {
    font-style: italic !important;
    color: var(--text-secondary, #666) !important;
    margin: 0 0 16px 0 !important;
    padding: 12px 0 12px 16px !important;
    font-size: 1em !important;
    line-height: 1.5 !important;
    border-left: 3px solid var(--accent-secondary, #6c757d) !important;
    background: var(--bg-primary, #ffffff) !important;
    border-radius: 0 4px 4px 0 !important;
}

.step-summary .file-info {
    background: var(--bg-primary, #ffffff) !important;
    border: 1px solid var(--border-primary, #e1e5e9) !important;
    border-radius: 6px !important;
    padding: 12px 16px !important;
    margin: 8px 0 !important;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif !important;
}

.step-summary .file-info:first-of-type {
    margin-top: 0 !important;
}

.step-summary .file-info:last-of-type {
    margin-bottom: 0 !important;
}

.step-summary .file-info strong {
    color: var(--text-primary, #1a1a1a) !important;
    font-weight: 600 !important;
    margin-right: 8px !important;
}

.step-summary .file-info .change-indicator {
    color: var(--success-color, #28a745) !important;
    font-weight: bold !important;
    margin: 0 8px !important;
}

.step-summary .file-info .change-indicator.negative {
    color: var(--error-color, #dc3545) !important;
}

.step-summary .file-info .file-stats {
    color: var(--text-muted, #6c757d) !important;
    font-size: 0.85em !important;
    margin-top: 4px !important;
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace !important;
}

.step-summary .file-info .file-stats.positive {
    color: var(--success-color, #28a745) !important;
}

.step-summary .file-info .file-stats.negative {
    color: var(--error-color, #dc3545) !important;
}

.step-summary code {
    background: var(--bg-accent, #e9ecef) !important;
    color: var(--text-primary, #1a1a1a) !important;
    padding: 4px 8px !important;
    border-radius: 4px !important;
    font-size: 0.85em !important;
    border: 1px solid var(--border-primary, #dee2e6) !important;
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace !important;
}

/* Remove any old styles that might conflict */
body .step-summary p {
    margin: 0 0 8px 0 !important;
    color: var(--text-secondary, #666) !important;
    line-height: 1.5 !important;
    float: none !important;
    text-align: left !important;
}

.step-summary ul,
.step-summary li {
    margin: 0 !important;
    padding: 0 !important;
    list-style: none !important;
}

#debug-error-input{width:100% !important}
#debug-prompt-input{width:100% !important}
/* --- Execution Choice Modal Fixes --- */
#execution-choice-modal .settings-item h3 {
    display: flex;
    align-items: center;
    gap: 0.75rem;
}
#execution-choice-modal .execution-icon {
    color: #fbbf24; /* Yellow color for the icons */
    flex-shrink: 0;
}
#execution-choice-modal .settings-item p {
    line-height: 1.6 !important; /* Improved line height for readability */
}
@media (max-width: 768px) {
    #execution-choice-modal .execution-style-grid {
        grid-template-columns: 1fr !important; /* Stack columns on mobile */
    }
}


.one-shot-icon-modal {
    color: #fbbf24; /* Yellow color */
    flex-shrink: 0;
}

/* --- Polish Modal Fixes (Comprehensive) --- */
.polish-modal-title {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
	line-height:1.3;
    color: var(--text-primary);
}
.polish-title-icon {
    color: #fbbf24; /* Yellow */
    flex-shrink: 0;
}
.polish-option h4 {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}
.polish-option p {
    line-height: 1.6 !important;
}
.polish-option-icon { flex-shrink: 0; }
.polish-option-icon.modern { color: var(--accent-secondary); }
.polish-option-icon.mobile { color: #60a5fa; } /* Blue */
.polish-option-icon.accessibility { color: #34d399; } /* Green */
.polish-option-icon.performance { color: #f472b6; } /* Pink */
#btn-start-polish {
    gap: 0.5rem;
}

@media (max-width: 768px) {
    .polish-section-grid,
    .polish-options {
        grid-template-columns: 1fr !important;
    }
    .polish-features-grid {
        grid-template-columns: 1fr !important;
    }
    .polish-intensity-options {
        flex-direction: column !important;
        align-items: flex-start !important;
        gap: 0.75rem !important;
    }
}
/* --- Polish Modal Mobile Layout Fix --- */
@media (max-width: 768px) {
    #polish-modal .modal-buttons {
        flex-direction: column;
        gap: 0.75rem;
    }
    #polish-modal .modal-buttons .modal-button,
    #polish-modal .modal-buttons .modal-button-secondary {
        width: 100%; /* Make buttons full-width */
    }
    .polish-intensity {
        text-align: left; /* Ensure the entire section is left-aligned */
    }
    .polish-intensity-options {
        flex-direction: column !important;
        align-items: flex-start !important;
        gap: 1rem !important; /* Increase gap for better touch targets */
    }
}
/* --- Polish Modal & Results Fixes (Comprehensive) --- */
.polish-modal-title { display: flex; align-items: center; justify-content: center; gap: 0.5rem; color: var(--text-primary); }
.polish-title-icon { color: #fbbf24; flex-shrink: 0; }
.polish-option h4 { display: flex; align-items: center; gap: 0.5rem; }
.polish-option p { line-height: 1.6 !important; }
.polish-option-icon { flex-shrink: 0; }
.polish-option-icon.modern { color: var(--accent-secondary); }
.polish-option-icon.mobile { color: #60a5fa; }
.polish-option-icon.accessibility { color: #34d399; }
.polish-option-icon.performance { color: #f472b6; }
#btn-start-polish { gap: 0.5rem; }

.polish-summary-grid { display: grid; grid-template-columns: 1fr 1fr; gap: 1.5rem; margin: 1.5rem 0; text-align: left; }
.polish-summary-header { display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.75rem; font-size: 1.1rem; }
.polish-summary-header.functionality { color: #60a5fa; }
.polish-summary-header.visual { color: #a78bfa; }
.polish-summary-icon { flex-shrink: 0; }
.polish-summary-list { list-style: none; padding-left: 0; margin: 0; font-size: 0.95rem; }
.polish-summary-list li { display: flex; align-items: flex-start; gap: 0.6rem; margin-bottom: 0.6rem; color: var(--text-secondary); line-height: 1.5; }
.check-icon-box {
    flex-shrink: 0;
    margin-top: 4px;
    width: 16px;
    height: 16px;
    background-color: #34d399;
    border-radius: 4px;
    display: flex;
    align-items: center;
    justify-content: center;
}
.check-icon-box::before {
    content: '✔';
    color: var(--bg-primary);
    font-size: 12px;
    font-weight: bold;
}
.polish-files-summary { background: var(--bg-accent); padding: 1rem; border-radius: var(--radius-md); margin: 1.5rem 0; }
.polish-files-header { margin: 0 0 0.75rem 0; color: var(--text-primary); font-weight: 600; }
.polish-files-summary p { margin: 0.25rem 0; font-size: 0.9rem; }
.polish-files-summary code { background: rgba(0,0,0,0.3); padding: 0.2rem 0.4rem; border-radius: 4px; }
.polish-footer-note { margin-top: 1.5rem; font-weight: 600; color: var(--text-primary); display: flex; align-items: center; gap: 0.5rem; justify-content: center; }
.polish-footer-note svg { color: var(--accent-primary); }
.polish-footer-subnote { margin-top: 0.5rem; font-size: 0.9rem; color: var(--text-tertiary); text-align: center; }
#polish-results-modal .modal-buttons button { gap: 0.5rem; }

@media (max-width: 768px) {
    .polish-options, .polish-features-grid, .polish-summary-grid {
        grid-template-columns: 1fr !important;
    }
    .polish-intensity-options {
        flex-direction: column !important;
        align-items: flex-start !important;
        gap: 1rem !important;
    }
    #polish-modal .modal-buttons, #polish-results-modal .modal-buttons {
        flex-direction: column;
        gap: 0.75rem;
    }
    #polish-modal .modal-button, #polish-modal .modal-button-secondary,
    #polish-results-modal .modal-button, #polish-results-modal .modal-button-secondary {
        width: 100%;
    }
}
/* --- Website Generated Modal Fix (Definitive) --- */
.website-generated-title {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
    font-size: 1.75rem; /* Match the visual style */
}
.website-generated-icon {
    color: #fbbf24 !important; /* Yellow color */
    font-size: 28px;
    /* Crucially, we must override the gradient fill */
    -webkit-text-fill-color: #fbbf24 !important;
    background: none !important;
}
#btn-view-website-link {
    text-decoration: none !important;
    color: white !important;
}

@media (max-width: 768px) {
    #view-website-modal .modal-buttons {
        flex-direction: column;
        gap: 0.75rem;
    }
    #view-website-modal .modal-button,
    #view-website-modal .modal-button-secondary {
        width: 100%; /* Make buttons full-width on mobile */
    }
}

/* --- Step Selection Modal Fixes (Comprehensive) --- */
.step-selection-title {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
    color: var(--text-primary) !important;
}
.step-selection-icon {
    color: #fbbf24 !important; /* Yellow color */
    font-size: 24px;
    /* Crucially, we must override the gradient fill */
    -webkit-text-fill-color: #fbbf24 !important;
    background: none !important;

}
@media (max-width: 768px) {
    #step-selection-modal .step-options {
        grid-template-columns: 1fr !important;
    }
    #step-selection-modal .modal-buttons {
        flex-direction: column;
        gap: 0.75rem;
    }
    #step-selection-modal .modal-button,
    #step-selection-modal .modal-button-secondary {
        width: 100%;
    }
}





/* --- Specification Modal Fixes (Comprehensive) --- */
.spec-modal-title {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.75rem;
    line-height: 1.3 !important;
}
.spec-modal-icon {
    color: #fbbf24 !important; /* Yellow */
    font-size: 26px;
    -webkit-text-fill-color: #fbbf24 !important;
    background: none !important;
}
.spec-textarea {
    width: 100%;
    height: 120px; /* Set a fixed default height */
    padding: 0.75rem;
    border: 1px solid var(--border-primary);
    border-radius: var(--radius-md);
    background: var(--bg-tertiary);
    color: var(--text-primary);
    font-family: var(--font-sans);
    resize: vertical; /* Allow vertical resizing */
    /* overflow-y: hidden is removed, scrollbar will now appear when needed */
}

@media (max-width: 768px) {
    #specification-modal .modal-buttons {
        flex-direction: column;
        align-items: stretch !important;
        gap: 0.75rem;
    }
    #specification-modal .modal-buttons > div {
        display: flex;
        flex-direction: column;
        width: 100%;
        gap: 0.75rem;
    }
    #specification-modal .modal-button,
    #specification-modal .modal-button-secondary {
        width: 100%;
        justify-content: center;
    }
}

#one-shot-input::-webkit-scrollbar-corner, 
.file-tree::-webkit-scrollbar-corner, 
.chat-messages::-webkit-scrollbar-corner, 
.code-area::-webkit-scrollbar-corner, 
#chat-input::-webkit-scrollbar-corner,
#image-plan-prompt::-webkit-scrollbar-corner,
.spec-textarea::-webkit-scrollbar-corner,
body::-webkit-scrollbar-corner {
    background: transparent !important;
    visibility: hidden !important;
    opacity: 0 !important;
    display: none !important;
}
/* --- Version Control Modal Fixes (Comprehensive) --- */
.version-control-title {
    color: var(--accent-primary) !important;
    background: none !important;
    -webkit-text-fill-color: var(--accent-primary) !important;
}
.version-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 1rem;
    margin-bottom: 0.75rem;
    background: var(--bg-tertiary);
    border-radius: var(--radius-md);
    border-left: 4px solid;
    transition: all 0.2s ease;
}
.version-item:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
}
.version-info { display: flex; align-items: center; gap: 1rem; }
.version-icon { font-size: 1.2rem; color: var(--text-secondary); }
.version-name { color: var(--text-primary); font-weight: 600; }
.version-date { color: var(--text-tertiary); font-size: 0.8rem; }
.version-actions { display: flex; align-items: center; gap: 0.75rem; }
.version-type-badge { font-size: 0.75rem; padding: 0.2rem 0.6rem; border-radius: 12px; font-weight: 600; }
.version-type-badge.auto { background: #fbbf24; color: #000; }
.version-type-badge.manual { background: var(--accent-primary); color: #fff; }
.version-actions .icon-button { width: 36px; height: 36px; border-radius: 8px; }
.version-actions .icon-button.restore { background: var(--success); color: white; }
.version-actions .icon-button.delete { background: var(--error); color: white; }

@media (max-width: 768px) {
    .version-control-actions {
        flex-direction: column;
        align-items: stretch;
    }
    .version-item {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.75rem;
    }
    .version-actions {
        width: 100%;
        justify-content: space-between;
    }
}
/* --- Debug Modal Mobile Layout Fix --- */
@media (max-width: 768px) {
    #debug-modal .modal-buttons {
        flex-direction: column;
        align-items: stretch; /* Allows buttons to stretch to full width */
        gap: 0.75rem;
    }
    #debug-modal .modal-button,
    #debug-modal .modal-button-secondary {
        width: 100%; /* Make buttons full-width */
    }
}
@media (max-width: 768px) {
    #one-shot-modal .modal-buttons {
        flex-direction: column;
        align-items: stretch; /* Make buttons take up full width */
        gap: 0.75rem;
    }
    #one-shot-modal .modal-button,
    #one-shot-modal .modal-button-secondary {
        width: 100%;
        justify-content: center;
    }
}


.zenlogo{color:#fff !important}

.fa-code{color:#b692ff;
}


/* Glow effect for the "AI" text */
.ai-glow {
  position: relative;
  color: #ffffff;
  text-shadow:
    0 0 4px rgba(255, 255, 255, 0.6),
    0 0 10px rgba(255, 255, 255, 0.3);
  animation: glowPulse 2.5s ease-in-out infinite;
}

/* Subtle pulsating glow animation */
@keyframes glowPulse {
  0%, 100% {
    text-shadow:
      0 0 4px rgba(255, 255, 255, 0.5),
      0 0 10px rgba(255, 255, 255, 0.3);
    opacity: 0.9;
  }
  50% {
    text-shadow:
      0 0 8px rgba(255, 255, 255, 0.9),
      0 0 20px rgba(255, 255, 255, 0.6);
    opacity: 1;
  }
}

   /* ADD these new styles for CodeMirror */
    #code-area { display: none; } /* Hide original textarea */

    .CodeMirror {
        font-family: var(--font-mono) !important;
        font-size: 14px !important;
        height: 100% !important;
        width: 100% !important;
        position: absolute !important;
        top: 0; left: 0; right: 0; bottom: 0;
    }

    .cm-s-dracula.CodeMirror {
        background: var(--bg-primary) !important;
        color: #f8f8f2 !important;
    }
    .cm-s-dracula .CodeMirror-gutters {
        background: var(--bg-primary) !important;
        border-right: 1px solid var(--border-primary) !important;
    }
    .cm-s-dracula .CodeMirror-cursor {
        border-left: 1px solid #f8f8f2 !important;
    }
    .cm-s-dracula .CodeMirror-selected {
        background: rgba(255, 255, 255, 0.15) !important;
    }
    .cm-s-dracula .CodeMirror-line::selection, 
    .cm-s-dracula .CodeMirror-line > span::selection, 
    .cm-s-dracula .CodeMirror-line > span > span::selection {
        background: rgba(255, 255, 255, 0.15) !important;
    }
    .cm-s-dracula .CodeMirror-activeline-background {
        background: rgba(255, 255, 255, 0.05) !important;
    }
 `;
    const styleEl = document.createElement('style');
    styleEl.textContent = css;
    document.head.appendChild(styleEl);
},
  

 injectHTML() {
    rootElement.innerHTML = `
        <div class="zen-app-subnav">
            <ul>
                <li class="nav-left">
                    <a href="${zencode_ai_app_vars.nav_links.home}">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"><polyline points="15 18 9 12 15 6"></polyline></svg>
                        Home
                    </a>
                </li>
                <li class="nav-center">
                    <a href="${zencode_ai_app_vars.nav_links.user_manual}">
                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M2 3h6a4 4 0 0 1 4 4v14a3 3 0 0 0-3-3H2z"></path><path d="M22 3h-6a4 4 0 0 0-4 4v14a3 3 0 0 1 3-3h7z"></path></svg>
                        User Manual
                    </a>
                </li>
                <li class="nav-right">
                    <a href="${zencode_ai_app_vars.nav_links.dashboard}">Dashboard</a>
                </li>
            </ul>
        </div>
        <div class="zencode-main-wrapper">
            <div id="mobile-overlay"></div>
         
			
			
			 <header class="header">
                <div class="header-left"><button id="sidebar-toggle-header" class="sidebar-toggle-header" title="Toggle Project Panel"><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect><line x1="9" y1="3" x2="9" y2="21"></line></svg></button></div>
                <div class="header-title">
				<div class="header-title" id="header-project-name" style="display:none !important"></div>
  <i class="fa-solid fa-code"></i>
  <span class="zenlogo">ZenCode <span class="ai-glow">AI</span></span>
</div>
                <div class="header-right"><div id="status-container" class="desktop-status-indicator" style="display:flex; align-items:center; gap: 0.5rem;"><span id="api-status" class="status-indicator"></span><span id="api-status-text" style="font-size: 0.85rem; color: var(--text-secondary);">AI Ready</span></div></div>
            </header>
			
			
			
			
			
            <main class="main-layout">
                <div class="sidebar" id="sidebar">
                    <div class="project-header">
					
					
      
                        <div class="project-title-bar"><span>Project</span><span id="save-status" class="save-status"></span><div class="project-actions"><button id="btn-projects" class="icon-button" title="Manage Projects"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"></path></svg></button><button id="btn-new-folder" class="icon-button" title="New Folder"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M4 20h16a2 2 0 0 0 2-2V8a2 2 0 0 0-2-2h-7.93a2 2 0 0 1-1.66-.9l-.82-1.2A2 2 0 0 0 7.93 3H4a2 2 0 0 0-2 2v13c0 1.1.9 2 2 2z"></path><line x1="12" y1="11" x2="12" y2="17"></line><line x1="9" y1="14" x2="15" y2="14"></line></svg></button><button id="btn-settings" class="icon-button" title="Project Actions"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="3"></circle><path d="M19.4 15a1.65 1.65 0 0 0 .33 1.82l.06.06a2 2 0 0 1 0 2.83 2 2 0 0 1-2.83 0l-.06-.06a1.65 1.65 0 0 0-1.82-.33 1.65 1.65 0 0 0-1 1.51V21a2 2 0 0 1-2 2 2 2 0 0 1-2-2v-.09A1.65 1.65 0 0 0 9 19.4a1.65 1.65 0 0 0-1.82.33l-.06.06a2 2 0 0 1-2.83 0 2 2 0 0 1 0-2.83l.06-.06a1.65 1.65 0 0 0 .33-1.82 1.65 1.65 0 0 0-1.51-1H3a2 2 0 0 1-2-2 2 2 0 0 1 2-2h.09A1.65 1.65 0 0 0 4.6 9a1.65 1.65 0 0 0-.33-1.82l-.06-.06a2 2 0 0 1 0-2.83 2 2 0 0 1 2.83 0l.06.06a1.65 1.65 0 0 0 1.82.33H9a1.65 1.65 0 0 0 1-1.51V3a2 2 0 0 1 2-2 2 2 0 0 1 2 2v.09a1.65 1.65 0 0 0 1 1.51 1.65 1.65 0 0 0 1.82-.33l.06-.06a2 2 0 0 1 2.83 0 2 2 0 0 1 0 2.83l-.06-.06a1.65 1.65 0 0 0-.33 1.82V9a1.65 1.65 0 0 0 1.51 1H21a2 2 0 0 1 2 2 2 2 0 0 1-2 2h-.09a1.65 1.65 0 0 0-1.51 1z"></path></svg></button><input type="file" id="file-upload-input" multiple style="display:none;" /><input type="file" id="zip-upload-input" accept=".zip" style="display:none;" /></div></div>
                        <input type="text" id="project-name-input" class="project-name-input" placeholder="Project Name">
                    </div>
					
					
                    <div class="file-tree" id="file-tree"></div>
                </div>
				
				
          <!-- REPLACE your entire <div class="content-area"> block with this FINAL version -->
<div class="content-area">
    <div class="feature-bar"><div class="feature-bar-main"><p class="feature-bar-text">A one-shot prompt to generate a complete, single-page website.</p><button id="btn-show-one-shot" class="build-website-button">🚀 Build a Website</button></div></div>
    <div class="panels-container">
        <div class="panel code-editor-panel" id="code-panel">
            <div class="panel-header">
                <span id="editor-filename">Editor</span>
                <div class="editor-actions">
                    <button id="btn-accept-changes" class="diff-btn accept" style="display:none;">Accept</button>
                    <button id="btn-reject-changes" class="diff-btn reject" style="display:none;">Reject</button>
                    <button id="btn-view-generated-website" class="icon-button" title="View Generated Website" style="display: none;"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg></button>
                    <button id="btn-save-file" class="icon-button" title="Download File" style="display: none;"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="7 10 12 15 17 10"></polyline><line x1="12" y1="15" x2="12" y2="3"></line></svg></button>
                </div>
            </div>
            <div class="editor-content">
                <textarea class="code-area" id="code-area" spellcheck="false"></textarea>
                <div class="code-preview-area" id="code-preview" style="display:none;"></div>
                
                <!-- THIS IS THE NEW, CORRECT LOCATION FOR THE DEBUG BUTTON -->
                <button id="debug-button" title="Add Debugging Info" class="icon-button">
                     <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M15 13H9V7c0-1.7 1.3-3 3-3s3 1.3 3 3v6Z"/>
        <path d="M12 20c-3.3 0-6-2.7-6-6v-3h12v3c0 3.3-2.7 6-6 6Z"/>
        <path d="M4 11h2"/>
        <path d="M18 11h2"/>
        <path d="M15 8h4"/>
        <path d="M5 8h4"/>
    </svg>
                </button>
            </div>
        </div>
        <div class="resizer" id="resizer"></div>
        <div class="panel chat-panel">
            <div class="panel-header">
                <div class="chat-panel-header-left">
                    <span>AI Assistant</span>
                    <div id="mobile-status-container">
                        <span id="mobile-api-status" class="status-indicator"></span>
                        <span id="mobile-api-status-text" style="font-size: 0.85rem; color: var(--text-secondary);">AI Ready</span>
                    </div>
                </div>
                <!-- The button has been removed from here -->
                <div class="chat-panel-header-right">
                    <div class="mode-toggle-container">
                        <label for="mode-toggle-switch">AI Mode</label>
                        <button id="mode-toggle-switch">Normal</button>
                    </div>
                </div>
            </div>
            <div class="chat-messages" id="chat-messages"></div>
            <div class="chat-input-container">
                <div id="error-input-container" style="display: none; width: 100%; margin-bottom: 8px; flex-basis: 100%;">
                    <textarea id="error-input" class="chat-input" rows="3" placeholder="Paste console error or debugging information here..."></textarea>
                </div>
                <textarea id="chat-input" class="chat-input" rows="1"></textarea>
                <button id="voice-button" class="voice-button">
                     <svg width="20" height="20" viewBox="0 0 24 24" fill="white"><path d="M12 14c1.66 0 3-1.34 3-3V5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3z"></path><path d="M17 11c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z"></path></svg>
                </button>
                   <!-- === START: UPDATED BUTTON === -->
                         <button id="send-button" class="send-button">
  <!-- The "Send" state (Default) -->
  <div class="send-button-content">
    <span class="runsa">Run</span>
    <div class="shortcut-keys">
      <kbd>Ctrl</kbd>
      <kbd class="arrow-key">
  <svg viewBox="0 0 24 24" width="18" height="18" stroke="currentColor" fill="none" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
    <path d="M9 7v4a3 3 0 0 0 3 3h6" />
    <path d="M15 10l3 4-3 4" />
  </svg>
</kbd>
    </div>
  </div>

  <!-- The "Stop" state (Hidden by default) -->
  <div class="stop-button-content" style="display: none;">
    <svg width="20" height="20" viewBox="0 0 24 24" fill="none"
         xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
      <!-- This group will contain everything that spins -->
      <g class="spinner-group">
        <circle cx="12" cy="12" r="10" stroke="#FFFFFF" stroke-opacity="0.3" stroke-width="3"/>
        <path class="countdown-arc"
              d="M12 2 a10 10 0 0 1 0 20 a10 10 0 0 1 0 -20"
              stroke="#FFFFFF" stroke-width="3" stroke-linecap="round"/>
      </g>
      <!-- The square is outside the group, so it won't spin -->
      <rect x="9" y="9" width="6" height="6" rx="1" fill="white"/>
    </svg>
    <span>Stop</span>
  </div>
</button>
            </div>
        </div>
    </div>
</div>

<!-- Progress Bar - Properly Integrated in Layout -->
<div id="footer-progress-bar" class="footer-progress-bar" style="display: none;">
    <div class="footer-progress-content">
        <div class="footer-progress-info">
            <div style="display: flex; align-items: center; gap: 0.75rem;">
                <span class="footer-progress-title">Executing Plan</span>
                <span class="footer-progress-step">Step <span id="footer-current-step">1</span> of <span id="footer-total-steps">5</span></span>
            </div>
            <button id="close-footer-progress" class="footer-progress-close" title="Close progress bar">×</button>
        </div>
        <div class="footer-progress-details">
            <span id="footer-progress-text" class="footer-progress-text">Initializing...</span>
            <span id="footer-progress-percentage" class="footer-progress-percentage">0%</span>
        </div>
        <div class="footer-progress-bar-track">
            <div id="footer-progress-bar-fill" class="footer-progress-bar-fill" style="width: 0%;"></div>
        </div>
    </div>
</div>

            </main>
        </div>
        <div id="one-shot-modal" class="modal-overlay">
            <div class="modal-content">
                <h3 style="display: flex; align-items: center; justify-content: center; gap: 0.75rem; color: var(--text-primary);">
    <svg class="one-shot-icon-modal" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
        <path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/>
        <path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/>
    </svg>
    <span  style="line-height:1.3">One-Shot Website Builder</span>
</h3>
                <p>Describe the website you want in detail. For more accurate colors and style, you can upload an inspirational image.</p>
                <div class="textarea-wrapper">
                    <textarea id="one-shot-input" placeholder="Describe your website..."></textarea>
                    <button id="btn-refine-prompt" class="icon-button" title="Refine prompt for better results">
                        <svg class="refine-icon" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 3a6 6 0 0 0 9 9 9 9 0 1 1-9-9Z"></path></svg>
                        <svg class="loading-spinner" style="display:none;" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><line x1="12" y1="2" x2="12" y2="6"></line><line x1="12" y1="18" x2="12" y2="22"></line><line x1="4.93" y1="4.93" x2="7.76" y2="7.76"></line><line x1="16.24" y1="16.24" x2="19.07" y2="19.07"></line><line x1="2" y1="12" x2="6" y2="12"></line><line x1="18" y1="12" x2="22" y2="12"></line><line x1="4.93" y1="19.07" x2="7.76" y2="16.24"></line><line x1="16.24" y1="7.76" x2="19.07" y2="4.93"></line></svg>
                    </button>
                </div>
                <div id="prompt-length-warning" class="modal-feedback error" style="display: none; text-align: left; margin-bottom: 1.5rem;"></div>
                <div id="image-upload-area">
                    <div id="image-upload-prompt">
                        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin: 0 auto 0.5rem; color: var(--text-tertiary);"><path d="M21.2 15c.7-1.2 1-2.5.7-3.9-.6-2.8-3.3-4.4-6.1-3.4-.6.2-1.1.5-1.6.9-.4-.3-.8-.6-1.3-.8-2.6-1.3-5.5.3-6.1 3-.3 1.2 0 2.5.7 3.7m13.8 0c.2.6.3 1.3.1 1.9-.5 1.7-2.1 2.8-3.8 2.3-.6-.2-1.1-.5-1.6-.9-.4.3-.8.6-1.2.8-1.8 1-4-.2-4.5-2.1-.3-1 .1-2 .6-2.9m10.4.1c-.4.8-1 1.5-1.8 1.9m-8.4-1.9c.8-.4 1.4-1.1 1.8-1.9"></path><path d="M12 12.5a3.5 3.5 0 1 1 0-7 3.5 3.5 0 0 1 0 7z"></path></svg>
                        <p>Drag & Drop an image or click to upload</p>
                    </div>
                    <div id="image-preview-container" style="display: none;"><img id="image-preview" src="" alt="Image Preview"><button id="remove-image-btn" title="Remove Image">×</button></div>
                </div>
                <input type="file" id="image-upload-input" accept="image/png, image/jpeg, image/webp" style="display:none;" />
                <div class="modal-buttons"><button id="btn-close-modal" class="modal-button-secondary">Cancel</button><button id="btn-generate-website" class="modal-button">Generate Website</button></div>
            </div>
        </div>
<div id="view-website-modal" class="modal-overlay">
    <div class="modal-content">
        <h2 class="website-generated-title">
            <i class="fas fa-rocket website-generated-icon"></i>
            <span class="gradient-texty" style="line-height:1.3">Website Generated!</span> 
        </h2>
        <p>Your new website is ready. Click the button below to open it in a new tab for a live preview.</p>
        <div class="modal-buttons">
            <button id="btn-close-view-modal" class="modal-button-secondary">Close</button>
            <a id="btn-view-website-link" href="#" target="_blank" class="modal-button" onclick="App.toggleViewWebsiteModal(false)">
                <i class="fas fa-rocket" style="margin-right: 0.5rem;"></i>
                View Website
            </a>
        </div>
    </div>
</div>
        <div id="confirmation-modal" class="confirm-modal-overlay">
            <div class="confirm-modal-content"><p id="confirmation-message">Are you sure?</p><div class="confirm-modal-buttons"><button id="btn-confirm-cancel" class="modal-button-secondary">Cancel</button><button id="btn-confirm-ok" class="modal-button">OK</button></div></div>
        </div>
        <div id="projects-modal" class="modal-overlay">
            <div class="modal-content">
                <h2>Project Manager</h2><p>Load a previous project, create a new one, or delete old ones.</p>
                <div id="projects-modal-feedback" class="modal-feedback"></div>
                <div class="modal-buttons"><button id="btn-new-project" class="modal-button">✨ New Project</button></div>
                <ul id="projects-list" class="projects-modal-list"></ul>
                <div id="project-list-controls" class="modal-buttons" style="margin-top: 1rem; justify-content: center;"></div>
                <div class="modal-buttons"><button id="btn-close-projects-modal" class="modal-button-secondary">Close</button></div>
            </div>
        </div>
        <div id="settings-modal" class="modal-overlay">
            <div class="modal-content">
                <h2>Project Actions</h2><p>Manage your project files and assets.</p>
                <div class="settings-grid">
                    
					<div id="btn-upload-files-modal" class="settings-item"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="17 8 12 3 7 8"></polyline><line x1="12" y1="3" x2="12" y2="15"></line></svg><span>Upload Files</span></div>
                        <!-- ADD THIS NEW POLISH TOOL ITEM -->
    <div id="btn-show-polish" class="settings-item">
        <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
            <path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/>
        </svg>
        <span>Polish Project</span>
    </div>
					
					<div id="btn-unzip-modal" class="settings-item"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21.44 11.05l-9.19 9.19a6 6 0 0 1-8.49-8.49l9.19-9.19a4 4 0 0 1 5.66 5.66l-9.2 9.19a2 2 0 0 1-2.83-2.83l8.49-8.48"></path></svg><span>Upload ZIP</span></div>
                    <!-- NEW: Add this BEFORE the download zip button: -->
<div id="btn-version-control" class="settings-item">
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M3 12a9 9 0 1 0 9-9 9.75 9.75 0 0 0-6.74 2.74L3 8"/>
        <path d="M3 3v5h5"/>
    </svg>
    <span>Version Control</span>
</div>
<div id="btn-save-version" class="settings-item">
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M19 21H5a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h11l5 5v11a2 2 0 0 1-2 2z"/>
        <polyline points="17 21 17 13 7 13 7 21"/>
        <polyline points="7 3 7 8 15 8"/>
    </svg>
    <span>Save Version</span>
</div>
					
					<div id="btn-download-zip-modal" class="settings-item"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 15v4a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2v-4"></path><polyline points="7 10 12 15 17 10"></polyline><line x1="12" y1="15" x2="12" y2="3"></line></svg><span>Download as ZIP</span></div>
					<div id="btn-show-learning-stats" class="settings-item">
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M3 3v18h18"/>
        <path d="M18 17V9"/>
        <path d="M13 17V5"/>
        <path d="M8 17v-3"/>
    </svg>
    <span>View Learning Stats</span>
</div>
                </div>
                <div class="settings-actions-destructive"><div id="btn-clear-kb-modal" class="settings-item settings-item-destructive"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M2.5 4.5h19"/><path d="M11.5 9.5v5"/><path d="M15.5 9.5v5"/><path d="M6.5 4.5v15c0 1.1.9 2 2 2h7c1.1 0 2-.9 2-2v-15"/><path d="M9.5 4.5l3-3 3 3"/></svg><span>Clear Knowledge Base</span></div>
				
				<div id="btn-cleanup-learnings" style="margin-left:15px; display:none !important" class="settings-item settings-item-destructive">
    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M3 6h18"/>
        <path d="M19 6v14c0 1-1 2-2 2H7c-1 0-2-1-2-2V6"/>
        <path d="M8 6V4c0-1 1-2 2-2h4c1 0 2 1 2 2v2"/>
        <line x1="10" y1="11" x2="10" y2="17"/>
        <line x1="14" y1="11" x2="14" y2="17"/>
    </svg>
    <span>Cleanup Old Learnings</span>
</div>
				
				
				</div>
               
 <div class="modal-buttons" style="margin-top: 1.5rem;"><button id="btn-close-settings-modal" class="modal-button-secondary">Close</button></div>
            </div>
        </div>
        <div id="one-shot-loading-overlay" style="display: none;">
            <div class="loading-content">
                <svg class="magic-spinner" width="80" height="80" viewBox="0 0 100 100">
                    <defs><linearGradient id="magic-gradient" x1="0%" y1="0%" x2="100%" y2="100%"><stop offset="0%" stop-color="var(--accent-primary)"/><stop offset="100%" stop-color="var(--accent-secondary)"/></linearGradient></defs>
                    <path d="M 50,50 m 0,-45 a 45,45 0 1 1 0,90 a 45,45 0 1 1 0,-90" stroke="url(#magic-gradient)" stroke-width="10" fill="none" stroke-linecap="round" stroke-dasharray="141.37" stroke-dashoffset="282.74"><animateTransform attributeName="transform" type="rotate" from="0 50 50" to="360 50 50" dur="2s" repeatCount="indefinite" /><animate attributeName="stroke-dashoffset" from="282.74" to="0" dur="1.5s" repeatCount="indefinite" /></path>
                </svg>
                <p id="loading-text-anim">Getting started...</p>
                <button id="cancel-generation-btn" class="modal-button-secondary">Cancel</button>
            </div>
        </div>
        <div id="image-preview-modal" class="modal-overlay">
            <div class="image-modal-content">
                <span id="close-image-modal" class="close-image-modal-btn">&times;</span>
                <img id="image-modal-preview-area" src="" alt="Image Preview" style="min-height:400px; background-color:#111318; border: 1px solid #2a2e38">
             <div id="image-modal-controls">
    <div id="image-plan-prompt-container" style="margin-bottom: 1rem; display: none;">
       <textarea id="image-plan-prompt" placeholder="Describe what you want to build based on this image (optional)&#10;e.g., 'Create an e-commerce site for books with this color scheme' or 'Build a restaurant website inspired by this layout'" style="width: 100%; min-height: 80px; padding: 0.75rem; border: 1px solid var(--border-primary); border-radius: var(--radius-md); background: var(--bg-tertiary); color: var(--text-primary); font-family: var(--font-sans); font-size: 0.9rem; resize: none;"></textarea>
	</div>
    <button id="btn-describe-image" class="modal-button-secondary">Describe Image</button>
    <button id="btn-plan-from-image" class="modal-button">Create Plan from Image</button>
</div>
		   </div>
        </div>
<!-- Polish Tool: Main Modal -->
<div id="polish-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 800px;">
        <h2 class="polish-modal-title">
            <svg class="polish-title-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path></svg>
            <svg class="polish-title-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/><path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/></svg>
            Integration & Polish Tool
        </h2>
        <p>Transform your project into a fully integrated, professionally polished website. This tool fixes functionality issues AND adds stunning visual design.</p>
        
        <div class="polish-section-grid" style="background: var(--bg-accent); padding: 1rem; border-radius: var(--radius-md); margin: 1.5rem 0;">
            <h4 style="margin: 0 0 0.5rem 0; color: var(--accent-primary); font-size:18px !important">What This Tool Does:</h4>
            <div class="polish-features-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; font-size: 0.9rem;">
                <div>
                    <strong>Integration Fixes:</strong>
                    <ul style="margin: 0.5rem 0; padding-left: 1.2rem;">
                        <li>Connects broken step-by-step code</li>
                        <li>Fixes missing functions & references</li>
                        <li>Resolves CSS conflicts</li>
                        <li>Ensures everything works together</li>
                    </ul>
                </div>
                <div>
                    <strong>Visual Polish:</strong>
                    <ul style="margin: 0.5rem 0; padding-left: 1.2rem;">
                        <li>Modern design enhancements</li>
                        <li>Professional visual effects</li>
                        <li>Mobile optimization</li>
                        <li>Accessibility improvements</li>
                    </ul>
                </div>
            </div>
        </div>
        
        <div class="polish-options" style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin: 1.5rem 0;">
            <div class="polish-option" data-polish-type="modern-ui">
                <h4><svg class="polish-option-icon modern" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="m12 3-1.45 4.1-3.85.4 2.95 2.5-.85 4L12 17l3.25 2 .85-4 2.95-2.5-3.85-.4L12 3z"/></svg>Modern UI Polish</h4>
                <p>Add glassmorphism, gradients, animations, and contemporary design elements.</p>
            </div>
            <div class="polish-option" data-polish-type="mobile-first">
                <h4><svg class="polish-option-icon mobile" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="5" y="2" width="14" height="20" rx="2" ry="2"></rect><line x1="12" y1="18" x2="12.01" y2="18"></line></svg>Mobile-First</h4>
                <p>Perfect mobile experience with optimized layouts and touch interactions.</p>
            </div>
            <div class="polish-option" data-polish-type="accessibility">
                <h4><svg class="polish-option-icon accessibility" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="10"></circle><path d="M12 6v6l4 2"></path></svg>Accessibility & UX</h4>
                <p>WCAG compliance, better contrast, focus states, and usability improvements.</p>
            </div>
            <div class="polish-option" data-polish-type="performance">
                <h4><svg class="polish-option-icon performance" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polygon points="13 2 3 14 12 14 11 22 21 10 12 10 13 2"></polygon></svg>Performance</h4>
                <p>Optimize CSS, reduce file sizes, and improve loading performance.</p>
            </div>
        </div>

        <div class="polish-intensity" style="margin: 1.5rem 0;">
            <label style="display: block; margin-bottom: 0.5rem; font-weight: 500; line-height:1.8; font-size:17px !important">Enhancement Intensity:</label>
            <div class="polish-intensity-options" style="display: flex; gap: 1.5rem; align-items: center;">
                <label><input type="radio" name="polish-intensity" value="subtle" checked> <strong>Subtle</strong> - Essential fixes + light polish</label>
                <label><input type="radio" name="polish-intensity" value="moderate"> <strong>Moderate</strong> - Good balance of fixes + design</label>
                <label><input type="radio" name="polish-intensity" value="dramatic"> <strong>Dramatic</strong> - Full transformation</label>
            </div>
        </div>
         <div style="background: rgba(139, 92, 246, 0.1); padding: 1rem; border-radius: var(--radius-md); border-left: 4px solid var(--accent-primary); margin: 1.5rem 0;">
            <p style="margin: 0; font-size: 0.95rem;"><strong>💡 Recommended:</strong> Start with <strong>Moderate</strong> intensity for the best balance of functionality fixes and visual improvements. You can always run it again with <strong>Dramatic</strong> for more polish!</p>
        </div>
        <div class="modal-buttons">
            <button id="btn-close-polish-modal" class="modal-button-secondary">Cancel</button>
            <button id="btn-start-polish" class="modal-button">
                <svg class="polish-title-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path></svg>
                <svg class="polish-title-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/><path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/></svg>
                Fix & Polish Project
            </button>
        </div>
    </div>
</div>

<div id="polish-progress-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 500px; text-align: center;">
        <h3>
             <svg class="polish-title-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/><path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/></svg>
          <span style="line-height:1.3">  Polishing Your Project</span>
        </h3>
        <div style="margin: 2rem 0;">
            <svg width="60" height="60" viewBox="0 0 100 100" style="display: block; margin: 0 auto;">
                <defs>
                    <linearGradient id="polish-gradient" x1="0%" y1="0%" x2="100%" y2="100%">
                        <stop offset="0%" stop-color="var(--accent-primary)"/>
                        <stop offset="100%" stop-color="var(--accent-secondary)"/>
                    </linearGradient>
                </defs>
                <circle cx="50" cy="50" r="40" stroke="url(#polish-gradient)" stroke-width="8" fill="none" stroke-linecap="round" stroke-dasharray="251.2" stroke-dashoffset="251.2">
                    <animateTransform attributeName="transform" type="rotate" from="0 50 50" to="360 50 50" dur="2s" repeatCount="indefinite" />
                    <animate attributeName="stroke-dashoffset" from="251.2" to="0" dur="2s" repeatCount="indefinite" />
                </circle>
            </svg>
        </div>
        <p id="polish-progress-text">Analyzing current design patterns...</p>
        <button id="btn-cancel-polish" class="modal-button-secondary" style="margin-top: 1.5rem;">Cancel</button>
    </div>
</div>

<div id="polish-results-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 800px;">
        <h2 class="polish-modal-title">
            <svg class="polish-title-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/><path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/></svg>
            Polish Complete!
        </h2>
        <p>Your project has been enhanced with modern design principles. Review the changes below:</p>
        
        <div id="polish-summary" style="background: var(--bg-tertiary); padding: 1rem; border-radius: var(--radius-md); margin: 1rem 0;">
            <!-- Summary will be populated by JS -->
        </div>

        <div class="modal-buttons" style="display: flex; gap: 0.75rem; justify-content: center; flex-wrap: wrap;">
            <button id="btn-preview-polish" class="modal-button-secondary" style="flex: 1; min-width: 140px;">
                <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 0.5rem;"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg>
                Preview Changes
            </button>
            <button id="btn-reject-polish" class="modal-button-secondary" style="flex: 1; min-width: 140px;">Revert Changes</button>
            <button id="btn-accept-polish" class="modal-button" style="flex: 1; min-width: 140px;">Keep Polish</button>
        </div>
    </div>
</div>

<footer class="app-footer">

<div id="step-selection-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 600px;">
        <h2 class="step-selection-title">
            <i class="fas fa-sitemap step-selection-icon"></i>
            <span class="gradient-texty" style="line-height:1.3">Choose Plan Complexity</span>
        </h2>
        <p style="color: var(--text-secondary); margin-bottom: 1.5rem;">Select how many steps to break your project into. More steps means more detailed execution but takes longer.</p>
        
        <!-- Analysis Results -->
        <div id="step-analysis-result" style="background:#23232d; padding: 1rem; border-radius: var(--radius-md); margin-bottom: 1.5rem; display: none;">
            <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                </svg>
                <strong>AI Analysis Complete</strong>
            </div>
            <p id="step-analysis-text" style="margin: 0; font-size: 0.9rem; color: var(--text-secondary);"></p>
        </div>
        
        <div class="step-options" style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin: 1.5rem 0;">
            <div class="step-option" data-step-mode="quick" data-steps="3" style="background:#23232d; padding: 1rem; border-radius: var(--radius-md); cursor: pointer; border: 2px solid transparent; transition: all 0.2s;">
                <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#F59E0B" stroke-width="2">
                        <path d="M13 10V3L4 14h7v7l9-11h-7z"></path>
                    </svg>
                    <h4 style="margin: 0; color: var(--text-primary);">Quick</h4>
                </div>
                <p style="margin: 0; font-size: 0.85rem; color: var(--text-secondary);">Fewer steps, faster execution</p>
                <p style="margin: 0.5rem 0 0 0; font-weight: 600; color: #F59E0B;">3 steps ≈ 4 min</p>
            </div>
            
            <div class="step-option recommended" data-step-mode="balanced" data-steps="5" style="background: var(--bg-tertiary); padding: 1rem; border-radius: var(--radius-md); cursor: pointer; border: 2px solid var(--accent-primary); transition: all 0.2s; position: relative;">
                <div style="position: absolute; top: -10px; right: 10px; background: var(--accent-gradient); color: white; padding: 0.25rem 0.75rem; border-radius: 12px; font-size: 0.75rem; font-weight: 600;">Best</div>
                <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="var(--accent-primary)" stroke-width="2">
                        <circle cx="12" cy="12" r="3"></circle>
                        <path d="M12 1v6m0 6v6m8.66-9.34l-5.2 3m-6.92 0l-5.2-3m0 10.68l5.2-3m6.92 0l5.2 3"></path>
                    </svg>
                    <h4 style="margin: 0; color: var(--text-primary);">Balanced</h4>
                </div>
                <p style="margin: 0; font-size: 0.85rem; color: var(--text-secondary);">AI optimizes step count for your project</p>
                <p style="margin: 0.5rem 0 0 0; font-weight: 600; color: var(--accent-primary);">5 steps ≈ 6 min</p>
            </div>
            
            <div class="step-option" data-step-mode="detailed" data-steps="8" style="background:#23232d; padding: 1rem; border-radius: var(--radius-md); cursor: pointer; border: 2px solid transparent; transition: all 0.2s;">
                <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#8B5CF6" stroke-width="2">
                        <path d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"></path>
                    </svg>
                    <h4 style="margin: 0; color: var(--text-primary);">Detailed</h4>
                </div>
                <p style="margin: 0; font-size: 0.85rem; color: var(--text-secondary);">More steps, thorough execution</p>
                <p style="margin: 0.5rem 0 0 0; font-weight: 600; color: #8B5CF6;">8 steps ≈ 12 min</p>
            </div>
            
            <div class="step-option" data-step-mode="custom" data-steps="5" style="background:#23232d; padding: 1rem; border-radius: var(--radius-md); cursor: pointer; border: 2px solid transparent; transition: all 0.2s;">
                <div style="display: flex; align-items: center; gap: 0.5rem; margin-bottom: 0.5rem;">
                    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="var(--text-secondary)" stroke-width="2">
                        <path d="M12 6V4m0 2a2 2 0 100 4m0-4a2 2 0 110 4m-6 8a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4m6 6v10m6-2a2 2 0 100-4m0 4a2 2 0 110-4m0 4v2m0-6V4"></path>
                    </svg>
                    <h4 style="margin: 0; color: var(--text-primary);">Custom</h4>
                </div>
                <p style="margin: 0; font-size: 0.85rem; color: var(--text-secondary);">Choose your own step count</p>
                <div id="custom-step-slider" style="margin-top: 0.75rem; display: none;">
                    <input type="range" min="3" max="15" value="5" step="1" id="step-count-slider" style="width: 100%;">
                    <div style="display: flex; justify-content: space-between; margin-top: 0.25rem; font-size: 0.75rem; color: var(--text-tertiary);">
                        <span>3</span>
                        <span id="slider-value" style="font-weight: 600; color: var(--accent-primary);">5 steps</span>
                        <span>15</span>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Warning Message -->
        <div id="step-warning" style="display: none; background: rgba(251, 191, 36, 0.1); border: 1px solid rgba(251, 191, 36, 0.3); padding: 1rem; border-radius: var(--radius-md); margin-bottom: 1rem;">
            <div style="display: flex; align-items: start; gap: 0.75rem;">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="#F59E0B" stroke-width="2" style="flex-shrink: 0; margin-top: 2px;">
                    <path d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
                </svg>
                <div>
                    <strong style="color: #F59E0B; display: block; margin-bottom: 0.25rem;">Complexity Warning</strong>
                    <p id="step-warning-text" style="margin: 0; font-size: 0.9rem; color: var(--text-secondary);"></p>
                </div>
            </div>
        </div>
        
        <!--Info Box -->
        <div style="background: rgba(59, 130, 246, 0.1); border: 1px solid rgba(59, 130, 246, 0.3); padding: 1rem; border-radius: var(--radius-md); margin-bottom: 1.5rem;">
            <p style="margin: 0; font-size: 0.9rem;"><strong>💡 Tip:</strong> The AI analyzes your project description to recommend the optimal step count. More complex projects (multi-page, backend features, etc.) benefit from more steps to avoid timeouts and ensure quality.</p>
        </div>
        
        <div class="modal-buttons" style="display: flex; gap: 0.75rem; justify-content: center;">
            <button id="btn-cancel-step-selection" class="modal-button-secondary">Cancel</button>
            <button id="btn-confirm-step-selection" class="modal-button">Continue with Plan</button>
        </div>
    </div>
</div>

    <div class="footer-content">
        <span class="footer-brand">ZenCode AI</span>
        <span class="footer-status">
            <span class="footer-idle">•</span>
        </span>
    </div>
</footer>

<div id="version-control-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 600px; text-align: center;">
        <h2 class="version-control-title" style="line-height:1.3">Project Version Control</h2>
        <p>Manage your project versions and rollback to previous states when needed.</p>
        
        <div class="modal-buttons version-control-actions" style="margin-bottom: 2rem;">
            <button id="btn-create-version" class="modal-button">Save Current Version</button>
            <button id="btn-refresh-versions" class="modal-button-secondary">Refresh List</button>
        </div>
        
        <div id="versions-list" style="max-height: 40vh; overflow-y: auto; padding-right: 10px;">
            <!-- Versions will be populated here -->
        </div>

        <div id="version-list-controls" class="modal-buttons" style="margin-top: 1rem; justify-content: center;"></div>
        
        <div class="modal-buttons" style="margin-top: 1.5rem;">
            <button id="btn-close-version-modal" class="modal-button-secondary">Close</button>
        </div>
    </div>
</div>
		<div id="specification-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 800px; text-align: left;">
	
        <h2>📝 Project Specification Sheet</h2>
        <p>Your request requires more detail. I've generated a specification based on your idea. Please review, edit, and approve it to create a high-quality project plan.</p>
<form id="spec-form" style="border-top: 1px solid var(--border-primary); border-bottom: 1px solid var(--border-primary); margin: 1.5rem 0; padding-top: 1.5rem;"></form>
        <div class="modal-buttons" style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 1rem;">
    <button id="btn-cancel-spec" class="modal-button-secondary">Cancel</button>
    <div style="display: flex; gap: 1rem; justify-content: flex-end; flex-wrap: wrap;">
        <button id="btn-build-standalone" class="modal-button-secondary">Generate Standalone App Plan</button>
        <button id="btn-approve-spec" class="modal-button">Generate Standard Plan</button>
    </div>
</div>
    </div>
</div>

 <!-- REPLACE the existing #debug-modal div with this one -->
<div id="debug-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 600px; text-align: left;">
        <h3><svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="vertical-align: -3px; margin-right: 8px;">
        <path d="M15 13H9V7c0-1.7 1.3-3 3-3s3 1.3 3 3v6Z"/>
        <path d="M12 20c-3.3 0-6-2.7-6-6v-3h12v3c0 3.3-2.7 6-6 6Z"/>
        <path d="M4 11h2"/>
        <path d="M18 11h2"/>
        <path d="M15 8h4"/>
        <path d="M5 8h4"/>
    </svg>  <span style="line-height:1.3">Debug & Repair Tool</span></h3>
       <p style="color: var(--text-secondary);">Use the AI to automatically find and fix errors in your project, or paste a specific console error for a targeted solution.</p>
        
        <div style="margin-top: 1.5rem;">
            <button id="btn-auto-debug" class="modal-button" style="width: 100%; justify-content: center;">
                <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="margin-right: 8px;"><path d="M12 2l3.09 6.26L22 9.27l-5 4.87 1.18 6.88L12 17.77l-6.18 3.25L7 14.14 2 9.27l6.91-1.01L12 2z"/></svg>
                Auto-Detect & Fix Errors
            </button>
        </div>

        <div style="text-align: center; color: var(--text-tertiary); margin: 1.5rem 0; font-weight: 500;">OR</div>

        <div>
            <label for="debug-error-input" style="display: block; font-weight: 500; margin-bottom: 0.5rem;">Paste a Specific Console Error</label>
            <p style="font-size: 0.85rem; color: var(--text-tertiary); margin-bottom: 0.75rem;">To find an error, preview your HTML file, then press <strong>F12</strong> to open the console.</p>
            <textarea id="debug-error-input" class="chat-input" rows="3" placeholder="Paste console error here..."></textarea>
        </div>
        
        <div style="margin-top: 1rem;">
            <label for="debug-prompt-input" style="display: block; font-weight: 500; margin-bottom: 0.5rem;">Your Prompt (Optional)</label>
            <textarea id="debug-prompt-input" class="chat-input" rows="2" placeholder="e.g., 'Fix this error' or 'Why isn't the button working?'"></textarea>
        </div>

        <div class="modal-buttons" style="margin-top: 1.5rem; justify-content: space-between;">
            <button id="btn-cancel-debug" class="modal-button-secondary">Cancel</button>
            <button id="btn-submit-manual-debug" class="modal-button">Submit Manual Debug</button>
        </div>
    </div>
</div>

<!-- Add this modal alongside your other modals -->
<div id="execution-choice-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 600px;">
        <h2 style="line-height:1.3">Choose Your Execution Style</h2>
        <div class="execution-style-grid" style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin: 1.5rem 0; text-align: left;">
            <div class="settings-item" id="btn-run-unsupervised" style="border-color: var(--accent-primary);">
                <h3>
                    <svg class="execution-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M5 12h14M12 5l7 7-7 7"/></svg>
                    Automated Run
                </h3>
                <p style="font-size: 0.9rem; color: var(--text-secondary);">Fastest option. The AI runs all steps automatically. Best for straightforward plans or when speed is a priority.</p>
            </div>
            <div class="settings-item" id="btn-run-supervised">
                <h3>
                    <svg class="execution-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 2a10 10 0 1 0 10 10c0-2.34-1-4.44-2.5-6s-3.66-2.5-6-2.5Z"/><path d="m10 10 2 2 4-4"/><path d="m9 15-2-2"/></svg>
                    Guided Run
                </h3>
                <p style="font-size: 0.9rem; color: var(--text-secondary);">Most accurate option. You will quickly confirm the AI's context before each step. Best for complex plans where precision is critical.</p>
            </div>
        </div>
        <div class="modal-buttons">
            <button id="btn-cancel-execution-choice" class="modal-button-secondary">Cancel</button>
        </div>
    </div>
</div>
<!-- ADD THIS ENTIRE NEW MODAL for the Guided Run feature -->
<div id="ai-suggestion-modal" class="modal-overlay">
    <div class="modal-content" style="max-width: 600px; text-align: left;">
        <h3>🧠 Confirm Context for Step</h3>
        <p>The AI has analyzed the next step and suggests using the following files. Review its selection, or use your own manually checked files.</p>
        
        <div style="background: var(--bg-tertiary); padding: 1rem; border-radius: var(--radius-md); margin: 1.5rem 0;">
            <h4 style="margin: 0 0 0.5rem 0; color: var(--accent-primary);">AI's Suggested Files:</h4>
            <ul id="ai-suggested-files-list" style="list-style: none; padding-left: 0; color: var(--text-secondary); font-family: var(--font-mono);">
                <!-- File list will be populated here by JS -->
            </ul>
        </div>

        <div class="modal-buttons" style="flex-direction: column; align-items: stretch; gap: 0.75rem;">
            <button id="btn-confirm-ai-selection" class="modal-button">Run with AI's Suggestion</button>
            <button id="btn-confirm-manual-selection" class="modal-button-secondary">Run with My Manual Selection</button>
        </div>

        <div class="modal-buttons" style="margin-top: 1rem;">
             <button id="btn-cancel-suggestion" class="modal-button-secondary" style="background: transparent; border: none;">Cancel Step</button>
        </div>
    </div>
</div>
    `;
},
// REPLACE your current cacheDOMElements function with this corrected version.
cacheDOMElements() {
    const s = (id) => document.getElementById(id);
    this.elements = {
        headerProjectName: s('header-project-name'),
        apiStatus: s('api-status'), apiStatusText: s('api-status-text'), mobileApiStatus: s('mobile-api-status'), mobileApiStatusText: s('mobile-api-status-text'),
        codeArea: s('code-area'), chatMessages: s('chat-messages'), chatInput: s('chat-input'), sendButton: s('send-button'),
        fileTree: s('file-tree'), sidebar: s('sidebar'), sidebarToggleHeader: s('sidebar-toggle-header'), resizer: s('resizer'), modeToggleSwitch: s('mode-toggle-switch'),
        projectNameInput: s('project-name-input'), btnSaveFile: s('btn-save-file'),
        fileUploadInput: s('file-upload-input'), btnNewFolder: s('btn-new-folder'), zipUploadInput: s('zip-upload-input'),
        codePanel: document.querySelector('.code-editor-panel'), mobileOverlay: s('mobile-overlay'), editorFilename: s('editor-filename'),
        btnAcceptChanges: s('btn-accept-changes'), btnRejectChanges: s('btn-reject-changes'), codePreview: s('code-preview'),
        oneShotModal: s('one-shot-modal'), btnShowOneShot: s('btn-show-one-shot'), btnCloseModal: s('btn-close-modal'), btnGenerateWebsite: s('btn-generate-website'), oneShotInput: s('one-shot-input'),
        viewWebsiteModal: s('view-website-modal'), btnCloseViewModal: s('btn-close-view-modal'), btnViewWebsiteLink: s('btn-view-website-link'), btnViewGeneratedWebsite: s('btn-view-generated-website'),
        debugButton: s('debug-button'),
        debugModal: s('debug-modal'),
        debugErrorInput: s('debug-error-input'),
        debugPromptInput: s('debug-prompt-input'),
        btnCancelDebug: s('btn-cancel-debug'),
        btnSubmitManualDebug: s('btn-submit-manual-debug'),
btnAutoDebug: s('btn-auto-debug'),
        imageUploadArea: s('image-upload-area'),
        imageUploadInput: s('image-upload-input'),
        voiceButton: s('voice-button'),
        imageUploadPrompt: s('image-upload-prompt'),
        imagePreviewContainer: s('image-preview-container'),
        imagePreview: s('image-preview'),
        removeImageBtn: s('remove-image-btn'),
        saveStatus: s('save-status'),
        btnProjects: s('btn-projects'),
        projectsModal: s('projects-modal'),
        projectsList: s('projects-list'),
        btnRefinePrompt: s('btn-refine-prompt'),
        btnCloseProjectsModal: s('btn-close-projects-modal'),
        btnNewProject: s('btn-new-project'),
        btnSettings: s('btn-settings'), // THIS LINE IS NOW CORRECT
        settingsModal: s('settings-modal'),
        btnCloseSettingsModal: s('btn-close-settings-modal'),
        btnUploadFiles: s('btn-upload-files-modal'),
        btnUnzip: s('btn-unzip-modal'),
        btnDownloadZip: s('btn-download-zip-modal'),
        btnClearkbModal: s('btn-clear-kb-modal'),
        featureBar: document.querySelector('.feature-bar'),
        projectsModalFeedback: s('projects-modal-feedback'),
        oneShotLoadingOverlay: s('one-shot-loading-overlay'),
        loadingTextAnim: s('loading-text-anim'),
        cancelGenerationBtn: s('cancel-generation-btn'),
        imagePreviewModal: s('image-preview-modal'),
        imageModalPreviewArea: s('image-modal-preview-area'),
        closeImageModal: s('close-image-modal'),
        projectListControls: s('project-list-controls'),
    };
    for (const key in this.elements) { if (!this.elements[key]) throw new Error(`Critical DOM element not found: ${key}`); }
},
   

     // REPLACE your entire, existing bindEvents function with this correct version.
bindEvents() {
    // This listener now correctly calls cancelAIRequest.
    this.elements.sendButton.addEventListener('click', () => {
        if (this.state.isAILoading) {
            this.cancelAIRequest();
        } else {
            this.sendMessage();
        }
    });
// ADD THIS NEW, CORRECT BLOCK OF LISTENERS
// This is the "Debug & Repair" button in the panel header that OPENS the modal.
this.elements.debugButton.addEventListener('click', () => {
    // Pre-fill the modal's prompt with whatever is in the main chat input.
    this.elements.debugPromptInput.value = this.elements.chatInput.value;
    // Show the modal.
    this.elements.debugModal.classList.add('visible');
    // Automatically focus the error textarea for the user.
    this.elements.debugErrorInput.focus();
});

// This is the "Auto-Detect & Fix Errors" button inside the modal.
this.elements.btnAutoDebug.addEventListener('click', () => {
    this.handleAutoDebug();
});

// This is the "Submit Manual Debug" button inside the modal.
this.elements.btnSubmitManualDebug.addEventListener('click', () => {
    // It calls sendMessage but passes `true` to indicate this is a special debug request.
    this.sendMessage(true); 
    // Hide the modal after sending.
    this.elements.debugModal.classList.remove('visible');
});

// This is the "Cancel" button inside the modal.
this.elements.btnCancelDebug.addEventListener('click', () => {
    this.elements.debugModal.classList.remove('visible');
});
	
 setTimeout(() => this.debugScrollIssues(), 1000);
    // The rest of the function remains the same, with the other fix applied.
    this.elements.chatInput.addEventListener('keydown', e => { if (e.key === 'Enter' && !e.shiftKey) { e.preventDefault(); this.sendMessage(); }});
    this.elements.sidebarToggleHeader.addEventListener('click', () => this.toggleSidebar());
	// Polish tool events
document.addEventListener('click', (e) => {
    if (e.target.closest('#btn-show-polish')) {
        this.showPolishModal();
    }
    if (e.target.closest('#btn-close-polish-modal')) {
        this.togglePolishModal(false);
    }
    if (e.target.closest('#btn-start-polish')) {
        this.startPolishProcess();
    }
    if (e.target.closest('#btn-cancel-polish')) {
        this.cancelPolishProcess();
    }
    if (e.target.closest('#btn-accept-polish')) {
        this.acceptPolishChanges();
    }
    if (e.target.closest('#btn-reject-polish')) {
        this.rejectPolishChanges();
    }
	
	  if (e.target.closest('#btn-preview-polish')) {
        this.previewPolishChanges();
    }
    if (e.target.closest('.polish-option')) {
        this.togglePolishOption(e.target.closest('.polish-option'));
    }
    
    // NEW: Step Selection Modal Handlers
    if (e.target.closest('#btn-cancel-step-selection')) {
        this.cancelStepSelection();
    }
    if (e.target.closest('#btn-confirm-step-selection')) {
        this.confirmStepSelection();
    }
    if (e.target.closest('.step-option')) {
        this.selectStepOption(e.target.closest('.step-option'));
    }
    // Plan configuration handlers
    if (e.target.closest('#btn-confirm-plan-config')) {
        this.handlePlanConfigConfirm();
    }
    if (e.target.closest('#btn-cancel-plan-config')) {
        this.handlePlanConfigCancel();
    }
    if (e.target.closest('#plan-steps-modal') === document.getElementById('plan-steps-modal') && e.target.id === 'plan-steps-modal') {
        this.handlePlanConfigCancel(); // Close on backdrop click
    }
});

    // NEW: Step count slider handler - wrapped in setTimeout to ensure DOM is ready
    setTimeout(() => {
        const slider = document.getElementById('step-count-slider');
        if (slider) {
            slider.addEventListener('input', (e) => {
                const value = e.target.value;
                const sliderValue = document.getElementById('slider-value');
                if (sliderValue) {
                    sliderValue.textContent = `${value} steps`;
                }
                this.state.selectedStepCount = parseInt(value);
                
                // Update warning if needed
                this.validateStepCount(parseInt(value));
            });
        }
    }, 100);
    
    this.elements.mobileOverlay.addEventListener('click', () => this.toggleSidebar(false));
	document.getElementById('version-list-controls').addEventListener('click', (e) => this.handleVersionAction(e));
    this.elements.oneShotInput.addEventListener('input', () => this.handlePromptLengthCheck());
    this.elements.resizer.addEventListener('mousedown', e => this.startResize(e));
    this.elements.resizer.addEventListener('touchstart', e => this.startResize(e));
    this.elements.fileTree.addEventListener('click', e => this.handleTreeClick(e));
    this.elements.btnRefinePrompt.addEventListener('click', () => this.handleRefinePrompt());
    this.elements.modeToggleSwitch.addEventListener('click', () => this.cycleAIMode());
    this.elements.btnSaveFile.addEventListener('click', () => this.downloadActiveFile());
    this.elements.btnNewFolder.addEventListener('click', () => this.createNewFolder());
    this.elements.voiceButton.addEventListener('click', () => this.toggleVoiceRecognition());
    this.elements.btnAcceptChanges.addEventListener('click', () => this.acceptChanges());
    this.elements.btnRejectChanges.addEventListener('click', () => this.rejectChanges());
    window.addEventListener('resize', () => this.handleResponsiveLayout());
    this.elements.fileTree.addEventListener('dragstart', e => this.handleTreeDragStart(e));
    this.elements.fileTree.addEventListener('dragover', e => this.handleTreeDragOver(e));
    this.elements.fileTree.addEventListener('dragleave', e => this.handleTreeDragLeave(e));
    this.elements.fileTree.addEventListener('drop', e => this.handleTreeDrop(e));
    this.elements.sidebar.addEventListener('dragover', e => { e.preventDefault(); this.elements.sidebar.classList.add('drag-over'); });
    this.elements.sidebar.addEventListener('dragleave', () => this.elements.sidebar.classList.remove('drag-over'));
    this.elements.sidebar.addEventListener('drop', e => this.handleExternalDrop(e));
    this.elements.codeArea.addEventListener('blur', () => this.syncEditorToFileState());
    this.elements.btnShowOneShot.addEventListener('click', () => this.handleOneShotButtonClick());
    this.elements.btnCloseModal.addEventListener('click', () => this.toggleOneShotModal(false));
    this.elements.oneShotModal.addEventListener('click', (e) => { if (e.target === this.elements.oneShotModal) this.toggleOneShotModal(false); });
    this.elements.btnGenerateWebsite.addEventListener('click', () => this.handleOneShotGeneration());
    this.elements.btnCloseViewModal.addEventListener('click', () => this.toggleViewWebsiteModal(false));
    this.elements.viewWebsiteModal.addEventListener('click', (e) => { if (e.target === this.elements.viewWebsiteModal) this.toggleViewWebsiteModal(false); });
    this.elements.btnViewGeneratedWebsite.addEventListener('click', () => {
        if (this.state.activeFile && this.state.activeFile.endsWith('.html')) {
            this.previewHTMLWithDependencies(this.state.activeFile);
        }
    });
    if (this.elements.modelSelector) {
        this.elements.modelSelector.addEventListener('change', (e) => {
            this.state.currentModel = e.target.value;
        });
    }
    this.elements.imageUploadArea.addEventListener('click', () => this.elements.imageUploadInput.click());
    this.elements.imageUploadInput.addEventListener('change', e => this.handleImageUpload(e));
    this.elements.removeImageBtn.addEventListener('click', e => { e.stopPropagation(); this.removeOneShotImage(); });
    this.elements.imageUploadArea.addEventListener('dragover', e => { e.preventDefault(); e.stopPropagation(); this.elements.imageUploadArea.style.borderColor = 'var(--border-accent)'; });
    this.elements.imageUploadArea.addEventListener('dragleave', e => { e.preventDefault(); e.stopPropagation(); this.elements.imageUploadArea.style.borderColor = 'var(--border-primary)'; });
    this.elements.imageUploadArea.addEventListener('drop', e => { e.preventDefault(); e.stopPropagation(); this.elements.imageUploadArea.style.borderColor = 'var(--border-primary)'; this.handleImageUpload(e); });
    this.elements.codeArea.addEventListener('input', () => this.scheduleAutoSave());
    this.elements.chatInput.addEventListener('input', () => this.scheduleAutoSave());
    this.elements.projectNameInput.addEventListener('input', () => this.scheduleAutoSave());
    this.elements.projectNameInput.addEventListener('input', () => this.updateHeaderProjectName());
    this.elements.btnProjects.addEventListener('click', () => this.openProjectsModal());
    this.elements.btnCloseProjectsModal.addEventListener('click', () => this.toggleProjectsModal(false));
    this.elements.projectsModal.addEventListener('click', (e) => { if (e.target === this.elements.projectsModal) this.toggleProjectsModal(false); });
    this.elements.projectsList.addEventListener('click', (e) => this.handleProjectAction(e));
    this.elements.projectListControls.addEventListener('click', (e) => this.handleProjectAction(e));
    this.elements.btnNewProject.addEventListener('click', () => this.handleNewProjectClick());
    this.elements.btnSettings.addEventListener('click', () => this.toggleSettingsModal(true));
    this.elements.btnCloseSettingsModal.addEventListener('click', () => this.toggleSettingsModal(false));
    this.elements.settingsModal.addEventListener('click', (e) => { if (e.target === this.elements.settingsModal) this.toggleSettingsModal(false); });
    this.elements.btnUploadFiles.addEventListener('click', () => { this.toggleSettingsModal(false); this.elements.fileUploadInput.click(); });
    this.elements.btnUnzip.addEventListener('click', () => { this.toggleSettingsModal(false); this.elements.zipUploadInput.click(); });
    this.elements.btnDownloadZip.addEventListener('click', () => { this.toggleSettingsModal(false); this.downloadProjectAsZip(); });
// REPLACE your existing file upload handlers with these optimized versions:
if (window.optimizeFileUpload) {
    window.optimizeFileUpload(this.elements.fileUploadInput, (files) => {
        this.handleFileUpload({ target: { files: files } });
    });
    window.optimizeFileUpload(this.elements.zipUploadInput, (files) => {
        this.handleZipUpload({ target: { files: files } });
    });
} else {
    // Fallback to original handlers if optimization not loaded
    this.elements.fileUploadInput.addEventListener('change', (e) => this.handleFileUpload(e));
    this.elements.zipUploadInput.addEventListener('change', (e) => this.handleZipUpload(e));
}
    this.elements.btnClearkbModal.addEventListener('click', () => this.handleClearKnowledgeBase()); 
	// ADD THESE TWO LINES:
document.getElementById('btn-show-learning-stats').addEventListener('click', () => {
    this.toggleSettingsModal(false);
    this.showLearningStats();
});

document.getElementById('btn-cleanup-learnings').addEventListener('click', () => {
    this.toggleSettingsModal(false);
    this.cleanupLowPerformingLearnings();
});

// Close footer progress bar
document.getElementById('close-footer-progress').addEventListener('click', () => {
    this.hideFooterProgress(true); // Hide immediately
});
	
document.getElementById('btn-version-control').addEventListener('click', () => {
    this.toggleSettingsModal(false);
    this.toggleVersionControl();
});

document.getElementById('btn-save-version').addEventListener('click', () => {
    this.toggleSettingsModal(false);
    this.createManualSnapshot();
});

// In the bindEvents() method, add this event listener:
document.getElementById('btn-refresh-versions').addEventListener('click', () => {
    this.loadProjectVersions();
});

document.getElementById('btn-close-version-modal').addEventListener('click', () => {
    this.toggleVersionControl();
});

document.getElementById('btn-create-version').addEventListener('click', () => {
    this.createManualSnapshot();
});

document.getElementById('btn-refresh-versions').addEventListener('click', () => {
    this.loadProjectVersions();
});

document.getElementById('version-control-modal').addEventListener('click', (e) => {
    if (e.target.id === 'version-control-modal') {
        this.toggleVersionControl();
    }
});


 // --- UPDATED: Event listeners for the enhanced specification modal ---
    document.getElementById('btn-approve-spec').addEventListener('click', () => this.handleSpecificationApproval());
    
    // THE FIX IS ON THIS LINE: The function name is now correct (no hyphen).
    document.getElementById('btn-build-standalone').addEventListener('click', () => this.handleStandaloneAppPlanGeneration());

    document.getElementById('btn-cancel-spec').addEventListener('click', () => {
        // Close the modal and do nothing else. The user can type a new prompt.
        document.getElementById('specification-modal').classList.remove('visible');
    });

    
// This listener for the one-shot builder's cancel button also now correctly calls cancelAIRequest.
this.elements.cancelGenerationBtn.addEventListener('click', () => this.cancelAIRequest());

this.elements.imagePreviewModal.addEventListener('click', (e) => {
    if (e.target === this.elements.imagePreviewModal || e.target.closest('#close-image-modal')) { this.toggleImageModal(false); }
    if (e.target.closest('#btn-describe-image')) { this.handleImageDescriptionRequest(); }
    if (e.target.closest('#btn-plan-from-image')) { this.handleImagePlanRequest(); }
});

// THIS IS THE ORIGINAL, BUGGY BLOCK IN YOUR FUNCTION
this.elements.chatMessages.addEventListener('click', e => {
    // Updated to handle both old and new button structures
    const executeBtn = e.target.closest('#execute-plan-btn, .execute-plan-btn-new');
    const stopBtn = e.target.closest('#stop-plan-btn, .stop-plan-btn-new');
    const stepBtn = e.target.closest('.execute-step-btn');
    const redoBtn = e.target.closest('.redo-step-btn');
    const retryBtn = e.target.closest('.retry-step-btn'); // ADD THIS LINE
    const polishBtn = e.target.closest('#polish-plan-btn, .polish-plan-btn-new');

    if (executeBtn) {
        this.executeFullPlan();
    }
    if (stopBtn) {
        this.state.isExecutingPlan = false;
        this.togglePlanControls(false);
        this.highlightCurrentStep(-1);
    }
    if (stepBtn) {
        const stepIndex = parseInt(stepBtn.dataset.stepIndex, 10);
        const step = this.state.currentPlan[stepIndex];
        if (step) {
            this.executeSingleStep(step.text, stepIndex);
        }
    }
    if (redoBtn) {
        const stepIndex = parseInt(redoBtn.dataset.stepIndex, 10);
        this.handleRedoStep(stepIndex);
    }
    // ADD THIS HANDLER FOR RETRY BUTTON
    if (retryBtn) {
        const stepIndex = parseInt(retryBtn.dataset.stepIndex, 10);
        const step = this.state.currentPlan[stepIndex];
        if (step) {
            this.executeSingleStep(step.text, stepIndex);
        }
    }
    if (polishBtn) {
        this.runDesignConsistencyPass();
    }
});

},
        toggleImageModal(show, url = null) {
            if (show && url) {
                const oldUrl = this.elements.imageModalPreviewArea.dataset.currentUrl;
                if (oldUrl) URL.revokeObjectURL(oldUrl);
                this.elements.imageModalPreviewArea.src = url;
                this.elements.imageModalPreviewArea.dataset.currentUrl = url;
                this.elements.imagePreviewModal.classList.add('visible');
            } else {
                const currentUrl = this.elements.imageModalPreviewArea.dataset.currentUrl;
                if (currentUrl) URL.revokeObjectURL(currentUrl);
                this.elements.imageModalPreviewArea.src = '';
                this.elements.imageModalPreviewArea.dataset.currentUrl = '';
                this.elements.imagePreviewModal.classList.remove('visible');
            }
        },

// FIXED VERSION - Replace your functions with these

showImagePreview(path) {
    const fileData = this.state.projectFiles[path];
    if (!fileData || !fileData.rawFile) {
        this.renderChatMessage('ai', `⚠️ I can only analyze actual image files. "${path}" looks like a text placeholder.`);
        return;
    }
    
    const url = URL.createObjectURL(fileData.rawFile);
    this.toggleImageModal(true, url);
    const container = document.querySelector('#image-preview-modal .image-modal-content');
    if (!container) return;
    
    // Clear previous state explicitly and verify
    this.state.tempVisionImage = null;
    this.state.tempVisionImagePath = null;
    
    const oldBadge = container.querySelector('.vision-badge');
    if (oldBadge) oldBadge.remove();
    const badge = document.createElement('div');
    badge.className = 'vision-badge';
    badge.style.cssText = 'position:absolute;left:10px;top:10px;padding:6px 10px;border-radius:8px;background:rgba(0,0,0,0.65);border:1px solid rgba(255,255,255,0.12);color:#fff;font-size:12px;display:flex;align-items:center;gap:6px;';
    
    const plan = this.getUserPlan();
    const controls = document.getElementById('image-modal-controls');
    const describeBtn = document.getElementById('btn-describe-image');
    const planBtn = document.getElementById('btn-plan-from-image');
    
    // Always disable buttons initially
    if (describeBtn) describeBtn.disabled = true;
    if (planBtn) planBtn.disabled = true;
    
    if (plan === 'visionary') {
        badge.innerHTML = '👁️ <strong>Processing image...</strong>';
        container.appendChild(badge);
        if (controls) controls.style.display = 'flex';
        
        this._fileToBase64(fileData.rawFile).then(base64 => {
            // Validate the base64 data
            if (!base64 || !base64.startsWith('data:image/')) {
                throw new Error('Invalid image data format');
            }
            
            // Set state atomically
            this.state.tempVisionImage = base64;
            this.state.tempVisionImagePath = path;
            
            badge.innerHTML = '👁️ <strong>AI Vision active</strong>';

// Show the prompt container and enable buttons
setTimeout(() => {
    const promptContainer = document.getElementById('image-plan-prompt-container');
    if (promptContainer) promptContainer.style.display = 'block';
    
    if (describeBtn) describeBtn.disabled = false;
    if (planBtn) planBtn.disabled = false;
}, 50);
            
        }).catch(err => {
            badge.innerHTML = '❌ Failed to process image';
            this.renderChatMessage('ai', `❌ Could not prepare the image for analysis: ${err.message || 'Unknown error'}`);
            // Ensure state is cleared on error
            this.state.tempVisionImage = null;
            this.state.tempVisionImagePath = null;
        });
    } else {
        badge.innerHTML = '🔒 Vision is a Visionary feature';
        container.appendChild(badge);
        if (controls) controls.style.display = 'none';
        this.renderChatMessage('ai', '👁️ Image analysis is a Visionary-plan feature. Please upgrade.');
    }
},
async handleAutoDebug() {
    this.elements.debugModal.classList.remove('visible');

    // --- THIS IS THE DEFINITIVE FIX ---
    // Check the AI mode at the moment the function is called.
    if (this.state.currentAIMode === 'plan') {
        this.renderChatMessage('ai', "💡 **Auto-Repair in Plan Mode:** The debugging and auto-repair process is handled automatically after each step during a plan's execution. To use the manual Debug & Repair tool, please switch back to **Normal Mode**.");
        return; // Stop the function completely.
    }
    // --- END OF FIX ---

    this.setLoadingState(true);

    const contextFiles = Array.from(this.state.contextFiles);
    if (contextFiles.length === 0) {
        this.renderChatMessage('ai', "⚠️ **Cannot Auto-Repair:** Please check the boxes next to the files you want me to analyze for errors.");
        this.setLoadingState(false);
        return;
    }

    this.renderChatMessage('user', "Run an automatic debugging scan on the selected files.");
    this.renderChatMessage('ai', `🤖 Starting automatic debugging scan on ${contextFiles.length} file(s)...`);

    const autoDebugPrompt = `
You are a specialist AI Debugging Bot. Your SOLE purpose is to find and fix errors in the provided code files.

**PRIMARY DIRECTIVE: FIX THE CODE, DO NOT PLAN.**

**YOUR MISSION:**
1.  **Analyze Provided Files:** Meticulously review every line of code in the context files.
2.  **Identify All Errors:** Find syntax errors, broken references (e.g., missing functions/CSS classes), and integration issues.
3.  **Fix the Code:** Rewrite the necessary files with the corrections to make them fully functional.

**CRITICAL RULES & CONSTRAINTS:**
1.  **NO PLANNING:** You are strictly forbidden from creating a development plan, a list of steps, or any conversational text that isn't part of the JSON summary.
2.  **CODE ONLY:** Your job is to correct the existing code, not to generate new, unrelated features.
3.  **TRUST THE CONTEXT:** Assume the user has provided all necessary files. Fix the errors within these files. Do not suggest that other files are needed.

**OUTPUT REQUIREMENTS (MANDATORY):**
1.  **JSON Summary First:** Your response MUST begin with a JSON block summarizing the repairs.
    \`\`\`json
    {
      "summary": "A brief, one-sentence summary of the repairs. Example: 'Fixed a missing function call in script.js and corrected an unclosed div tag in index.html.'",
      "errors_found": [
        "Description of the first error found.",
        "Description of the second error found."
      ],
      "files_used": ["path/to/repaired_file1.js", "path/to/repaired_file2.html"]
    }
    \`\`\`
2.  **Corrected Files Only:** After the JSON block, provide the complete, corrected versions of ONLY the files you changed.
3.  **If No Errors:** If you find no errors, your JSON summary MUST state that, and you MUST NOT output any file blocks.

Analyze and repair the provided code files now.`;

    try {
        await this._executeSend(autoDebugPrompt, true, null, "Auto-debug request");
    } catch (error) {
        console.error("Auto-debug failed:", error);
        this.renderChatMessage('ai', `❌ The auto-repair process encountered an unexpected error: ${error.message}`);
    } finally {
        this.setLoadingState(false);
    }
},
previewPolishChanges() {
    document.getElementById('polish-results-modal').classList.remove('visible');
    
    // Find the first HTML file that was changed
    const changedFiles = this.getChangedFilesSincePolish();
    const htmlFile = changedFiles.find(f => f.endsWith('.html'));
    
    if (htmlFile) {
        // Activate the file in the editor
        this.activateFile(htmlFile);
        
        // Auto-open browser preview after a short delay
        setTimeout(() => {
            this.previewHTMLWithDependencies(htmlFile);
        }, 500);
        
        this.renderChatMessage('ai', '👁️ **Browser preview opened!** Check the new tab to see your polished website. The polish modal will reappear in 5 seconds.', true);
    } else {
        // No HTML file, just show in editor
        if (changedFiles.length > 0) {
            this.activateFile(changedFiles[0]);
        }
        this.renderChatMessage('ai', '👁️ **Preview Mode Active** - Inspect the polished files in the editor. The polish modal will reappear in 3 seconds.', true);
    }
    
    const quickReturnBtn = document.createElement('button');
    quickReturnBtn.className = 'modal-button';
    quickReturnBtn.style.cssText = 'margin: 0.5rem auto; display: block;';
    quickReturnBtn.textContent = '↩️ Return to Polish Decision';
    quickReturnBtn.onclick = () => {
        this.reopenPolishResults();
        quickReturnBtn.remove();
    };
    this.elements.chatMessages.appendChild(quickReturnBtn);
    this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
    
    // Auto-reopen after 5 seconds (more time to check browser preview)
    setTimeout(() => {
        if (this.state.polishModalWasOpen && !document.getElementById('polish-results-modal').classList.contains('visible')) {
            this.reopenPolishResults();
            if (quickReturnBtn.parentNode) quickReturnBtn.remove();
        }
    }, 5000);
},

reopenPolishResults() {
    document.getElementById('polish-results-modal').classList.add('visible');
    this.state.polishModalWasOpen = false;
},
// ===================== PUT THIS WHOLE FUNCTION IN PLACE OF YOUR CURRENT ONE =====================
// REPLACE your current handleImagePlanRequest function with this FINAL, corrected version.
handleImagePlanRequest() {
    // --- THE FIX IS HERE ---
    // 1. Manually set the application's state to 'plan'.
    this.state.currentAIMode = 'plan';
    // 2. Call the function that updates the button's color and text.
    this.updateModeToggleVisuals();
    // --- END OF FIX ---

    // The rest of the function remains exactly the same.
    if (!this.state.tempVisionImage || !this.state.tempVisionImagePath) {
        this.renderChatMessage('ai', '🔄 Please wait for the image to finish processing before creating a plan.');
        return;
    }
    if (typeof this.state.tempVisionImage !== 'string' || !this.state.tempVisionImage.startsWith('data:image/')) {
        this.renderChatMessage('ai', '❌ Image data is invalid. Please try opening the image again.');
        return;
    }

    const userPromptEl = document.getElementById('image-plan-prompt');
    const userText = (userPromptEl ? userPromptEl.value : '').trim();

    let specPrompt = `Based on the provided image of "${this.state.tempVisionImagePath}", generate a detailed project specification sheet.`;
    if (userText) {
        specPrompt += ` The user has provided these additional requirements: "${userText}"`;
    }
    specPrompt += ` Pay close attention to the visual style, layout, color palette, and implied functionality in the image to inform the specification.`;

    this.toggleImageModal(false);

    const displayMessage = userText 
        ? `Create a spec sheet from the image "${this.state.tempVisionImagePath}" with my notes.`
        : `Create a spec sheet from the image "${this.state.tempVisionImagePath}".`;
    
    // We now create a message object to be compatible with our updated renderChatMessage function
    const messageContent = {
        text: displayMessage,
        // We could even add a thumbnail of the image here if we wanted in the future
    };
    this.renderChatMessage('user', messageContent);
    
    this.startSpecificationProcess(specPrompt, this.state.tempVisionImage);
    
    this.state.tempVisionImage = null;
    this.state.tempVisionImagePath = null;
    if (userPromptEl) userPromptEl.value = '';
    this.elements.chatInput.value = '';
},


// ===================== END REPLACEMENT ==========================================================

        async handleClearKnowledgeBase() {
            this.toggleSettingsModal(false);
            const confirmed = await this.showConfirmation(
                "Are you sure you want to permanently delete all vectorized data? This will erase the AI's long-term memory of your files and cannot be undone."
            );
             if (confirmed) {
                if (window.innerWidth <= 768) {
                    this.toggleSidebar(false);
                }
                this.setLoadingState(true);
                this.renderChatMessage('ai', '🗑️ Clearing your personal knowledge base...');
                const formData = new FormData();
                formData.append('action', 'zencode_ai_clear_user_vectors');
                formData.append('nonce', zencode_ai_app_vars.nonce);
                try {
                    const response = await fetch(zencode_ai_app_vars.ajaxurl, { method: 'POST', body: formData });
                    const data = await response.json();
                    if (data.success) {
                        this.renderChatMessage('ai', `✅ ${data.data.message}`);
                    } else {
                        throw new Error(data.data.message || 'An unknown error occurred.');
                    }
                } catch (error) {
                    this.renderChatMessage('ai', `❌ Error clearing knowledge base: ${error.message}`);
                } finally {
                    this.setLoadingState(false);
                }
            }
        },
        
        cycleAIMode() {
              let modes = ['normal', 'plan']; 
            if (zencode_ai_app_vars.user_plan !== 'spark') {
                modes.push('vector');
            }
            const currentIndex = modes.indexOf(this.state.currentAIMode);
            const nextIndex = (currentIndex === -1) ? 0 : (currentIndex + 1) % modes.length;
            this.state.currentAIMode = modes[nextIndex];
            this.updateModeToggleVisuals();
            this.updateChatPlaceholder();
        },

        // REPLACE your existing updateModeToggleVisuals function with this one.
updateModeToggleVisuals() {
    const button = this.elements.modeToggleSwitch;
    const mode = this.state.currentAIMode;
    button.className = 'mode-toggle-button';
    
    // Also grab the debug button
    const debugButton = this.elements.debugButton;

    switch (mode) {
        case 'plan': 
            button.textContent = 'Plan'; 
            button.classList.add('mode-plan'); 
            // --- THIS IS THE FIX ---
            // Disable the debug button in Plan Mode
            debugButton.disabled = true;
            debugButton.title = 'Debugging is handled automatically within Plan Mode steps.';
            debugButton.style.opacity = '0.5';
            debugButton.style.cursor = 'not-allowed';
            break;
        case 'vector': 
            button.textContent = 'Vector'; 
            button.classList.add('mode-vector');
            // Re-enable for Vector Mode
            debugButton.disabled = false;
            debugButton.title = 'Add Debugging Info';
            debugButton.style.opacity = '1';
            debugButton.style.cursor = 'pointer';
            break;
        default: // 'normal'
            button.textContent = 'Normal'; 
            button.classList.add('mode-normal');
            // Re-enable for Normal Mode
            debugButton.disabled = false;
            debugButton.title = 'Add Debugging Info';
            debugButton.style.opacity = '1';
            debugButton.style.cursor = 'pointer';
            break;
    }
},

        showModalFeedback(message, type = 'success') {
            const feedbackEl = this.elements.projectsModalFeedback;
            feedbackEl.textContent = message;
            feedbackEl.className = `modal-feedback ${type}`;
            setTimeout(() => {
                feedbackEl.style.transition = 'opacity 0.5s ease';
                feedbackEl.style.opacity = '0';
                setTimeout(() => {
                    feedbackEl.style.display = 'none';
                    feedbackEl.style.opacity = '1';
                }, 500);
            }, 4000);
        },
        
        updateSaveStatus(status) {
            this.state.saveState = status;
            const statusMap = { 'saved': '✓ Saved', 'unsaved': 'Unsaved changes', 'saving': 'Saving...' };
            if (this.elements.saveStatus) {
                this.elements.saveStatus.textContent = statusMap[status] || '';
                this.elements.saveStatus.className = `save-status ${status}`;
            }
        },

            scheduleAutoSave() {
            // PERFORMANCE: Skip auto-saves during plan execution
            if (this.state.planExecutionMode) {
                // Just mark that save is pending, don't actually save
                this.state.pendingSaveOperation = Date.now();
                console.log('⚡ Auto-save deferred during plan execution');
                return;
            }
            
            this.updateSaveStatus('unsaved');
            clearTimeout(this.state.autoSaveTimer);

            // If AI is busy, check again in a moment instead of saving immediately
            if (this.state.isAILoading || this.state.isTyping) {
                this.state.autoSaveTimer = setTimeout(() => this.scheduleAutoSave(), this.config.AUTOSAVE_DELAY_MS + 1000);
                return;
            }

            this.state.autoSaveTimer = setTimeout(() => this.saveProject(), this.config.AUTOSAVE_DELAY_MS);
        },

      // OLD saveProject function - REPLACE with this updated version:

async saveProject(createAutoSnapshot = false, maxRetries = 3) {
    if (this.state.isSaving) {
        console.log('Save already in progress, skipping...');
        return false;
    }

    this.state.isSaving = true;
    this.syncEditorToFileState();
    const projectName = this.elements.projectNameInput.value.trim();
    if (!projectName) {
        this.updateSaveStatus('unsaved');
        this.state.isSaving = false;
        return false;
    }
    
    this.updateSaveStatus('saving');

    // CRITICAL: Clear CodeMirror history before serializing to reduce payload size
    if (this.state.codeMirrorInstance) {
        this.state.codeMirrorInstance.clearHistory();
    }

    const serializableProjectFiles = {};
    let totalSize = 0;
    
    for (const path in this.state.projectFiles) {
        const fileData = this.state.projectFiles[path];
        
        // Get clean content without CodeMirror metadata
        let content = fileData.content;
        
        // If this is the active file in CodeMirror, ensure we have the latest content
        if (this.state.activeFile === path && this.state.codeMirrorInstance) {
            content = this.state.codeMirrorInstance.getValue();
        }
        
        serializableProjectFiles[path] = {
            content: content,
            isBinary: !!fileData.isBinary,
            // Only include base64Content if it exists and is reasonably sized
            ...(fileData.base64Content && fileData.base64Content.length < 100000 && { 
                base64Content: fileData.base64Content 
            })
        };
        
        // Track total size for validation
        totalSize += content.length;
        if (fileData.base64Content) {
            totalSize += fileData.base64Content.length;
        }
    }

    // Check if project is getting too large
    if (totalSize > 2000000) { // 2MB warning
        console.warn(`⚠️ Project size large: ${(totalSize / 1024 / 1024).toFixed(2)}MB`);
        
        // Optionally trim chat history if too large
        if (this.state.chatHistory.length > 20) {
            this.state.chatHistory = this.state.chatHistory.slice(-20);
            console.log('Trimmed chat history to reduce size');
        }
    }

    const projectData = {
        projectFiles: serializableProjectFiles,
        fileOrder: this.state.fileOrder,
        chatHistory: this.state.chatHistory,
        collapsedFolders: Array.from(this.state.collapsedFolders),
        projectContext: this.state.projectContext,
        currentPlan: this.state.currentPlan || []
    };
    
    const payloadString = JSON.stringify(projectData);
    
    // Log size for debugging
    console.log(`📊 Save payload: ${(payloadString.length / 1024).toFixed(2)}KB`);
    
    if (payloadString.length > 5000000) { // 5MB hard limit
        this.handleOversizedProject(payloadString.length);
        this.state.isSaving = false;
        return false;
    }
    
    const formData = new FormData();
    formData.append('action', 'zencode_ai_save_project');
    formData.append('nonce', zencode_ai_app_vars.nonce);
    formData.append('project_name', projectName);
    formData.append('project_data', payloadString);
    formData.append('project_context', JSON.stringify(this.state.projectContext));
    
    if (createAutoSnapshot) {
        formData.append('create_auto_snapshot', 'true');
    }
    
    // RETRY LOGIC WITH BETTER USER FEEDBACK
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
        try {
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 30000);
            
            const response = await fetch(zencode_ai_app_vars.ajaxurl, { 
                method: 'POST', 
                body: formData,
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            
            if (response.status === 413) {
                throw new Error('Payload too large (413). Project size exceeds server limits.');
            }
            
            if (!response.ok) throw new Error(`Server responded with status ${response.status}`);
            
            const data = await response.json();
            
            if (data.success) {
                this.updateSaveStatus('saved');
                this.lastSaveFailed = false;

                if (this.state._postUploadSave > 0) {
                    this.renderChatMessage('ai', `✅ Successfully saved all ${this.state._postUploadSave} new file(s) to the server.`);
                    this.state._postUploadSave = 0;
                }
                
                this.state.isSaving = false;
                return true; // SUCCESS
                
            } else {
                throw new Error(data.data?.message || 'Unknown server error during save.');
            }
        } catch (error) {
            const isLastAttempt = attempt === maxRetries;
            
            console.error(`Save attempt ${attempt}/${maxRetries} failed:`, error);
            
            if (isLastAttempt) {
                // FINAL FAILURE - Tell user clearly
                this.handleSaveError(`Failed to save after ${maxRetries} attempts`);
                
                const errorMsg = error.name === 'AbortError' 
                    ? '⏱️ Save timed out. Check your internet connection.' 
                    : error.message.includes('413') 
                        ? '📁 Project too large to save. Please remove some large files or clear chat history.'
                        : '❌ Could not save your work. Please try again.';
                    
                if (this.showNotification) {
                    this.showNotification(errorMsg, 'error', 8000);
                }
                
                this.state.isSaving = false;
                return false; // FAILURE
                
            } else {
                // RETRYING - Show user
                this.updateSaveStatus('retrying');
                console.log(`Retrying save (attempt ${attempt + 1}/${maxRetries})...`);
                
                if (this.showNotification && attempt === 1) {
                    this.showNotification('Save failed, retrying...', 'warning', 2000);
                }
                
                // Wait before retry
                await new Promise(resolve => setTimeout(resolve, 1000 * attempt));
            }
        }
    }
},



showLimitNotification(data) {
    const { remaining_queries, plan, reset_time, warning_type } = data;
    
    const messages = {
        warning: {
            title: "You're on fire!",
            message: "You're really making progress with AI development! You've used most of your 5-hour query limit.",
            icon: "⚡"
        },
        critical: {
            title: remaining_queries === 0 ? "Limit Reached!" : "Final queries remaining!",
            message: remaining_queries === 0 
                ? "No queries remaining in this 5-hour window! Please wait for your limit to reset."
                : remaining_queries === 1
                ? "Only 1 query left in this 5-hour window. Use it wisely!"
                : `Only ${remaining_queries} queries left in this 5-hour window. Use them wisely!`,
            icon: remaining_queries === 0 ? "🚫" : "🎯"
        }
    };
    const config = messages[warning_type] || messages.warning;
    
    // Different actions for critical vs warning
    let actions;
    if (warning_type === 'critical') {
        actions = [
            { text: 'Upgrade for Unlimited', type: 'primary', action: () => window.open('/pricing/', '_blank') },
            { 
                text: remaining_queries === 0 ? 'Wait for Reset' : 'Use Final Queries', 
                type: 'secondary', 
                action: () => this.dismissNotification() 
            }
        ];
    } else {
        actions = [
            { text: 'Upgrade to Unlimited', type: 'primary', action: () => window.open('/pricing/', '_blank') },
            { text: 'Continue for now', type: 'secondary', action: () => this.dismissNotification() }
        ];
    }
    
    this.displayNotification({
        type: warning_type,
        title: config.title,
        message: config.message,
        icon: config.icon,
        stats: { remaining: remaining_queries, resets: reset_time, plan: plan.charAt(0).toUpperCase() + plan.slice(1) },
        actions: actions
    });
},
setLimitReachedState(isLimitReached) {
    this.state.isLimitReached = isLimitReached;
    
    if (isLimitReached) {
        // Disable the send button and show limit message
        this.elements.sendButton.disabled = true;
        this.elements.chatInput.disabled = true;
        this.elements.chatInput.placeholder = "Query limit reached. Please upgrade or wait for reset.";
    } else {
        // Re-enable when limit resets
        this.elements.sendButton.disabled = false;
        this.elements.chatInput.disabled = false;
        this.updateChatPlaceholder();
    }
},
 showLimitReached(message) {
    this.displayNotification({
        type: 'reached',
        title: 'Incredible productivity!',
        message: message,
        icon: '🏆',
        actions: [
            { 
                text: 'Upgrade to Visionary', 
                type: 'primary', 
                action: () => window.open('/pricing/', '_blank') 
            },
            { 
                text: 'Got it', 
                type: 'secondary', 
                action: () => {
                    this.dismissNotification();
                    this.setLimitReachedState(true); // Add this line
                }
            }
        ],
        persistent: true
    });
},

    displayNotification(config) {
        this.dismissNotification();
        const notification = document.createElement('div');
        notification.className = `limit-notification ${config.type}`;
        notification.innerHTML = `
            <div class="notification-header">
                <div class="notification-icon ${config.type}">${config.icon}</div>
                <button class="notification-close">&times;</button>
            </div>
            <div class="notification-body">
                <h4 class="notification-title">${config.title}</h4>
                <p class="notification-message">${config.message}</p>
                ${config.stats ? `<div class="notification-stats"><span>Queries left: <strong>${config.stats.remaining}</strong></span><span>Resets in: <strong>${config.stats.resets}</strong></span></div>` : ''}
                <div class="notification-actions">
                    ${config.actions.map(action => `<button class="notification-btn ${action.type}">${action.text}</button>`).join('')}
                </div>
            </div>
        `;

        document.body.appendChild(notification);
        this.currentNotification = notification;
        setTimeout(() => notification.classList.add('show'), 100);

        notification.querySelector('.notification-close').addEventListener('click', () => this.dismissNotification());
        notification.querySelectorAll('.notification-btn').forEach((btn, index) => {
            btn.addEventListener('click', () => config.actions[index].action());
        });

        if (!config.persistent) {
            this.notificationTimer = setTimeout(() => this.dismissNotification(), 8000);
        }
    },

dismissNotification() {
    if (this.currentNotification) {
        this.currentNotification.classList.remove('show');
        setTimeout(() => {
            if (this.currentNotification && this.currentNotification.parentNode) {
                this.currentNotification.remove();
            }
            this.currentNotification = null;
        }, 500);
    }
    
    if (this.notificationTimer) {
        clearTimeout(this.notificationTimer);
        this.notificationTimer = null;
    }
    
    // Force re-enable inputs when dismissing any notification
    setTimeout(() => {
        if (this.elements.chatInput) {
            this.elements.chatInput.disabled = false;
            this.elements.chatInput.removeAttribute('disabled');
            this.updateChatPlaceholder(); // Restore proper placeholder
        }
        if (this.elements.sendButton) {
            this.elements.sendButton.disabled = false;
        }
        
        // Clear any limit reached state
        this.state.isLimitReached = false;
        
        // Ensure proper focus can be restored
        document.body.style.pointerEvents = 'auto';
    }, 100);
},
            async loadProject(projectName = '') {
    let url = `${zencode_ai_app_vars.ajaxurl}?action=zencode_ai_load_project&nonce=${zencode_ai_app_vars.nonce}`;
    if (projectName) {
        url += `&project_name=${encodeURIComponent(projectName)}`;
    }
    try {
        const response = await fetch(url);
        if (!response.ok) throw new Error(`Server responded with status ${response.status}`);
        const data = await response.json();
        if (data.success) {
            // Load project files
            const loadedProjectFiles = data.data.projectFiles || {};
            for (const path in loadedProjectFiles) {
                const fileData = loadedProjectFiles[path];
                if (fileData && fileData.base64Content) {
                    const blob = this._dataURLtoBlob(fileData.base64Content);
                    fileData.rawFile = new File([blob], path.split('/').pop(), { type: blob.type });
                }
            }
            this.state.projectFiles = loadedProjectFiles;
            this.state.fileOrder = data.data.fileOrder || [];
            
            // 👇 Load the currentPlan
            this.state.currentPlan = data.data.currentPlan || [];
            
            // 👇 Detect plan status with completion count
            const hasPlan = this.state.currentPlan.length > 0;
            const allStepsDone = hasPlan && this.state.currentPlan.every(step => step.status === 'done');
            const completedCount = hasPlan ? this.state.currentPlan.filter(step => step.status === 'done').length : 0;
            const planIncomplete = hasPlan && !allStepsDone;
            
            // 👇 Clean chat history - remove old plan HTML
            let chatHistory = data.data.chatHistory || [];
            if (hasPlan) {
                console.log('🧹 Cleaning old plan HTML from chat history');
                chatHistory = chatHistory.filter(msg => {
                    // Keep all user messages
                    if (msg.role === 'user') return true;
                    
                    // Remove AI messages that contain plan HTML
                    if (typeof msg.content === 'string') {
                        const hasPlanHTML = msg.content.includes('plan-step') || 
                                           msg.content.includes('plan-container') ||
                                           msg.content.includes('execute-plan-btn') ||
                                           msg.content.includes('plan-controls');
                        return !hasPlanHTML;
                    }
                    return true;
                });
            }
            
            // 👇 Clear chat for ANY saved plan
            if (hasPlan) {
                console.log(`✨ Saved plan detected - clearing chat for clean presentation`);
                chatHistory = [];
            }
            
            this.state.chatHistory = chatHistory;
            
            this.state.collapsedFolders = new Set(data.data.collapsedFolders || []);
            this.elements.projectNameInput.value = data.data.projectName || 'Loaded Project';
            
            // ========== Restore project context ==========
            if (data.data.projectContext) {
                this.state.projectContext = {
                    originalVision: '',
                    designRequirements: '',
                    technicalNotes: '',
                    targetAudience: '',
                    createdAt: null,
                    recentSteps: [],
                    keyFiles: {},
                    architecture: [],
                    lastUpdated: null,
                    ...data.data.projectContext
                };
                const steps = this.state.projectContext.recentSteps?.length || 0;
                console.log(`✅ Restored project context with ${steps} remembered steps`);
            } else {
                this.ProjectContextManager.clear();
                console.log('🆕 No saved context found, initialized fresh');
            }
            // ========== END ==========
            
            // 👇 Render everything in the correct order
            this.renderFileTree();
            this.elements.chatMessages.innerHTML = '';
            
            // First render all the cleaned chat messages (empty if plan exists)
            this.state.chatHistory.forEach(msg => this.renderChatMessage(msg.role, msg.content, true));
            
            // 👇 Show appropriate box based on plan status
            if (allStepsDone) {
                // ✅ GREEN BOX: Plan completed successfully
                console.log('🔄 Rendering completed plan summary (GREEN BOX)');
                const stepsList = this.state.currentPlan
                    .map((step, i) => `<div style="padding: 0.25rem 0;">${i + 1}. ${this.escapeHtml(step.text)}</div>`)
                    .join('');
                    
                const completedMessage = `
                    <div style="background: rgba(72, 187, 120, 0.1); padding: 1rem; border-radius: 8px; border: 1px solid rgba(72, 187, 120, 0.3);">
                        <strong style="color: #48bb78;">✅ Plan Completed Successfully!</strong>
                        <p style="margin: 0.5rem 0;">All ${this.state.currentPlan.length} steps were executed. Your project files are ready to use.</p>
                        <details style="margin-top: 0.5rem;">
                            <summary style="cursor: pointer; color: var(--text-secondary); font-size: 0.9rem;">View completed steps</summary>
                            <div style="margin-top: 0.5rem; font-size: 0.9rem;">
                                ${stepsList}
                            </div>
                        </details>
                    </div>
                `;
                this.renderChatMessage('ai', completedMessage, true);
                
            } else if (planIncomplete) {
                // ⚠️ RED BOX: Incomplete plan - different messages for 0 vs partial completion
                const totalSteps = this.state.currentPlan.length;
                
                let incompleteMessage;
                if (completedCount === 0) {
                    // Plan was created but never executed
                    console.log('🔄 Rendering never-executed plan warning (RED BOX)');
                    incompleteMessage = `
                        <div style="background: rgba(245, 101, 101, 0.1); padding: 1rem; border-radius: 8px; border: 1px solid rgba(245, 101, 101, 0.3);">
                            <strong style="color: #f56565;">⚠️ Incomplete Plan Detected</strong>
                            <p style="margin: 0.5rem 0;">This project has a plan that was never executed (0 of ${totalSteps} steps completed).</p>
                            <p style="margin: 0.5rem 0; font-weight: 600; color: #f56565;">⚡ Please start a new project to begin fresh.</p>
                        </div>
                    `;
                } else {
                    // Plan was partially executed
                    console.log('🔄 Rendering partially-executed plan warning (RED BOX)');
                    incompleteMessage = `
                        <div style="background: rgba(245, 101, 101, 0.1); padding: 1rem; border-radius: 8px; border: 1px solid rgba(245, 101, 101, 0.3);">
                            <strong style="color: #f56565;">⚠️ Incomplete Plan Detected</strong>
                            <p style="margin: 0.5rem 0;">This project has an incomplete plan (${completedCount} of ${totalSteps} steps completed).</p>
                            <p style="margin: 0.5rem 0; font-weight: 600; color: #f56565;">⚡ Please start a new project to begin fresh.</p>
                            <p style="margin: 0.5rem 0 0 0; font-size: 0.9rem; color: var(--text-secondary);">
                                <em>💡 Your project files from completed steps are still available if you want to work on them manually in normal mode.</em>
                            </p>
                        </div>
                    `;
                }
                
                this.renderChatMessage('ai', incompleteMessage, true);
                
                // Clear the plan so it doesn't interfere
                this.state.currentPlan = [];
            }
            
            this.updateHeaderProjectName();
            this.updateSaveStatus('saved');
        } else {
            if (data.data?.message !== 'No saved project found.') {
               throw new Error(data.data?.message || 'Unknown server error during load.');
            }
        }
    } catch (error) {
        console.error('Failed to load project:', error);
        this.renderChatMessage('ai', `❌ Load Error: ${error.message}`);
    }
},
        
        toggleVoiceRecognition() {
            if (!('webkitSpeechRecognition' in window) && !('SpeechRecognition' in window)) {
                this.renderChatMessage('ai', "Voice input is not supported in your browser. Please use Chrome or Edge.");
                return;
            }
            const SpeechRecognition = window.SpeechRecognition || window.webkitSpeechRecognition;
            if (!this.recognition) {
                this.recognition = new SpeechRecognition();
                this.recognition.continuous = false;
                this.recognition.interimResults = true;
                this.recognition.lang = 'en-US';
                this.recognition.onstart = () => { this.elements.voiceButton.classList.add('active'); this.elements.chatInput.placeholder = "Listening..."; };
                this.recognition.onresult = (event) => {
                    let transcript = '';
                    for (let i = event.resultIndex; i < event.results.length; i++) {
                        transcript += event.results[i][0].transcript;
                    }
                    this.elements.chatInput.value = transcript;
                };
                this.recognition.onerror = (event) => { this.elements.voiceButton.classList.remove('active'); this.updateChatPlaceholder(); this.renderChatMessage('ai', `Voice input error: ${event.error}`); };
                this.recognition.onend = () => { this.elements.voiceButton.classList.remove('active'); this.updateChatPlaceholder(); };
            }
            if (this.isListening) { this.recognition.stop(); this.isListening = false; } else { this.recognition.start(); this.isListening = true; }
        },
        
        async vectorizeFile(path) {
            this.syncEditorToFileState();
            const fileData = this.state.projectFiles[path];
            if (!fileData || fileData.rawFile) {
                this.renderChatMessage('ai', `❌ Cannot vectorize binary file: ${path}`);
                return;
            }
            const fileContent = fileData.content;
            this.setLoadingState(true);
            this.renderChatMessage('ai', `🧠 Preparing '${path}' for vectorization...`);
            const chunkSize = 6000;
            const chunks = [];
            for (let i = 0; i < fileContent.length; i += chunkSize) {
                chunks.push(fileContent.substring(i, i + chunkSize));
            }
            this.renderChatMessage('ai', `Splitting into ${chunks.length} chunk(s). This may take a moment...`);
            try {
                for (let i = 0; i < chunks.length; i++) {
                    const chunkContent = chunks[i];
                    const chunkIdentifier = `${path} (Part ${i + 1}/${chunks.length})`;
                    const formData = new FormData();
                    formData.append('action', 'zencode_ai_vectorize_file');
                    formData.append('nonce', zencode_ai_app_vars.nonce);
                    formData.append('file_path', chunkIdentifier);
                    formData.append('file_content', chunkContent);
                    const response = await fetch(zencode_ai_app_vars.ajaxurl, { method: 'POST', body: formData });
                    const data = await response.json();
                    if (!data.success) { throw new Error(data.data?.message || `A chunk failed to vectorize on the server.`); }
                }
                this.renderChatMessage('ai', `✅ Successfully vectorized all ${chunks.length} chunks of '${path}'!`);
            } catch (error) {
                this.renderChatMessage('ai', `❌ ${error.message}`);
            } finally {
                this.setLoadingState(false);
            }
        },
		
		handleImageUpload(event) {
            const file = event.dataTransfer ? event.dataTransfer.files[0] : event.target.files[0];
            if (!file || !file.type.startsWith('image/')) return;
            const reader = new FileReader();
            reader.onload = (e) => {
                this.state.oneShotImage = e.target.result;
                this.elements.imagePreview.src = e.target.result;
                this.elements.imageUploadPrompt.style.display = 'none';
                this.elements.imagePreviewContainer.style.display = 'block';
                this.elements.imageUploadArea.classList.add('has-image');
            };
            reader.readAsDataURL(file);
        },

        removeOneShotImage() {
            this.state.oneShotImage = null;
            this.elements.imageUploadInput.value = '';
            this.elements.imagePreview.src = '';
            this.elements.imageUploadPrompt.style.display = 'block';
            this.elements.imagePreviewContainer.style.display = 'none';
            this.elements.imageUploadArea.classList.remove('has-image');
        },
// Add missing handlePlanConfigCancel function
handlePlanConfigCancel() {
    this.togglePlanStepsModal(false);
    this.state.pendingPlanPrompt = null;
    this.state._pendingPlan = null;
    this.state.requestedStepCount = null;
},

// Add missing togglePolishOption function
togglePolishOption(optionElement) {
    optionElement.classList.toggle('selected');
},
        async showConfirmation(message) {
            const modal = document.getElementById('confirmation-modal');
            const msgEl = document.getElementById('confirmation-message');
            const okBtn = document.getElementById('btn-confirm-ok');
            const cancelBtn = document.getElementById('btn-confirm-cancel');
            msgEl.textContent = message;
            modal.classList.add('visible');
            return new Promise(resolve => {
                const cleanup = () => { okBtn.onclick = null; cancelBtn.onclick = null; };
                okBtn.onclick = () => { modal.classList.remove('visible'); cleanup(); resolve(true); };
                cancelBtn.onclick = () => { modal.classList.remove('visible'); cleanup(); resolve(false); };
            });
        },

        updateContextWithAllFiles() {
            this.state.contextFiles.clear();
            this.state.fileOrder.forEach(path => this.state.contextFiles.add(path));
            this.renderFileTree();
        },

        toggleOneShotModal(show) { this.elements.oneShotModal.classList.toggle('visible', show); },
toggleViewWebsiteModal(show) {
    this.elements.viewWebsiteModal.classList.toggle('visible', show);
    
    // When the modal is hidden (after the user clicks "View Website" or "Close"),
    // restore the feature bar's visibility.
    if (!show) {
        this.elements.featureBar.style.setProperty('display', 'flex', 'important');
    }
},
        toggleSettingsModal(show) { this.elements.settingsModal.classList.toggle('visible', show); },
        
        updatePreviewButtonVisibility() {
            const activeFileIsHtml = this.state.activeFile && this.state.activeFile.toLowerCase().endsWith('.html');
            if (activeFileIsHtml) {
                this.elements.btnViewGeneratedWebsite.style.display = 'flex';
            } else {
                this.elements.btnViewGeneratedWebsite.style.display = 'none';
            }
        },

async handleOneShotGeneration() {
    const userInput = this.elements.oneShotInput.value.trim();
    if (!userInput && !this.state.oneShotImage) {
        alert("Please describe the website or upload an image.");
        return;
    }
    
    const meaningfulFiles = Object.keys(this.state.projectFiles).filter(path => 
        path !== 'README.md' && 
        !path.endsWith('/.keep') &&
        this.state.projectFiles[path].content.trim().length > 50
    );
    
    if (meaningfulFiles.length > 0) {
        this.toggleOneShotModal(false);
        this.state.pendingOneShot = {
            userInput: userInput,
            image: this.state.oneShotImage
        };
        this.toggleProjectsModal(true);
        this.showModalFeedback('You already have content in this project. Please create a new project for your one-shot website build.', 'error');
        return;
    }
    
    this.toggleOneShotModal(false);
    this.elements.featureBar.style.setProperty('display', 'none', 'important');
    
    const loadingPhrases = [
        // Phase 1: The Magical Kick-off (The first ~30 seconds)
        "Igniting the creativity engine...",
        "Gathering cosmic ingredients for your design...",
        "Consulting the digital muses...",
        "Translating your vision into ones and zeros...",
        "Booting up the AI's imagination core...",
        "Uncorking a bottle of pure inspiration...",
        "Let the digital alchemy begin!",
        
        // Phase 2: The Work Begins & First Reassurance (~1-2 minutes)
        "Analyzing the structure of your idea...",
        "This is a brilliant concept! The AI is thinking deeply.",
        "Drafting the foundational HTML blueprint...",
        "Thanks for your patience. Great things are happening.",
        "Crafting the perfect color palette based on your prompt...",
        "Good things come to those who wait. Great websites take a little longer.",
        "Your creativity is impressive. The AI is working hard to match it.",

        // Phase 3: The "Go Do Something" Break (~2-4 minutes)
        "Okay, this requires some serious brainpower.",
        "Now might be a perfect time to go make a fresh cup of coffee or tea.",
        "The AI is in deep concentration mode. We'll keep things running for you.",
        "Feel free to stretch your legs! We're on it.",
        "Patience is a virtue, especially when crafting digital masterpieces.",
        "Building the responsive layout now... this is the really tricky part.",

        // Phase 4: The Value Proposition & Empowerment (~4-7 minutes)
        "Remember, a human designer could take a week for this.",
        "You're the creative director here. The AI is your expert developer.",
        "Condensing days of design work into just a few minutes...",
        "Your vision is guiding the entire process.",
        "This is still way faster (and cheaper!) than finding a freelancer.",
        "We're building something unique, just for you.",
        "The AI is now compiling all the JavaScript logic. Almost like magic.",
        
        // Phase 5: The Final Push & Anticipation (~7-10 minutes)
        "Just a little longer now... it's going to be worth it.",
        "Putting the final, beautiful touches on the CSS.",
        "Optimizing all the assets for lightning-fast loading...",
        "The wait is nearly over. Prepare to be amazed.",
        "Running final quality and design checks...",
        "Polishing every pixel to perfection...",
        "Wrapping everything up in a neat little package for you.",
        "Preparing for the grand reveal..."
    ];
    
    this.startLoadingAnimation(loadingPhrases);
    this.state.isOneShotBuild = true;
    this.state.chatHistory = [];
    this.state.projectFiles = {};
    this.state.fileOrder = [];
    this.elements.chatMessages.innerHTML = '';
    if (this.elements.btnViewGeneratedWebsite) { this.elements.btnViewGeneratedWebsite.style.display = 'none'; }
    this.addFile('README.md', { content: `# Your new project is being generated...` }, true);
    this.updateContextWithAllFiles();
    this.activateFile('README.md');
    
    let fullPromptForAI = `You are a world-class web developer and UX designer, obsessed with creating pixel-perfect, visually stunning websites that make users say 'wow'.
CRITICAL REQUIREMENTS:
1. Output ONLY complete HTML with embedded CSS and JavaScript - no explanations or markdown
2. Make it fully responsive and mobile-first
3. Extract and incorporate visual elements from the image: colors, typography style, aesthetic feel, branding elements
4. Use modern design trends: subtle gradients, clean typography, appropriate shadows, contemporary layouts
5. Include working interactive elements where appropriate (navigation, buttons, forms)
6. Use professional color schemes inspired by the image
7. No external dependencies except Font Awesome CDN - everything else must be inline
8. Ensure accessibility with proper alt text and semantic HTML
9. Make it look professional and production-ready for client presentations
10. Focus on capturing the essence and branding of the business shown in the image
11. ALWAYS include Font Awesome icons for professional iconography - use the CDN link in the <head>

// --- MOBILE NAVIGATION (CRITICAL DETAIL) ---
-   **You MUST include a fully functional and beautifully designed mobile navigation menu.**
-   **Desktop View:** The menu should be a standard, visible navigation bar.
-   **Mobile View (e.g., under 768px):** The desktop menu must disappear. It must be replaced by a "hamburger" icon.
-   **Behavior:** When the hamburger icon is clicked, the menu must smoothly slide in from the side (or fade in) as a full-screen overlay. Clicking a menu link or a "close" (X) icon within the overlay must close it.
-   **Styling:** The links inside the mobile menu overlay must be large, easy to tap, and have generous spacing/padding.

FONT AWESOME USAGE:
- Include Font Awesome 6 CDN: <link rel=\"stylesheet\" href=\"https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css\">
- Use appropriate Font Awesome icons throughout the design (navigation, features, contact info, social media, etc.)
- Icons should enhance the professional appearance and user experience
- Common useful classes: fas fa-home, fas fa-phone, fas fa-envelope, fas fa-map-marker-alt, fas fa-star, fas fa-shopping-cart, fas fa-user, fas fa-menu, etc.

// --- VISUAL IMPACT DIRECTIVE ---
Prioritize bold, memorable design over safe, conventional choices. The website must feel cutting-edge and premium.
// --- CRITICAL TECHNICAL CONSTRAINTS ---
1.  The entire website **MUST** be contained in a single HTML file with embedded CSS (in a <style> tag) and JavaScript (in a <script> tag).
2.  The code **MUST** be clean, semantic, well-structured, and optimized for performance and accessibility.
3.  You **MUST** hotlink all images from royalty-free sources like Unsplash and include proper alt text.
4.  You **MUST** wrap your final output in the standard file format, starting with the line "File: index.html (NEW)". Do not include any other text or explanation outside of this file block.

--- STRICT OUTPUT RULES (MANDATORY) ---
- Output **exactly one** file block in this format (no prose before/after):
  File: index.html (NEW)
  \`\`\`html
  <!DOCTYPE html>
  <html lang="en">
    <head>
      <meta charset="utf-8" />
      <meta name="viewport" content="width=device-width, initial-scale=1" />
      <title>...</title>
      <style>
        /* All CSS here. Avoid huge comments. */
      </style>
    </head>
    <body>
      <!-- Full site markup here -->
      <script>
        // All JS here. Implement mobile menu open/close and prefers-reduced-motion.
      </script>
    </body>
  </html>
  \`\`\`
- **Close the final code fence** (\`\`\`) at the very end of your message.
- **No additional files**, no JSON, and no markdown headings outside the code fence.
- Keep CSS+JS concise; avoid long comments so the full file fits in one response.
- Do not import external JS/CSS frameworks; everything must be inline in this single file.
- Include accessible labels, alt text, focus states, and ARIA where appropriate.
`;

    let promptForHistory = '';
    if (userInput) {
        fullPromptForAI += `\n\nUser's Text Request: "${userInput}"`;
        promptForHistory = `Build a website with the prompt: "${userInput}"`;
    }
    if (this.state.oneShotImage) {
        fullPromptForAI += "\n\nVisual Instructions: Take direct and strong inspiration from the provided image for the color palette, style, layout, and overall mood.";
        promptForHistory = userInput ? `Build a website based on an image and the prompt: "${userInput}"` : `Build a website based on the uploaded image.`;
    }
    await this._executeSend(fullPromptForAI, false, this.state.oneShotImage, promptForHistory);
    this.stopLoadingAnimation();
    this.removeOneShotImage();
},
        startLoadingAnimation(phrases) {
            this.state.loadingIntervalId && clearInterval(this.state.loadingIntervalId);
            const textElement = this.elements.loadingTextAnim;
            const overlayElement = this.elements.oneShotLoadingOverlay;
            let currentIndex = 0;
            textElement.style.opacity = '0';
            overlayElement.style.display = 'flex';
            setTimeout(() => { textElement.textContent = phrases[currentIndex]; textElement.style.opacity = '1'; }, 100);
            this.state.loadingIntervalId = setInterval(() => {
                currentIndex = (currentIndex + 1) % phrases.length;
                textElement.style.opacity = '0';
                setTimeout(() => { textElement.textContent = phrases[currentIndex]; textElement.style.opacity = '1'; }, 300);
            }, 2500);
        },

        stopLoadingAnimation() {
            if (this.state.loadingIntervalId) {
                clearInterval(this.state.loadingIntervalId);
                this.state.loadingIntervalId = null;
                this.elements.oneShotLoadingOverlay.style.display = 'none';
            }
        },

// DELETE your old cancelGeneration function and REPLACE it with this one.
cancelAIRequest() {
    // 1. Abort the network request (this is the universal part).
    if (this.state.currentRequestController) {
        this.state.currentRequestController.abort();
        this.state.currentRequestController = null;
    }

    // 2. Immediately reset the main UI state (also universal).
    this.setLoadingState(false);
    this.state.isTyping = false;

    // 3. Handle feature-specific cleanup.
    // This part is smart: it only stops the one-shot loading screen IF it's actually running.
    if (this.state.isOneShotBuild) {
        this.stopLoadingAnimation();
        this.state.isOneShotBuild = false; // Ensure we reset the state
        if (this.elements.featureBar) {
            this.elements.featureBar.style.setProperty('display', 'flex', 'important');
        }
    }

    // Clean up plan mode state
    this.finishPlanMode();
},
  async executePolish(polishPrompt, selectedOptions, intensity) {
    const formData = new FormData();
    formData.append('action', 'zencode_ai_call_ai_api');
    formData.append('nonce', zencode_ai_app_vars.nonce);
    formData.append('prompt', polishPrompt);
    formData.append('user_text_input', polishPrompt);
    formData.append('is_automated', 'true');
    formData.append('model', this.state.currentModel);
    formData.append('chat_history', JSON.stringify([...this.state.chatHistory, { 
        role: 'user', 
        content: `Polish project: ${selectedOptions.join(', ')} (${intensity} intensity)` 
    }]));
    formData.append('is_plan_mode', 'false');
    formData.append('use_knowledge_base', 'false');
    formData.append('is_one_shot_build', 'false');
    formData.append('project_structure', this.getProjectStructureString());
    
    this.state.currentRequestController = new AbortController();
    const response = await fetch(zencode_ai_app_vars.ajaxurl, {
        method: 'POST',
        body: formData,
        signal: this.state.currentRequestController.signal
    });
    
    const data = await response.json();
    if (data.success) {
        const aiResponseText = typeof data.data === 'string' ? data.data : data.data.ai_response;
        this.state.chatHistory.push({ role: 'user', content: `Polish project: ${selectedOptions.join(', ')} (${intensity} intensity)` });
        this.state.chatHistory.push({ role: 'assistant', content: aiResponseText });
        await this.handleParsedFiles(aiResponseText, true);
    } else {
        throw new Error(data.data?.message || 'Polish request failed');
    }
},
handlePolishError(error) {
    this.state.polishSessionActive = false;
    
    if (error.name === 'AbortError') {
        this.renderChatMessage('ai', 'Polish process cancelled by user.');
        return;
    }
    
    const errorMessage = error.message || '';
    
    if (errorMessage.includes('timeout') || 
        errorMessage.includes('413') || 
        errorMessage.includes('Request Entity Too Large') ||
        errorMessage.includes('systems. Please try again') ||
        errorMessage.includes('Overloaded')) {
        this.renderChatMessage('ai', 'Project too large for polish tool. Try one of these alternatives:\n\n1. Polish individual files manually\n2. Reduce project complexity\n3. Use the "heal" tool on specific files\n4. Break your project into smaller parts');
    } else if (errorMessage.includes('upgrading our systems')) {
        this.renderChatMessage('ai', 'High demand right now - that means you\'re using the latest AI technology! The servers are getting a workout from all the creative projects. Try again in a few minutes when traffic lightens up.');
    } else {
        this.renderChatMessage('ai', `Polish process failed: ${errorMessage}\n\nTip: For large projects, try polishing files individually using direct AI commands.`);
    }
},
    // REPLACE your entire, existing _executeSend function with this FINAL, corrected version.
// REPLACE your entire _executeSend function with this FINAL, CORRECT version.
// REPLACE your entire _executeSend function with this FINAL, COMPLETE, and VERIFIED version.
// This is your complete, original _executeSend function with the definitive fix integrated.


// ============================================================================
// ✨ FEATURE 2: STREAMING RESPONSES
// ============================================================================
/**
 * Handles streaming AI responses using Server-Sent Events (SSE)
 * Provides real-time token-by-token display for better UX
 */
async executeStreamingRequest(prompt, contextFiles, userMessage, metadata = {}) {
    const formData = new FormData();
    formData.append('action', 'zencode_ai_stream_response');
    formData.append('nonce', zencode_ai_app_vars.nonce);
    formData.append('prompt', prompt);
    formData.append('model', this.state.currentModel);
    
    // Create a temporary message bubble for the streaming output
    const messageId = `stream-msg-${Date.now()}`;
    const msgEl = this.renderChatMessage('ai', '<div class="stream-cursor">▋</div>', false, true);
    msgEl.id = messageId;
    msgEl.classList.add('streaming');
    
    const streamContent = msgEl.querySelector('.message-bubble');
    const streamCursor = msgEl.querySelector('.stream-cursor');

    try {
        // Await the full text from the fetch operation
        const fullText = await this.executeStreamingWithFetch(formData, streamContent, streamCursor, msgEl);
        
        // The stream is done. Remove the temporary streaming bubble.
        // The final content will be processed and rendered by the caller (_executeSend).
        msgEl.remove();
        
        return fullText; // Return the complete string for parsing.
    } catch (error) {
        // If the stream fails, remove the temporary bubble and re-throw the error.
        msgEl.remove();
        throw error;
    }
},

/**
 * Executes streaming using fetch API with chunked transfer
 */
// REPLACE your current executeStreamingWithFetch function with this definitive version.

async executeStreamingWithFetch(formData, contentElement, cursorElement, messageElement) {
    try {
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData,
        });

        if (!response.ok) {
            throw new Error(`Server error: ${response.status}`);
        }

        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        let fullText = '';
        let buffer = '';

        while (true) {
            const { done, value } = await reader.read();
            if (done) break;

            buffer += decoder.decode(value, { stream: true });
            
            let boundary;
            while ((boundary = buffer.indexOf('\n\n')) !== -1) {
                const eventText = buffer.substring(0, boundary);
                buffer = buffer.substring(boundary + 2);

                if (eventText.startsWith('data: ')) {
                    try {
                        const data = JSON.parse(eventText.slice(6));
                        if (data.token) {
                            fullText += data.token;
                            // Update the temporary bubble's content in real-time
                            contentElement.innerHTML = this.markdownToHtml(fullText) + '<div class="stream-cursor">▋</div>';
                            this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
                        }
                        if (data.error) {
                            throw new Error(data.error);
                        }
                    } catch (e) { /* Ignore parsing errors of incomplete chunks */ }
                }
            }
        }
        
        return fullText; // Return the final, complete string

    } catch (error) {
        console.error('Streaming execution failed:', error);
        contentElement.innerHTML = `<span style="color: #ef4444;">Request failed: ${error.message}</span>`;
        throw error;
    }
},

markdownToHtml(text) {
    return this.escapeHtml(text)
        .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
        .replace(/\n/g, '<br>');
},
/**
 * Fallback to regular non-streaming request
 */
async executeRegularRequest(prompt, contextFiles, userMessage, metadata) {
    // Use existing _executeSend method
    return await this._executeSend(prompt, false, null, userMessage, metadata);
},

// ============================================================================
// [REPLACE] your entire _executeSend function with this FINAL, CORRECTED version.
// [REPLACE] your entire _executeSend function with this FINAL, CORRECTED version.
 // [REPLACE] your existing _executeSend function with this final, correct version.
   // [REPLACE] your entire _executeSend function with this FINAL, STABLE, and CORRECTED version.
async _executeSend(prompt, isAutomated = false, imageData = null, historyPrompt = null, options = {}) {
    const { specData = null, isStepExecution = false } = options;
    const isOneShot = this.state.isOneShotBuild === true;

    if (isOneShot) {
        this.state.projectFiles = {};
        this.state.fileOrder = [];
        this.state.contextFiles.clear();
        this.renderFileTree();
    }

    const getPlan = () => (this.getUserPlan ? this.getUserPlan() : (zencode_ai_app_vars?.user_plan || '')).toString().trim().toLowerCase();
    const isVisionEnabled = () => ['visionary'].includes(getPlan()) || (zencode_ai_app_vars?.is_admin === true);
    const isImagePath = (p) => /\.(png|jpe?g|webp|gif|bmp|tiff?)$/i.test(p || '');
    const isImageRequest = (txt = '') => { const t = (txt || '').toLowerCase(); const asksToDescribe = /(describe|analy[sz]e|what(?:'s| is) in|explain|caption|ocr|read text)/i.test(t); const mentionsImage = /(image|photo|picture|screenshot|diagram|chart|scan)/i.test(t); return asksToDescribe && mentionsImage; };
    const listContextImages = () => { return Array.from(this.state.contextFiles || []).filter(p => isImagePath(p)).map(p => { const f = this.state.projectFiles[p]; const bytes = f?.rawFile?.size ?? (f?.content ? new TextEncoder().encode(f.content).length : 0); const kb = bytes ? (bytes / 1024).toFixed(1) + 'KB' : ''; return { path: p, size: kb }; }); };
    const fileToImage = (file) => new Promise((resolve, reject) => { const img = new Image(); img.onload = () => resolve(img); img.onerror = () => reject(new Error('Could not load image for compression')); img.src = URL.createObjectURL(file); });
    const canvasToBlob = (canvas, type, quality) => new Promise((resolve) => { canvas.toBlob((blob) => resolve(blob), type, quality); });
    const compressFileToDataURL = async (file, tiers) => {
        const img = await fileToImage(file);
        for (const tier of tiers) {
            const { maxDim, qualities, targetBytes, mime = 'image/jpeg' } = tier;
            let { width, height } = img;
            if (width > height && width > maxDim) { height = Math.round(height * (maxDim / width)); width = maxDim; }
            else if (height >= width && height > maxDim) { width = Math.round(width * (maxDim / height)); height = maxDim; }
            const canvas = document.createElement('canvas'); canvas.width = width; canvas.height = height;
            const ctx = canvas.getContext('2d'); ctx.drawImage(img, 0, 0, width, height);
            for (const q of qualities) {
                const blob = await canvasToBlob(canvas, mime, q);
                if (!blob) continue;
                const estimatedBase64Bytes = Math.ceil(blob.size * 1.35);
                if (estimatedBase64Bytes <= targetBytes) {
                    const fr = new FileReader();
                    const dataURL = await new Promise((res, rej) => { fr.onload = () => res(fr.result); fr.onerror = () => rej(new Error('Failed to read compressed blob')); fr.readAsDataURL(blob); });
                    return { dataURL, sizeBytes: estimatedBase64Bytes, usedTier: tier };
                }
            }
        }
        return null;
    };
    const tiersPrimary = [{ maxDim: 1024, qualities: [0.75,0.65,0.55,0.45,0.35], targetBytes: 350_000, mime: 'image/jpeg' }];
    const tiersFallback1 = [{ maxDim: 800, qualities: [0.70,0.60,0.50,0.40,0.30], targetBytes: 250_000, mime: 'image/jpeg' }];
    const tiersFallback2 = [{ maxDim: 640, qualities: [0.65,0.55,0.45,0.35,0.25,0.20], targetBytes: 180_000, mime: 'image/jpeg' }];

    const promptForDisplay = historyPrompt || prompt;

    if (!isAutomated) {
        const contentToRender = { text: promptForDisplay };
        if(imageData) contentToRender.image = imageData;
        this.renderChatMessage('user', contentToRender);
        this.elements.chatInput.value = '';
        this.elements.chatInput.style.height = '48px';
    }

    const wantsVision = !!imageData || isImageRequest(promptForDisplay) || listContextImages().length > 0;
    if (!isVisionEnabled() && wantsVision) {
        const plan = getPlan() || 'spark';
        this.renderChatMessage('ai', `👁️ AI Vision is available on the <strong>Visionary</strong> plan. Your current plan is <strong>${plan}</strong>.`);
        const imgs = listContextImages();
        if (imgs.length) {
            const list = imgs.slice(0, 12).map(i => `• ${i.path}${i.size ? ` (${i.size})` : ''}`).join('\n');
            this.renderChatMessage('ai', `I can still work with your image <em>filenames</em>. Checked images:\n\n${list}`);
        } else {
            this.renderChatMessage('ai', `Tip: check image files in the Project panel to include them in context.`);
        }
        const guardNote = `[NON-VISION MODE]\nYou cannot see images. You MAY reference image file names, but never describe image contents.\nAvailable image filenames:\n${imgs.map(i => `- ${i.path}`).join('\n') || '- (none selected)'}`;
        prompt = `${prompt}\n\n${guardNote}`;
    }

    this.setLoadingState(true);
    clearTimeout(this.state.longRequestTimer);
    const timeoutDelay = (this.state.polishSessionActive || this.state.isExecutingProcess) ? 90000 : 30000;
    this.state.longRequestTimer = setTimeout(() => { 
        if (this.state.isAILoading) { 
            const message = (this.state.polishSessionActive || this.state.isExecutingProcess) ? 
                "Large project processing takes time. Still working..." : 
                "This seems to be a complex request. I'm still working on it; thanks for your patience!";
            this.renderChatMessage('ai', message); 
        } 
    }, timeoutDelay);

    const currentMessageHistory = [...this.state.chatHistory];
    if (promptForDisplay) {
        currentMessageHistory.push({ role: 'user', content: promptForDisplay });
    }

    const isPlanMode = this.state.currentAIMode === 'plan' && !isStepExecution;
    const useKnowledgeBase = this.state.currentAIMode === 'vector';
    const clearRequestedStepsGlobal = () => { this.state.requestedStepCount = null; };
    
    // ================== THE FIX: PART 1 ==================
    // This block correctly gathers the content of checked files for the API call.
    const projectFilesData = {};
    const contextFiles = Array.from(this.state.contextFiles);
    contextFiles.forEach(path => {
        const fileData = this.state.projectFiles[path];
        if (fileData && !fileData.rawFile && fileData.content) {
            projectFilesData[path] = fileData.content;
        }
    });
    // ================== END FIX: PART 1 ==================

    const buildFormData = (projectStructure, imageDataToSend = null, imageSource = null, executionOptions = {}) => {
        const fd = new FormData();
        fd.append('action', 'zencode_ai_call_ai_api');
        fd.append('nonce', zencode_ai_app_vars.nonce);
        fd.append('prompt', prompt);
        fd.append('user_text_input', promptForDisplay);
        let shouldBypassGatekeeper = isAutomated;
        if (this.state._pendingPlan && this.state._pendingPlan.bypassGatekeeper) {
            shouldBypassGatekeeper = true;
        }
        fd.append('is_automated', shouldBypassGatekeeper.toString());
        fd.append('model', this.state.currentModel);
        fd.append('chat_history', JSON.stringify(currentMessageHistory));
        fd.append('is_plan_mode', isPlanMode.toString());
        fd.append('use_knowledge_base', useKnowledgeBase.toString());
        fd.append('is_one_shot_build', this.state.isOneShotBuild.toString());
        
        // ================== THE FIX: PART 2 ==================
        // Pass both the file contents AND the structure to the backend.
        fd.append('project_files', JSON.stringify(projectFilesData));
        fd.append('project_structure', projectStructure);
        // ================== END FIX: PART 2 ==================
        
        if (this.state.projectContext && this.state.projectContext.originalVision) {
            fd.append('project_context', JSON.stringify(this.state.projectContext));
        }

      if (executionOptions.numSteps != null) {
    fd.append('requested_steps', String(executionOptions.numSteps));
} else if (this.state.requestedStepCount != null) {
    fd.append('requested_steps', String(this.state.requestedStepCount));
}

        if (specData) {
            fd.append('specification_sheet', JSON.stringify(specData));
        }

        if (executionOptions.isStepExecution || isStepExecution) {
            fd.append('is_step_execution', 'true');
        }

        if (imageDataToSend && (isVisionEnabled() || isOneShot)) { 
            fd.append('image_data', imageDataToSend); 
            if (imageSource) fd.append('image_source', imageSource); 
        }
        if (this.state.refinementToken) { 
            fd.append('refinement_token', this.state.refinementToken); 
            this.state.refinementToken = null; 
        }
        return fd;
    };

    const pickFirstContextImageFile = () => {
        const firstPath = Array.from(this.state.contextFiles || []).find(p => isImagePath(p));
        if (!firstPath) return null;
        const f = this.state.projectFiles[firstPath]?.rawFile;
        return f ? { path: firstPath, file: f } : null;
    };

    let attachedVision = null;
    if (imageData && isVisionEnabled()) {
        attachedVision = { dataURL: imageData, from: 'direct_passthrough' };
    } else if (isOneShot && this.state.oneShotImage) {
        try {
            const resp = await fetch(this.state.oneShotImage);
            const blob = await resp.blob();
            const file = new File([blob], 'oneshot-image.jpg', { type: blob.type });
            const websiteGenTiers = [
                { maxDim: 1920, qualities: [0.85, 0.75, 0.65], targetBytes: 800_000, mime: 'image/jpeg' },
                { maxDim: 1600, qualities: [0.80, 0.70, 0.60], targetBytes: 600_000, mime: 'image/jpeg' },
                { maxDim: 1200, qualities: [0.75, 0.65, 0.55], targetBytes: 400_000, mime: 'image/jpeg' }
            ];
            const compressed = await compressFileToDataURL(file, websiteGenTiers);
            attachedVision = { dataURL: compressed ? compressed.dataURL : this.state.oneShotImage, from: 'one_shot' };
        } catch (e) {
            attachedVision = { dataURL: this.state.oneShotImage, from: 'one_shot' };
            console.warn('One-shot compression failed, using original:', e);
        }
    } else if (isVisionEnabled()) {
        const picked = pickFirstContextImageFile();
        if (picked) {
            const compressed = await compressFileToDataURL(picked.file, tiersPrimary);
            if (compressed) {
                attachedVision = { dataURL: compressed.dataURL, from: 'file_tree' };
                this.renderChatMessage('ai', `👁️ AI Vision Active on "${picked.path}" (compressed for analysis).`);
            } else {
                this.renderChatMessage('ai', `⚠️ That image is large. I'll try a smaller version if the server rejects it.`);
            }
        }
    }

    const delay = ms => new Promise(res => setTimeout(res, ms));

    const trySend = async (visionPayload, note) => {
        if (note) this.renderChatMessage('ai', note);
        const MAX_RETRIES = 3; let lastError = null;

       for (let attempt = 1; attempt <= MAX_RETRIES; attempt++) {
    try {
        const projectStructureString = isOneShot ? "The project is currently empty." : this.getProjectStructureString();
        const formData = buildFormData(projectStructureString, visionPayload?.dataURL ?? null, visionPayload?.from ?? null, { isStepExecution });
        this.state.currentRequestController = new AbortController();
        const response = await fetch(zencode_ai_app_vars.ajaxurl, { method: 'POST', body: formData, signal: this.state.currentRequestController.signal });
        const contentType = response.headers.get('content-type');
        
     if (!contentType || !contentType.includes('application/json')) {
    const htmlResponse = await response.text();
    let friendlyError = "Request failed. ";
    
    if (response.status === 504 || response.status === 524) {
        friendlyError = "⏱️ Request timed out. This usually happens with large projects or complex requests.\n\n" +
                       "Try these solutions:\n" +
                       "• Use Plan Mode and break into smaller steps (3-5 files per step)\n" +
                       "• Uncheck some context files (reduce to 3-5 files)\n" +
                       "• Simplify your request\n" +
                       "• Wait 30 seconds and try again";
                       
    } else if (response.status === 413) {
        friendlyError = "📦 Request too large. Your project or context has too much data.\n\n" +
                       "Try:\n" +
                       "• Uncheck most files, keep only 3-5 relevant files\n" +
                       "• Use Plan Mode with smaller steps\n" +
                       "• Reduce the size of individual files";
                       
    } else if (response.status === 502 || response.status === 503) {
        friendlyError = "🔧 Server temporarily unavailable. This could be:\n" +
                       "• Server maintenance\n" +
                       "• Hosting provider issue\n" +
                       "• Temporary overload\n\n" +
                       "Wait a few minutes and try again.";
                       
    } else if (response.status === 500) {
        if (htmlResponse.includes('memory') || htmlResponse.includes('Allowed memory size')) {
            friendlyError = "💾 Server memory limit exceeded.\n\n" +
                           "Try:\n" +
                           "• Uncheck files to reduce context (3-5 files max)\n" +
                           "• Break your request into smaller steps\n" +
                           "• Use simpler prompts";
        } else if (htmlResponse.includes('Maximum execution time')) {
            friendlyError = "⏱️ PHP execution timeout.\n\n" +
                           "Try:\n" +
                           "• Use Plan Mode with smaller steps\n" +
                           "• Select fewer context files\n" +
                           "• Simplify your request";
        } else {
            friendlyError = "❌ Server error. Something went wrong on the server.\n\n" +
                           "Try:\n" +
                           "• Refresh the page and try again\n" +
                           "• Check if the issue persists\n" +
                           "• Contact support if this continues";
        }
    } else {
        friendlyError = `Server error (${response.status}). Please try again or contact support if this persists.`;
    }
    
    console.error('Non-JSON response:', {
        status: response.status,
        contentType: contentType,
        errorPreview: htmlResponse.substring(0, 300),
        projectFiles: Object.keys(this.state.projectFiles).length,
        contextFiles: this.state.contextFiles?.size || 0
    });
    
    lastError = friendlyError;
    break; 
}
        
        const data = await response.json();
        if (data.success) {
            const responseData = data.data;
            if (responseData.show_limit_warning) {
                this.showLimitNotification({
                    remaining_queries: responseData.remaining_queries,
                    plan: responseData.plan,
                    reset_time: responseData.reset_time,
                    warning_type: responseData.warning_type
                });
                if (!responseData.ai_response) { return { kind: 'ok', aiResponse: '' }; }
            }
            if (responseData.show_limit_reached) {
                this.showLimitReached(responseData.message);
                return { kind: 'ok', aiResponse: '' };
            }
            if (responseData.retry_with_delay) {
                const retryMessage = responseData.retry_message || "Processing your request with priority...";
                const retryDelay = responseData.retry_delay || 3000;
                this.renderChatMessage('ai', retryMessage);
                setTimeout(() => {
                    this.renderChatMessage('ai', '🔄 Retrying your request now...');
                    trySend(visionPayload, null).then(result => {
                    });
                }, retryDelay);
                return { kind: 'retry_scheduled' };
            }
            const aiResponseText = typeof responseData === 'string' ? responseData : responseData.ai_response;
if (!isStepExecution && aiResponseText.trim().startsWith('{') && aiResponseText.trim().endsWith('}')) {
    try {
        const parsedJson = JSON.parse(aiResponseText);
        if (parsedJson.is_incompatible === true && parsedJson.explanation) {
            this.renderChatMessage('ai', parsedJson.explanation);
            if (this.state.isOneShotBuild) {
                this.stopLoadingAnimation();
                this.state.isOneShotBuild = false;
                this.elements.featureBar.style.setProperty('display', 'flex', 'important');
            }
            return { kind: 'ok', aiResponse: '' };
        }
    } catch (e) { 
    }
}
     
                    if (responseData.refinement_token) this.state.refinementToken = responseData.refinement_token;
                    return { kind: 'ok', aiResponse: aiResponseText };
                }
                const errorMessage = data.data?.message || '';
                if (errorMessage.includes('Overloaded')) {
                    if (attempt < MAX_RETRIES) {
                        const waitTime = attempt * 2;
                        const positiveMessages = [
                            `High demand right now - that means you're using cutting-edge AI! Retrying in ${waitTime} seconds...`,
                            `Popular service means quality - retrying your request in ${waitTime} seconds...`,
                            `Lots of creative minds at work today! Trying again in ${waitTime} seconds...`
                        ];
                        const randomMessage = positiveMessages[Math.floor(Math.random() * positiveMessages.length)];
                        if (this.elements.loadingTextAnim) this.elements.loadingTextAnim.textContent = randomMessage;
                        await delay(waitTime * 1000);
                        continue;
                    } else {
                        lastError = `High demand right now! The AI servers are getting a workout. Try again in a few minutes - your request will go through when traffic lightens up.`;
                    }
                }
                lastError = `Error: ${errorMessage}`;
                break;
            } catch (error) {
                if (error.name === 'AbortError') { return { kind: 'aborted' }; }
                lastError = `Network Error: ${error.message}`;
                break;
            }
        }
        this.renderChatMessage('ai', `❌ ${lastError}`);
        return { kind: 'fatal' };
    };

    try {
        let result = await trySend(attachedVision, null);

        if (result && result.kind === 'ok') {
            const rawResponse = result.aiResponse;
            let isPlanResponse = false;
            if (this.state.currentAIMode === 'plan' && rawResponse.trim().startsWith('{')) {
                try {
                    const planData = JSON.parse(rawResponse);
                    if (planData.introduction && Array.isArray(planData.plan)) {
                        isPlanResponse = true;
                        this.renderChatMessage('ai', rawResponse);
                        this.state.chatHistory.push({ role: 'assistant', content: rawResponse });
                    }
                } catch (e) {
                    console.warn("Received a JSON-like response in plan mode that wasn't a valid plan.", e);
                }
            }
            
         if (!isPlanResponse) {
    const shouldBypassDiff = this.state.isOneShotBuild || this.state.isExecutingProcess || this.state.polishSessionActive;
    const isAutoRepair = options.isAutoRepair || false;
    
    const conversationalResponse = await this.handleParsedFiles(rawResponse, shouldBypassDiff, isAutoRepair);

    if (conversationalResponse && !isAutoRepair) {
        this.renderChatMessage('ai', conversationalResponse);
        // THE FIX: Saves the clean conversational response instead of the raw one.
        this.state.chatHistory.push({ role: 'assistant', content: conversationalResponse });
    }
}

        } else if (result && result.kind === 'aborted') {
            this.renderChatMessage('ai', 'OK, I\'ve cancelled that request.');
        }

        if (result && result.kind === 'too_large' && (isVisionEnabled() || isOneShot)) {
            if (isOneShot && this.state.oneShotImage) {
                const resp = await fetch(imageData || this.state.oneShotImage);
                const blob = await resp.blob();
                const file = new File([blob], 'oneshot-image.jpg', { type: blob.type });
                const compressed1 = await compressFileToDataURL(file, tiersFallback1);
                attachedVision = compressed1 ? { dataURL: compressed1.dataURL, from: 'one_shot' } : null;
            } else {
                const picked = pickFirstContextImageFile();
                if (picked) {
                    const compressed1 = await compressFileToDataURL(picked.file, tiersFallback1);
                    attachedVision = compressed1 ? { dataURL: compressed1.dataURL, from: 'file_tree' } : null;
                }
            }
            result = await trySend(attachedVision, '🪄 Retrying with extra image compression...');
            if (result && result.kind === 'ok') {
                 const rawResponse = result.aiResponse;
                let isPlanResponse = false;
                if (this.state.currentAIMode === 'plan' && rawResponse.trim().startsWith('{')) { try { const d = JSON.parse(rawResponse); if (d.introduction && Array.isArray(d.plan)) { isPlanResponse = true; this.renderChatMessage('ai', rawResponse); this.state.chatHistory.push({ role: 'assistant', content: rawResponse }); } } catch(e){} }
                if (!isPlanResponse) { const conversationalResponse = await this.handleParsedFiles(rawResponse, true); if (conversationalResponse) { this.renderChatMessage('ai', conversationalResponse); this.state.chatHistory.push({ role: 'assistant', content: conversationalResponse }); } }
            }
        }
        
        if (result && result.kind === 'too_large' && (isVisionEnabled() || isOneShot)) {
            if (isOneShot && this.state.oneShotImage) {
                const resp = await fetch(imageData || this.state.oneShotImage);
                const blob = await resp.blob();
                const file = new File([blob], 'oneshot-image.jpg', { type: blob.type });
                const compressed2 = await compressFileToDataURL(file, tiersFallback2);
                attachedVision = compressed2 ? { dataURL: compressed2.dataURL, from: 'one_shot' } : null;
            } else {
                const picked = pickFirstContextImageFile();
                if (picked) {
                    const compressed2 = await compressFileToDataURL(picked.file, tiersFallback2);
                    attachedVision = compressed2 ? { dataURL: compressed2.dataURL, from: 'file_tree' } : null;
                }
            }
            result = attachedVision
                ? await trySend(attachedVision, '🧪 One more try with ultra-small image…')
                : await trySend(null, '🧪 One more try without an image…');
            if (result && result.kind === 'ok') {
                 const rawResponse = result.aiResponse;
                let isPlanResponse = false;
                if (this.state.currentAIMode === 'plan' && rawResponse.trim().startsWith('{')) { try { const d = JSON.parse(rawResponse); if (d.introduction && Array.isArray(d.plan)) { isPlanResponse = true; this.renderChatMessage('ai', rawResponse); this.state.chatHistory.push({ role: 'assistant', content: rawResponse }); } } catch(e){} }
                if (!isPlanResponse) { const conversationalResponse = await this.handleParsedFiles(rawResponse, true); if (conversationalResponse) { this.renderChatMessage('ai', conversationalResponse); this.state.chatHistory.push({ role: 'assistant', content: conversationalResponse }); } }
            }
        }

        if (result && result.kind === 'too_large') {
            this.renderChatMessage('ai', `❌ Vision analysis failed: payload was too large for the server (HTTP 413).`);
            this.renderChatMessage('ai', `💡 Tip: uncheck very large images in the context, or resize them smaller before asking me to analyze.`);
        }

    } catch (error) {
        if (error.name === 'AbortError') {
            this.renderChatMessage('ai', 'Process cancelled.');
        } else {
            this.renderChatMessage('ai', `Network Error: ${error.message}`);
        }
    } finally {
        this.setLoadingState(false);
        this.state.currentRequestController = null;
        clearTimeout(this.state.longRequestTimer);
        this.state.longRequestTimer = null;
        clearRequestedStepsGlobal();
    }
},

async executeStreamingWithFetch(formData, contentElement, cursorElement, messageElement) {
    try {
        console.log('🔵 Starting streaming request...');
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData,
        });

        if (!response.ok) {
            throw new Error(`Server error: ${response.status}`);
        }

        console.log('🔵 Stream connected, reading data...');
        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        let fullText = '';
        let buffer = '';
        let chunkCount = 0;

        while (true) {
            const { done, value } = await reader.read();
            if (done) {
                console.log(`🔵 Stream complete. Total chunks: ${chunkCount}, Full text length: ${fullText.length}`);
                break;
            }

            buffer += decoder.decode(value, { stream: true });
            
            let boundary;
            while ((boundary = buffer.indexOf('\n\n')) !== -1) {
                const eventText = buffer.substring(0, boundary);
                buffer = buffer.substring(boundary + 2);

                if (eventText.startsWith('data: ')) {
                    try {
                        const data = JSON.parse(eventText.slice(6));
                        if (data.token) {
                            chunkCount++;
                            fullText += data.token;
                            // Update the temporary bubble's content in real-time
                            contentElement.innerHTML = this.markdownToHtml(fullText) + '<div class="stream-cursor">▋</div>';
                            this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
                        }
                        if (data.error) {
                            throw new Error(data.error);
                        }
                        if (data.done) {
                            console.log('🔵 Server sent done signal');
                        }
                    } catch (e) { 
                        console.warn('⚠️ Failed to parse chunk:', e);
                    }
                }
            }
        }
        
        console.log('🔵 Returning full text:', fullText.substring(0, 100) + '...');
        return fullText; // Return the final, complete string

    } catch (error) {
        console.error('🔴 Streaming execution failed:', error);
        contentElement.innerHTML = `<span style="color: #ef4444;">Request failed: ${error.message}</span>`;
        // No need to set loading state here, the caller will handle it.
        throw error;
    }
},

    // REPLACE your sendMessage function with this FINAL, REORDERED version.
// REPLACE your current sendMessage function with this FINAL, SIMPLIFIED version.
// Add this to your state initialization (where other state properties are defined)
// REPLACE your finishPlanMode function with this corrected version
// In input_file_1.js, inside the finishPlanMode function...

finishPlanMode() {
    document.body.classList.remove('plan-mode');
    this.state.isExecutingProcess = false;
    this.state.isExecutingPlan = false;
    
    // Clear context cache when plan finishes
    this.state.planContextCache = null;
    this.state.planContextCacheTimestamp = null;
    
    // Reset any plan-specific UI state
    this.highlightCurrentStep(-1);
    this.togglePlanControls(false);

    // --- ADD THIS BLOCK TO RE-ENABLE THE BUTTON ---
    const debugButton = this.elements.debugButton;
    debugButton.disabled = false;
    debugButton.title = 'Add Debugging Info';
    debugButton.style.opacity = '1';
    debugButton.style.cursor = 'pointer';
    // --- END OF NEW BLOCK ---
},

// REPLACE your entire finishPlanModeWithCompletion function with this:
async finishPlanModeWithCompletion() {
    console.log('🏁 Finishing plan mode with completion...');
    
    try {
        console.log('🧠 Starting final plan reflection...');
        await this.reflectOnEntirePlan();
        console.log('✅ Final reflection completed successfully');
    } catch (error) {
        console.error('❌ Final reflection failed:', error);
        this.renderChatMessage('ai', '🎉 **Plan execution complete!** All steps have been successfully implemented.');
    }
    
    this.finishPlanMode(); // Cleans up UI state like the "Stop" button
    
    const projectFiles = Object.keys(this.state.projectFiles).filter(path => 
        path !== 'README.md' && !path.endsWith('/.keep')
    );
    
    if (projectFiles.length > 2) {
        this.renderChatMessage('ai', '💾 Saving successful plan result as version snapshot...');
        this.saveProject(true); // createAutoSnapshot = true
        this.renderChatMessage('ai', '✅ Plan result saved. You can restore to this working state from Version Control if needed.');
    }
    
    if (!this.state.reflectionMemory || this.state.reflectionMemory.length === 0) {
        this.renderChatMessage('ai', '🎉 **Plan execution complete!** All steps have been successfully implemented.');
    }

    // NEW: Optional Polish & Unify with guardrails
    await new Promise(resolve => setTimeout(resolve, 1500)); 
    
    // Check if polish/unify should be offered
    const shouldOfferPolish = await this.shouldOfferPolishAndUnify();
    
    if (shouldOfferPolish.offer) {
        this.offerOptionalPolishAndUnify(shouldOfferPolish);
    } else {
        this.renderChatMessage('ai', shouldOfferPolish.reason);
    }
},

// NEW: Determine if polish/unify should be offered based on guardrails
async shouldOfferPolishAndUnify() {
    const editableFiles = this.getEditableFiles();
    const filesToProcess = this.prioritizeAndFilterFiles(editableFiles);
    const totalSize = this.getTotalFileSize(filesToProcess);
    
    // Guardrail 1: Too many files
    if (filesToProcess.length > 5) {
        return {
            offer: false,
            reason: `✅ Plan complete! Your project has ${filesToProcess.length} files - too many for the automatic polish tool (max 5 files). You can manually polish individual files if needed.`
        };
    }
    
    // Guardrail 2: Files too large
    if (totalSize > 300000) {
        const sizeMB = (totalSize / 1000000).toFixed(2);
        return {
            offer: false,
            reason: `✅ Plan complete! Your project is ${sizeMB}MB - too large for automatic polish (max 300KB). Consider polishing sections separately.`
        };
    }
    
    // Guardrail 3: Very small projects don't need polish
    if (filesToProcess.length === 0) {
        return {
            offer: false,
            reason: '✅ Plan complete! No files eligible for polishing at this time.'
        };
    }
    
    return {
        offer: true,
        fileCount: filesToProcess.length,
        totalSize: totalSize,
        files: filesToProcess
    };
},

// NEW: Offer optional polish with clear explanation
async offerOptionalPolishAndUnify(polishInfo) {
    const sizeKB = Math.round(polishInfo.totalSize / 1000);
    
    this.renderChatMessage('ai', `

✨ **Optional: Polish & Unify**

Your plan is complete! Would you like to run an optional design consistency pass?

📊 **What it does:**
- Fixes integration issues (broken links, missing CSS classes, JS errors)
- Unifies visual design (colors, fonts, spacing, buttons)
- Ensures components work together seamlessly

📁 **Your project:** ${polishInfo.fileCount} files (${sizeKB}KB total)

⏱️ **Estimated time:** ~2-3 minutes

**Note:** This step is completely optional. Your project is already functional without it.

Would you like to proceed with polish & unify?`);
    
    // Show action buttons
    this.showPolishUnifyPrompt();
},

// NEW: Show interactive polish/unify buttons
showPolishUnifyPrompt() {
    const buttonHtml = `
        <div id="polish-unify-prompt" style="display: flex; gap: 1rem; justify-content: center; margin: 1rem 0; padding: 1rem; background: var(--bg-secondary); border-radius: var(--radius-md); border: 1px solid var(--border-primary);">
            <button id="btn-skip-polish-unify" class="modal-button-secondary" style="flex: 1; max-width: 200px;">
                ❌ Skip Polish
            </button>
            <button id="btn-run-polish-unify" class="modal-button" style="flex: 1; max-width: 200px;">
                ✨ Run Polish & Unify
            </button>
        </div>
    `;
    
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = buttonHtml;
    this.elements.chatMessages.appendChild(tempDiv.firstElementChild);
    this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
    
    // Add event listeners
    document.getElementById('btn-skip-polish-unify')?.addEventListener('click', () => {
        this.skipPolishUnify();
    });
    
    document.getElementById('btn-run-polish-unify')?.addEventListener('click', () => {
        this.runPolishUnify();
    });
},

// NEW: Skip polish/unify
skipPolishUnify() {
    const promptDiv = document.getElementById('polish-unify-prompt');
    if (promptDiv) {
        promptDiv.remove();
    }
    this.renderChatMessage('ai', '✅ Polish skipped. Your project is ready to use!');
},

// NEW: Run polish/unify
async runPolishUnify() {
    const promptDiv = document.getElementById('polish-unify-prompt');
    if (promptDiv) {
        promptDiv.remove();
    }
    
    console.log('🤖 User chose to run polish & unify');
    await this.runDesignConsistencyPass();
},
// Add this function to your App object
autoSelectFilesForStep(stepText) {
    const stepLower = stepText.toLowerCase();
    const availableFiles = this.state.fileOrder;
    const selectedFiles = new Set();
    
    // Always include HTML files if the step mentions pages, forms, or structure
    if (stepLower.includes('html') || stepLower.includes('page') || stepLower.includes('form') || 
        stepLower.includes('structure') || stepLower.includes('navigation') || stepLower.includes('header') || 
        stepLower.includes('footer')) {
        availableFiles.filter(f => f.endsWith('.html')).forEach(f => selectedFiles.add(f));
    }
    
    // Always include CSS files if the step mentions styling, design, or visual elements
    if (stepLower.includes('css') || stepLower.includes('style') || stepLower.includes('design') || 
        stepLower.includes('color') || stepLower.includes('layout') || stepLower.includes('responsive') ||
        stepLower.includes('button') || stepLower.includes('hover') || stepLower.includes('animation')) {
        availableFiles.filter(f => f.endsWith('.css')).forEach(f => selectedFiles.add(f));
    }
    
    // Always include JS files if the step mentions functionality, interaction, or behavior
    if (stepLower.includes('js') || stepLower.includes('javascript') || stepLower.includes('function') || 
        stepLower.includes('click') || stepLower.includes('event') || stepLower.includes('interactive') ||
        stepLower.includes('dynamic') || stepLower.includes('api') || stepLower.includes('validation')) {
        availableFiles.filter(f => f.endsWith('.js')).forEach(f => selectedFiles.add(f));
    }
    
    // Specific file mentions
    const fileKeywords = {
        'index': ['index.html', 'index.js', 'index.css'],
        'main': ['main.html', 'main.js', 'main.css'],
        'style': ['style.css', 'styles.css'],
        'script': ['script.js', 'scripts.js'],
        'app': ['app.js', 'app.css', 'app.html']
    };
    
    for (const [keyword, files] of Object.entries(fileKeywords)) {
        if (stepLower.includes(keyword)) {
            files.forEach(filename => {
                const matchingFile = availableFiles.find(f => f.endsWith(filename) || f.includes(filename));
                if (matchingFile) selectedFiles.add(matchingFile);
            });
        }
    }
    
    // If step is very specific about a file, try to find it
    const filePattern = /(?:create|modify|update|edit)\s+([a-zA-Z0-9_.-]+\.[a-zA-Z]{2,4})/gi;
    let match;
    while ((match = filePattern.exec(stepText)) !== null) {
        const filename = match[1];
        const exactFile = availableFiles.find(f => f.endsWith(filename));
        if (exactFile) selectedFiles.add(exactFile);
    }
    
    // Smart defaults: if nothing specific was mentioned, include key files
    if (selectedFiles.size === 0) {
        // Include main entry points
        const mainFiles = availableFiles.filter(f => 
            f.includes('index') || f.includes('main') || f.includes('app') ||
            (f.endsWith('.html') && availableFiles.filter(h => h.endsWith('.html')).length <= 2)
        );
        mainFiles.forEach(f => selectedFiles.add(f));
        
        // Also include any CSS/JS if we have HTML files
        if (mainFiles.some(f => f.endsWith('.html'))) {
            availableFiles.filter(f => f.endsWith('.css') || f.endsWith('.js')).slice(0, 2).forEach(f => selectedFiles.add(f));
        }
    }
    
    // Always preserve manually selected images
    this.state.contextFiles.forEach(file => {
        if (this.isImageFile(file)) {
            selectedFiles.add(file);
        }
    });
    
    return Array.from(selectedFiles);
},

// REPLACE your entire existing sendMessage function with this definitive version.
// REPLACE your entire sendMessage function with this FINAL version.
async sendMessage(isDebugSend = false) {
    if (this.state.isAILoading || this.state.isLimitReached) return;
    this.syncEditorToFileState();

    let prompt, errorContext, imageToSend;
    let finalPromptForAI, historyPrompt;

    if (isDebugSend) {
        prompt = this.elements.debugPromptInput.value.trim();
        errorContext = this.elements.debugErrorInput.value.trim();
        imageToSend = null;
        this.elements.debugPromptInput.value = '';
        this.elements.debugErrorInput.value = '';
    } else {
        prompt = this.elements.chatInput.value.trim();
        errorContext = null;
        imageToSend = this.state.pendingChatImage || null;
        this.state.pendingChatImage = null;
        this.elements.chatInput.value = '';
        this.elements.chatInput.style.height = '48px';
    }

    if (!prompt && !errorContext && !imageToSend) {
        return;
    }

    if (errorContext) {
        finalPromptForAI = `User prompt: "${prompt || 'Fix this error.'}"\n\nCRITICAL CONTEXT: The user is reporting the following error. Use this to find the root cause of the problem:\n--- ERROR START ---\n${errorContext}\n--- ERROR END ---`;
        historyPrompt = prompt || `Debug: ${errorContext.substring(0, 50)}...`;
    } else {
        finalPromptForAI = prompt;
        historyPrompt = prompt;
    }

    // --- START OF THE FINAL, SIMPLIFIED LOGIC ---
    if (this.state.currentAIMode === 'plan') {
        const hasExistingPlan = this.state.chatHistory.some(msg =>
            typeof msg.content === 'string' && msg.content.trim().startsWith('{')
        );
        if (hasExistingPlan) {
            this.state.pendingPlanPrompt = finalPromptForAI;
            this.toggleProjectsModal(true);
            this.showModalFeedback('Only one plan per project is permitted. To create a new plan, please start a new project.', 'error');
            return;
        }

        // 1. Render the user's message immediately.
        this.renderChatMessage('user', historyPrompt || finalPromptForAI);

        // 2. Capture the project context.
        this.captureProjectContext(finalPromptForAI);

        // 3. ALWAYS go directly to the specification sheet process.
        // All "shouldRunSpecificationCheck" logic is now gone.
        this.startSpecificationProcess(finalPromptForAI, imageToSend);
        
    } else {
        // NORMAL MODE - This remains unchanged.
        await this.executeNormalModeWithAutoSelection(finalPromptForAI, imageToSend, historyPrompt);
    }
	},



// [REPLACE THIS ENTIRE FUNCTION]
// [REPLACE] your entire validateStepOutput function in your App object.

// [REPLACE] your entire validateStepOutput function with this final version.

// [REPLACE] your current validateStepOutput function with this one.
async validateStepOutput(stepIndex, createdFiles, modifiedFiles, isScaffoldingStep = false) {
    const issues = {
        missingReferences: [],
        syntaxErrors: [],
        warnings: [],
        hasIssues: false
    };

    // ✅ HELPER: Check if a file reference is external (CDN/URL)
    const isExternalReference = (filePath) => {
        if (!filePath) return false;
        return filePath.startsWith('http://') || 
               filePath.startsWith('https://') ||
               filePath.startsWith('//') ||
               filePath.includes('cdnjs.cloudflare.com') ||
               filePath.includes('fonts.googleapis.com') ||
               filePath.includes('unpkg.com') ||
               filePath.includes('cdn.jsdelivr.net') ||
               filePath.includes('fontawesome.com') ||
               filePath.includes('fonts.gstatic.com');
    };

    // ✅ HELPER: Fuzzy file matching (handles style.css vs styles.css)
    const findSimilarFile = (requestedFile, availableFiles) => {
        // If it's an external reference, skip file validation
        if (isExternalReference(requestedFile)) {
            console.log(`🌐 External reference detected: ${requestedFile} - skipping validation`);
            return requestedFile; // Return the original to indicate it's external
        }
        
        // Exact match first
        if (availableFiles.includes(requestedFile)) {
            return requestedFile;
        }
        
        // Try plural/singular variations
        const base = requestedFile.replace(/\.css$|\.js$/, '');
        const ext = requestedFile.match(/\.(css|js)$/)?.[0] || '';
        
        // Check variations
        const variations = [
            requestedFile,                    // exact: style.css
            `${base}s${ext}`,                 // plural: styles.css
            base.replace(/s$/, '') + ext,     // singular: style.css (remove trailing s)
            `${base.toLowerCase()}${ext}`,    // lowercase
        ];
        
        for (const variation of variations) {
            if (availableFiles.includes(variation)) {
                console.log(`ℹ️ Found similar file: ${requestedFile} → ${variation}`);
                return variation;
            }
        }
        
        return null;
    };

    const allFiles = Object.keys(this.state.projectFiles);

    // Get all HTML files to check
    const htmlFiles = allFiles.filter(path => path.endsWith('.html'));

    for (const htmlFile of htmlFiles) {
        let content = this.state.projectFiles[htmlFile].content;
        
        // ✅ FILTER OUT META TAGS - they cause false positives
        content = content.replace(/<meta[^>]*>/gi, '');
        
        // Extract linked CSS files
        const cssRegex = /<link[^>]*href=["']([^"']*\.css)["']/gi;
        const linkedCss = [];
        let match;
        while ((match = cssRegex.exec(content)) !== null) {
            linkedCss.push(match[1]);
        }

        // Extract linked JS files
        const jsRegex = /<script[^>]*src=["']([^"']*\.js)["']/gi;
        const linkedJs = [];
        while ((match = jsRegex.exec(content)) !== null) {
            linkedJs.push(match[1]);
        }

        // ✅ IMPROVED: Better scaffolding-aware validation for CSS
        if (linkedCss.length > 0) {
            linkedCss.forEach(cssFile => {
                // Skip external references (CDN links)
                if (isExternalReference(cssFile)) {
                    console.log(`✅ External CSS reference (skipping validation): ${cssFile}`);
                    return; // Skip to next iteration
                }
                
                const actualFile = findSimilarFile(cssFile, allFiles);
                
                if (!actualFile) {
                    // File doesn't exist - apply scaffolding-aware logic
                    const isDefaultCss = (cssFile.toLowerCase() === 'style.css' || cssFile.toLowerCase() === 'styles.css');
                    
                    // ==================== IMPROVED LOGIC ====================
                    if (isScaffoldingStep) {
                        // In scaffolding step, missing default files are EXPECTED - they're being created now
                        console.log(`🏗️ Scaffolding mode: '${cssFile}' will be created in this step`);
                        issues.warnings.push(`Creating '${cssFile}' as part of project foundation`);
                    } 
                    else if (isDefaultCss && stepIndex < 3) {
                        // Early steps in new project - give grace period for default files
                        console.log(`💡 Grace period for '${cssFile}' in early step ${stepIndex + 1}`);
                        issues.warnings.push(`Default file '${cssFile}' referenced but not found yet - will be created soon`);
                    }
                    else if (this.isFilePlannedInFutureSteps(cssFile, stepIndex)) {
                        // File is planned in future steps - this is normal workflow
                        console.log(`✅ ${cssFile} not found yet, but planned in upcoming step - OK`);
                        issues.warnings.push(`${cssFile} will be created in a future step`);
                    }
                    else {
                        // This is a genuine missing reference that needs attention
                        issues.missingReferences.push({
                            file: htmlFile,
                            missing: cssFile,
                            type: 'CSS',
                            line: 'link tag reference'
                        });
                    }
                    // ==================== END IMPROVED LOGIC ====================
                } else if (actualFile !== cssFile) {
                    // File exists but name is slightly different - just warn, don't error
                    console.log(`⚠️ Name mismatch: HTML references "${cssFile}" but file is "${actualFile}"`);
                    issues.warnings.push(`Using ${actualFile} (HTML references ${cssFile})`);
                    // Don't treat as missing - file exists!
                }
            });
        }

        // ✅ IMPROVED: Better scaffolding-aware validation for JS
        if (linkedJs.length > 0) {
            linkedJs.forEach(jsFile => {
                // Skip external references (CDN links)
                if (isExternalReference(jsFile)) {
                    console.log(`✅ External JS reference (skipping validation): ${jsFile}`);
                    return; // Skip to next iteration
                }
                
                const actualFile = findSimilarFile(jsFile, allFiles);
                
                if (!actualFile) {
                    // File doesn't exist - apply scaffolding-aware logic
                    const isDefaultJs = (jsFile.toLowerCase() === 'script.js' || jsFile.toLowerCase() === 'scripts.js');
                    
                    // ==================== IMPROVED LOGIC ====================
                    if (isScaffoldingStep) {
                        // In scaffolding step, missing default files are EXPECTED - they're being created now
                        console.log(`🏗️ Scaffolding mode: '${jsFile}' will be created in this step`);
                        issues.warnings.push(`Creating '${jsFile}' as part of project foundation`);
                    } 
                    else if (isDefaultJs && stepIndex < 3) {
                        // Early steps in new project - give grace period for default files
                        console.log(`💡 Grace period for '${jsFile}' in early step ${stepIndex + 1}`);
                        issues.warnings.push(`Default file '${jsFile}' referenced but not found yet - will be created soon`);
                    }
                    else if (this.isFilePlannedInFutureSteps(jsFile, stepIndex)) {
                        // File is planned in future steps - this is normal workflow
                        console.log(`✅ ${jsFile} not found yet, but planned in upcoming step - OK`);
                        issues.warnings.push(`${jsFile} will be created in a future step`);
                    }
                    else {
                        // This is a genuine missing reference that needs attention
                        issues.missingReferences.push({
                            file: htmlFile,
                            missing: jsFile,
                            type: 'JavaScript',
                            line: 'script tag reference'
                        });
                    }
                    // ==================== END IMPROVED LOGIC ====================
                } else if (actualFile !== jsFile) {
                    console.log(`⚠️ Name mismatch: HTML references "${jsFile}" but file is "${actualFile}"`);
                    issues.warnings.push(`Using ${actualFile} (HTML references ${jsFile})`);
                }
            });
        }

        // ✅ FIXED: Check for function calls in HTML event handlers ONLY (not meta tags)
        // Match actual event handlers: onclick, onload, onsubmit, etc.
        const eventHandlerRegex = /\s(onclick|onload|onsubmit|onchange|oninput|onmouseover|onmouseout|onfocus|onblur|onkeyup|onkeydown|onkeypress)=["']([^"']+)["']/gi;
        while ((match = eventHandlerRegex.exec(content)) !== null) {
            const eventType = match[1];
            const functionCall = match[2];
            const functionName = functionCall.split('(')[0].trim();
            
            // Skip if it's not actually a function call (e.g., just "return false")
            if (!functionName || functionName === 'return' || functionName.length < 2) {
                continue;
            }
            
            // Check if function exists in any JS file
            const jsFiles = allFiles.filter(p => p.endsWith('.js'));
            let functionFound = false;
            
            for (const jsFile of jsFiles) {
                const jsContent = this.state.projectFiles[jsFile].content;
                if (jsContent.includes(`function ${functionName}`) || 
                    jsContent.includes(`const ${functionName}`) ||
                    jsContent.includes(`let ${functionName}`) ||
                    jsContent.includes(`${functionName} =`) ||
                    jsContent.includes(`${functionName}:`)) {
                    functionFound = true;
                    break;
                }
            }
            
            if (!functionFound && jsFiles.length > 0) {
                // For scaffolding steps, be more lenient about missing functions
                if (isScaffoldingStep) {
                    console.log(`🏗️ Scaffolding mode: function '${functionName}' not found yet - will be implemented soon`);
                    issues.warnings.push(`Function '${functionName}' will be implemented in upcoming steps`);
                } else {
                    issues.missingReferences.push({
                        file: htmlFile,
                        missing: functionName,
                        type: 'JavaScript function',
                        line: `${eventType}="${functionCall}"`
                    });
                }
            }
        }
    }

    // Check JS files for syntax errors
    const jsFiles = allFiles.filter(path => path.endsWith('.js'));

    for (const jsFile of jsFiles) {
        const content = this.state.projectFiles[jsFile].content;
        
        // Check for common syntax errors
        try {
            // Check for unmatched braces
            const openBraces = (content.match(/{/g) || []).length;
            const closeBraces = (content.match(/}/g) || []).length;
            if (openBraces !== closeBraces) {
                issues.syntaxErrors.push({
                    file: jsFile,
                    error: 'Unmatched braces',
                    severity: 'high'
                });
            }

            // Check for unmatched parentheses
            const openParens = (content.match(/\(/g) || []).length;
            const closeParens = (content.match(/\)/g) || []).length;
            if (openParens !== closeParens) {
                issues.syntaxErrors.push({
                    file: jsFile,
                    error: 'Unmatched parentheses',
                    severity: 'high'
                });
            }

        } catch (e) {
            issues.syntaxErrors.push({
                file: jsFile,
                error: e.message,
                severity: 'high'
            });
        }
    }

    // Determine if there are actual issues (not just warnings)
    // For scaffolding steps, be more lenient - only syntax errors are critical
    if (isScaffoldingStep) {
        issues.hasIssues = issues.syntaxErrors.length > 0;
        console.log(`🏗️ Scaffolding validation: ${issues.syntaxErrors.length} syntax errors, ${issues.missingReferences.length} missing references (ignored for scaffolding)`);
    } else {
        issues.hasIssues = issues.missingReferences.length > 0 || issues.syntaxErrors.length > 0;
    }

    console.log('🔍 VALIDATION RESULT:', {
        step: stepIndex + 1,
        scaffolding: isScaffoldingStep,
        missingReferences: issues.missingReferences.length,
        syntaxErrors: issues.syntaxErrors.length,
        warnings: issues.warnings.length,
        hasIssues: issues.hasIssues
    });

    return issues;
},
// NEW: Add this helper function to predict what files will be created
predictFutureFileCreations(futureSteps) {
    const predictions = [];
    const allStepsText = futureSteps.map(s => s.text).join(' ').toLowerCase();
    
    // Common file patterns
    const patterns = {
        'style.css': ['css', 'styling', 'stylesheet'],
        'styles.css': ['css', 'styling', 'stylesheet'],
        'script.js': ['javascript', 'js', 'interactive', 'functionality'],
        'main.js': ['javascript', 'js'],
        'app.js': ['javascript', 'js']
    };
    
    for (const [fileName, keywords] of Object.entries(patterns)) {
        if (keywords.some(keyword => allStepsText.includes(keyword))) {
            predictions.push(fileName);
        }
    }
    
    // Also check for explicit file mentions
    const filePattern = /(?:create|add|make)\s+([a-zA-Z0-9_.-]+\.(css|js|html))/gi;
    let match;
    while ((match = filePattern.exec(allStepsText)) !== null) {
        predictions.push(match[1]);
    }
    
    return [...new Set(predictions)]; // Remove duplicates
},

async autoFixStepIssues(stepIndex, validation) {
    // ✅ FILTER OUT EXTERNAL REFERENCES from missing references
    const internalIssues = validation.missingReferences.filter(ref => {
        // Skip external URLs (CDN links)
        const isExternal = ref.missing.startsWith('http://') || 
                          ref.missing.startsWith('https://') ||
                          ref.missing.startsWith('//') ||
                          ref.missing.includes('cdnjs.cloudflare.com') ||
                          ref.missing.includes('fonts.googleapis.com') ||
                          ref.missing.includes('unpkg.com') ||
                          ref.missing.includes('fontawesome.com');
        
        if (isExternal) {
            console.log(`🌐 Skipping external reference in auto-fix: ${ref.missing}`);
            return false;
        }
        return true;
    });
    
    // ✅ FILTER OUT FUTURE-PLANNED FILES from remaining issues
    const immediateIssues = internalIssues.filter(ref => {
        return !this.isFilePlannedInFutureSteps(ref.missing, stepIndex);
    });
    
    const futureIssues = internalIssues.filter(ref => {
        return this.isFilePlannedInFutureSteps(ref.missing, stepIndex);
    });
    
    // Show info about future-planned files (not errors)
    if (futureIssues.length > 0) {
        const filesList = futureIssues.map(f => f.missing).join(', ');
        console.log(`ℹ️ ${futureIssues.length} reference(s) will be resolved in future steps: ${filesList}`);
    }
    
    // If no immediate issues, we're done
    if (immediateIssues.length === 0 && validation.syntaxErrors.length === 0) {
        console.log('✅ No immediate issues to fix');
        return true; // 🆕 FIXED: Return true (success)
    }
    
    // Update validation object to only include immediate issues
    const filteredValidation = {
        ...validation,
        missingReferences: immediateIssues
    };
    
    // Show warning for REAL issues
    const issueCount = immediateIssues.length + validation.syntaxErrors.length;
    this.renderChatMessage('ai', `⚠️ Found ${issueCount} integration issue(s). Running auto-repair...`);
    
    const issues = [];
    
    // Group issues by type
    const missingCss = immediateIssues.filter(i => i.type === 'CSS');
    const missingJs = immediateIssues.filter(i => i.type === 'JavaScript');
    const missingFunctions = immediateIssues.filter(i => i.type === 'JavaScript function');
    
    // Build issue list for display
    if (missingCss.length > 0) {
        missingCss.forEach(issue => {
            issues.push(`Create the missing file: ${issue.missing}`);
        });
    }
    
    if (missingJs.length > 0) {
        missingJs.forEach(issue => {
            issues.push(`Create the missing file: ${issue.missing}`);
        });
    }
    
    if (missingFunctions.length > 0) {
        missingFunctions.forEach(issue => {
            issues.push(`Add the missing function "${issue.missing}" referenced in ${issue.file}`);
        });
    }
    
    if (validation.syntaxErrors.length > 0) {
        validation.syntaxErrors.forEach(error => {
            issues.push(`Fix syntax error in ${error.file}: ${error.error}`);
        });
    }
    
    // Display issues
    const issuesText = '**Issues to fix:**\n' + issues.map(i => `- ${i}`).join('\n');
    this.renderChatMessage('ai', issuesText);
    
    // ⚡ Call the 3-attempt aggressive auto-fix system and RETURN the result
    const repairSuccess = await this.aggressiveAutoFix(stepIndex, filteredValidation);
    return repairSuccess; // 🆕 FIXED: Return the result (true/false)
},
// ADD this helper function to detect external references
isExternalReference(filePath) {
    if (!filePath) return false;
    return filePath.startsWith('http://') || 
           filePath.startsWith('https://') ||
           filePath.startsWith('//') ||
           filePath.includes('cdnjs.cloudflare.com') ||
           filePath.includes('fonts.googleapis.com') ||
           filePath.includes('unpkg.com') ||
           filePath.includes('cdn.jsdelivr.net') ||
           filePath.includes('fontawesome.com') ||
           filePath.includes('fonts.gstatic.com') ||
           filePath.includes('ajax.googleapis.com') ||
           filePath.includes('code.jquery.com') ||
           filePath.includes('maxcdn.bootstrapcdn.com');
},
// Add this new function to your App object
// Replace the executeNormalModeWithAutoSelection function with this corrected version
async executeNormalModeWithAutoSelection(prompt, imageToSend = null, historyPrompt = null) {
    // Check if this request would benefit from auto-selection
    const shouldUseAutoSelection = !imageToSend && // Don't auto-select if there's an image
                                   this.state.fileOrder.length > 2 && // Only if we have multiple files
                                   !this.isSimpleQuestion(prompt); // Skip for simple questions
    
    if (shouldUseAutoSelection) {
        // Track selection changes for feedback
        const previouslySelected = Array.from(this.state.contextFiles);
        
        // Use auto-selection instead of API call
        const autoSelectedFiles = this.autoSelectFilesForNormalMode(prompt);
        const finalMergedContext = this.updateCheckboxesFromAI(autoSelectedFiles);
        this.state.contextFiles = finalMergedContext;
        
        // Visual feedback
        this.elements.fileTree.classList.add('file-tree-updating');
        this.renderFileTree();
        
        const nowSelected = Array.from(this.state.contextFiles);
        const added = nowSelected.filter(f => !previouslySelected.includes(f));
        const removed = previouslySelected.filter(f => !nowSelected.includes(f));
        
        // Show what changed (only if significant changes)
        if (added.length > 0 || removed.length > 0) {
            let selectionMessage = `🤖 Auto-selected ${nowSelected.length} relevant files.`;
            if (added.length > 0) {
                selectionMessage += ` Added: ${added.join(', ')}.`;
            }
            if (removed.length > 0) {
                selectionMessage += ` Removed: ${removed.join(', ')}.`;
            }
            this.renderChatMessage('ai', selectionMessage);
        }
        
        setTimeout(() => {
            this.elements.fileTree.classList.remove('file-tree-updating');
        }, 600);
        
        // --- REFACTOR LOGIC MOVED HERE ---
        const isCreatingNewPage = /(create|add|make).*(page|\.html)/i.test(prompt);
        const htmlFileCount = Object.keys(this.state.projectFiles).filter(p => p.endsWith('.html')).length;
        const hasBeenRefactored = !!(this.state.projectFiles['style.css'] || this.state.projectFiles['script.js']);

        // Check if user is adding a new page to a single-file project
        if (isCreatingNewPage && htmlFileCount === 1 && !hasBeenRefactored) {
            const shouldRefactor = await this.showConfirmation(
                "It looks like you're adding a new page to a single-file project. For better management, I recommend refactoring the CSS and JS into separate files first. Shall I do this now?"
            );
            
            if (shouldRefactor) {
                this.renderChatMessage('ai', "Understood. I will refactor the project first. Once complete, please ask for the new page again.");
                const refactorPrompt = "Refactor the project to extract embedded CSS and JavaScript into separate style.css and script.js files. Update index.html to link to them correctly.";
                // Execute the refactor and stop the original request
                await this._executeSend(refactorPrompt, true, null, `Refactor project before creating '${prompt}'`); 
                return; // IMPORTANT: Stop further execution
            } else {
                this.renderChatMessage('ai', "Refactoring has been cancelled. Your original request was not sent. What would you like to do instead?");
                return; // IMPORTANT: Stop further execution
            }
        }
        // --- END OF MOVED LOGIC ---
        
        // If refactor wasn't needed, execute the original request
        this._executeSend(prompt, false, imageToSend, historyPrompt);
        
    } else {
        // For simple cases or when auto-selection isn't suitable, execute directly
        // The refactor check is also needed here for projects with few files (e.g., just index.html)
        
        // --- REFACTOR LOGIC COPIED HERE FOR FALLBACK ---
        const isCreatingNewPage = /(create|add|make).*(page|\.html)/i.test(prompt);
        const htmlFileCount = Object.keys(this.state.projectFiles).filter(p => p.endsWith('.html')).length;
        const hasBeenRefactored = !!(this.state.projectFiles['style.css'] || this.state.projectFiles['script.js']);

        if (isCreatingNewPage && htmlFileCount === 1 && !hasBeenRefactored) {
            const shouldRefactor = await this.showConfirmation(
                "It looks like you're adding a new page to a single-file project. For better management, I recommend refactoring the CSS and JS into separate files first. Shall I do this now?"
            );
            
            if (shouldRefactor) {
                this.renderChatMessage('ai', "Understood. I will refactor the project first. Once complete, please ask for the new page again.");
                const refactorPrompt = "Refactor the project to extract embedded CSS and JavaScript into separate style.css and script.js files. Update index.html to link to them correctly.";
                await this._executeSend(refactorPrompt, true, null, `Refactor project before creating '${prompt}'`); 
                return;
            } else {
                this.renderChatMessage('ai', "Refactoring has been cancelled. Your original request was not sent. What would you like to do instead?");
                return;
            }
        }
        // --- END OF FALLBACK LOGIC ---

        this._executeSend(prompt, false, imageToSend, historyPrompt);
    }
},

async getAISelectedContextFiles(prompt, fileList, triageMode = 'general') {
    // This is a fallback that uses our auto-selection instead of making an API call
    console.log('Using auto-selection fallback instead of AI triage');
    
    if (this.state.currentAIMode === 'plan' || this.state.isExecutingProcess) {
        return this.autoSelectFilesForStep(prompt);
    } else {
        return this.autoSelectFilesForNormalMode(prompt);
    }
},

// Add this function for normal mode auto-selection
autoSelectFilesForNormalMode(prompt) {
    const promptLower = prompt.toLowerCase();
    const availableFiles = this.state.fileOrder;
    const selectedFiles = new Set();
    
    // Detect what type of request this is
    const isStyleRequest = /\b(style|css|color|design|layout|look|appearance|theme|visual)\b/i.test(prompt);
    const isJSRequest = /\b(javascript|js|function|click|event|interactive|dynamic|behavior)\b/i.test(prompt);
    const isHTMLRequest = /\b(html|page|structure|content|text|header|footer|navigation|form)\b/i.test(prompt);
    const isCreateRequest = /\b(create|add|make|build|generate)\b/i.test(prompt);
    const isModifyRequest = /\b(modify|update|change|edit|fix|improve|enhance)\b/i.test(prompt);
    
    // Always preserve manually selected images
    this.state.contextFiles.forEach(file => {
        if (this.isImageFile(file)) {
            selectedFiles.add(file);
        }
    });
    
    // Smart selection based on request type
    if (isCreateRequest) {
        // For creation requests, include relevant file types
        if (isHTMLRequest || (!isStyleRequest && !isJSRequest)) {
            availableFiles.filter(f => f.endsWith('.html')).slice(0, 2).forEach(f => selectedFiles.add(f));
        }
        if (isStyleRequest) {
            availableFiles.filter(f => f.endsWith('.css')).forEach(f => selectedFiles.add(f));
        }
        if (isJSRequest) {
            availableFiles.filter(f => f.endsWith('.js')).forEach(f => selectedFiles.add(f));
        }
    } else if (isModifyRequest) {
        // For modification requests, be more inclusive
        if (isHTMLRequest) {
            availableFiles.filter(f => f.endsWith('.html')).forEach(f => selectedFiles.add(f));
        }
        if (isStyleRequest) {
            availableFiles.filter(f => f.endsWith('.css')).forEach(f => selectedFiles.add(f));
        }
        if (isJSRequest) {
            availableFiles.filter(f => f.endsWith('.js')).forEach(f => selectedFiles.add(f));
        }
    }
    
    // Check for specific file mentions
    const filePattern = /\b([a-zA-Z0-9_.-]+\.(html|css|js|json|md))\b/gi;
    let match;
    while ((match = filePattern.exec(prompt)) !== null) {
        const filename = match[1];
        const exactFile = availableFiles.find(f => f.endsWith(filename) || f.includes(filename));
        if (exactFile) selectedFiles.add(exactFile);
    }
    
    // If still no specific selection, use smart defaults
    if (selectedFiles.size === 0) {
        // Include main files
        const mainFiles = availableFiles.filter(f => 
            f.includes('index') || f.includes('main') || f.includes('app')
        ).slice(0, 3);
        mainFiles.forEach(f => selectedFiles.add(f));
        
        // If it's a general request, include one of each type
        if (mainFiles.length === 0) {
            const htmlFile = availableFiles.find(f => f.endsWith('.html'));
            const cssFile = availableFiles.find(f => f.endsWith('.css'));
            const jsFile = availableFiles.find(f => f.endsWith('.js'));
            
            if (htmlFile) selectedFiles.add(htmlFile);
            if (cssFile) selectedFiles.add(cssFile);
            if (jsFile) selectedFiles.add(jsFile);
        }
    }
    
    return Array.from(selectedFiles);
},

captureProjectContext(prompt) {
    // Extract design and technical requirements from the user's prompt
    const designKeywords = this.extractDesignElements(prompt);
    const technicalKeywords = this.extractTechnicalElements(prompt);
    const audienceInfo = this.extractAudienceInfo(prompt);
    
    this.state.projectContext = {
        originalVision: prompt,
        designRequirements: designKeywords,
        technicalNotes: technicalKeywords,
        targetAudience: audienceInfo,
        createdAt: new Date().toISOString()
    };
    
    console.log('Project context captured:', this.state.projectContext);
},

// Add this helper function
isSimpleQuestion(prompt) {
    const simplePatterns = [
        /^(what|how|why|when|where|can you|could you|please)\b/i,
        /\b(explain|describe|tell me|show me)\b.*\?$/i,
        /^(help|assist|guide)\b/i
    ];
    
    // Short prompts are likely questions
    if (prompt.length < 100) {
        return simplePatterns.some(pattern => pattern.test(prompt));
    }
    
    return false;
},
extractDesignElements(prompt) {
    const promptLower = prompt.toLowerCase();
    const designElements = [];
    
    // Color schemes
    const colors = ['dark', 'light', 'blue', 'red', 'green', 'purple', 'orange', 'yellow', 'black', 'white', 'gray', 'colorful', 'monochrome', 'neon', 'pastel'];
    colors.forEach(color => {
        if (promptLower.includes(color)) designElements.push(`${color} color scheme`);
    });
    
    // Design styles
    const styles = ['modern', 'minimalist', 'retro', 'vintage', 'futuristic', 'elegant', 'professional', 'playful', 'bold', 'clean', 'rustic', 'industrial', 'organic', 'geometric', 'cyberpunk', 'brutalist'];
    styles.forEach(style => {
        if (promptLower.includes(style)) designElements.push(`${style} aesthetic`);
    });
    
    // Layout preferences
    if (promptLower.includes('grid')) designElements.push('grid-based layout');
    if (promptLower.includes('sidebar')) designElements.push('sidebar navigation');
    if (promptLower.includes('header')) designElements.push('prominent header');
    if (promptLower.includes('footer')) designElements.push('detailed footer');
    
    return designElements.length > 0 ? designElements.join(', ') : 'Clean, professional design';
},

extractTechnicalElements(prompt) {
    const promptLower = prompt.toLowerCase();
    const techElements = [];
    
    // Technologies mentioned
    const techs = ['react', 'vue', 'angular', 'javascript', 'php', 'python', 'node', 'database', 'api', 'responsive', 'mobile', 'desktop'];
    techs.forEach(tech => {
        if (promptLower.includes(tech)) techElements.push(tech);
    });
    
    // Functionality
    if (promptLower.includes('login') || promptLower.includes('auth')) techElements.push('user authentication');
    if (promptLower.includes('search')) techElements.push('search functionality');
    if (promptLower.includes('form')) techElements.push('form handling');
    if (promptLower.includes('animation')) techElements.push('animations and transitions');
    
    return techElements.length > 0 ? techElements.join(', ') : 'Modern web standards (HTML5, CSS3, JavaScript)';
},

extractAudienceInfo(prompt) {
    const promptLower = prompt.toLowerCase();
    
    if (promptLower.includes('business') || promptLower.includes('corporate')) return 'Business professionals';
    if (promptLower.includes('e-commerce') || promptLower.includes('shop')) return 'Online shoppers';
    if (promptLower.includes('blog') || promptLower.includes('news')) return 'Content readers';
    if (promptLower.includes('portfolio')) return 'Potential clients/employers';
    if (promptLower.includes('game') || promptLower.includes('entertainment')) return 'Entertainment seekers';
    if (promptLower.includes('education') || promptLower.includes('learning')) return 'Students and learners';
    
    return 'General web users';
},
// REPLACE your current handleParsedFiles function with this FINAL, MERGED version.

// REPLACE your current handleParsedFiles function with this FINAL, MERGED version.

// REPLACE your current handleParsedFiles function with this FINAL version.
// Its main purpose is now to return the clean conversational text.

// [REPLACE] your entire handleParsedFiles function in your App object.

async handleParsedFiles(response, bypassDiff = false, isAutoRepair = false) {
    // If this is an auto-repair, we want to completely suppress conversational output
    if (isAutoRepair) {
        // Parse files silently without any chat output
        const filesRegex = /File:\s*(.+?)\s*(?:\((NEW|MODIFY)\))?\s*\r?\n```(?:[a-zA-Z0-9_-]+)?\r?\n([\s\S]*?)\r?\n```/g;
        let filesFound = false;
        const newFiles = [];
        const modifiedFiles = [];
        
        const leftoverText = (response.replace(filesRegex, (m, path, type, content) => {
            filesFound = true;
            const file = { path: path.trim(), content: content.trim() };
            (type && type.toUpperCase() === 'MODIFY' ? modifiedFiles : newFiles).push(file);
            return '';
        }) || '').trim();

        // Apply changes silently
        if (newFiles.length > 0) {
            this.state.isBatchUpdating = true;
            for (const file of newFiles) {
                this.addFile(file.path, { content: '' }, true);
            }
            this.renderFileTree();
            for (const file of newFiles) {
                this.activateFile(file.path, true);
                // Skip typing animation for auto-repair to make it faster
                this.state.projectFiles[file.path].content = file.content;
                this.elements.codeArea.value = file.content;
            }
            this.state.isBatchUpdating = false;
        }
        
        if (modifiedFiles.length > 0) {
            this.state.isBatchUpdating = true;
            for (const file of modifiedFiles) {
                this.state.projectFiles[file.path].content = file.content;
                if (this.state.activeFile === file.path) {
                    this.elements.codeArea.value = file.content;
                }
            }
            this.state.isBatchUpdating = false;
        }
        
        this.scheduleAutoSave();
        return ''; // Return empty string to suppress all conversational output
    }

    // Normal processing for non-auto-repair responses
    let conversationalResponse = (response || '').trim();
    let aiSummary = null;
    let filesUsedByAI = [];

    const summaryRegex = /^```json\s*([\s\S]*?)\s*```/;
    const summaryMatch = conversationalResponse.match(summaryRegex);

    if (summaryMatch) {
        try {
            const jsonData = JSON.parse(summaryMatch[1]);
            if (jsonData.summary) {
                aiSummary = jsonData.summary;
            }
            if (jsonData.files_used && Array.isArray(jsonData.files_used)) {
                filesUsedByAI = jsonData.files_used;
                this.elements.fileTree.classList.add('file-tree-updating');
                this.state.contextFiles = new Set(filesUsedByAI);
                this.renderFileTree();
                setTimeout(() => {
                    this.elements.fileTree.classList.remove('file-tree-updating');
                }, 400);
            }
        } catch (e) {
            console.error("Failed to parse AI summary/context JSON:", e);
        }
        conversationalResponse = conversationalResponse.replace(summaryRegex, '').trim();
    }
    
    this.state.lastStepSummary = aiSummary; 

    const newFiles = [];
    const modifiedFiles = [];
    let filesFound = false;
    let parseSource = conversationalResponse;
    const fenceCount = (parseSource.match(/```/g) || []).length;
    if (fenceCount % 2 === 1) parseSource += '\n```';
    const filesRegex = /File:\s*(.+?)\s*(?:\((NEW|MODIFY)\))?\s*\r?\n```(?:[a-zA-Z0-9_-]+)?\r?\n([\s\S]*?)\r?\n```/g;
    const leftoverText = (parseSource.replace(filesRegex, (m, path, type, content) => {
        filesFound = true;
        const file = { path: path.trim(), content: content.trim() };
        (type && type.toUpperCase() === 'MODIFY' ? modifiedFiles : newFiles).push(file);
        return '';
    }) || '').trim();

    let finalConversationalText = '';
    // FIXED: Always show conversational text if it exists, regardless of files
    if (leftoverText) {
        finalConversationalText = leftoverText;
    } else if (newFiles.length > 0 && modifiedFiles.length === 0 && !leftoverText) {
        finalConversationalText = `I've created ${newFiles.length} new file(s) for you.`;
    }

    // HTML recovery logic
    if (!filesFound) {
        const htmlDocMatch = parseSource.match(/<!DOCTYPE html[\s\S]*<\/html>/i) || parseSource.match(/<html[\s\S]*<\/html>/i);
        if (htmlDocMatch) {
            filesFound = true;
            newFiles.push({ path: 'index.html', content: htmlDocMatch[0] });
            finalConversationalText = "I created a full HTML document and saved it as index.html.";
        }
    }

    if (newFiles.length > 0) await this.updateProjectWithTyping(newFiles);
    if (modifiedFiles.length > 0) {
        if (bypassDiff) await this.updateProjectWithTyping(modifiedFiles);
        else this.proposeModifications(modifiedFiles);
    }
    
    if (this.state.isOneShotBuild && filesFound) {
        const htmlFile = newFiles.find(f => f.path.toLowerCase().endsWith('.html'));
        if (htmlFile) {
            finalConversationalText = "Success! I've built your website. The preview will now appear.";
            const blob = new Blob([htmlFile.content], { type: 'text/html;charset=utf-8' });
            const previewURL = URL.createObjectURL(blob);
            this.state.generatedWebsiteResources = { htmlURL: previewURL, dependencies: {} };
            if (this.elements.btnViewWebsiteLink) {
                this.elements.btnViewWebsiteLink.href = previewURL;
            }
            this.toggleViewWebsiteModal(true);
            this.state.isOneShotBuild = false;
        }
    }

    this.updatePreviewButtonVisibility();
    this.scheduleAutoSave();

    return finalConversationalText;
},

startResize(e) { 
    e.preventDefault(); 
    
  
    
    const isDesktop = window.innerWidth >= 1200;
    const codePanel = this.elements.codePanel;
    const chatPanel = document.querySelector('.chat-panel');
    
    if (isDesktop) {
        // Desktop horizontal resize
        const startX = e.touches ? e.touches[0].clientX : e.clientX;
        const containerRect = document.querySelector('.panels-container').getBoundingClientRect();
        const startCodeWidth = codePanel.getBoundingClientRect().width;
        
        const moveHandler = (moveEvent) => {
            const currentX = moveEvent.touches ? moveEvent.touches[0].clientX : moveEvent.clientX;
            const deltaX = currentX - startX;
            const newCodeWidth = startCodeWidth + deltaX;
            const newCodePercent = (newCodeWidth / containerRect.width) * 100;
            
            // Constrain between 20% and 80%
            if (newCodePercent >= 20 && newCodePercent <= 80) {
                const newChatPercent = 100 - newCodePercent;
                // Force override the CSS with important
                codePanel.style.setProperty('flex-basis', `${newCodePercent}%`, 'important');
                chatPanel.style.setProperty('flex-basis', `${newChatPercent}%`, 'important');
            }
        };
        
        const endHandler = () => {
            document.removeEventListener('mousemove', moveHandler);
            document.removeEventListener('touchmove', moveHandler);
            document.removeEventListener('mouseup', endHandler);
            document.removeEventListener('touchend', endHandler);
        };
        
        document.addEventListener('mousemove', moveHandler);
        document.addEventListener('touchmove', moveHandler);
        document.addEventListener('mouseup', endHandler);
        document.addEventListener('touchend', endHandler);
        
    } else {
        // Mobile vertical resize
        const startY = e.touches ? e.touches[0].clientY : e.clientY;
        const containerRect = document.querySelector('.panels-container').getBoundingClientRect();
        const startCodeHeight = codePanel.getBoundingClientRect().height;
        
        const moveHandler = (moveEvent) => {
            const currentY = moveEvent.touches ? moveEvent.touches[0].clientY : moveEvent.clientY;
            const deltaY = currentY - startY;
            const newCodeHeight = startCodeHeight + deltaY;
            const newCodePercent = (newCodeHeight / containerRect.height) * 100;
            
            // Constrain between 20% and 80%
            if (newCodePercent >= 20 && newCodePercent <= 80) {
                const newChatPercent = 100 - newCodePercent;
                // Force override the CSS with important
                codePanel.style.setProperty('flex-basis', `${newCodePercent}%`, 'important');
                chatPanel.style.setProperty('flex-basis', `${newChatPercent}%`, 'important');
            }
        };
        
        const endHandler = () => {
            document.removeEventListener('mousemove', moveHandler);
            document.removeEventListener('touchmove', moveHandler);
            document.removeEventListener('mouseup', endHandler);
            document.removeEventListener('touchend', endHandler);
        };
        
        document.addEventListener('mousemove', moveHandler);
        document.addEventListener('touchmove', moveHandler);
        document.addEventListener('mouseup', endHandler);
        document.addEventListener('touchend', endHandler);
    }
},
 initializeWelcome() { let welcomeHTML = `<div class="chat-message ai"><div class="chat-avatar">🤖</div><div class="message-bubble">Welcome to ZenCode AI! You can start by asking me to create a project, or open the <code>README.md</code> file for more info.`; if (zencode_ai_app_vars.app_page_url) { welcomeHTML += `<div style="margin-top: 1rem; padding-top: 1rem; border-top: 1px solid var(--border-primary);"><p style="font-weight: 500;">🚀 Your Assistant is Live</p><p style="font-size: 0.9rem; color: var(--text-secondary); margin-top: 0.25rem;">You can access this page directly via the link below.</p><a href="${this.escapeHtml(zencode_ai_app_vars.app_page_url)}" target="_blank" style="display: inline-block; margin-top: 0.75rem; padding: 0.5rem 1rem; background: var(--accent-gradient); color: white; text-decoration: none; border-radius: var(--radius-md); font-weight: 500;">${this.escapeHtml(zencode_ai_app_vars.app_page_title)} →</a></div>`; } welcomeHTML += `</div></div>`; this.elements.chatMessages.innerHTML = welcomeHTML; },
        updateChatPlaceholder() { const placeholders = { normal: 'Ask AI to create or modify code...', plan: 'Describe the plan for the AI to create...', vector: 'Ask questions about your project...' }; this.elements.chatInput.placeholder = placeholders[this.state.currentAIMode]; },
               addFile(path, data, suppressSave = false) {
            const isNewFile = !this.state.projectFiles[path];
            if (isNewFile) this.state.fileOrder.push(path);
            this.state.projectFiles[path] = typeof data === 'string' ? { content: data } : data;
            this.state.contextFiles.add(path);
            
            // This is the only change. We check the batch flag.
            if (!suppressSave && !this.state.isBatchUpdating) {
                this.scheduleAutoSave();
            }
            
            // We no longer call renderFileTree() here for performance.
            // It will be called once at the end of the batch.
        },

	   deleteFile(path) { delete this.state.projectFiles[path]; const index = this.state.fileOrder.indexOf(path); if (index > -1) this.state.fileOrder.splice(index, 1); this.state.contextFiles.delete(path); this.scheduleAutoSave(); },
        getParentPath(path) { const parts = path.split('/'); return parts.length <= 1 ? '' : parts.slice(0, -1).join('/'); },
        getIconForFile(path) { const extension = path.split('.').pop().toLowerCase(); switch (extension) { case 'js': return '📒'; case 'html': return '📙'; case 'css': return '📘'; case 'php': return '🐘'; case 'json': return '📋'; case 'md': return '📝'; case 'png': case 'jpg': case 'jpeg': case 'gif': case 'svg': return '🖼️'; default: return '📄'; }},
        renderFileTree() {
            const fileStructure = {};
            const tempOrder = [...this.state.fileOrder];
            for (const path in this.state.projectFiles) { if (!tempOrder.includes(path)) tempOrder.push(path); }
            this.state.fileOrder = tempOrder.filter(p => this.state.projectFiles[p]);
            this.state.fileOrder.forEach(path => {
                let currentLevel = fileStructure;
                path.split('/').forEach((part, index, arr) => {
                    if (index === arr.length - 1) { currentLevel[part] = path; } 
                    else { if (!currentLevel[part]) { currentLevel[part] = {}; } currentLevel = currentLevel[part]; }
                });
            });
            const generateHtml = (structure, pathPrefix = '') => {
                let html = '';
                Object.entries(structure).forEach(([name, content]) => {
                    const currentPath = pathPrefix ? `${pathPrefix}/${name}` : name;
                    if (typeof content === 'string') {
                        if (name === '.keep') return;
                        const icon = this.getIconForFile(name);
                        const isChecked = this.state.contextFiles.has(content);
                        const fileData = this.state.projectFiles[content];
                        const isBinary = !!fileData.rawFile || fileData.isBinary === true;
                        html += `
                            <div class="file-tree-item ${this.state.activeFile === content ? 'active' : ''}" data-path="${content}" draggable="true">
                                <div class="file-name"><input type="checkbox" class="context-toggle" data-path="${content}" ${isChecked ? 'checked' : ''} title="Include in AI context">${icon} ${this.escapeHtml(name)}</div>
                                <div class="file-actions-container">
                                    ${isBinary ? `<span class="preview-btn" data-path="${content}" title="Preview File"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg></span>` : ''}
                                    ${!isBinary ? `` : ''}
                                    <span class="vectorize-btn" data-path="${content}" title="Save to Knowledge Base"><svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 8.5c-1.5 0-2.5 1-2.5 2.5S10.5 13.5 12 13.5s2.5-1 2.5-2.5S13.5 8.5 12 8.5z"/><path d="M20.2 18.8c.8-1.3 1.2-2.8 1-4.3-.4-2.8-2.6-5-5.4-5.2-1.6-.1-3.1.5-4.3 1.5-1.2-1-2.7-1.6-4.3-1.5-2.8.2-5 2.4-5.4 5.2-.2 1.5.2 3 .9 4.3"/><path d="M12 13.5V16c0 1.1.9 2 2 2h3.5"/><path d="M12 8.5V6c0-1.1-.9-2-2-2h-1"/><path d="M12 8.5V6c0-1.1.9-2 2-2h1"/><path d="M4.8 18.8c.2-.6.3-1.2.3-1.8V12c0-1.1.9-2 2-2h1.4"/><path d="M19.2 18.8c-.2-.6-.3-1.2-.3-1.8V12c0-1.1-.9-2-2-2h-1.4"/><path d="M12 13.5V16c0 1.1-.9 2-2 2H6.5"/></svg></span>
                                    <span class="delete-btn" data-path="${content}"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg></span>
                                </div>
                            </div>`;
                    } else {
                        const isCollapsed = this.state.collapsedFolders.has(currentPath);
                        html += `<div class="folder-item ${!isCollapsed ? 'open' : ''}" data-folder-path="${currentPath}" draggable="true"><div class="folder-name"><span class="folder-arrow">▶</span> <span style="color: #FFD700;">📂</span> ${this.escapeHtml(name)}</div><div class="file-actions-container"><span class="delete-btn" data-folder-path="${currentPath}"><svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><polyline points="3 6 5 6 21 6"></polyline><path d="M19 6v14a2 2 0 0 1-2 2H7a2 2 0 0 1-2-2V6m3 0V4a2 2 0 0 1 2-2h4a2 2 0 0 1 2 2v2"></path></svg></span></div></div><div class="folder-content" ${isCollapsed ? 'style="display: none;"' : ''}>${generateHtml(content, currentPath)}</div>`;
                    }
                });
                return html;
            };
            this.elements.fileTree.innerHTML = generateHtml(fileStructure);
        },
        createNewFolder() { const folderPath = prompt("Enter the new folder path (e.g., 'assets/images'):"); if (folderPath) { const cleanedPath = folderPath.trim().replace(/\/$/, ''); if (cleanedPath) { const placeholderPath = `${cleanedPath}/.keep`; if (Object.keys(this.state.projectFiles).some(p => p.startsWith(cleanedPath))) { alert('A folder or file with this name already exists.'); return; } this.addFile(placeholderPath, ''); this.renderChatMessage('ai', `📁 Folder \`${cleanedPath}\` created.`); }}},
// REPLACE your current animateTyping function with this FINAL, "chunk-aware" version.
// [REPLACE THIS ENTIRE FUNCTION IN app.js]
async animateTyping(element, text) {
    if (!this.state.codeMirrorInstance) return; // Safety check

    const cm = this.state.codeMirrorInstance;
    const typingId = Date.now() + Math.random();
    this.state.currentTypingId = typingId;
    this.state.isTyping = true;
    
    // Instantly set text if no delay
    if (this.config.TYPING_DELAY_MS <= 0) {
        cm.setValue(text);
        this.state.isTyping = false;
        this.state.currentTypingId = null;
        return;
    }

    const chunkSize = this.config.TYPING_CHUNK_SIZE || 1;
    const delay = this.config.TYPING_DELAY_MS;
    
    cm.setValue('');
    cm.focus();
    
    return new Promise(resolve => {
        let i = 0;
        const type = () => {
            if (this.state.isTyping && this.state.currentTypingId === typingId && i < text.length) {
                const chunk = text.slice(i, i + chunkSize);
                // Use CodeMirror's efficient replaceRange to append text
                cm.replaceRange(chunk, cm.getDoc().posFromIndex(Infinity));
                i += chunkSize;
                
                setTimeout(type, delay);
            } else {
                if (this.state.currentTypingId === typingId && this.state.isTyping) {
                    cm.setValue(text); // Ensure final content is perfect
                }
                
                if (this.state.currentTypingId === typingId) {
                    this.state.isTyping = false;
                    this.state.currentTypingId = null;
                }
                resolve();
            }
        };
        
        type();
    });
},
// In your app initialization or user data loading
updateUsageDisplay() {
    const userPlan = zencode_ai_app_vars.user_plan || 'spark'; // from your PHP localized data
    const planName = {
        'spark': 'Spark',
        'forge': 'Forge', 
        'visionary': 'Visionary'
    }[userPlan];

    // You'll need to add an AJAX call to get current usage
    fetch(zencode_ai_app_vars.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            action: 'zencode_ai_get_user_usage',
            nonce: zencode_ai_app_vars.nonce
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            document.querySelector('.plan-name').textContent = `${planName} Plan`;
            document.querySelector('.usage-count').textContent = `${data.data.used}/${data.data.limit}`;
        }
    });
},	
		async updateProjectWithTyping(files) {
            this.state.isBatchUpdating = true;
            for (const file of files) { this.addFile(file.path, { content: '' }, true); }
            this.renderFileTree();
           // [REPLACE THE LOOP with this new version]
for (const file of files) {
    this.activateFile(file.path, true);
    await this.animateTyping(this.elements.codeArea, file.content);
    this.state.projectFiles[file.path].content = file.content;
    
    // --- THIS IS THE FIX ---
    // Give the editor a "nudge" to re-apply syntax highlighting
    // after the animation is complete.
    if (this.state.codeMirrorInstance) {
        this.state.codeMirrorInstance.setOption('mode', this.getCodeMirrorMode(file.path));
    }
    // --- END OF FIX ---
}
            this.scheduleAutoSave();
            this.state.isBatchUpdating = false;
        },
      proposeModifications(modifiedFiles) {
    if (!this.state.hasJsDiff || modifiedFiles.length === 0) {
        this.updateProjectWithTyping(modifiedFiles);
        return;
    }

    // --- THE DEFINITIVE FIX: SORT THE QUEUE BY PRIORITY ---

    // 1. Define a helper to assign a priority score to each file type.
    //    Lower numbers will be processed first.
    const getPriority = (path) => {
        const lowerPath = path.toLowerCase();
        if (lowerPath.endsWith('.css')) return 1;    // CSS is highest priority.
        if (lowerPath.endsWith('.js')) return 2;     // JavaScript is second.
        if (lowerPath.endsWith('.html')) return 3;   // HTML is third.
        return 4;                                    // Everything else is last.
    };

    // 2. Sort the incoming `modifiedFiles` array based on our priority scores.
    const sortedModifiedFiles = [...modifiedFiles].sort((a, b) => {
        return getPriority(a.path) - getPriority(b.path);
    });

    // --- END OF THE FIX ---

    // 3. Start the review session using the CORRECTLY ORDERED array.
    this.state.modificationQueue = sortedModifiedFiles;
    this.state.currentModificationSessionFiles = sortedModifiedFiles.map(file => file.path);

    // 4. Begin the review process. It will now present files in the right order.
    this._showNextModification();
},
// [ADD] this new helper function to your App object.
_showNextModification() {
    // If the queue is empty, it means we've reviewed all files. Clean up and exit.
    if (this.state.modificationQueue.length === 0) {
        this.toggleModificationControls(false); // Hide Accept/Reject buttons
        this.state.pendingModification = null;
        this.renderChatMessage('ai', 'All changes have been reviewed.');
        return;
    }

    // Get the next file from the front of the queue.
    const file = this.state.modificationQueue.shift();

    if (this.state.projectFiles[file.path]) {
        // --- All of your original logic now lives here ---
        this.renderChatMessage('ai', `I have suggested changes for \`${file.path}\`. Please review (${this.state.modificationQueue.length} more file(s) in queue).`);
        
        const oldContent = this.state.projectFiles[file.path].content;
        const newContent = file.content;
        
        // Store this file as the currently pending modification.
        this.state.pendingModification = { path: file.path, oldContent, newContent };
        
        const diff = Diff.diffLines(oldContent, newContent, { newlineIsToken: true });
        let html = '';
        diff.forEach(part => {
            const escapedValue = this.escapeHtml(part.value);
            if (part.added) { html += `<span class="line-highlight-add">${escapedValue}</span>`; } 
            else if (part.removed) { html += `<span class="line-highlight-remove">${escapedValue}</span>`; } 
            else { html += escapedValue; }
        });
        
        this.elements.codePreview.innerHTML = html;
        this.activateFile(file.path, true);
        this.toggleModificationControls(true); // Show Accept/Reject buttons.
        
    } else {
        // If for some reason the file to be modified doesn't exist, treat it as a new file.
        this.updateProjectWithTyping([file]);
        // Immediately try to show the next file in the queue.
        this._showNextModification();
    }
},
    // [REPLACE THESE two functions in app.js]
acceptChanges() {
    if (!this.state.pendingModification) return;
    const { path, newContent } = this.state.pendingModification;
    
    this.state.projectFiles[path].content = newContent;
    if (this.state.codeMirrorInstance) {
        this.state.codeMirrorInstance.setValue(newContent);
    }
    
    this.renderChatMessage('ai', `✅ Changes for \`${path}\` accepted.`);
    this.state.pendingModification = null;
    this.scheduleAutoSave();
    this._showNextModification();
},

     rejectChanges() {
    if (!this.state.pendingModification) return;
    const { path, oldContent } = this.state.pendingModification;
    
    // Restore the old content to the editor
    if (this.state.codeMirrorInstance) {
        this.state.codeMirrorInstance.setValue(oldContent);
    }
    
    this.renderChatMessage('ai', `❌ Changes for \`${path}\` rejected.`);
    this.state.pendingModification = null;
    this._showNextModification();
},
	// [REPLACE THIS ENTIRE FUNCTION IN app.js]
toggleModificationControls(show) {
    this.elements.btnAcceptChanges.style.display = show ? 'inline-flex' : 'none';
    this.elements.btnRejectChanges.style.display = show ? 'inline-flex' : 'none';
    this.elements.btnSaveFile.style.display = show ? 'none' : (this.state.activeFile ? 'inline-flex' : 'none');
    
    // Hide/show CodeMirror vs the diff preview
    if (this.state.codeMirrorInstance) {
        this.state.codeMirrorInstance.getWrapperElement().style.display = show ? 'none' : 'block';
    }
    this.elements.codePreview.style.display = show ? 'block' : 'none';
},

renderChatMessage(role, content, isHistory = false, returnElement = false) {
    let msgEl;
    let finalContent = '';
    let textContent = '';
    let imageContent = '';

    if (typeof content === 'object' && content !== null) {
        textContent = content.text || '';
        imageContent = content.image || '';
    } else {
        textContent = content || '';
    }

    let isPlanHandled = false;

    // IMPROVED: More strict JSON detection
    if (role === 'ai' && textContent.includes('{') && textContent.includes('}')) {
        try {
            const firstBrace = textContent.indexOf('{');
            const lastBrace = textContent.lastIndexOf('}');
            
            if (firstBrace !== -1 && lastBrace > firstBrace) {
                const jsonString = textContent.substring(firstBrace, lastBrace + 1);
                
                // NEW: Validate this looks like a plan before trying to parse
                const looksLikePlan = jsonString.includes('"introduction"') && 
                                     jsonString.includes('"plan"') &&
                                     jsonString.includes('[');
                
                if (looksLikePlan) {
                    const planData = JSON.parse(jsonString);

                    // Validate that it's a real plan structure
                    if (planData.introduction && Array.isArray(planData.plan)) {
                        isPlanHandled = true;

                        // Render the intro as a separate message first
                        this.renderChatMessage('ai', planData.introduction);
                        
                        // Create the interactive plan steps
                        const planContent = this.makePlanInteractive(planData.plan);
                        
                        msgEl = document.createElement('div');
                        msgEl.classList.add('chat-message', 'ai');
                        msgEl.innerHTML = `<div class="chat-avatar">🤖</div><div class="message-bubble">${planContent}</div>`;
                    }
                }
            }
        } catch (error) {
            // Silently fail - this wasn't a plan, just regular text with braces
            console.log("Text contained braces but wasn't a valid plan:", error.message);
            isPlanHandled = false;
        }
    }

    // If the response was NOT a valid plan, render it as a standard message
    if (!isPlanHandled) {
        finalContent = textContent
            .replace(/```([\s\S]*?)```/g, (match, p1) => `<pre><code>${this.escapeHtml(p1)}</code></pre>`)
            .replace(/`([^`]+)`/g, '<code>$1</code>')
            .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
            .replace(/\n/g, '<br>');

        if (imageContent) {
            finalContent = `<img src="${imageContent}" alt="User upload" style="max-width: 100%; border-radius: 8px; margin-bottom: 8px;" /><br>` + finalContent;
        }
        
        msgEl = document.createElement('div');
        msgEl.classList.add('chat-message', role);
        msgEl.innerHTML = `<div class="chat-avatar">${role === 'user' ? '<svg width="24" height="24" viewBox="0 0 24 24" fill="white" xmlns="http://www.w3.org/2000/svg"><path d="M12 12C14.21 12 16 10.21 16 8C16 5.79 14.21 4 12 4C9.79 4 8 5.79 8 8C8 10.21 9.79 12 12 12ZM12 14C9.33 14 4 15.34 4 18V20H20V18C20 15.34 14.67 14 12 14Z"/></svg>' : '🤖'}</div><div class="message-bubble">${finalContent}</div>`;
    }

    if (returnElement) {
        return msgEl;
    } else {
        this.elements.chatMessages.appendChild(msgEl);
        if (!isHistory) {
            this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
        }
    }
},
       escapeHtml(text) {
    if (typeof text !== 'string') return '';
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#x27;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
},
        async handleDelete(path, isFolder) {
            const message = isFolder ? `Are you sure you want to delete the folder "${path}" and all its contents?` : `Are you sure you want to delete ${path}?`;
            if (await this.showConfirmation(message)) {
                if (window.innerWidth <= 768) { this.toggleSidebar(false); }
                if (isFolder) {
                    const pathsToDelete = this.state.fileOrder.filter(p => p.startsWith(path + '/'));
                    pathsToDelete.push(`${path}/.keep`);
                    pathsToDelete.forEach(p => this.deleteFile(p));
                    if (this.state.activeFile && this.state.activeFile.startsWith(path + '/')) { this.activateFile(null); }
                } else {
                    if (this.state.pendingModification && this.state.pendingModification.path === path) { this.rejectChanges(); }
                    this.deleteFile(path);
                    if (this.state.activeFile === path) { this.activateFile(null); }
                }
                this.renderFileTree();
                this.updatePreviewButtonVisibility();
            }
        },
        handleTreeClick(e) {
            const fileItem = e.target.closest('.file-tree-item');
            const folderItem = e.target.closest('.folder-item');
            const deleteBtn = e.target.closest('.delete-btn');
            const contextToggle = e.target.closest('.context-toggle');
            const vectorizeBtn = e.target.closest('.vectorize-btn');
            const previewBtn = e.target.closest('.preview-btn');
            
            
            if (previewBtn) { e.stopPropagation(); this.showImagePreview(previewBtn.dataset.path); return; }
            if (contextToggle) { const path = contextToggle.dataset.path; if (contextToggle.checked) this.state.contextFiles.add(path); else this.state.contextFiles.delete(path); return; }
            if (vectorizeBtn) { e.stopPropagation(); if (window.innerWidth <= 768) { this.toggleSidebar(false); } this.vectorizeFile(vectorizeBtn.dataset.path); return; }
            if (deleteBtn) { e.stopPropagation(); this.handleDelete(deleteBtn.dataset.path || deleteBtn.dataset.folderPath, !!deleteBtn.dataset.folderPath); }
            else if (fileItem) { this.activateFile(fileItem.dataset.path); }
            else if (folderItem) { const folderPath = folderItem.dataset.folderPath; if (this.state.collapsedFolders.has(folderPath)) this.state.collapsedFolders.delete(folderPath); else this.state.collapsedFolders.add(folderPath); this.renderFileTree(); this.scheduleAutoSave(); }
        },
      // [REPLACE THIS ENTIRE FUNCTION IN app.js]
activateFile(path, force = false) {
    if (this.state.pendingModification && !force) {
        this.renderChatMessage('ai', 'Please accept or reject the pending changes before switching files.');
        return;
    }
    if (!this.state.pendingModification) {
        this.toggleModificationControls(false);
    }

    this.syncEditorToFileState();
    const cm = this.state.codeMirrorInstance;

    if (path === null) {
        this.state.activeFile = null;
        if (cm) {
            cm.setValue('');
            cm.setOption('readOnly', true);
        }
        this.elements.editorFilename.textContent = 'Editor';
        this.elements.btnSaveFile.style.display = 'none';
    } else if (this.state.projectFiles[path]) {
        this.state.activeFile = path;
        const fileData = this.state.projectFiles[path];
        
        if (cm) {
            cm.setValue(fileData.content || '');
            const isReadOnly = !!fileData.rawFile;
            cm.setOption('readOnly', isReadOnly);
            cm.setOption('mode', this.getCodeMirrorMode(path));
            setTimeout(() => cm.refresh(), 1);
        }
        
        this.elements.editorFilename.textContent = this.escapeHtml(path);
        if (!this.state.pendingModification) {
            this.elements.btnSaveFile.style.display = 'inline-flex';
        }
    }
    this.renderFileTree();
    this.updatePreviewButtonVisibility();
},
        toggleSidebar(forceState) { const isMobile = window.innerWidth <= 768; if (isMobile) { this.state.isSidebarOpen = typeof forceState === 'boolean' ? forceState : !this.state.isSidebarOpen; this.elements.sidebar.classList.toggle('open', this.state.isSidebarOpen); this.elements.mobileOverlay.classList.toggle('visible', this.state.isSidebarOpen); } else { this.state.isSidebarOpen = !this.elements.sidebar.classList.contains('hidden'); this.elements.sidebar.classList.toggle('hidden', this.state.isSidebarOpen); }},
        handleResponsiveLayout() { const isMobile = window.innerWidth <= 768; this.elements.sidebarToggleHeader.style.display = 'flex'; if(!isMobile) { this.elements.sidebar.classList.remove('open'); this.elements.mobileOverlay.classList.remove('visible'); } else { this.elements.sidebar.classList.remove('hidden'); }},
        async downloadProjectAsZip() { if (this.state.isBatchUpdating) { alert("Please wait for the AI to finish generating files before downloading the project."); return; } if (!this.state.hasJszip) { this.renderChatMessage('ai', 'Error: JSZip library failed to load, so this feature is disabled.'); return; } this.syncEditorToFileState(); const zip = new JSZip(); const projectName = this.elements.projectNameInput.value.trim() || 'zencode-ai-project'; for (const path in this.state.projectFiles) { if (path.endsWith('/.keep')) continue; const fileData = this.state.projectFiles[path]; if (fileData.rawFile) { zip.file(path, fileData.rawFile); } else { zip.file(path, fileData.content); } } const blob = await zip.generateAsync({type:"blob"}); const link = document.createElement('a'); link.href = URL.createObjectURL(blob); link.download = `${projectName}.zip`; link.click(); URL.revokeObjectURL(link.href); },
       



	   
// Update your sync method to handle CodeMirror properly
syncEditorToFileState() {
    if (this.state.activeFile && this.state.codeMirrorInstance) {
        const currentContent = this.state.codeMirrorInstance.getValue();
        if (this.state.projectFiles[this.state.activeFile]) {
            this.state.projectFiles[this.state.activeFile].content = currentContent;
        }
    }
},
		
		
		downloadActiveFile() { if (!this.state.activeFile || this.state.pendingModification) return; const fileData = this.state.projectFiles[this.state.activeFile]; const filename = this.state.activeFile.split('/').pop(); const blob = fileData.rawFile ? new Blob([fileData.rawFile], { type: fileData.rawFile.type }) : new Blob([fileData.content], { type: 'text/plain;charset=utf-8' }); const link = document.createElement('a'); link.href = URL.createObjectURL(blob); link.download = filename; link.click(); URL.revokeObjectURL(link.href); },
        async handleFileUpload(event) {
            const files = event.target.files;
            if (!files.length) return;

            const fileCount = files.length;

            // Show a temporary loading message for large uploads
            if (fileCount > 5) {
                this.renderChatMessage('ai', `⚙️ Processing ${fileCount} files... please wait.`);
            }

            // Set the flag *before* adding files. We store the count.
            this.state._postUploadSave = fileCount;

            for (const file of Array.from(files)) {
                const isEditable = this.config.EDITABLE_FILE_EXTENSIONS.has(file.name.split('.').pop().toLowerCase());
                if (isEditable) {
                    const content = await file.text();
                    this.addFile(file.name, { content });
                } else {
                    const base64Content = await this._fileToBase64(file);
                    const placeholderContent = `[Binary File: ${file.name}]\n[Size: ${(file.size / 1024).toFixed(2)} KB]\n\nThis file is not editable. Click the eye icon to preview.`;
                    this.addFile(file.name, { 
                        content: placeholderContent, 
                        rawFile: file,
                        isBinary: true,
                        base64Content: base64Content
                    });
                }
            }

            this.elements.fileUploadInput.value = '';
            this.renderFileTree();
            
            // NEW, MORE ACCURATE MESSAGE
            // This tells the user the files are ready and a save is imminent.
            this.renderChatMessage('ai', `✅ ${fileCount} file(s) have been added to the workspace. Saving project...`);
            
            // The `addFile` function automatically triggers the auto-save, so we don't need to call it here.
            if (window.innerWidth <= 768) {
                setTimeout(() => this.toggleSidebar(false), 2000);
            }
        },
	 
        async handleZipUpload(event) {
            const file = event.target.files[0];
            if (!file) return;

            if (!await this.showConfirmation("This will unzip the project. Existing files with the same name will be overwritten. Continue?")) {
                this.elements.zipUploadInput.value = '';
                return;
            }

            this.setLoadingState(true);
            this.renderChatMessage('ai', 'Unzipping project...');

            this.state.isBatchUpdating = true;

            try {
                const zip = await JSZip.loadAsync(file);
                const fileEntries = [];
                
                zip.forEach((relativePath, zipEntry) => {
                    if (!zipEntry.dir && !relativePath.startsWith('__MACOSX')) {
                        fileEntries.push({ relativePath, zipEntry });
                    }
                });

                const totalFiles = fileEntries.length;
                if (totalFiles === 0) {
                    this.renderChatMessage('ai', '⚠️ The ZIP file appears to be empty or contains only folders.');
                    return;
                }

                // <-- THE FIX: Arm the save function with the file count.
                this.state._postUploadSave = totalFiles; 

                const progressModal = document.createElement("div");
                progressModal.style.cssText = 'position: fixed; top: 50%; left: 50%; transform: translate(-50%, -50%); background: var(--bg-tertiary); color: white; padding: 2rem; border-radius: var(--radius-lg); z-index: 10001; text-align: center; border: 1px solid var(--border-primary); box-shadow: var(--shadow-xl);';
                progressModal.innerHTML = `
                    <p style="margin: 0 0 1rem 0; font-weight: 500;">Extracting <span id="extract-count">0</span> of ${totalFiles} files...</p>
                    <div style="background: var(--bg-accent); height: 10px; border-radius: 5px; margin: 10px 0;">
                        <div id="extract-bar" style="background: var(--accent-primary); height: 100%; width: 0%; border-radius: 5px; transition: width 0.1s linear;"></div>
                    </div>
                `;
                document.body.appendChild(progressModal);

                const BATCH_SIZE = 10;
                let processedCount = 0;

                for (let i = 0; i < totalFiles; i += BATCH_SIZE) {
                    const batch = fileEntries.slice(i, i + BATCH_SIZE);
                    
                    await Promise.all(batch.map(async ({ relativePath, zipEntry }) => {
                        const blob = await zipEntry.async('blob');
                        const isEditable = this.config.EDITABLE_FILE_EXTENSIONS.has(relativePath.split('.').pop().toLowerCase());

                        if (isEditable) {
                            const content = await blob.text();
                            this.addFile(relativePath, { content });
                        } else {
                            const fileObject = new File([blob], zipEntry.name.split('/').pop(), { type: blob.type });
                            const base64Content = await this._fileToBase64(fileObject);
                            const placeholderContent = `[Binary File: ${fileObject.name}]\n[Size: ${(fileObject.size / 1024).toFixed(2)} KB]`;
                            this.addFile(relativePath, { 
                                content: placeholderContent, 
                                rawFile: fileObject,
                                isBinary: true,
                                base64Content: base64Content
                            });
                        }
                    }));
                    
                    processedCount += batch.length;
                    
                    const percent = (processedCount / totalFiles) * 100;
                    progressModal.querySelector("#extract-count").textContent = processedCount;
                    progressModal.querySelector("#extract-bar").style.width = percent + "%";

                    await new Promise(resolve => setTimeout(resolve, 50)); 
                }

                document.body.removeChild(progressModal);
                
                this.renderFileTree();
                this.renderChatMessage('ai', `✅ Project unzipped successfully. Saving ${totalFiles} files...`);
                this.updatePreviewButtonVisibility();

            } catch (error) {
                this.renderChatMessage('ai', `Error: Failed to unzip file. ${error.message}`);
                console.error("ZIP Error:", error);
            } finally {
                this.state.isBatchUpdating = false;
                this.saveProject();
                this.setLoadingState(false);
                this.elements.zipUploadInput.value = '';
            }
        },
		handleTreeDragStart(e) { const target = e.target.closest('.file-tree-item, .folder-item'); if (target) { const isFolder = target.classList.contains('folder-item'); const path = isFolder ? target.dataset.folderPath : target.dataset.path; e.dataTransfer.setData('application/json', JSON.stringify({ path, isFolder })); e.dataTransfer.effectAllowed = 'move'; }},
        handleTreeDragOver(e) { e.preventDefault(); this.removeDropIndicator(); const target = e.target.closest('.file-tree-item, .folder-item, .file-tree'); if (!target) return; if (target.classList.contains('folder-item')) { target.classList.add('drag-over'); } else { const rect = target.getBoundingClientRect(); const isTopHalf = (e.clientY - rect.top) < (rect.height / 2); const indicator = document.createElement('div'); indicator.className = `drop-indicator ${isTopHalf ? 'top' : 'bottom'}`; target.appendChild(indicator); }},
        handleTreeDragLeave(e) { const target = e.target.closest('.folder-item'); if(target) target.classList.remove('drag-over'); this.removeDropIndicator(); },
        removeDropIndicator() { const indicator = this.elements.fileTree.querySelector('.drop-indicator'); if (indicator) indicator.remove(); },
        handleTreeDrop(e) { e.preventDefault(); this.removeDropIndicator(); const targetEl = e.target.closest('.folder-item, .file-tree-item, .file-tree'); const droppedOnFolder = targetEl && targetEl.classList.contains('folder-item'); if (droppedOnFolder) targetEl.classList.remove('drag-over'); const dragData = JSON.parse(e.dataTransfer.getData('application/json')); if (!dragData) return; const draggedPath = dragData.path; let newParentPath = ''; let targetPath = targetEl.dataset.path || targetEl.dataset.folderPath; if (droppedOnFolder) { newParentPath = targetEl.dataset.folderPath; } else { newParentPath = targetPath ? this.getParentPath(targetPath) : ''; } if (dragData.isFolder && (newParentPath.startsWith(draggedPath) || newParentPath === draggedPath)) { alert("Cannot move a folder into itself."); return; } this.moveAndReorderItems(dragData, newParentPath, targetEl); },
        moveAndReorderItems(dragData, newParentPath, targetEl) { const oldPath = dragData.path; const baseName = oldPath.split('/').pop(); const newPath = newParentPath ? `${newParentPath}/${baseName}` : baseName; const itemsToMove = dragData.isFolder ? this.state.fileOrder.filter(p => p.startsWith(oldPath + '/')) : []; itemsToMove.unshift(oldPath); if (this.state.projectFiles[newPath] && oldPath !== newPath && !itemsToMove.includes(newPath)) { alert(`An item named "${baseName}" already exists in the destination.`); return; } const originalActiveFile = this.state.activeFile; const movedData = itemsToMove.map(p => ({ oldPath: p, newPath: p.replace(oldPath, newPath), data: this.state.projectFiles[p] })); let currentOrder = this.state.fileOrder.filter(p => !itemsToMove.includes(p)); let targetIndex; const targetPath = targetEl.dataset.path || (targetEl.dataset.folderPath ? `${targetEl.dataset.folderPath}/.keep` : null); const droppedOnFolder = targetEl && targetEl.classList.contains('folder-item'); if (droppedOnFolder) { const children = currentOrder.filter(p => this.getParentPath(p) === targetEl.dataset.folderPath); if (children.length > 0) { targetIndex = currentOrder.indexOf(children[children.length - 1]) + 1; } else { const folderPlaceholder = `${targetEl.dataset.folderPath}/.keep`; const folderIndex = currentOrder.indexOf(folderPlaceholder); targetIndex = folderIndex !== -1 ? folderIndex + 1 : currentOrder.length; }} else if (targetPath) { targetIndex = currentOrder.indexOf(targetPath); } else { targetIndex = currentOrder.length; } currentOrder.splice(targetIndex, 0, ...movedData.map(d => d.newPath)); this.state.fileOrder = currentOrder; itemsToMove.forEach(p => delete this.state.projectFiles[p]); movedData.forEach(d => this.state.projectFiles[d.newPath] = d.data); if (originalActiveFile && itemsToMove.includes(originalActiveFile)) { this.state.activeFile = originalActiveFile.replace(oldPath, newPath); } this.renderFileTree(); if (this.state.activeFile) this.activateFile(this.state.activeFile, true); this.updatePreviewButtonVisibility(); this.scheduleAutoSave(); },
        async handleExternalDrop(e) { e.preventDefault(); this.elements.sidebar.classList.remove('drag-over'); const items = e.dataTransfer.items; const promises = []; for (let i = 0; i < items.length; i++) { const entry = items[i].webkitGetAsEntry(); if (entry) { promises.push(this.traverseFileTree(entry)); } } await Promise.all(promises); this.renderFileTree(); this.renderChatMessage('ai', '✅ Files and folders have been uploaded.'); },
        async traverseFileTree(item, path = '') { if (item.isFile) { return new Promise(resolve => { item.file(async file => { const fullPath = path ? `${path}/${file.name}` : file.name; const isEditable = this.config.EDITABLE_FILE_EXTENSIONS.has(file.name.split('.').pop().toLowerCase()); if (isEditable) { const content = await file.text(); this.addFile(fullPath, { content }); } else { const placeholderContent = `[Binary File: ${file.name}]\n[Size: ${(file.size / 1024).toFixed(2)} KB]\n\nThis file is not editable. Click the eye icon to preview.`; this.addFile(fullPath, { content: placeholderContent, rawFile: file }); } resolve(); }); }); } else if (item.isDirectory) { const dirReader = item.createReader(); const entries = await new Promise(resolve => dirReader.readEntries(resolve)); const fullPath = path ? `${path}/${item.name}` : item.name; if (entries.length === 0) { this.addFile(`${fullPath}/.keep`, { content: '' }); } else { const promises = entries.map(entry => this.traverseFileTree(entry, fullPath)); await Promise.all(promises); } } },
        toggleProjectsModal(show) { this.elements.projectsModal.classList.toggle('visible', show); },
        async openProjectsModal() {
            this.toggleProjectsModal(true);
            this.elements.projectsModalFeedback.style.display = 'none';
            this.elements.projectsList.innerHTML = '<li>Loading projects...</li>';
            this.elements.projectListControls.innerHTML = '';
            try {
                const response = await fetch(`${zencode_ai_app_vars.ajaxurl}?action=zencode_ai_get_projects_list&nonce=${zencode_ai_app_vars.nonce}`);
                if (!response.ok) throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                const data = await response.json();
                if (data.success) { this.state.allProjects = data.data; this.state.visibleProjectCount = 5; this.renderProjectsList(); } 
                else { throw new Error(data.data?.message || 'Failed to list projects.'); }
            } catch (error) { this.elements.projectsList.innerHTML = `<li>Network Error: ${error.message}</li>`; }
        },
        renderProjectsList() {
            const allProjects = this.state.allProjects || [];
            const visibleCount = this.state.visibleProjectCount || 5;
            if (allProjects.length === 0) { this.elements.projectsList.innerHTML = '<li>No saved projects found.</li>'; this.elements.projectListControls.innerHTML = ''; return; }
            const projectsToRender = allProjects.slice(0, visibleCount);
            this.elements.projectsList.innerHTML = projectsToRender.map(p => `<li><div class="project-info"><div class="project-name">${this.escapeHtml(p.project_name)}</div><div class="project-date">Last saved: ${new Date(p.last_modified).toLocaleString()}</div></div><div class="project-actions-modal"><button class="modal-button-secondary" data-action="load" data-project-name="${this.escapeHtml(p.project_name)}">Load</button><button class="modal-button-secondary" data-action="delete" data-project-name="${this.escapeHtml(p.project_name)}">Delete</button></div></li>`).join('');
            const controlsContainer = this.elements.projectListControls;
            controlsContainer.innerHTML = '';
            if (allProjects.length > visibleCount) { const remaining = allProjects.length - visibleCount; const loadNextCount = Math.min(5, remaining); const loadMoreBtn = document.createElement('button'); loadMoreBtn.className = 'modal-button-secondary'; loadMoreBtn.dataset.action = 'load-more'; loadMoreBtn.textContent = `Load ${loadNextCount} More`; controlsContainer.appendChild(loadMoreBtn); }
            if (visibleCount > 5) { const showLessBtn = document.createElement('button'); showLessBtn.className = 'modal-button-secondary'; showLessBtn.dataset.action = 'show-less'; showLessBtn.textContent = 'Show Less'; controlsContainer.appendChild(showLessBtn); }
        },
        async handleProjectAction(e) {
            const target = e.target.closest('button');
            if (!target) return;
            const action = target.dataset.action;
            const projectName = target.dataset.projectName;
            if (action === 'load') { this.toggleProjectsModal(false); if (await this.showConfirmation(`This will overwrite your current workspace. Are you sure you want to load "${projectName}"?`)) { this.loadSpecificProject(projectName); } } 
            else if (action === 'delete') { this.toggleProjectsModal(false); if (await this.showConfirmation(`Are you sure you want to permanently delete "${projectName}"? This cannot be undone.`)) { await this.deleteProjectSilently(projectName); } } 
            else if (action === 'load-more') { this.state.visibleProjectCount += 5; this.renderProjectsList(); } 
            else if (action === 'show-less') { this.state.visibleProjectCount = 5; this.renderProjectsList(); }
        },
        async deleteProjectSilently(projectName) {
            const formData = new FormData();
            formData.append('action', 'zencode_ai_delete_project');
            formData.append('nonce', zencode_ai_app_vars.nonce);
            formData.append('project_name', projectName);
            try {
                const response = await fetch(zencode_ai_app_vars.ajaxurl, { method: 'POST', body: formData });
                if (!response.ok) throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                const data = await response.json();
                if (data.success) { this.renderChatMessage('ai', `✅ Project "${projectName}" deleted successfully.`); } 
                else { throw new Error(data.data?.message || 'Failed to delete project.'); }
            } catch (error) { this.renderChatMessage('ai', `❌ Error deleting project: ${error.message}`); }
        },
async handleNewProjectClick() {
    this.toggleProjectsModal(false);
    
    // Check if there's meaningful work in progress
    const hasActiveWork = this.hasUnsavedWork();
    
    if (hasActiveWork) {
        const shouldProceed = await this.showConfirmation(
            "Creating a new project will discard any ongoing AI processes and reset your workspace.\n\nYour current files will be auto-saved, but any active AI conversations or processes will be cancelled.\n\nAre you sure you want to start a new project?"
        );
        if (!shouldProceed) {
            return;
        }
    }
    
    const projectName = prompt("Please enter a name for your new project:", "New Project");
    if (projectName && projectName.trim() !== "") {
        if (window.innerWidth <= 768) { 
            this.toggleSidebar(false); 
        }
        this.startNewProject(projectName.trim());
        this.renderChatMessage('ai', `New project "${projectName.trim()}" created successfully.`);
        
        // Handle pending plan or one-shot - ONLY clear after successful execution
        if (this.state.pendingPlanPrompt) {
            const prompt = this.state.pendingPlanPrompt;
            this.state.currentAIMode = 'plan';
            this.updateModeToggleVisuals();
            document.body.classList.add('plan-mode');
            this.state.isExecutingProcess = true;
            this.showPlanConfigModal(prompt);
            // Clear only after successful trigger
            this.state.pendingPlanPrompt = null;
        } else if (this.state.pendingOneShot) {
            const pending = this.state.pendingOneShot;
            this.elements.oneShotInput.value = pending.userInput || '';
            this.state.oneShotImage = pending.image || null;
            // Clear only after successful setup
            this.state.pendingOneShot = null;
            setTimeout(() => this.handleOneShotGeneration(), 100);
        }
    }
},

// Add this helper method to check for unsaved work
hasUnsavedWork() {
    // Check for meaningful files (not just README)
    const meaningfulFiles = Object.keys(this.state.projectFiles).filter(path => 
        path !== 'README.md' && 
        !path.endsWith('/.keep') &&
        this.state.projectFiles[path].content &&
        this.state.projectFiles[path].content.trim().length > 100
    );
    
    // Check for ongoing AI processes
    const hasActiveProcess = this.state.isAILoading || 
                           this.state.isExecutingProcess || 
                           this.state.isExecutingPlan || 
                           this.state.polishSessionActive ||
                           this.state.isTyping ||
                           this.state.isOneShotBuild;
    
    // Check for chat history (more than just welcome message)
    const hasChatHistory = this.state.chatHistory.length > 0;
    
    // Check for unsaved changes
    const hasUnsavedChanges = this.state.saveState === 'unsaved' || this.state.saveState === 'saving';
    
    return meaningfulFiles.length > 0 || hasActiveProcess || hasChatHistory || hasUnsavedChanges;
},


 startNewProject(projectName, doSave = true) {
    console.log('🔄 Starting new project:', projectName);
    
    // 1. IMMEDIATELY clear activeFile and editor
    this.state.activeFile = null;
    this.state.projectSpecification = null;
    this.elements.codeArea.value = '';
    this.elements.codeArea.disabled = true;
    this.state.planContextCache = null;
    this.state.planContextCacheTimestamp = null;
    this.elements.editorFilename.textContent = 'Editor';
    
    // 2. Cancel all operations and clear timers
    if (this.state.currentRequestController) {
        this.state.currentRequestController.abort();
        this.state.currentRequestController = null;
    }
    this.setLoadingState(false);
    clearTimeout(this.state.autoSaveTimer);
    clearTimeout(this.state.longRequestTimer);
    if (this.state.loadingIntervalId) {
        clearInterval(this.state.loadingIntervalId);
        this.state.loadingIntervalId = null;
    }
    
    // 3. Cleanup all resources
    this.cleanupPreviewResources();
    
    // 4. FORCE clear all project data with new objects (prevent reference issues)
    this.state.projectFiles = Object.create(null);
    this.state.fileOrder = [];
    this.state.chatHistory = [];
    this.state.collapsedFolders = new Set();
    this.state.contextFiles = new Set();
    
    // 5. Clear UI immediately
    this.elements.chatMessages.innerHTML = '';
    this.elements.fileTree.innerHTML = '';
    
    // 6. Reset all other states completely
    this.state.isAILoading = false;
    this.state.isTyping = false;
    this.state.isBatchUpdating = false;
    this.state.isExecutingProcess = false;
    this.state.isExecutingPlan = false;
    this.state.currentAIMode = 'normal';
    this.state.currentPlan = [];
    this.state.currentStepIndex = 0;
    this.state._lastResponseWasImagePlan = false;
    this.state.pendingModification = null;
    this.state.polishSessionActive = false;
    this.state.polishResults = null;
    this.state.prePolishFiles = null;
    this.state.tempVisionImage = null;
    this.state.tempVisionImagePath = null;
    this.state.oneShotImage = null;
    this.state.isOneShotBuild = false;
    this.state.currentVisionPrompt = null;
    this.state.pendingPlanPrompt = null;
    this.state.pendingOneShot = null;
    this.state._pendingPlan = null;
    this.state.requestedStepCount = null;
    this.state.refinementToken = null;
    this.state.versionControlOpen = false;
    this.state.projectVersions = [];
    this.state.isVersionLoading = false;
    this.state.visibleVersionCount = 5;
    this.state.autoSaveTimer = null;
    this.state.longRequestTimer = null;
    this.state.generatedWebsiteURL = null;
    this.state.generatedWebsiteResources = null;
    this.state.saveState = 'unsaved';
    
    // Initialize project context
    this.state.projectContext = {
        originalVision: '',
        designRequirements: '',
        technicalNotes: '',
        targetAudience: '',
        createdAt: Date.now(),
        recentSteps: [],
        keyFiles: {},
        architecture: [],
        lastUpdated: Date.now()
    };
    
    // 7. Close all modals
    document.querySelectorAll('.modal-overlay, .confirm-modal-overlay').forEach(modal => {
        modal.classList.remove('visible');
    });
    document.body.classList.remove('plan-mode');
    
    // 8. Reset UI elements
    this.elements.projectNameInput.value = projectName;
    this.updateHeaderProjectName();
    this.toggleModificationControls(false);
    this.removeOneShotImage();
    if (this.elements.oneShotInput) {
        this.elements.oneShotInput.value = '';
    }
    this.updateModeToggleVisuals();
    this.updateChatPlaceholder();
    if (this.elements.featureBar) {
        this.elements.featureBar.style.setProperty('display', 'flex', 'important');
    }
    this.stopLoadingAnimation();
    
    // 9. Wait a tick to ensure all clearing is complete
    setTimeout(() => {
        console.log('📝 Creating README.md with fresh content');
        
        const readmeContent = `# Welcome to ZenCode AI!

This is your interactive, AI-powered development environment. This \`README.md\` file is your guide to the powerful features at your fingertips.

### 🚀 Build & Create

*   **One-Shot Website Builder:** Instantly generate a complete, single-page website from a simple description. For stunning results, you can **upload an inspirational image** to guide the AI's design, color palette, and layout.
*   **AI-Powered Planning & Execution:** Switch to **Plan Mode** to have the AI create a detailed, step-by-step development plan for complex projects. Once the plan is generated, you can execute each step individually or run the entire plan automatically.

### 🧠 AI Modes & Context

ZenCode AI has three distinct modes to supercharge your workflow:

1.  **Normal Mode:** The default mode. Directly ask the AI to write, modify, or explain code.
2.  **Plan Mode:** A strategic mode for high-level project architecture. The AI will create an interactive plan with executable steps.
3.  **Vector Mode (Premium):** Ask questions about your project's codebase. The AI uses your personalized Knowledge Base for highly context-aware answers.

*   **Context Control (Critical!):** To manage costs and focus the AI, use the **checkboxes** next to each file. Only checked files are included in the context for the AI's response. This is essential for efficient development.

### 👁️ AI Vision (Visionary Plan)

*   **Image Analysis:** Upload images and ask the AI to describe, analyze, or create development plans based on visual content.
*   **Plan from Image:** Upload a design mockup or screenshot and have the AI create a step-by-step development plan to recreate it.

### 🧰 Project & File Management

This is a fully-featured file manager. You can:
*   Create new folders and organize your project structure.
*   Drag and drop to organize your project structure.
*   Upload individual files or entire \`.zip\` archives.
*   Download your complete project as a \`.zip\` file at any time.
*   Preview binary files (images, etc.) with the eye icon.

### ✨ Advanced Features

*   **Vector Knowledge Base:** Save key files to the AI's long-term memory using the pink brain icon. This creates a personalized knowledge base, allowing the AI to understand your project's architecture and provide more accurate assistance.
*   **Integration & Polish Tool:** Located in Project Actions, this powerful tool transforms step-by-step built projects into fully integrated, professionally polished websites. It fixes functionality issues AND adds stunning visual design with multiple enhancement options and intensity levels.
*   **Plan Execution Controls:** When in Plan Mode, execute individual steps, run the entire plan automatically, or re-do failed steps.
*   **Live Preview:** Click the eye icon next to HTML files to open them in a new tab with all dependencies properly linked.

### 🎯 Getting Started

*   **New users:** Take the guided tour (click "Tour" in the bottom-left corner on desktop)
*   **Quick start:** Use the "Build a Website" button for instant results
*   **Complex projects:** Switch to Plan Mode for structured development
*   **Large projects:** Use context checkboxes to include only relevant files
*   **Polish projects:** Use the Integration & Polish Tool after step-by-step builds for professional results

Happy Building!`;

        console.log('📄 Adding README.md file...');
        this.state.projectFiles['README.md'] = { 
            content: readmeContent 
        };
        this.state.fileOrder = ['README.md'];
        this.state.contextFiles.add('README.md');
        
        console.log('📄 README.md content length:', readmeContent.length);
        
        this.renderFileTree();
        this.initializeWelcome();
        this.activateFile('README.md');
        this.updatePreviewButtonVisibility();
        this.updateSaveStatus('unsaved');
        
        if (doSave) {
            this.saveProject();
        }
        
        console.log('✅ New project setup complete');
    }, 10);
},
      updateHeaderProjectName() {
    const projectName = this.elements.projectNameInput.value.trim() || 'Untitled Project';
    // This SVG creates a folder icon styled to match your application's theme.
    const iconSVG = `<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="margin-right: 0.6rem; vertical-align: -4px; color: var(--text-tertiary);"><path d="M22 19a2 2 0 0 1-2 2H4a2 2 0 0 1-2-2V5a2 2 0 0 1 2-2h5l2 3h9a2 2 0 0 1 2 2z"></path></svg>`;
    // We use innerHTML to add the icon and escape the project name to ensure security.
    this.elements.headerProjectName.innerHTML = iconSVG + this.escapeHtml(projectName);
},
        _dataURLtoBlob(dataurl) { const arr = dataurl.split(','); const mime = arr[0].match(/:(.*?);/)[1]; const bstr = atob(arr[1]); let n = bstr.length; const u8arr = new Uint8Array(n); while (n--) { u8arr[n] = bstr.charCodeAt(n); } return new Blob([u8arr], { type: mime }); },
        _fileToBase64(file) { return new Promise((resolve, reject) => { const reader = new FileReader(); reader.readAsDataURL(file); reader.onload = () => resolve(reader.result); reader.onerror = error => reject(error); }); },
 // [REPLACE] your previewHTMLWithDependencies function and its helpers with these.
previewHTMLWithDependencies(htmlPath) {
    this.cleanupPreviewResources();
    const htmlFile = this.state.projectFiles[htmlPath];
    if (!htmlFile) return;

    let modifiedHtml = htmlFile.content;
    const blobUrlsToRevoke = [];
    const assetMap = new Map();

    for (const path in this.state.projectFiles) {
        const fileData = this.state.projectFiles[path];
        if (path.endsWith('.css') || fileData.rawFile) {
            const blob = fileData.rawFile
                ? new Blob([fileData.rawFile], { type: fileData.rawFile.type })
                : new Blob([fileData.content], { type: this.getMimeType(path) });
            
            const blobUrl = URL.createObjectURL(blob);
            assetMap.set(path, blobUrl);
            blobUrlsToRevoke.push(blobUrl);
        }
    }

    for (const [path, url] of assetMap.entries()) {
        const regex = new RegExp(`(href|src)=["'](./)?${path}["']`, 'g');
        modifiedHtml = modifiedHtml.replace(regex, `$1="${url}"`);
    }
    
    let bundledJs = '';
    const allJsFiles = this.state.fileOrder.filter(p => p.endsWith('.js'));
    
    const modules = allJsFiles.filter(path => {
        const content = this.state.projectFiles[path].content;
        return content.includes('export');
    });

    const scripts = allJsFiles.filter(path => !modules.includes(path));

    modules.forEach(path => {
        let content = this.state.projectFiles[path].content;
        content = content.replace(/export default\s+/g, 'var ');
        content = content.replace(/export\s+/g, '');
        bundledJs += `\n// --- From Module: ${path} ---\n${content}\n`;
    });

    scripts.forEach(path => {
        let content = this.state.projectFiles[path].content;
        content = content.replace(/^import .* from '.*';\r?\n?/gm, '');
        content = content.replace(/const\s+module\s*=\s*await\s+import\((['"]).*?\1\);/g, '');
        content = content.replace(/const\s+\w+\s*=\s*module.default;/g, '');
        bundledJs += `\n// --- From Script: ${path} ---\n${content}\n`;
    });

    const parser = new DOMParser();
    const doc = parser.parseFromString(modifiedHtml, 'text/html');
    
    doc.querySelectorAll('script[src]').forEach(script => script.remove());
    
    const finalScript = doc.createElement('script');
    finalScript.textContent = bundledJs;
    finalScript.defer = true; 
    doc.body.appendChild(finalScript);
    
    const finalHtmlContent = doc.documentElement.outerHTML;

    const htmlBlob = new Blob([finalHtmlContent], { type: 'text/html' });
    const htmlURL = URL.createObjectURL(htmlBlob);
    blobUrlsToRevoke.push(htmlURL);
    
    this.state.generatedWebsiteResources = { blobUrls: [htmlURL, ...blobUrlsToRevoke] };
    window.open(htmlURL, '_blank');
},

getMimeType(path) {
    const extension = path.split('.').pop().toLowerCase();
    switch (extension) {
        case 'html': return 'text/html';
        case 'css': return 'text/css';
        case 'js': return 'text/javascript';
        default: return 'text/plain';
    }
},

cleanupPreviewResources() {
    if (this.state.generatedWebsiteResources && this.state.generatedWebsiteResources.blobUrls) {
        this.state.generatedWebsiteResources.blobUrls.forEach(url => {
            URL.revokeObjectURL(url);
        });
        this.state.generatedWebsiteResources = null;
    }
},
        async loadSpecificProject(projectName) { this.cleanupPreviewResources(); await this.loadProject(projectName); this.activateFile(this.state.fileOrder[0] || null); },
// REPLACE your entire makePlanInteractive function with this version.
// REPLACE your entire makePlan-Interactive function with this FINAL, COMPLETE version.
// REPLACE your entire makePlan-Interactive function with this FINAL, COMPLETE version.
makePlanInteractive(planStepsArray) {
    // Sanitize the input to create a clean array of step objects.
	 // --- ADD THIS BLOCK AT THE TOP OF THE FUNCTION ---
    const debugButton = this.elements.debugButton;
    debugButton.disabled = true;
    debugButton.title = 'Debugging is handled automatically within Plan Mode steps.';
    debugButton.style.opacity = '0.5';
    debugButton.style.cursor = 'not-allowed';
    // --- END OF NEW BLOCK ---

    // Sanitize the input to create a clean array of step objects.
    const steps = planStepsArray
        .map(line => ({ text: line.trim(), status: 'pending' }))
        .filter(step => step.text);

    let displayHtml = '';
    this.state.currentPlan = steps;

    // Create a container for the plan steps that won't interfere with main page scrolling.
    displayHtml += `<div class="plan-container" style="max-height: none; overflow: visible;">`;

    // Loop through each step to generate its HTML.
    steps.forEach((step, stepIndex) => {
        const stepTextWithNumber = `${stepIndex + 1}. ${step.text}`;

        displayHtml += `
            <div class="plan-step" data-step-index="${stepIndex}" style="display: flex; flex-direction: column; margin: 0.5rem 0; padding: 0.75rem; border: 1px solid var(--border-primary); border-radius: 8px; background: var(--bg-secondary); transition: all 0.3s ease; position: static !important; overflow: visible !important; max-height: none !important;">
                <div style="display: flex; justify-content: space-between; align-items: flex-start; width: 100%;">
                    <span style="flex: 1; margin-right: 1rem; word-wrap: break-word; line-height: 1.5;">${this.escapeHtml(stepTextWithNumber)}</span>
                    <div class="plan-step-controls" data-step-index="${stepIndex}" style="flex-shrink: 0;">
                        <button class="execute-step-btn modal-button-secondary" data-step-index="${stepIndex}" style="padding: 0.5rem 1rem; font-size: 0.9rem; white-space: nowrap; position: static !important;">Do It</button>
                    </div>
                </div>
                <div class="step-progress-container" data-step-index="${stepIndex}" style="display: none; margin-top: 0.75rem; width: 100%;">
                    <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 0.375rem;">
                        <span class="step-progress-text" style="font-size: 0.8125rem; color: var(--text-secondary); line-height: 1.25;">Initializing...</span>
                        <span class="step-progress-percentage" style="font-size: 0.8125rem; font-weight: 600; color: var(--text-primary); line-height: 1.25;">0%</span>
                    </div>
                    <div style="width: 100%; height: 6px; background: rgba(0,0,0,0.1); border-radius: 3px; overflow: hidden; box-shadow: inset 0 1px 2px rgba(0,0,0,0.1);">
                        <div class="step-progress-bar" style="width: 0%; height: 100%; background: linear-gradient(90deg, #8b5cf6, #a78bfa); transition: width 0.3s ease, background 0.3s ease; border-radius: 3px;"></div>
                    </div>
                </div>
            </div>`;
    });
    
    // Generate unique IDs for this specific plan's controls.
    const planId = Date.now();
    const executeButtonId = `execute-plan-btn-${planId}`;
    const stopButtonId = `stop-plan-btn-${planId}`;
    
    // Generate the HTML for the main plan controls without the polish button.
    displayHtml += `
        <div class="plan-controls" style="margin-top: 1.5rem; border-top: 1px solid var(--border-primary); padding-top: 1rem; text-align: center; position: static !important;">
            <button id="${executeButtonId}" class="modal-button execute-plan-btn-new" style="display: none;">🚀 Execute Full Plan</button>
            <button id="${stopButtonId}" class="modal-button-secondary stop-plan-btn-new" style="display: none;">⏹️ Stop Execution</button>
        </div>
    </div>`;
    
    // Store the unique IDs in the app's state.
    this.state.currentPlanButtonIds = {
        execute: executeButtonId,
        stop: stopButtonId,
    };
    
    setTimeout(() => {
        const executeBtn = document.getElementById(executeButtonId);
        const stopBtn = document.getElementById(stopButtonId);
        
        document.querySelectorAll('.execute-plan-btn-new, .stop-plan-btn-new').forEach(btn => {
            if (btn.id !== executeButtonId && btn.id !== stopButtonId) {
                btn.style.display = 'none';
            }
        });
        
        if (executeBtn) executeBtn.style.display = 'inline-flex';
        if (stopBtn) stopBtn.style.display = 'none';
        
    }, 500);
    
    return displayHtml;
},
 updateStepUI(stepIndex, status, progressData = null) {
    const controlDiv = document.querySelector(`.plan-step-controls[data-step-index="${stepIndex}"]`);
    if (!controlDiv) return;
    
    const stepContainer = controlDiv.closest('.plan-step');
    const progressContainer = stepContainer?.querySelector('.step-progress-container');
    
    if (status === 'running') {
        if (stepContainer) {
            stepContainer.style.background = 'rgba(139, 92, 246, 0.2)';
            stepContainer.style.borderColor = 'rgba(139, 92, 246, 0.5)';
        }
        controlDiv.innerHTML = `<button class="modal-button-secondary" disabled style="padding: 0.5rem 1rem; font-size: 0.9rem; flex-shrink: 0;">Running...</button>`;
        
        // Show progress bar
        if (progressContainer) {
            progressContainer.style.display = 'block';
            this.updateStepProgress(stepIndex, progressData || { text: 'Starting...', percentage: 5 });
        }
    } else if (status === 'done') {
        if (stepContainer) {
            stepContainer.style.background = 'rgba(72, 187, 120, 0.1)';
            stepContainer.style.borderColor = 'rgba(72, 187, 120, 0.5)';
        }
        controlDiv.innerHTML = `<button class="redo-step-btn modal-button-secondary" data-step-index="${stepIndex}" style="padding: 0.5rem 1rem; font-size: 0.9rem; flex-shrink: 0;">Re-do</button>`;
        
        // Show 100% completion briefly then hide
        if (progressContainer) {
            this.updateStepProgress(stepIndex, { text: 'Complete!', percentage: 100, color: '#48bb78' });
            setTimeout(() => {
                progressContainer.style.display = 'none';
            }, 2000);
        }
    } else if (status === 'failed') {
        if (stepContainer) {
            stepContainer.style.background = 'rgba(245, 101, 101, 0.1)';
            stepContainer.style.borderColor = 'rgba(245, 101, 101, 0.5)';
        }
        controlDiv.innerHTML = `<button class="retry-step-btn modal-button-secondary" data-step-index="${stepIndex}" style="padding: 0.5rem 1rem; font-size: 0.9rem; flex-shrink: 0;">Retry</button>`;
        
        // Show failure state
        if (progressContainer) {
            this.updateStepProgress(stepIndex, { text: 'Failed - click Retry', percentage: 0, color: '#f56565' });
            // Keep visible to show error
        }
    } else {
        // Reset to pending state
        if (stepContainer) {
            stepContainer.style.background = 'var(--bg-secondary)';
            stepContainer.style.borderColor = 'var(--border-primary)';
        }
        controlDiv.innerHTML = `<button class="execute-step-btn modal-button-secondary" data-step-index="${stepIndex}" style="padding: 0.5rem 1rem; font-size: 0.9rem; flex-shrink: 0;">Do It</button>`;
        
        // Hide progress bar
        if (progressContainer) {
            progressContainer.style.display = 'none';
        }
    }
},

updateStepProgress(stepIndex, { text, percentage, color = null }) {
    const stepContainer = document.querySelector(`.plan-step[data-step-index="${stepIndex}"]`);
    if (!stepContainer) return;
    
    const progressText = stepContainer.querySelector('.step-progress-text');
    const progressPercentage = stepContainer.querySelector('.step-progress-percentage');
    const progressBar = stepContainer.querySelector('.step-progress-bar');
    
    if (progressText) progressText.textContent = text;
    if (progressPercentage) progressPercentage.textContent = `${Math.round(percentage)}%`;
    if (progressBar) {
        progressBar.style.width = `${percentage}%`;
        if (color) {
            progressBar.style.background = color;
        } else {
            progressBar.style.background = 'linear-gradient(90deg, #8b5cf6, #a78bfa)';
        }
    }
    
    // Also update footer progress bar
    this.updateFooterProgress(stepIndex, text, percentage, color);
},

// Footer Progress Bar Functions
showFooterProgress() {
    const footerProgress = document.getElementById('footer-progress-bar');
    if (footerProgress) {
        footerProgress.style.display = 'block';
        // Update total steps
        const totalSteps = this.state.currentPlan?.length || 0;
        const totalStepsEl = document.getElementById('footer-total-steps');
        if (totalStepsEl) totalStepsEl.textContent = totalSteps;
    }
},

hideFooterProgress(immediate = false) {
    const footerProgress = document.getElementById('footer-progress-bar');
    if (footerProgress) {
        if (immediate) {
            footerProgress.style.display = 'none';
        } else {
            setTimeout(() => {
                footerProgress.style.display = 'none';
            }, 2000); // Keep visible for 2 seconds after completion
        }
    }
},

updateFooterProgress(stepIndex, text, percentage, color = null) {
    const footerProgressText = document.getElementById('footer-progress-text');
    const footerProgressPercentage = document.getElementById('footer-progress-percentage');
    const footerProgressFill = document.getElementById('footer-progress-bar-fill');
    const footerCurrentStep = document.getElementById('footer-current-step');
    
    if (footerProgressText) footerProgressText.textContent = text;
    if (footerProgressPercentage) footerProgressPercentage.textContent = `${Math.round(percentage)}%`;
    if (footerCurrentStep) footerCurrentStep.textContent = stepIndex + 1;
    
    if (footerProgressFill) {
        footerProgressFill.style.width = `${percentage}%`;
        if (color) {
            footerProgressFill.style.background = color;
        } else {
            footerProgressFill.style.background = 'linear-gradient(90deg, #8b5cf6, #a78bfa, #c4b5fd)';
        }
    }
},

// 8. UPDATED executePlanStep function
// REPLACE your entire executePlanStep function with this one.
// This is YOUR working version, with one crucial line added for stability.
// [REPLACE] your executePlanStep function with this FINAL, CORRECTED version.
// [REPLACE] your executePlanStep function with this FINAL, MERGED version.
// REPLACE your entire executePlanStep function with this:
// REPLACE your entire, overly-complex executePlanStep function with this CLEAN version.
// =============================================================================
// COMPLETE executePlanStep FUNCTION WITH REFLECTION-AFTER-REPAIR
// Replace your entire executePlanStep function (line 8995-9097) with this
// =============================================================================


// ============================================================================
// ✨ FEATURE 1: PARALLEL STEP PREPARATION
// ============================================================================
/**
 * Prepares the next step in parallel while current step executes
 * This saves 2-3 seconds per step by doing work in advance
 */
async prepareStepInBackground(stepIndex) {
    if (stepIndex >= this.state.currentPlan.length) return null;
    
    const step = this.state.currentPlan[stepIndex];
    if (!step) return null;
    
    console.log(`🔄 Background: Preparing step ${stepIndex + 1}...`);
    
    try {
        // Prepare context in parallel
        const [prunedSpec, adaptiveContext] = await Promise.all([
            this.getPrunedSpecificationForStep(step.text),
            this.getAdaptiveContextForStep(stepIndex, step.text)
        ]);
        
        const preparedData = {
            stepIndex,
            prunedSpec,
            adaptiveContext,
            prompt: this.buildContextAwareStepPrompt(step.text, stepIndex + 1),
            timestamp: Date.now()
        };
        
        this.state.preparedSteps.set(stepIndex, preparedData);
        console.log(`✅ Background: Step ${stepIndex + 1} prepared and cached`);
        
        return preparedData;
    } catch (error) {
        console.error(`❌ Background preparation failed for step ${stepIndex + 1}:`, error);
        return null;
    }
},

/**
 * Gets prepared step data or prepares it on-demand
 */
async getPreparedStepData(stepIndex) {
    // Check if already prepared
    if (this.state.preparedSteps.has(stepIndex)) {
        console.log(`⚡ Using pre-prepared data for step ${stepIndex + 1}`);
        const prepared = this.state.preparedSteps.get(stepIndex);
        this.state.preparedSteps.delete(stepIndex); // Clear after use
        return prepared;
    }
    
    // Prepare now if not ready
    console.log(`⏳ Preparing step ${stepIndex + 1} on-demand...`);
    return await this.prepareStepInBackground(stepIndex);
},

/**
 * Starts background preparation of next step
 */
startNextStepPreparation(currentStepIndex) {
    const nextIndex = currentStepIndex + 1;
    
    // Don't prepare if already done or if we're at the end
    if (nextIndex >= this.state.currentPlan.length) return;
    if (this.state.preparedSteps.has(nextIndex)) return;
    if (this.state.isPreparingNextStep) return;
    
    this.state.isPreparingNextStep = true;
    this.state.preparationQueue = this.prepareStepInBackground(nextIndex)
        .finally(() => {
            this.state.isPreparingNextStep = false;
        });
},

// ============================================================================

async executePlanStep(stepIndex, promptOverride = null) {
    const step = this.state.currentPlan[stepIndex];
    if (!step) return false;
    
    if (this.state.isExecutingProcess) {
        this.renderChatMessage('ai', 'Please wait for the current AI task to complete.');
        return false;
    }
    
    this.state.isExecutingProcess = true;
    const filesBefore = { ...this.state.projectFiles };
    
    // ===================================================================
    // IMPROVED SCAFFOLDING LOGIC
    // ===================================================================
    let isScaffoldingStep = false;
    let finalPromptForAI = promptOverride || this.buildContextAwareStepPrompt(step.text, stepIndex + 1);

    // More accurate check for truly empty projects
    const isEmptyProject = this.state.fileOrder.length <= 1 || 
                          (this.state.fileOrder.length === 1 && 
                           Object.keys(this.state.projectFiles).length === 0);
    
    const isFirstStepOfNewProject = (stepIndex === 0 && isEmptyProject);
    
    if (isFirstStepOfNewProject) {
        isScaffoldingStep = true;
        this.renderChatMessage('ai', "🏗️ Setting up project foundation with intelligent scaffolding...");
        
        // Enhanced prompt that understands this is foundation + first feature
        const fullPlanText = this.state.currentPlan.map((s, i) => `${i + 1}. ${s.text}`).join('\n');
        finalPromptForAI = `PROJECT SETUP & FIRST FEATURE IMPLEMENTATION:

FULL DEVELOPMENT PLAN:
${fullPlanText}

YOUR TASK FOR THIS STEP: "${step.text}"

IMPLEMENTATION STRATEGY:
1. Create essential project files if they don't exist (index.html, style.css, script.js)
2. Implement the core functionality requested in this step
3. Ensure everything works together - this is the foundation for the entire project
4. Focus on working functionality first, polish can come later

Remember: This is step 1 of ${this.state.currentPlan.length} - build a solid foundation!`;
    }
    // ===================================================================
    // END OF IMPROVED LOGIC
    // ===================================================================
    
    try {
        // Progress: 10% - Applying learnings
        this.updateStepProgress(stepIndex, { text: 'Applying past learnings...', percentage: 10 });
        
        const enhancedContext = await this.applyReflectionLearnings(step.text, this.state.contextFiles);
        
        if (enhancedContext.size !== this.state.contextFiles.size) {
            this.state.contextFiles = enhancedContext;
            this.renderFileTree();
            this.renderChatMessage('ai', `🎓 Applied past learnings to context selection for step ${stepIndex + 1}`);
        }
        
        this.state.currentPlan[stepIndex].status = 'running';
        this.updateStepUI(stepIndex, 'running');
        this.highlightCurrentStep(stepIndex);
        
        // Progress: 20% - Preparing context
        this.updateStepProgress(stepIndex, { text: 'Preparing step context...', percentage: 20 });
        
        let prunedSpec;
        // ✅ FIXED: Revert to original logic that works
        if (!isScaffoldingStep) {
            prunedSpec = await this.getPrunedSpecificationForStep(step.text);
        } else {
            prunedSpec = null;
        }
        
        this.startNextStepPreparation(stepIndex);
        
        // Progress: 30% - Executing AI request
        this.updateStepProgress(stepIndex, { text: 'Sending request to AI...', percentage: 30 });
        
        // Enhanced step execution with better scaffolding support
        await this._executeSend(
            finalPromptForAI, 
            true, 
            null, 
            isScaffoldingStep ? "Project Foundation Setup" : `Step ${stepIndex + 1}: ${step.text}`,
            { 
                isStepExecution: true, 
                specData: prunedSpec, 
                stepIndex: stepIndex,
                isScaffoldingStep: isScaffoldingStep
            }
        );
        
        // Progress: 60% - Processing results
        this.updateStepProgress(stepIndex, { text: 'Processing AI response...', percentage: 60 });
        
        const { newFiles, modifiedFiles } = this.debugFileCreation(stepIndex, filesBefore, this.state.projectFiles);
        
        // Progress: 70% - Validating output
        this.updateStepProgress(stepIndex, { text: 'Validating step output...', percentage: 70 });
        
        // ✅ UPDATED: Pass isScaffoldingStep parameter to enable scaffolding-aware validation
        const validation = await this.validateStepOutput(stepIndex, newFiles, modifiedFiles, isScaffoldingStep);
        
        // Progress: 80% - Running reflection
        this.updateStepProgress(stepIndex, { text: 'Learning from execution...', percentage: 80 });
        
        await this.safeReflectOnStep(
            stepIndex,
            isScaffoldingStep ? "Project Foundation" : step.text,
            !validation.hasIssues,
            validation,
            filesBefore,
            this.state.projectFiles
        );
        
        if (validation.hasIssues) {
            this.updateStepProgress(stepIndex, { text: `Found ${validation.missingReferences.length + validation.syntaxErrors.length} issue(s), auto-repairing...`, percentage: 85 });
            await new Promise(resolve => setTimeout(resolve, 3000));
            this.renderChatMessage('ai', `🔍 Step ${stepIndex + 1} Validation: Found ${validation.missingReferences.length + validation.syntaxErrors.length} integration issue(s). Attempting auto-repair...`);
            const filesBeforeRepair = { ...this.state.projectFiles };
            this.updateStepProgress(stepIndex, { text: 'Running auto-repair...', percentage: 90 });
            const repairResult = await this.autoFixStepIssues(stepIndex, validation);
            if (repairResult === true) {
                this.updateStepProgress(stepIndex, { text: 'Finalizing repairs...', percentage: 95 });
                await this.rateLimitDelay(2000);
                await this.safeReflectOnStep(
                    stepIndex,
                    isScaffoldingStep ? "[REPAIRED] Project Foundation" : `[AUTO-REPAIR] ${step.text}`,
                    true,
                    { hasIssues: false, missingReferences: [], syntaxErrors: [], warnings: [] },
                    filesBeforeRepair,
                    this.state.projectFiles
                );
            }
        }
        
        this.state.currentPlan[stepIndex].status = 'done';
        this.updateStepUI(stepIndex, 'done');
        
        this.updatePlanContextAfterStep(stepIndex, step.text, filesBefore, this.state.projectFiles);
        
        // Use appropriate summary title
        const stepTitle = isScaffoldingStep ? "Project Foundation" : step.text;
        this.generateStepSummary(stepIndex + 1, stepTitle, filesBefore, this.state.projectFiles);
        
        return true;
        
    } catch (error) {
        console.error(`❌ Step ${stepIndex + 1} failed:`, error);
        this.renderChatMessage('ai', `❌ An error occurred on step ${stepIndex + 1}: ${error.message}`);
        this.state.currentPlan[stepIndex].status = 'failed';
        this.updateStepUI(stepIndex, 'failed');
        const failureValidation = { 
            hasIssues: true, 
            missingReferences: [], 
            syntaxErrors: [{ error: error.message, file: 'Unknown' }] 
        };
        await this.safeReflectOnStep(
            stepIndex, 
            isScaffoldingStep ? "Project Foundation" : step.text, 
            false, 
            failureValidation, 
            filesBefore, 
            this.state.projectFiles
        );
        return false;
        
    } finally {
        this.highlightCurrentStep(-1);
        this.state.isExecutingProcess = false;
    }
},

// NEW FUNCTION - Enhanced prompt builder
buildEnhancedStepPrompt(stepText, stepIndex, preCheck) {
    const basePrompt = this.buildContextAwareStepPrompt(stepText, stepIndex);
    
    // Add pre-validation insights
    let enhancedSection = '\n\n**PRE-EXECUTION ANALYSIS:**\n';
    
    if (preCheck.missingDefinitions.length > 0) {
        enhancedSection += 'Critical issues to address:\n';
        preCheck.missingDefinitions.forEach(def => {
            enhancedSection += `- ${def.issue}\n`;
        });
    }
    
    if (preCheck.recommendations.length > 0) {
        enhancedSection += '\nRecommendations:\n';
        preCheck.recommendations.forEach(rec => {
            enhancedSection += `- ${rec.suggestion}\n`;
        });
    }
    
    enhancedSection += '\n**YOU MUST ADDRESS ALL CRITICAL ISSUES IN YOUR OUTPUT.**\n';
    
    return basePrompt + enhancedSection;
},

// Helper function to expand context based on validation failures
getExpandedContextForFailedStep(stepIndex, validation) {
    const current = Array.from(this.state.contextFiles);
    const expanded = [...current];
    const availableFiles = this.state.fileOrder;
    
    console.log(`🔍 Analyzing ${validation.missingReferences.length} missing references to expand context...`);
    
    // Strategy 1: Add files that define missing references
    validation.missingReferences.forEach(ref => {
        // Look for files that contain the missing item
        const filesWithDefinition = availableFiles.filter(path => {
            if (expanded.includes(path)) return false; // Already included
            
            const content = this.state.projectFiles[path]?.content || '';
            
            // Check if this file defines what's missing
            if (ref.type === 'CSS' && path.endsWith('.css')) {
                // Look for CSS class/id definition
                const selector = ref.missing.replace(/^[.#]/, '');
                const hasClass = new RegExp(`\\.${selector}\\s*{`).test(content);
                const hasId = new RegExp(`#${selector}\\s*{`).test(content);
                return hasClass || hasId;
            }
            
            if (ref.type === 'JavaScript' && path.endsWith('.js')) {
                // Look for function definition
                const funcName = ref.missing.replace('.js', '');
                const hasFunction = new RegExp(`function\\s+${funcName}|const\\s+${funcName}|let\\s+${funcName}`).test(content);
                return hasFunction;
            }
            
            return content.includes(ref.missing);
        });
        
        if (filesWithDefinition.length > 0) {
            console.log(`  ✓ Found ${filesWithDefinition.length} file(s) defining ${ref.missing}`);
            expanded.push(...filesWithDefinition);
        } else {
            console.log(`  ⚠️ No existing file defines ${ref.missing} - may need creation`);
        }
    });
    
    // Strategy 2: Add all files of specific types based on error patterns
    const hasCSSIssues = validation.missingReferences.some(r => r.type === 'CSS');
    const hasJSIssues = validation.missingReferences.some(r => r.type === 'JavaScript') || 
                        validation.warnings.some(w => w.type === 'missing_function');
    const hasSyntaxErrors = validation.syntaxErrors.length > 0;
    
    if (hasCSSIssues) {
        const cssFiles = availableFiles.filter(f => f.endsWith('.css') && !expanded.includes(f));
        if (cssFiles.length > 0) {
            console.log(`  📄 Adding ${cssFiles.length} CSS files due to style issues`);
            expanded.push(...cssFiles);
        }
    }
    
    if (hasJSIssues) {
        const jsFiles = availableFiles.filter(f => f.endsWith('.js') && !expanded.includes(f));
        if (jsFiles.length > 0) {
            console.log(`  📄 Adding ${jsFiles.length} JS files due to function issues`);
            expanded.push(...jsFiles);
        }
    }
    
    if (hasSyntaxErrors) {
        // Add utility/helper files that might provide needed functions
        const utilFiles = availableFiles.filter(f => 
            (f.includes('util') || f.includes('helper') || f.includes('common')) && 
            !expanded.includes(f)
        );
        if (utilFiles.length > 0) {
            console.log(`  📄 Adding ${utilFiles.length} utility files due to syntax errors`);
            expanded.push(...utilFiles);
        }
    }
    
    // Strategy 3: Add files that reference the problematic file
    const problematicFiles = [...new Set(validation.missingReferences.map(r => r.file))];
    problematicFiles.forEach(problemFile => {
        const referencingFiles = availableFiles.filter(path => {
            if (expanded.includes(path) || path === problemFile) return false;
            
            const content = this.state.projectFiles[path]?.content || '';
            return content.includes(problemFile);
        });
        
        if (referencingFiles.length > 0) {
            console.log(`  🔗 Adding ${referencingFiles.length} files that reference ${problemFile}`);
            expanded.push(...referencingFiles);
        }
    });
    
    // Remove duplicates and preserve images
    const uniqueExpanded = [...new Set(expanded)];
    
    console.log(`📊 Context expansion: ${current.length} → ${uniqueExpanded.length} files`);
    
    return uniqueExpanded;
},

// NEW: More aggressive repair for stubborn integration issues
async aggressiveAutoFix(stepIndex, validation) {
    const MAX_REPAIR_ATTEMPTS = 3;
    this.state.repairAttempts = this.state.repairAttempts || {};
    
    const attemptKey = `step_${stepIndex}`;
    
    // Save original context to restore later
    const originalContext = new Set(this.state.contextFiles);
    
    // Track files before repair for validation
    const filesBefore = { ...this.state.projectFiles };

    // ==================== START MINIMAL FIX ====================
    // 1. DETECT THE SPECIFIC "MISSING FILE" SCENARIO
    const isMissingFileOnlyIssue = validation.missingReferences.length > 0 && validation.syntaxErrors.length === 0;
    
    // 2. SET THE STARTING POINT FOR THE LOOP
    let startAttempt = 1;
    if (isMissingFileOnlyIssue) {
        // If it's just missing files, jump straight to the brute-force strategy.
        startAttempt = 3; 
        this.renderChatMessage('ai', `🔧 Found missing file references. Expanding context to create them...`);
    }
    // ===================== END MINIMAL FIX =====================
    
    // ⚡ MAIN REPAIR LOOP - Try up to 3 attempts
    for (let attemptNumber = startAttempt; attemptNumber <= MAX_REPAIR_ATTEMPTS; attemptNumber++) {
        this.state.repairAttempts[attemptKey] = attemptNumber;
        
        const issues = [];
        const criticalIssues = [];
        const minorIssues = [];
        
        // Categorize issues by severity
        validation.missingReferences.forEach(ref => {
            const issue = {
                description: `${ref.file} references ${ref.missing} (${ref.type}) which doesn't exist`,
                file: ref.file,
                missing: ref.missing,
                type: ref.type
            };
            
            if (ref.type === 'JavaScript Module' || ref.type === 'JavaScript') {
                criticalIssues.push(issue);
            } else {
                minorIssues.push(issue);
            }
            
            issues.push(`❌ ${issue.description}`);
        });
        
        validation.syntaxErrors.forEach(err => {
            const issue = {
                description: `${err.file} has syntax error: ${err.error}`,
                file: err.file,
                error: err.error
            };
            criticalIssues.push(issue);
            issues.push(`❌ ${issue.description}`);
        });
        
        validation.warnings.forEach(warn => {
            if (warn.type === 'missing_function') {
                minorIssues.push({
                    description: warn.issue,
                    type: 'warning'
                });
                issues.push(`⚠️ ${warn.issue}`);
            }
        });
        
        if (issues.length === 0) {
            // No issues - success!
            delete this.state.repairAttempts[attemptKey];
            this.state.contextFiles = originalContext;
            return true;
        }
        
        // DETERMINE CONTEXT STRATEGY based on attempt number
        const contextFiles = this.getRepairContextFiles(attemptNumber, validation, criticalIssues, minorIssues);
        
        // TEMPORARILY set the context for this repair attempt
        this.state.contextFiles = new Set(contextFiles);
        
        // ==================== START MINIMAL FIX ====================
        // 3. ADJUST THE USER-FACING MESSAGES AND STRATEGY NAME
        let strategyName = '';
        let strategyDescription = '';
        let displayAttempt = attemptNumber;
        let displayMax = MAX_REPAIR_ATTEMPTS;

        if (isMissingFileOnlyIssue) {
            strategyName = 'BRUTE FORCE (Fast-Tracked)';
            strategyDescription = 'Using full project context to create missing files';
            displayAttempt = 1; // Always show "1/1" for this specific case
            displayMax = 1;
        } else if (attemptNumber === 1) {
            strategyName = 'SURGICAL';
            strategyDescription = 'Targeting only the affected files';
        } else if (attemptNumber === 2) {
            strategyName = 'SURGICAL+';
            strategyDescription = 'Expanding to related files and dependencies';
        } else {
            strategyName = 'BRUTE FORCE';
            strategyDescription = 'Using full project context for comprehensive analysis';
        }
        // ===================== END MINIMAL FIX =====================
        
        // Build the repair prompt (using the adjusted display variables)
        const repairPrompt = `CRITICAL INTEGRATION REPAIR - ATTEMPT ${displayAttempt}/${displayMax} [${strategyName}]

**REPAIR STRATEGY: ${strategyName}**
${strategyDescription}
Context files available: ${contextFiles.length} file(s)

**ISSUE SUMMARY:**
- Critical Issues: ${criticalIssues.length}
- Minor Issues: ${minorIssues.length}
- Total: ${issues.length}

**CRITICAL ISSUES (MUST FIX):**
${criticalIssues.map((issue, i) => `${i + 1}. ${issue.description}`).join('\n')}

${minorIssues.length > 0 ? `**MINOR ISSUES (SHOULD FIX):**
${minorIssues.map((issue, i) => `${i + 1}. ${issue.description}`).join('\n')}` : ''}

**YOUR MANDATORY REPAIR TASKS:**
${this.generateRepairTasks(criticalIssues, minorIssues)}

**REPAIR APPROACH FOR ${strategyName}:**
${this.getRepairApproachInstructions(attemptNumber)}

**OUTPUT REQUIREMENTS:**
- First line: JSON summary block with your repair plan
- Then ONLY the files that need to be created or fixed
- Make files complete, functional, and properly integrated
- Preserve ALL existing functionality
- DO NOT create placeholder or stub functions - implement real working code

**CRITICAL:** This is attempt ${displayAttempt} of ${displayMax}. ${attemptNumber === MAX_REPAIR_ATTEMPTS ? 'This is your FINAL attempt - if you cannot fix these issues completely, explain what manual steps are needed.' : 'Focus on precision and completeness.'}

Fix these issues NOW with complete attention to detail:`;
        
        try {
            this.renderChatMessage('ai', `🔧 Running repair attempt ${displayAttempt}/${displayMax} [${strategyName}] - ${contextFiles.length} file(s) in context...`);
            
            await this._executeSend(repairPrompt, true, null, `Auto-repair attempt ${displayAttempt} (${strategyName}) for step ${stepIndex + 1}`, { 
                isStepExecution: true,
                isAutoRepair: true 
            });
            
            this.renderChatMessage('ai', `✅ Repair attempt ${displayAttempt} complete. Verifying fixes...`);
            
            // ⚡ VALIDATE AFTER THIS ATTEMPT
            const newFiles = Object.keys(this.state.projectFiles).filter(p => !filesBefore[p]);
            const revalidation = await this.validateStepOutput(stepIndex, newFiles, []);
            
            if (!revalidation.hasIssues) {
                // SUCCESS! All issues fixed
                this.renderChatMessage('ai', `✅ Auto-repair successful! All integration issues resolved on attempt ${displayAttempt}.`);
                delete this.state.repairAttempts[attemptKey];
                this.state.contextFiles = originalContext;
                return true;
            } else {
                // Issues remain - update validation for next attempt
                validation = revalidation;
                
                // For the fast-tracked case, we jump straight to failure if it doesn't work the first time.
                if (attemptNumber < MAX_REPAIR_ATTEMPTS && !isMissingFileOnlyIssue) {
                    this.renderChatMessage('ai', `⚠️ ${revalidation.missingReferences.length + revalidation.syntaxErrors.length} issue(s) remain. Escalating to attempt ${attemptNumber + 1}...`);
                } else {
                    // Final attempt failed
                    this.renderChatMessage('ai', '⚠️ Some issues remain after all repair attempts. Manual review needed.');
                    this.generateManualRepairGuide(stepIndex, revalidation);
                    this.state.contextFiles = originalContext;
                    return false;
                }
            }
            
        } catch (error) {
            this.renderChatMessage('ai', `❌ Repair attempt ${displayAttempt} failed: ${error.message}`);
            
            if (attemptNumber >= MAX_REPAIR_ATTEMPTS) {
                this.generateManualRepairGuide(stepIndex, validation);
                this.state.contextFiles = originalContext;
                return false;
            }
            // Continue to next attempt on error
        }
    }
    
    // Cleanup and restore context
    delete this.state.repairAttempts[attemptKey];
    this.state.contextFiles = originalContext;
    return false;
},


getRepairContextFiles(attemptNumber, validation, criticalIssues, minorIssues) {
    const contextFiles = new Set();
    
    // Collect all files mentioned in issues
    const affectedFiles = new Set();
    [...criticalIssues, ...minorIssues].forEach(issue => {
        if (issue.file) {
            affectedFiles.add(issue.file);
        }
    });
    
    if (attemptNumber === 1) {
        // ATTEMPT 1: SURGICAL - Only the affected files
        affectedFiles.forEach(file => contextFiles.add(file));
        
    } else if (attemptNumber === 2) {
        // ATTEMPT 2: SURGICAL+ - Affected files + related files
        affectedFiles.forEach(file => contextFiles.add(file));
        
        // Add related files based on type and location
        affectedFiles.forEach(affectedFile => {
            const ext = affectedFile.split('.').pop();
            const folder = affectedFile.includes('/') ? affectedFile.substring(0, affectedFile.lastIndexOf('/')) : '';
            
            // Add files of same type in same folder
            Object.keys(this.state.projectFiles).forEach(path => {
                const pathExt = path.split('.').pop();
                const pathFolder = path.includes('/') ? path.substring(0, path.lastIndexOf('/')) : '';
                
                if (pathExt === ext && pathFolder === folder && !contextFiles.has(path)) {
                    contextFiles.add(path);
                }
            });
            
            // Add common dependencies
            if (affectedFile.endsWith('.html')) {
                // Add all CSS and JS files
                Object.keys(this.state.projectFiles).forEach(path => {
                    if (path.endsWith('.css') || path.endsWith('.js')) {
                        contextFiles.add(path);
                    }
                });
            } else if (affectedFile.endsWith('.js')) {
                // Add other JS files that might be dependencies
                Object.keys(this.state.projectFiles).forEach(path => {
                    if (path.endsWith('.js') && path !== affectedFile) {
                        contextFiles.add(path);
                    }
                });
            }
        });
        
    } else {
        // ATTEMPT 3: BRUTE FORCE - All editable text files
        Object.keys(this.state.projectFiles).forEach(path => {
            const file = this.state.projectFiles[path];
            
            // Include all text files (not binary)
            if (!file.rawFile && file.content !== undefined) {
                contextFiles.add(path);
            }
        });
    }
    
    return Array.from(contextFiles);
},
// ============================================================================
// FUNCTION 4: getRepairApproachInstructions
// Location: Add after getRepairContextFiles
// Purpose: Provides strategy-specific instructions to AI
// ============================================================================
getRepairApproachInstructions(attemptNumber) {
    if (attemptNumber === 1) {
        return `1. Focus ONLY on the files with errors
2. Make minimal, precise fixes
3. If a function is missing, create it properly
4. If a file is referenced incorrectly, fix the reference
5. Don't modify unrelated code`;
    } else if (attemptNumber === 2) {
        return `1. Review the affected files AND their dependencies
2. Check for cross-file integration issues
3. Ensure all CSS classes referenced in HTML are defined
4. Ensure all JS functions called are properly defined
5. Look for missing imports or script tags
6. Fix any cascading issues from the first attempt`;
    } else {
        return `1. Analyze the ENTIRE project structure
2. Identify ALL integration points between files
3. Create any missing files that should exist
4. Fix all references, imports, and dependencies comprehensively
5. Ensure complete end-to-end functionality
6. This is your LAST chance - be thorough and complete`;
    }
},


// NEW HELPER - Generate specific repair tasks
generateRepairTasks(criticalIssues, minorIssues) {
    const tasks = [];
    
    // Group issues by file
    const fileIssues = {};
    [...criticalIssues, ...minorIssues].forEach(issue => {
        if (issue.file) {
            if (!fileIssues[issue.file]) fileIssues[issue.file] = [];
            fileIssues[issue.file].push(issue);
        }
    });
    
    // Generate specific tasks for each file
    Object.entries(fileIssues).forEach(([file, issues]) => {
        tasks.push(`\n**${file}:**`);
        issues.forEach(issue => {
            if (issue.missing && issue.type === 'JavaScript') {
                tasks.push(`  - Create function "${issue.missing}" with working logic`);
            } else if (issue.missing && issue.type === 'CSS') {
                tasks.push(`  - Define CSS class/ID "${issue.missing}"`);
            } else if (issue.error) {
                tasks.push(`  - Fix syntax error: ${issue.error}`);
            } else {
                tasks.push(`  - Resolve: ${issue.description}`);
            }
        });
    });
    
    return tasks.join('\n');
},

// NEW HELPER - Generate manual repair guide when auto-fix fails
generateManualRepairGuide(stepIndex, validation) {
    let guide = `\n\n📋 **MANUAL REPAIR GUIDE FOR STEP ${stepIndex + 1}**\n\n`;
    guide += `Auto-repair could not fully resolve the issues. Here's what needs manual attention:\n\n`;
    
    if (validation.missingReferences.length > 0) {
        guide += `**Missing References:**\n`;
        validation.missingReferences.forEach((ref, i) => {
            guide += `${i + 1}. In \`${ref.file}\`: Add missing ${ref.type} "${ref.missing}"\n`;
            
            // Provide specific guidance
            if (ref.type === 'JavaScript') {
                guide += `   → Create function \`${ref.missing}\` in a .js file\n`;
            } else if (ref.type === 'CSS') {
                guide += `   → Define \`${ref.missing}\` in your CSS file\n`;
            }
        });
        guide += '\n';
    }
    
    if (validation.syntaxErrors.length > 0) {
        guide += `**Syntax Errors:**\n`;
        validation.syntaxErrors.forEach((err, i) => {
            guide += `${i + 1}. In \`${err.file}\`: ${err.error}\n`;
            guide += `   → Review the code around this error and fix the syntax\n`;
        });
        guide += '\n';
    }
    
    if (validation.warnings.length > 0) {
        guide += `**Warnings:**\n`;
        validation.warnings.forEach((warn, i) => {
            guide += `${i + 1}. ${warn.issue}\n`;
        });
        guide += '\n';
    }
    
    guide += `**Recommended Actions:**\n`;
    guide += `1. Review the files listed above\n`;
    guide += `2. Manually add the missing definitions\n`;
    guide += `3. Test your changes in the preview\n`;
    guide += `4. Use the "Re-do" button on this step if needed\n`;
    
    this.renderChatMessage('ai', guide);
},
async initiateAIRequestWithTriage(prompt, imageToSend = null, triageMode = 'general', historyPrompt = null) {
    // This now uses the clean historyPrompt for the UI, or falls back to the main prompt.
    const userMessageContent = { text: (historyPrompt || prompt) };
    if (imageToSend) {
        userMessageContent.image = imageToSend;
    }
    this.renderChatMessage('user', userMessageContent);

    if (this.state.isExecutingProcess) {
        await this._executeSend(prompt, false, imageToSend, historyPrompt);
        return;
    }

    if (imageToSend) {
        this.renderChatMessage('ai', 'Image detected. Bypassing automatic context selection to ensure full access to all manually selected files for accurate visual analysis.');
        await this._executeSend(prompt, false, imageToSend, historyPrompt);
        return;
    }

    const triageMessage = this.renderChatMessage('ai', ' analyzing context...', false, true);
    this.elements.chatMessages.appendChild(triageMessage);
    this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
    const triageBubble = triageMessage.querySelector('.message-bubble');

    try {
        const fileList = this.state.fileOrder;
        // Pass the triageMode down to the AJAX call function.
        const relevantFiles = await this.getAISelectedContextFiles(prompt, fileList, triageMode);
        
        const newContext = this.updateCheckboxesFromAI(relevantFiles);
        this.state.contextFiles = newContext;
        this.renderFileTree();

        triageBubble.innerHTML = ` selected ${relevantFiles.length} relevant file(s) for context. Generating response...`;

        const isCreatingNewPage = /(create|add|make).*(page|\.html)/i.test(prompt);
        const htmlFileCount = Object.keys(this.state.projectFiles).filter(p => p.endsWith('.html')).length;
        const hasBeenRefactored = !!(this.state.projectFiles['style.css'] || this.state.projectFiles['script.js']);

        if (isCreatingNewPage && htmlFileCount === 1 && !hasBeenRefactored) {
            const shouldRefactor = await this.showConfirmation(
                "It looks like you're adding a new page to a single-file project. For better management, I recommend refactoring the CSS and JS into separate files first. Shall I do this now?"
            );
            
            if (shouldRefactor) {
                triageBubble.innerHTML = `Understood. I will refactor the project first. Once complete, please ask for the new page again.`;
                const refactorPrompt = "Refactor the project to extract embedded CSS and JavaScript into separate style.css and script.js files. Update index.html to link to them correctly.";
                await this._executeSend(refactorPrompt, true, null, `Refactor project before creating '${prompt}'`); 
                return;
            } else {
                triageMessage.remove(); 
                this.renderChatMessage('ai', "Refactoring has been cancelled. Your original request was not sent. What would you like to do instead?");
                return;
            }
        }

        if (this.elements.projectNameInput.value.trim() && Object.keys(this.state.projectFiles).length > 0) {
            this.saveProject(true);
        }

        await this._executeSend(prompt, true, null, historyPrompt);

    } catch (error) {
        console.error('Triage or execution failed:', error);
        triageBubble.innerHTML = `An error occurred: ${error.message}. Please select files manually and try your request again.`;
        this.setLoadingState(false);
    }
},

// 2. MODIFY the `getAISelectedContextFiles` function.
//    - Add 'triageMode' to the function signature.
//    - Append 'triageMode' to the formData.


// REPLACE your current updateCheckboxesFromAI function with this one.
updateCheckboxesFromAI(filesToSelect) {
    // 1. Preserve manually checked images.
    const manuallySelectedImages = new Set();
    for (const file of this.state.contextFiles) {
        if (this.isImageFile(file)) {
            manuallySelectedImages.add(file);
        }
    }

    // 2. Get the AI's suggestions for text/code files.
    const aiSelectedTextFiles = new Set(filesToSelect);

    // 3. Create the final merged context.
    const newContext = new Set([...manuallySelectedImages, ...aiSelectedTextFiles]);

    // 4. THE KEY CHANGE: Return the result.
    return newContext;
},


// COMPLETE executeFullPlan function
// REPLACE your current executeFullPlan function with this one.
// REPLACE your current executeFullPlan function with this FINAL version.
// REPLACE your current executeFullPlan function with this FINAL, streamlined version.
// REPLACE your current executeFullPlan function with this FINAL, CORRECT version.
// REPLACE your current executeFullPlan function with this FINAL version.
// REPLACE your executeFullPlan function with this corrected version
// REPLACE your entire, existing executeFullPlan function with this definitive, corrected version.

// [REPLACE] your entire executeFullPlan function with this FINAL, COMPLETE version.


// 7. UPDATED executeFullPlan function
// REPLACE your current executeFullPlan function with this one.
async executeFullPlan() {
    if (this.state.isExecutingPlan) return;

    // Show the choice modal
    const modal = document.getElementById('execution-choice-modal');
    modal.classList.add('visible');

    // Add event listeners for the new buttons
    document.getElementById('btn-run-unsupervised').onclick = () => {
        modal.classList.remove('visible');
        this.executePlanUnsupervised();
    };
    document.getElementById('btn-run-supervised').onclick = () => {
        modal.classList.remove('visible');
        this.executePlanSupervised();
    };
    document.getElementById('btn-cancel-execution-choice').onclick = () => {
        modal.classList.remove('visible');
    };
},

// ADD THESE TWO NEW FUNCTIONS to your App object in app.js

/**
 * Executes the entire plan automatically without user intervention.
 * Optimized for speed.
 */
// REPLACE your current executePlanUnsupervised function with this one.
async executePlanUnsupervised() {
    // ===== 🆕 PERFORMANCE OPTIMIZATIONS =====
    console.log('⚡ Performance mode: ENABLED');
    this.state.planExecutionMode = true; // Batch operations
    const originalReflection = this.state.enableReflection;
    this.state.enableReflection = false; // Disable reflection during plan
    const startTime = Date.now();
    // ===== END PERFORMANCE SETUP =====
    
    // ===== 🆕 NEW: Reset reflection counters at start of new plan =====
    this.state.reflectionCount = 0;
    this.state.rateLimitHit = false;
    this.state.lastAPICallTime = null;
    console.log('🔄 Reset reflection counters for new plan');
    // ===== END NEW CODE =====
    
    if (this.state.isExecutingPlan) return;

    this.state.isExecutingPlan = true;
    this.togglePlanControls(true);
    
    // Show footer progress bar
    this.showFooterProgress();

    // Create safety backup
    const meaningfulFiles = Object.keys(this.state.projectFiles).filter(path => 
        path !== 'README.md' && 
        !path.endsWith('/.keep') &&
        this.state.projectFiles[path].content &&
        this.state.projectFiles[path].content.trim().length > 50
    );
    
    if (meaningfulFiles.length > 0) {
        this.renderChatMessage('ai', `💾 Creating safety backup of ${meaningfulFiles.length} existing file(s)...`);
        await this.saveProject(true);
    }

    this.renderChatMessage('ai', '🚀 Starting automated plan execution...');
    
    // Initialize intelligent caching with proper error handling
    try {
        this.initializePlanContextCache();
    } catch (error) {
        console.error('❌ Cache initialization failed:', error);
        this.renderChatMessage('ai', '⚠️ Context caching unavailable. Using fallback mode.');
    }

    // Execute all steps with proper context management
    for (let i = 0; i < this.state.currentPlan.length; i++) {
        if (!this.state.isExecutingPlan) {
            this.renderChatMessage('ai', 'Execution stopped by user.');
            break;
        }
        
        try {
            // Execute the step (context selection handled inside executePlanStep)
            const stepSucceeded = await this.executePlanStep(i);

            if (!stepSucceeded) {
                this.renderChatMessage('ai', `❌ Halting execution due to step ${i + 1} failure.`);
                this.state.isExecutingPlan = false;
                break;
            }
            
            // Brief pause between steps (except last step)
            if (this.state.isExecutingPlan && i < this.state.currentPlan.length - 1) {
                await new Promise(resolve => setTimeout(resolve, 500)); // Reduced from 1000ms
            }
            
        } catch (error) {
            console.error(`❌ Step ${i + 1} execution error:`, error);
            this.renderChatMessage('ai', `❌ Error in step ${i + 1}: ${error.message}`);
            this.state.currentPlan[i].status = 'failed';
            this.updateStepUI(i, 'failed');
            this.state.isExecutingPlan = false;
            break;
        }
    }

    // Clear cache
    this.state.planContextCache = null;

    // --- THIS IS THE FIX ---
    // Hide the progress bar immediately after the loop finishes.
    this.hideFooterProgress();
    // --- END FIX ---

    // ===== PERFORMANCE CLEANUP =====
    try {
        this.state.planExecutionMode = false;
        this.state.enableReflection = originalReflection;
        
        // ONE save at the end instead of many during execution
        this.renderChatMessage('ai', '💾 Saving project...');
        await this.saveProject();
        
        const duration = Math.round((Date.now() - startTime) / 1000);
        const minutes = Math.floor(duration / 60);
        const seconds = duration % 60;
        console.log(`⚡ Plan completed in ${minutes}m ${seconds}s`);
        this.renderChatMessage('ai', `⏱️ Execution time: ${minutes}m ${seconds}s`);
    } catch (saveError) {
        console.error('Save error:', saveError);
        this.renderChatMessage('ai', '⚠️ Project save encountered an issue, but execution completed.');
    }
    // ===== END PERFORMANCE CLEANUP =====

    // Now, proceed with the final completion steps (which includes the auto-polish).
    if (this.state.isExecutingPlan) {
        await this.finishPlanModeWithCompletion();
    } else {
        this.finishPlanMode();
    }
    
    this.state.isExecutingPlan = false;
},


// Add this function to ensure file order synchronization
synchronizeFileOrderWithProjectFiles() {
    const projectFilePaths = Object.keys(this.state.projectFiles);
    const fileOrderSet = new Set(this.state.fileOrder);
    
    // Add missing files to fileOrder
    projectFilePaths.forEach(path => {
        if (!fileOrderSet.has(path)) {
            this.state.fileOrder.push(path);
            console.log(`🔄 Added missing file to fileOrder: ${path}`);
        }
    });
    
    // Remove files from fileOrder that don't exist in projectFiles
    this.state.fileOrder = this.state.fileOrder.filter(path => 
        this.state.projectFiles[path]
    );
    
    console.log(`🔄 FileOrder synchronized: ${this.state.fileOrder.length} files`);
},
/**
 * Executes the plan step-by-step, asking the user to confirm context each time.
 * Optimized for accuracy.
 */
async executePlanSupervised() {
    this.state.isExecutingPlan = true;
    this.togglePlanControls(true);
    
    // Show footer progress bar
    this.showFooterProgress();
    
    // ✅ SMART SNAPSHOT: Only if project has meaningful content
    const meaningfulFiles = Object.keys(this.state.projectFiles).filter(path => 
        path !== 'README.md' && 
        !path.endsWith('/.keep') &&
        this.state.projectFiles[path].content &&
        this.state.projectFiles[path].content.trim().length > 100
    );
    
    if (meaningfulFiles.length > 0) {
        this.renderChatMessage('ai', `💾 Creating safety backup of ${meaningfulFiles.length} existing file(s) before guided execution...`);
        await this.saveProject(true); // createAutoSnapshot = true
        this.renderChatMessage('ai', '✅ Backup complete. Your existing work is protected.');
    }
    
    this.renderChatMessage('ai', '🧠 Starting guided plan execution. Please confirm context for each step.');

    // Execute steps with user confirmation
    for (let i = 0; i < this.state.currentPlan.length; i++) {
        if (!this.state.isExecutingPlan) {
            this.renderChatMessage('ai', 'Execution stopped by user.');
            break;
        }

        this.highlightCurrentStep(i);
        
        // The confirmAndExecuteStep function handles everything:
        // showing the modal, getting user choice, and running the step
        const stepSucceeded = await this.confirmAndExecuteStep(i, true);

        if (!stepSucceeded) {
            this.renderChatMessage('ai', `Halting guided execution due to step ${i + 1} failure or cancellation.`);
            this.state.isExecutingPlan = false;
        }
    }
    
    this.highlightCurrentStep(-1); // Clear highlight at the end

    // --- THIS IS THE FIX ---
    // Hide the progress bar immediately after the loop finishes.
    this.hideFooterProgress();
    // --- END FIX ---

    // Finish up
    if (this.state.isExecutingPlan) {
        await this.finishPlanModeWithCompletion();
    } else {
        this.finishPlanMode();
    }
    
    this.state.isExecutingPlan = false;
},

// Add this function to update the cache during plan execution
async updatePlanContextCache() {
    if (this.state.currentPlan && this.state.fileOrder.length > 0) {
        const planSummary = this.state.currentPlan.map(step => step.text).join(' ');
        this.state.planContextCache = await this.analyzeFullPlanContext(planSummary);
        this.state.planContextCacheTimestamp = Date.now();
    }
},

async analyzeFullPlanContext(planSummary) {
    const availableFiles = this.state.fileOrder;
    const selectedFiles = new Set();
    
    // ✨ FEATURE 3: Try AI-powered file selection first
    try {
        const aiSelected = await this.selectFilesWithAI(planSummary, availableFiles);
        if (aiSelected && aiSelected.length > 0) {
            console.log('✅ Using AI-selected files:', aiSelected);
            aiSelected.forEach(f => selectedFiles.add(f));
            
            // Also preserve manually selected images
            this.state.contextFiles.forEach(file => {
                if (this.isImageFile(file)) {
                    selectedFiles.add(file);
                }
            });
            
            return Array.from(selectedFiles);
        }
    } catch (error) {
        console.warn('⚠️ AI file selection failed, using keyword fallback:', error);
    }
    
    // Fallback to keyword-based selection
    const planLower = planSummary.toLowerCase();
    
    // HTML files - include if plan mentions pages, structure, content
    
    // Always preserve manually selected images
    this.state.contextFiles.forEach(file => {
        if (this.isImageFile(file)) {
            selectedFiles.add(file);
        }
    });
    
    // Analyze entire plan for comprehensive context
    
    // HTML files - include if plan mentions pages, structure, content
    if (planLower.includes('html') || planLower.includes('page') || planLower.includes('structure') || 
        planLower.includes('content') || planLower.includes('navigation')) {
        availableFiles.filter(f => f.endsWith('.html')).forEach(f => selectedFiles.add(f));
    }
    
    // CSS files - include if plan mentions styling, design, visual elements
    if (planLower.includes('css') || planLower.includes('style') || planLower.includes('design') || 
        planLower.includes('color') || planLower.includes('layout') || planLower.includes('responsive')) {
        availableFiles.filter(f => f.endsWith('.css')).forEach(f => selectedFiles.add(f));
    }
    
    // JS files - include if plan mentions functionality, interaction, behavior
    if (planLower.includes('js') || planLower.includes('javascript') || planLower.includes('function') || 
        planLower.includes('interactive') || planLower.includes('animation') || planLower.includes('event') ||
        planLower.includes('click') || planLower.includes('behavior') || planLower.includes('logic')) {
        availableFiles.filter(f => f.endsWith('.js')).forEach(f => selectedFiles.add(f));
    }
    
    // If nothing specific mentioned, include main files
    if (selectedFiles.size === 0) {
        const mainFiles = availableFiles.filter(f => 
            f.includes('index') || f.includes('main') || f.includes('app')
        );
        mainFiles.forEach(f => selectedFiles.add(f));
        
        // Also include one of each type if available
        const htmlFile = availableFiles.find(f => f.endsWith('.html'));
        const cssFile = availableFiles.find(f => f.endsWith('.css'));
        const jsFile = availableFiles.find(f => f.endsWith('.js'));
        
        if (htmlFile) selectedFiles.add(htmlFile);
        if (cssFile) selectedFiles.add(cssFile);
        if (jsFile) selectedFiles.add(jsFile);
    }
    
    return Array.from(selectedFiles);
},
// REPLACE your scrollToUnifyButtonWithRetries function with this corrected version
scrollToUnifyButtonWithRetries() {
    let attempts = 0;
    const maxAttempts = 5;
    
    const tryScroll = () => {
        // Try to find the current plan's polish button
        let polishBtn;
        if (this.state.currentPlanButtonIds) {
            polishBtn = document.getElementById(this.state.currentPlanButtonIds.polish);
        } else {
            polishBtn = document.getElementById('polish-plan-btn');
        }
        
        if (polishBtn && polishBtn.style.display !== 'none') {
            // Scroll the button into view
            polishBtn.scrollIntoView({ behavior: 'smooth', block: 'center' });
            
            // Add a subtle highlight animation
            polishBtn.style.animation = 'gentle-pulse 2s ease-in-out 3';
            
            // Show next step message WITHOUT "Plan execution complete" (as history to prevent auto-scroll)
            setTimeout(() => {
                this.renderChatMessage('ai', 'Next Step: Use the Polish & Unify Design button to create visual consistency and fix any integration issues between the generated components.', true);
            }, 1000);
            return;
        }
        
        attempts++;
        if (attempts < maxAttempts) {
            setTimeout(tryScroll, 500);
        }
    };
    
    setTimeout(tryScroll, 1000); // Initial delay
},

buildContextAwareStepPrompt(stepText, stepIndex) {
    const context = this.state.projectContext;
    const specification = this.state.projectSpecification;
    
    // Analyze what type of task this is
    const taskType = this.identifyTaskType(stepText);
    
    // BUILD SPECIFICATION SECTION (enhanced)
    let specSection = '';
    if (specification) {
        specSection = `
**YOU MUST FOLLOW THIS PROJECT SPECIFICATION EXACTLY:**
PROJECT TYPE: ${specification.project_type || 'Web Application'}
KEY FEATURES: ${specification.key_features || 'Standard features'}
DESIGN STYLE: ${specification.design_style || 'Modern'}
TARGET AUDIENCE: ${specification.target_audience || 'General users'}
TECHNICAL REQUIREMENTS: ${specification.technical_requirements || 'Standard web technologies'}
**CRITICAL**: Every file you create must perfectly align with these requirements.
`;
    }
    
    // NEW: Add future context awareness
    let futureContext = '';
    if (stepIndex < this.state.currentPlan.length - 2) {
        const upcomingSteps = this.state.currentPlan
            .slice(stepIndex + 1, Math.min(stepIndex + 3, this.state.currentPlan.length))
            .map((s, i) => `${i + 1}. ${s.text}`)
            .join('\n');
        
        futureContext = `
**UPCOMING STEPS (for planning compatibility):**
${upcomingSteps}
⚠️ Important: Design your implementation to be compatible with these future tasks.`;
    }
    
    // NEW: Add project snapshot
    const projectSnapshot = this.getProjectSnapshot();
    
    // 🚨 CRITICAL FIX: ADD FILE CONTEXT SECTION
    const fileContext = this.buildFileContextSection();
    
    const prompt = `STEP ${stepIndex} OF ${this.state.currentPlan.length} - ${taskType.toUpperCase()} TASK

**YOUR SINGLE, FOCUSED TASK:**
"${stepText}"

${specSection}

**CRITICAL FILE NAMING CONVENTIONS:**
- CSS file MUST be named: style.css (singular, NOT styles.css)
- JavaScript file MUST be named: script.js (singular, NOT scripts.js)
- These exact names are mandatory for consistency across all projects

**CURRENT PROJECT STATE:**
${projectSnapshot}

${fileContext}  // 🚨 THIS IS WHAT'S MISSING!

${futureContext}

**MANDATORY RESPONSE FORMAT:**
Your entire response MUST begin with a JSON block containing a summary, followed by file blocks.

\`\`\`json
{
  "summary": "One-sentence description of what you did. Example: 'Created mobile navigation menu with hamburger icon.'",
  "files_used": ["path/to/file1.js", "path/to/file2.html"]
}
\`\`\`

File: path/to/file.ext (NEW|MODIFY)
\`\`\`[language]
[Complete, functional code]
\`\`\`

**CRITICAL RULES:**
1. Start with JSON summary block - NO other text before it
2. Create ONLY files needed for THIS specific task
3. Use SINGULAR filenames: style.css (not styles.css), script.js (not scripts.js)
4. Preserve ALL existing functionality from previous steps
5. For JavaScript: ALWAYS wrap DOM access in DOMContentLoaded
6. Never break existing integrations between files

**INTEGRATION CHECKLIST (verify before responding):**
✅ All CSS classes I reference exist in CSS files
✅ All JavaScript functions I call exist in JS files
✅ All onclick handlers have corresponding functions
✅ All href links point to valid targets
✅ Mobile navigation works (if applicable)
✅ Filenames use singular convention (style.css, script.js)

**TASK-SPECIFIC GUIDANCE:**
${this.getTaskSpecificGuidance(taskType, stepText)}

**PROJECT DESIGN GUIDELINES:**
- Original Vision: ${context.originalVision}
- Design Requirements: ${context.designRequirements}
- Technical Notes: ${context.technicalNotes}
- Target Audience: ${context.targetAudience}

Complete this task now with perfect attention to integration:`;
    
    return prompt.trim();
},
buildFileContextSection() {
    const contextFiles = Array.from(this.state.contextFiles || []);
    
    if (contextFiles.length === 0) {
        return "**CURRENT FILES:** No files in context for this step.";
    }
    
    let fileContext = "**CURRENT FILES (You MUST reference these in your implementation):**\n\n";
    
    contextFiles.forEach(filePath => {
        const file = this.state.projectFiles[filePath];
        if (file) {
            fileContext += `FILE: ${filePath}\n\`\`\`${this.getFileLanguage(filePath)}\n${file.content}\n\`\`\`\n\n`;
        }
    });
    
    return fileContext;
},

getFileLanguage(filePath) {
    if (filePath.endsWith('.html')) return 'html';
    if (filePath.endsWith('.css')) return 'css';
    if (filePath.endsWith('.js')) return 'javascript';
    return 'text';
},

// NEW HELPER: Provides current project state overview
getProjectSnapshot() {
    const htmlFiles = this.state.fileOrder.filter(f => f.endsWith('.html'));
    const cssFiles = this.state.fileOrder.filter(f => f.endsWith('.css'));
    const jsFiles = this.state.fileOrder.filter(f => f.endsWith('.js'));
    const otherFiles = this.state.fileOrder.filter(f => 
        !f.endsWith('.html') && !f.endsWith('.css') && !f.endsWith('.js')
    );
    
    let snapshot = `📁 Project Files (${this.state.fileOrder.length} total):\n`;
    
    if (htmlFiles.length > 0) {
        snapshot += `   HTML: ${htmlFiles.join(', ')}\n`;
    }
    if (cssFiles.length > 0) {
        snapshot += `   CSS: ${cssFiles.join(', ')}\n`;
    }
    if (jsFiles.length > 0) {
        snapshot += `   JS: ${jsFiles.join(', ')}\n`;
    }
    if (otherFiles.length > 0) {
        snapshot += `   Other: ${otherFiles.join(', ')}\n`;
    }
    
    if (this.state.fileOrder.length === 0) {
        snapshot = '📁 Project is empty (this is the first step)';
    }
    
    return snapshot;
},

identifyTaskType(stepText) {
    const lower = stepText.toLowerCase();
    
    // Order matters - check more specific patterns first
    if (lower.includes('api') || lower.includes('fetch') || lower.includes('backend')) {
        return 'api_integration';
    }
    if ((lower.includes('create') || lower.includes('add')) && lower.includes('page')) {
        return 'page_creation';
    }
    if (lower.includes('style') || lower.includes('css') || lower.includes('design')) {
        return 'styling';
    }
    if (lower.includes('interactive') || lower.includes('function') || 
        lower.includes('click') || lower.includes('event')) {
        return 'functionality';
    }
    if (lower.includes('form') || lower.includes('input') || lower.includes('validation')) {
        return 'form_handling';
    }
    if (lower.includes('navigation') || lower.includes('menu') || lower.includes('nav')) {
        return 'navigation';
    }
    if (lower.includes('responsive') || lower.includes('mobile')) {
        return 'responsive';
    }
    
    return 'general';
},

getTaskSpecificGuidance(taskType, stepText) {
    const guidance = {
        page_creation: `- Create a complete HTML5 document with proper structure
- Include <!DOCTYPE html>, <head>, and <body>
- Link to existing style.css or create it if needed
- Use semantic HTML5 elements (header, nav, main, section, footer)
- Make it mobile-responsive with viewport meta tag
- Add meaningful content, not just placeholders`,
        
        styling: `- Work with existing CSS file or create a new one with clear purpose
- Use CSS variables (--color-primary, etc.) for consistency
- Include responsive breakpoints (@media queries)
- Don't remove or override existing styles unless specifically asked
- Use modern CSS (flexbox, grid) where appropriate
- Maintain consistent spacing and typography`,
        
        functionality: `- Add JavaScript in a proper DOMContentLoaded wrapper
- Use querySelector/getElementById with null checks
- Add event listeners properly (addEventListener, not onclick attributes)
- Test for existing elements before manipulating them
- Add proper error handling
- Use meaningful variable and function names`,
        
        api_integration: `- Create clear, well-named API functions
- Use async/await pattern consistently
- Add comprehensive error handling (try/catch)
- Show loading states to users
- Handle network failures gracefully
- Document API endpoint structure`,
        
        form_handling: `- Add proper form validation (HTML5 + JavaScript)
- Prevent default form submission if handling with JS
- Show clear error messages to users
- Validate on both client and prepare for server validation
- Make forms accessible (labels, aria attributes)
- Handle form submission state (disable button, show loading)`,
        
        navigation: `- Create accessible navigation (semantic HTML)
- Add mobile hamburger menu if needed
- Implement smooth scrolling if applicable
- Highlight active/current page
- Make it keyboard accessible
- Ensure all links work and point to existing pages`,
        
        responsive: `- Use mobile-first approach (base styles, then @media min-width)
- Test breakpoints: 320px (mobile), 768px (tablet), 1024px (desktop)
- Make touch targets at least 44x44px
- Use flexible units (%, em, rem) instead of fixed px
- Ensure text is readable without zooming
- Make images responsive (max-width: 100%, height: auto)`,
        
        general: `- Follow web development best practices
- Write clean, maintainable code
- Use consistent naming conventions
- Comment complex logic
- Test that code works as expected`
    };
    
    return guidance[taskType] || guidance.general;
},

// REPLACE your entire existing calculateRelevanceScore function with this:
calculateRelevanceScore(file, stepText) {
    let score = 0;
    const fileName = file.toLowerCase();
    const fileBaseName = fileName.replace(/\.[^.]+$/, '');
    const stepLower = stepText.toLowerCase();
    
    // EXISTING: Exact file name mention = very high score
    if (stepLower.includes(fileBaseName)) {
        score += 100;
    }
    
    // EXISTING: File type relevance based on keywords
    if (stepLower.includes('style') || stepLower.includes('css') || 
        stepLower.includes('design') || stepLower.includes('color')) {
        if (fileName.endsWith('.css')) score += 50;
    }
    
    if (stepLower.includes('page') || stepLower.includes('html') || 
        stepLower.includes('content') || stepLower.includes('text')) {
        if (fileName.endsWith('.html')) score += 50;
    }
    
    if (stepLower.includes('function') || stepLower.includes('interactive') || 
        stepLower.includes('click') || stepLower.includes('event') ||
        stepLower.includes('javascript') || stepLower.includes('js')) {
        if (fileName.endsWith('.js')) score += 50;
    }
    
    // NEW: Dependency graph analysis
    const dependencies = this.analyzeDependencyGraph([file]);
    if (dependencies[file]) {
        const deps = dependencies[file];
        
        // If this file imports others mentioned in step, boost score
        deps.imports.forEach(importedFile => {
            if (stepLower.includes(importedFile.toLowerCase())) {
                score += 35;
            }
        });
        
        // If this file exports something mentioned in step, boost score
        deps.exports.forEach(exportName => {
            if (stepLower.includes(exportName.toLowerCase())) {
                score += 40;
            }
        });
        
        // If this file references classes/IDs mentioned in step, boost score
        deps.references.forEach(ref => {
            if (stepLower.includes(ref.toLowerCase())) {
                score += 30;
            }
        });
    }
    
    // NEW: Cross-file reference scoring
    if (this.state.planContextCache) {
        const recentFiles = this.state.planContextCache.stepHistory
            .slice(-2)
            .flatMap(s => [...s.newFiles, ...s.modifiedFiles]);
        
        if (recentFiles.includes(file)) {
            score += 50; // Recently touched files are highly relevant
        }
        
        // Check if this file references recently created files
        const content = this.state.projectFiles[file]?.content || '';
        recentFiles.forEach(recentFile => {
            if (content.includes(recentFile)) {
                score += 30; // This file imports/references recent work
            }
        });
    }
    
    // NEW: Semantic relationship scoring
    const relationships = {
        'index.html': ['style.css', 'script.js', 'main.js'],
        'style.css': ['index.html', 'main.html'],
        'script.js': ['index.html', 'utils.js', 'api.js'],
        'main.css': ['index.html', 'main.html'],
        'app.js': ['index.html', 'main.html']
    };
    
    if (relationships[fileName]) {
        const relatedFiles = relationships[fileName];
        const stepMentionsRelated = relatedFiles.some(rf => stepLower.includes(rf));
        if (stepMentionsRelated) score += 25;
    }
    
    // EXISTING: Recently modified files are more relevant
    if (this.state.planContextCache) {
        const recentSteps = this.state.planContextCache.stepHistory.slice(-2);
        recentSteps.forEach(step => {
            if (step.newFiles.includes(file)) {
                score += 40; // Recently created
            }
            if (step.modifiedFiles.includes(file)) {
                score += 30; // Recently modified
            }
        });
    }
    
    // EXISTING: Core files get a small boost
    if (fileName.includes('index.html') || fileName.includes('main.') || 
        fileName.includes('style.css') || fileName.includes('app.')) {
        score += 10;
    }
    
    return score;
},

// NEW FUNCTION - Dependency graph analyzer
analyzeDependencyGraph(files) {
    const graph = {};
    
    files.forEach(file => {
        const content = this.state.projectFiles[file]?.content || '';
        graph[file] = {
            imports: this.extractImports(content, file),
            exports: this.extractExports(content, file),
            references: this.extractReferences(content, file)
        };
    });
    
    return graph;
},


// NEW HELPER - Extract imports from file
extractImports(content, filePath) {
    const imports = [];
    
    // CSS imports in HTML - REMOVED DUPLICATE DECLARATION
    const cssMatches = content.matchAll(/href=["']([^"']*\.css)["']/gi);
    for (const match of cssMatches) {
        imports.push(match[1].replace(/^\.\//, ''));
    }
    
    // JS imports in HTML
    const jsMatches = content.matchAll(/src=["']([^"']*\.js)["']/gi);
    for (const match of jsMatches) {
        imports.push(match[1].replace(/^\.\//, ''));
    }
    
    // ES6 imports in JS
    const es6Matches = content.matchAll(/import\s+.*?from\s+["']([^"']+)["']/gi);
    for (const match of es6Matches) {
        imports.push(match[1].replace(/^\.\//, ''));
    }
    
    return [...new Set(imports)]; // Remove duplicates
},

// NEW HELPER - Extract exports from file
extractExports(content, filePath) {
    const exports = [];
    
    if (filePath.endsWith('.js')) {
        // Function declarations
        const functionMatches = content.matchAll(/(?:export\s+)?function\s+(\w+)/gi);
        for (const match of functionMatches) {
            exports.push(match[1]);
        }
        
        // Const/let exports
        const constMatches = content.matchAll(/(?:export\s+)?(?:const|let)\s+(\w+)/gi);
        for (const match of constMatches) {
            exports.push(match[1]);
        }
        
        // Class exports
        const classMatches = content.matchAll(/(?:export\s+)?class\s+(\w+)/gi);
        for (const match of classMatches) {
            exports.push(match[1]);
        }
    }
    
    if (filePath.endsWith('.css')) {
        // CSS class names
        const classMatches = content.matchAll(/\.([a-zA-Z_-][a-zA-Z0-9_-]*)\s*{/g);
        for (const match of classMatches) {
            exports.push(`.${match[1]}`);
        }
        
        // CSS IDs
        const idMatches = content.matchAll(/#([a-zA-Z_-][a-zA-Z0-9_-]*)\s*{/g);
        for (const match of idMatches) {
            exports.push(`#${match[1]}`);
        }
    }
    
    return [...new Set(exports)];
},

// NEW HELPER - Extract references from file
extractReferences(content, filePath) {
    const references = [];
    
    if (filePath.endsWith('.html')) {
        // Class references
        const classMatches = content.matchAll(/class=["']([^"']+)["']/gi);
        for (const match of classMatches) {
            const classes = match[1].split(/\s+/);
            classes.forEach(cls => references.push(`.${cls}`));
        }
        
        // ID referencesAPI
        const idMatches = content.matchAll(/id=["']([^"']+)["']/gi);
        for (const match of idMatches) {
            references.push(`#${match[1]}`);
        }
        
        // Function calls in onclick
        const onclickMatches = content.matchAll(/onclick=["']([^"'(]+)\(/gi);
        for (const match of onclickMatches) {
            references.push(match[1].trim());
        }
    }
    
    if (filePath.endsWith('.js')) {
        // querySelector/getElementById references
        const querySelectorMatches = content.matchAll(/querySelector\(["']([^"']+)["']\)/gi);
        for (const match of querySelectorMatches) {
            references.push(match[1]);
        }
        
        const getElementByIdMatches = content.matchAll(/getElementById\(["']([^"']+)["']\)/gi);
        for (const match of getElementByIdMatches) {
            references.push(`#${match[1]}`);
        }
        
        // Function calls
        const functionCallMatches = content.matchAll(/(\w+)\s*\(/g);
        for (const match of functionCallMatches) {
            const funcName = match[1];
            // Exclude common keywords
            if (!['if', 'for', 'while', 'function', 'return', 'const', 'let', 'var'].includes(funcName)) {
                references.push(funcName);
            }
        }
    }
    
    return [...new Set(references)];
},


// REPLACE your _makeAPICall function with this COMPLETE version:



// NEW HELPER: Explain why specific files were selected
explainFileSelection(files, stepText) {
    const htmlFiles = files.filter(f => f.endsWith('.html'));
    const cssFiles = files.filter(f => f.endsWith('.css'));
    const jsFiles = files.filter(f => f.endsWith('.js'));
    const otherFiles = files.filter(f => 
        !f.endsWith('.html') && !f.endsWith('.css') && !f.endsWith('.js')
    );
    
    const taskType = this.identifyTaskType(stepText);
    let explanation = `This is a <strong>${taskType}</strong> task. `;
    
    if (htmlFiles.length > 0) {
        explanation += `Included <strong>${htmlFiles.length} HTML</strong> file(s) for page structure. `;
    }
    if (cssFiles.length > 0) {
        explanation += `Included <strong>${cssFiles.length} CSS</strong> file(s) for styling. `;
    }
    if (jsFiles.length > 0) {
        explanation += `Included <strong>${jsFiles.length} JavaScript</strong> file(s) for functionality. `;
    }
    if (otherFiles.length > 0) {
        explanation += `Included <strong>${otherFiles.length} other</strong> file(s) as supporting files. `;
    }
    
    return explanation || 'Files selected based on step analysis and project history.';
},

// ADD THIS ENTIRE NEW FUNCTION to your App object.
// This is the function for the SUPERVISED path.
// [REPLACE] your entire confirmAndExecuteStep function with this one.
async confirmAndExecuteStep(stepIndex, isAutoRun = false) {
    return new Promise(async (resolve) => {
        const step = this.state.currentPlan[stepIndex];
        if (!step) {
            resolve(false);
            return;
        }
        
        this.renderChatMessage('ai', `Analyzing context for Step ${stepIndex + 1}: ${step.text}`);
        let aiSuggestedFiles;
        try {
            aiSuggestedFiles = await this.getAISelectedContextFiles(step.text, this.state.fileOrder);
        } catch (error) {
            this.renderChatMessage('ai', `Context analysis failed: ${error.message}. Please select files manually and retry.`);
            resolve(false);
            return;
        }

        const manualSelection = Array.from(this.state.contextFiles);
        const modal = document.getElementById('ai-suggestion-modal');
        const fileListEl = document.getElementById('ai-suggested-files-list');
        const aiButton = document.getElementById('btn-confirm-ai-selection');
        const manualButton = document.getElementById('btn-confirm-manual-selection');
        const cancelButton = document.getElementById('btn-cancel-suggestion');

        fileListEl.innerHTML = aiSuggestedFiles.length > 0
            ? `<div style="margin-bottom: 1rem; padding: 0.75rem; background: var(--bg-accent); border-radius: 6px; border-left: 3px solid var(--accent-primary);">
                 <strong style="color: var(--accent-primary);">📋 Why these files?</strong><br>
                 <span style="color: var(--text-secondary); font-size: 0.9rem;">${this.explainFileSelection(aiSuggestedFiles, step.text)}</span>
               </div>
               ${aiSuggestedFiles.map(file => `<li style="font-family: var(--font-mono); color: var(--text-primary);">- ${this.escapeHtml(file)}</li>`).join('')}`
            : '<li>The AI suggests no files are needed for this step.</li>';

        aiButton.innerHTML = `Run with AI's Suggestion (${aiSuggestedFiles.length} files)`;
        manualButton.innerHTML = `Run with My Manual Selection (${manualSelection.length} files)`;
        
        const selectionsAreIdentical = JSON.stringify([...aiSuggestedFiles].sort()) === JSON.stringify([...manualSelection].sort());
        manualButton.style.display = selectionsAreIdentical ? 'none' : 'block';
        
        modal.classList.add('visible');

        const execute = async (useAISelection) => {
            modal.classList.remove('visible');
            aiButton.onclick = null;
            manualButton.onclick = null;
            cancelButton.onclick = null;

            if (useAISelection) {
                const finalMergedContext = this.updateCheckboxesFromAI(aiSuggestedFiles);
                this.state.contextFiles = finalMergedContext;
                this.renderFileTree();
                this.renderChatMessage('ai', `Using AI-selected context for Step ${stepIndex + 1}.`);
            } else {
                this.renderChatMessage('ai', `Using manually-selected context for Step ${stepIndex + 1}.`);
            }
            
            // ✅ ADD THESE TWO LINES - Set the flag to bypass diff
            this.state.isExecutingProcess = true;
            
            this.state.currentPlan[stepIndex].status = 'running';
            this.updateStepUI(stepIndex, 'running');
            this.highlightCurrentStep(stepIndex);

            const planExecutionPrompt = this.buildContextAwareStepPrompt(step.text, stepIndex + 1);

            try {
                await this._executeSend(planExecutionPrompt, true, null, `Step ${stepIndex + 1}: ${step.text}`, { isStepExecution: true });
                
                this.state.currentPlan[stepIndex].status = 'done';
                this.updateStepUI(stepIndex, 'done');
                resolve(true);
            } catch (error) {
                this.state.currentPlan[stepIndex].status = 'failed';
                this.updateStepUI(stepIndex, 'failed');
                this.renderChatMessage('ai', `An error occurred on step ${stepIndex + 1}.`);
                resolve(false);
            } finally {
                this.highlightCurrentStep(-1);
                // ✅ ADD THIS LINE - Reset the flag
                this.state.isExecutingProcess = false;
                
                if (!isAutoRun) {
                    this.finishPlanMode();
                }
            }
        };

        aiButton.onclick = () => execute(true);
        manualButton.onclick = () => execute(false);
        cancelButton.onclick = () => {
            modal.classList.remove('visible');
            this.renderChatMessage('ai', 'Execution cancelled by user.');
            resolve(false);
        };
    });
},
togglePlanControls(isExecuting) {
    let executeBtn, stopBtn;
    
    if (this.state.currentPlanButtonIds) {
        executeBtn = document.getElementById(this.state.currentPlanButtonIds.execute);
        stopBtn = document.getElementById(this.state.currentPlanButtonIds.stop);
    } else {
        executeBtn = document.getElementById('execute-plan-btn');
        stopBtn = document.getElementById('stop-plan-btn');
    }
    
    if (executeBtn && stopBtn) {
        if (isExecuting) {
            executeBtn.style.display = 'none';
            stopBtn.style.display = 'inline-flex';
        } else {
            executeBtn.style.display = 'none';
            stopBtn.style.display = 'none';
        }
    }
    // Polish button logic is now completely removed.
},


        highlightCurrentStep(index) {
            const steps = document.querySelectorAll('.plan-step');
            steps.forEach((stepEl, i) => {
                stepEl.style.background = i === index ? 'rgba(139, 92, 246, 0.2)' : 'transparent';
                stepEl.style.fontWeight = i === index ? 'bold' : 'normal';
            });
        },
        handlePromptLengthCheck() {
            const PROMPT_CHAR_LIMIT = 2000;
            const userInput = this.elements.oneShotInput.value;
            const warningElement = document.getElementById('prompt-length-warning');
            if (userInput.length > PROMPT_CHAR_LIMIT) {
                warningElement.textContent = `Prompt is long. Consider using the ✨ Refine button. (Current: ${userInput.length}/${PROMPT_CHAR_LIMIT})`;
                warningElement.style.display = 'block';
                this.elements.btnGenerateWebsite.disabled = true;
            } else {
                warningElement.style.display = 'none';
                this.elements.btnGenerateWebsite.disabled = false;
            }
        },
      // REPLACE your current handleImageDescriptionRequest function with this one.
handleImageDescriptionRequest() {
    this.toggleImageModal(false);
    this.renderChatMessage('user', `Describe the image "${this.state.tempVisionImagePath}"`);
    this.renderChatMessage('ai', `👁️ AI Vision Active on "${this.state.tempVisionImagePath}". I will now describe this image.`);
    const analysisPrompt = `Describe the visual contents of the provided image in detail. DO NOT write code. Focus on objects, colors, text, composition, and overall style. The output should be a paragraph or a simple list, not a numbered development plan.`;
    
    // --- THIS IS THE FIX ---
    // We now pass a 'false' flag for the new 'allowPlannning' parameter.
    this._executeSend(analysisPrompt, false, this.state.tempVisionImage, `Describe image ${this.state.tempVisionImagePath}`, false);
},
     // REPLACE your current handleImagePlanRequest function with this one.



		
async runDesignConsistencyPass() {
   if (this.state.isAILoading) {
       return;
   }
   
   const polishBtn = document.getElementById('polish-plan-btn');
   if (polishBtn) {
       polishBtn.scrollIntoView({ behavior: 'smooth', block: 'center' });
   }
   
   this.state.isExecutingProcess = true;
   this.showUnifyLoading();
   
   const allFiles = this.getEditableFiles();
   const filesToProcess = this.prioritizeAndFilterFiles(allFiles);
   
   if (filesToProcess.length === 0) {
       this.hideUnifyLoading();
       this.state.isExecutingProcess = false;
       this.renderChatMessage('ai', 'No suitable files found for design unification.', true);
       return;
   }
   
   const totalSize = this.getTotalFileSize(filesToProcess);
   if (totalSize > 300000 || filesToProcess.length > 5) {
       this.hideUnifyLoading();
       this.state.isExecutingProcess = false;
       this.renderChatMessage('ai', `Project too large for unify tool (${filesToProcess.length} files, ${Math.round(totalSize/1000)}KB). Try polishing individual files manually or reducing project complexity.`, true);
       return;
   }
   
   this.renderChatMessage('ai', `Running design consistency check on ${filesToProcess.length} files...`, true);
   
   const consistencyPrompt = `DESIGN CONSISTENCY & INTEGRATION AUDIT

**MISSION: Create visual harmony and fix integration issues across these ${filesToProcess.length} files.**

**FILES TO UNIFY:**
${filesToProcess.map(file => `- ${file} (${Math.round(this.state.projectFiles[file].content.length/1000)}KB)`).join('\n')}

**CRITICAL ANALYSIS AREAS:**

**1. INTEGRATION FIXES (PRIORITY 1)**
- Fix all broken JavaScript function calls
- Ensure all CSS classes referenced in HTML actually exist
- Repair missing onclick handlers and event listeners
- Fix broken file references and imports
- Resolve naming conflicts between files

**2. VISUAL CONSISTENCY (PRIORITY 2)**
- Unify color palette across all components
- Standardize typography and spacing patterns
- Create consistent button and form styling
- Harmonize animation timing and transitions
- Ensure responsive breakpoints are consistent

**3. COMPONENT HARMONY**
- Make all similar elements look and behave the same
- Consistent navigation styling across pages
- Unified card/section design patterns
- Standard hover effects and focus states

**PRESERVATION RULES:**
1. Keep ALL existing content and functionality
2. Maintain the original purpose of each page
3. Fix broken code before adding visual improvements
4. Ensure mobile responsiveness works perfectly

**OUTPUT REQUIREMENT:**
Return complete, working versions of files that need changes. Focus on making everything work together seamlessly while looking professionally designed.

The goal is a cohesive, fully-functional website where all components feel like they belong together.`;

   try {
       await this._executeSend(
           consistencyPrompt, 
           true, 
           null, 
           'Design consistency pass',
           { isStepExecution: true }
       );
       this.renderChatMessage('ai', 'Design consistency check complete! Your project now has unified, professional styling.', true);
   } catch (error) {
       this.handleUnifyError(error);
   } finally {
       this.hideUnifyLoading();
       this.state.isExecutingProcess = false;
   }
},
showPlanGenerationLoading() {
    const executeBtn = document.getElementById('execute-plan-btn');
    if (executeBtn) {
        executeBtn.style.display = 'none';
    }
    
    const loadingHtml = `
        <div id="plan-generation-loading" style="
            text-align: center; 
            padding: 1rem; 
            background: var(--bg-secondary); 
            border-radius: var(--radius-md); 
            margin: 1rem 0;
            border: 1px solid var(--border-primary);
            display: flex;
            align-items: center;
            justify-content: center;
        ">
            <div style="
                display: inline-block;
                width: 20px;
                height: 20px;
                border: 2px solid var(--border-primary);
                border-top: 2px solid var(--accent-primary);
                border-radius: 50%;
                animation: spin 1s linear infinite;
                margin-right: 0.5rem;
            "></div>
            <span style="color: var(--text-secondary);">Generating development plan...</span>
        </div>
    `;
    
    const tempDiv = document.createElement('div');
    tempDiv.innerHTML = loadingHtml;
    this.elements.chatMessages.appendChild(tempDiv.firstElementChild);
    this.elements.chatMessages.scrollTop = this.elements.chatMessages.scrollHeight;
},
hidePlanGenerationLoading() {
    const loadingElement = document.getElementById('plan-generation-loading');
    if (loadingElement) {
        loadingElement.remove();
    }
    
    // Show Execute Plan button when generation is complete
    const executeBtn = document.getElementById('execute-plan-btn');
    if (executeBtn) {
        executeBtn.style.display = 'inline-flex';
    }
},

// 4. REPLACE the `showUnifyLoading` function with this chat-based version:
showUnifyLoading() {
    this.renderChatMessage('ai', '✨ Running automated design unification & integration pass...');
},


// 5. REPLACE the `hideUnifyLoading` function with this empty version:
hideUnifyLoading() {
    // This function no longer needs to manipulate UI buttons.
    // Success/failure messages are handled in the calling function.
},


// ADD THIS NEW HELPER FUNCTION inside the App object
isImageFile(path) {
    // A simple regex to check for common image file extensions.
    return /\.(png|jpe?g|webp|gif|svg)$/i.test(path);
},
// [REPLACE] your handleRedoStep function with this one.
async handleRedoStep(stepIndex) {
    if (this.state.isExecutingPlan) {
        this.renderChatMessage('ai', 'Please wait for the current task to complete.');
        return;
    }

    const originalStep = this.state.currentPlan[stepIndex];
    if (!originalStep) return;

    const modification = prompt(`You are re-doing this step:\n\n"${originalStep.text}"\n\nPlease describe your desired change:`, "");
    if (modification === null || modification.trim() === "") {
        return;
    }

    // Reset UI state for plan controls
    const designUnifiedBtn = document.getElementById('design-unified-btn');
    if (designUnifiedBtn) designUnifiedBtn.remove();
    const polishBtn = document.getElementById('polish-plan-btn');
    if (polishBtn) polishBtn.style.display = 'none';

    this.state.isExecutingPlan = true;
    this.state.isExecutingProcess = true;
    this.state.currentPlan[stepIndex].status = 'running';
    this.updateStepUI(stepIndex, 'running');
    this.highlightCurrentStep(stepIndex);

    this.renderChatMessage('ai', `Re-doing Step ${stepIndex + 1}: ${originalStep.text}\nWith Modification: ${modification}`);

    const redoPrompt = `EXECUTE CODE ONLY - NO PLANNING

You are re-executing step: "${originalStep.text}"
User's modification: "${modification}"

CRITICAL RULES:
1. Write ONLY the code files needed for this modified step.
2. Do NOT create any plans, lists, or numbered steps.
3. Output format: File: filename.ext followed by code block.
4. NO other text outside the file blocks.

Write the code now:`;

    try {
        // --- THIS IS THE FIX ---
        // We now explicitly tell _executeSend that this is a step execution,
        // which will prevent it from trying to generate a new plan.
        await this._executeSend(redoPrompt, true, null, `Re-doing: ${originalStep.text}`, { isStepExecution: true });
        // --- END OF FIX ---

        this.state.currentPlan[stepIndex].status = 'done';
        this.updateStepUI(stepIndex, 'done');
        this.renderChatMessage('ai', `Step ${stepIndex + 1} has been re-executed with your changes.`);
    } catch (error) {
        this.state.currentPlan[stepIndex].status = 'failed';
        this.updateStepUI(stepIndex, 'failed');
        this.renderChatMessage('ai', `An error occurred while re-doing the step.`);
    } finally {
        this.state.isExecutingPlan = false;
        this.state.isExecutingProcess = false;
        this.highlightCurrentStep(-1);
        this.togglePlanControls(false); // We can keep this for a clean finish
    }
},
   getProjectStructureString() {
    if (!this.state.fileOrder || this.state.fileOrder.length === 0) { 
        return "The project is currently empty."; 
    }
    
    let result = "PROJECT STRUCTURE:\n\n";
    
    // Add file tree structure first
    const structure = new Set();
    this.state.fileOrder.forEach(path => {
        const parts = path.split('/');
        let currentPath = '';
        for (let i = 0; i < parts.length; i++) {
            currentPath += (i > 0 ? '/' : '') + parts[i];
            if (i < parts.length - 1) { 
                structure.add(`- ${parts.slice(0, i + 1).join('/')}/ (folder)`); 
            } else { 
                structure.add(`- ${currentPath}${this.state.contextFiles.has(currentPath) ? ' ✓' : ''}`); 
            }
        }
    });
    result += Array.from(structure).sort().join('\n');
    
    // Add file contents for checked files
    const checkedFiles = Array.from(this.state.contextFiles);
    if (checkedFiles.length > 0) {
        result += "\n\nFILE CONTENTS (checked files only):\n\n";
        
        checkedFiles.forEach(path => {
            const fileData = this.state.projectFiles[path];
            if (fileData) {
                if (fileData.rawFile) {
                    // Binary file
                    result += `--- ${path} (BINARY FILE) ---\n`;
                    result += `File type: ${fileData.rawFile.type || 'unknown'}\n`;
                    result += `File size: ${(fileData.rawFile.size / 1024).toFixed(2)} KB\n`;
                    result += "This is a binary file and cannot display text content.\n\n";
                } else if (fileData.content) {
                    // Text file
                    result += `--- ${path} ---\n`;
                    result += fileData.content;
                    result += "\n\n";
                }
            }
        });
    } else {
        result += "\n\nNo files are currently checked for AI context.";
    }
    
    return result;
},
		showPolishModal() {
    // Only show if we have HTML files to polish
    const htmlFiles = Object.keys(this.state.projectFiles).filter(path => 
        path.toLowerCase().endsWith('.html') && !this.state.projectFiles[path].rawFile
    );
    
    if (htmlFiles.length === 0) {
        this.renderChatMessage('ai', '❌ Polish tool requires at least one HTML file in your project.');
        return;
    }
    
    // Reset selections
    document.querySelectorAll('.polish-option').forEach(opt => opt.classList.remove('selected'));
    document.querySelector('input[name="polish-intensity"][value="subtle"]').checked = true;
    
    this.togglePolishModal(true);
},

togglePolishModal(show) {
    document.getElementById('polish-modal').classList.toggle('visible', show);
},
togglePlanStepsModal(show) {
    document.getElementById('plan-steps-modal').classList.toggle('visible', show);
},

// ========================================
// NEW: Step Selection Modal Functions
// ========================================

showStepSelectionModal(projectDescription) {
    this.state.currentProjectDescription = projectDescription;
    
    // Analyze project complexity
    this.analyzeProjectComplexity(projectDescription);
    
    // Show modal
    document.getElementById('step-selection-modal').classList.add('visible');
    
    // Set default selection to balanced
    const balanced = document.querySelector('.step-option[data-step-mode="balanced"]');
    if (balanced) {
        this.selectStepOption(balanced);
    }
},

// [REPLACE] your existing analyzeProjectComplexity function with this corrected version.
async analyzeProjectComplexity(description) {
    const numFiles = this.state.fileOrder.length;
    const hasExisting = numFiles > 0;
    
    try {
        // Call backend to analyze complexity
        const formData = new FormData();
        formData.append('action', 'zencode_ai_validate_step_count');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('project_description', description);
        formData.append('num_files', numFiles);
        formData.append('has_existing', hasExisting ? '1' : '0');
        formData.append('chosen_steps', this.state.selectedStepCount);
        
        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        
        if (data.success) {
            this.state.minimumRecommendedSteps = data.data.minimum_steps;
            
            // Update UI with analysis result
            const resultDiv = document.getElementById('step-analysis-result');
            const resultText = document.getElementById('step-analysis-text');
            
            if (resultDiv && resultText) {
                resultText.textContent = `Recommended minimum: ${data.data.minimum_steps} steps based on project complexity`;
                resultDiv.style.display = 'block';
            }
            
            // Update balanced option to use recommended steps
            const balancedOption = document.querySelector('.step-option[data-step-mode="balanced"]');
            if (balancedOption) {
                balancedOption.setAttribute('data-steps', data.data.minimum_steps);
                const stepsText = balancedOption.querySelector('p:last-child');
                if (stepsText) {
                    const estimatedTime = Math.ceil(data.data.minimum_steps * 1.2);
                    stepsText.textContent = `${data.data.minimum_steps} steps ≈ ${estimatedTime} min`;
                }

                // --- THIS IS THE FIX ---
                // If "Balanced" is still the selected mode, programmatically re-select it
                // to force an update to this.state.selectedStepCount with the new value.
                if (this.state.stepSelectionMode === 'balanced') {
                    this.selectStepOption(balancedOption);
                }
                // --- END OF FIX ---
            }
            
            // Re-validate the current selection after the update
            this.validateStepCount(this.state.selectedStepCount);
        }
    } catch (error) {
        console.error('Failed to analyze project complexity:', error);
        // Continue with defaults if analysis fails
    }
},

selectStepOption(optionElement) {
    // Remove previous selection
    document.querySelectorAll('.step-option').forEach(opt => {
        opt.classList.remove('selected');
        opt.style.borderColor = 'transparent';
    });
    
    // Mark as selected
    optionElement.classList.add('selected');
    optionElement.style.borderColor = 'var(--accent-primary)';
    
    const mode = optionElement.getAttribute('data-step-mode');
    const steps = parseInt(optionElement.getAttribute('data-steps'));
    
    this.state.stepSelectionMode = mode;
    
    // FIX: For custom mode, preserve slider value if it exists and was previously in custom mode
    // Otherwise use data-steps value for non-custom modes or initial custom selection
    if (mode === 'custom') {
        // Keep current value if already in custom mode, otherwise use default
        const slider = document.getElementById('step-count-slider');
        const currentSliderValue = slider ? parseInt(slider.value) : steps;
        
        // Only reset to data-steps if we're coming from a different mode
        if (this.state.stepSelectionMode !== 'custom') {
            this.state.selectedStepCount = steps;
        }
        // Otherwise keep the existing selectedStepCount (user's slider choice)
    } else {
        // For non-custom modes, always use the data-steps value
        this.state.selectedStepCount = steps;
    }
    
    // Show/hide custom slider
    const customSlider = document.getElementById('custom-step-slider');
    if (customSlider) {
        customSlider.style.display = mode === 'custom' ? 'block' : 'none';
    }
    
    // Update slider value if custom
    if (mode === 'custom') {
        const slider = document.getElementById('step-count-slider');
        const sliderValue = document.getElementById('slider-value');
        if (slider && sliderValue) {
            // Use current state value, not hardcoded steps
            slider.value = this.state.selectedStepCount;
            sliderValue.textContent = `${this.state.selectedStepCount} steps`;
        }
    }
    
    // Validate selection
    this.validateStepCount(this.state.selectedStepCount);
},

validateStepCount(chosenSteps) {
    const warningDiv = document.getElementById('step-warning');
    const warningText = document.getElementById('step-warning-text');
    
    if (!warningDiv || !warningText) return;
    
    if (chosenSteps < this.state.minimumRecommendedSteps) {
        warningDiv.style.display = 'block';
        warningText.textContent = `You've selected ${chosenSteps} steps, but we recommend at least ${this.state.minimumRecommendedSteps} steps for this project. Proceeding with fewer steps may result in rushed implementation, incomplete features, or timeout errors.`;
    } else {
        warningDiv.style.display = 'none';
    }
},

cancelStepSelection() {
    document.getElementById('step-selection-modal').classList.remove('visible');
    this.state.selectedStepCount = 5; // Reset to default
    this.state.stepSelectionMode = 'balanced';
},

confirmStepSelection() {
    document.getElementById('step-selection-modal').classList.remove('visible');
    
    console.log(`✅ User selected: ${this.state.selectedStepCount} steps (${this.state.stepSelectionMode} mode)`);
    
    // Continue with plan generation using selected step count
    this.proceedWithPlanGeneration();
},

// ========================================
// END: Step Selection Modal Functions
// ========================================


isFilePlannedInFutureSteps(filename, currentStepIndex) {
    // Check if this file will be created in upcoming steps
    if (!this.state.currentPlan || currentStepIndex >= this.state.currentPlan.length - 1) {
        return false; // No future steps
    }
    
    const remainingSteps = this.state.currentPlan.slice(currentStepIndex + 1);
    const filenameLower = filename.toLowerCase();
    
    // Extract base name without path (e.g., "style.css" from "css/style.css")
    const baseFilename = filename.split('/').pop().toLowerCase();
    
    for (const step of remainingSteps) {
        const stepText = step.text.toLowerCase();
        
        // Check multiple patterns for file creation
        const patterns = [
            filenameLower,                                    // exact match
            baseFilename,                                     // just filename
            'create ' + filenameLower,                        // "create style.css"
            'create ' + baseFilename,                         // "create style.css"
            'add ' + filenameLower,                           // "add style.css"
            'add ' + baseFilename,                            // "add style.css"
            filenameLower.replace('.css', ' css'),            // "style css"
            filenameLower.replace('.js', ' js'),              // "script js"
            filenameLower.replace('.css', ' stylesheet'),     // "style stylesheet"
            filenameLower.replace('.js', ' javascript'),      // "script javascript"
            baseFilename.replace('.css', ''),                 // "style" (without extension)
            baseFilename.replace('.js', ''),                  // "script" (without extension)
        ];
        
        // Check if any pattern matches
        for (const pattern of patterns) {
            if (stepText.includes(pattern)) {
                return true;
            }
        }
    }
    
    return false;
},

handleUnifyError(error) {
    if (error.name === 'AbortError') {
        this.renderChatMessage('ai', 'Design unification cancelled by user.', true);
        return;
    }
    
    const errorMessage = error.message || '';
    
    if (errorMessage.includes('timeout') || 
        errorMessage.includes('413') || 
        errorMessage.includes('Request Entity Too Large') ||
        errorMessage.includes('systems. Please try again') ||
        errorMessage.includes('Overloaded')) {
        this.renderChatMessage('ai', 'Project too large for unify tool. Alternative approaches:\n\n1. Polish individual files using direct AI commands\n2. Use the main polish tool on smaller file selections\n3. Break your project into smaller modules\n4. Focus on one page at a time', true);
    } else if (errorMessage.includes('upgrading our systems')) {
        this.renderChatMessage('ai', 'High demand right now - that means you\'re using the latest AI technology! The servers are getting a workout from all the creative projects. Try again in a few minutes when traffic lightens up.');
    } else {
        this.renderChatMessage('ai', `Design unification failed: ${errorMessage}\n\nFor large projects, consider polishing files one at a time with specific requests.`, true);
    }
},
// 1. UPDATE the handlePlanConfigConfirm function
// COMPLETE handlePlanConfigConfirm function

// REPLACE your current executeSingleStep function with this FINAL, DEFINITIVE version.
// REPLACE your current executeSingleStep function with this FINAL version.
// REPLACE your executeSingleStep function with this corrected version

// 9. UPDATED executeSingleStep function for manual step execution



getEditableFiles() {
    return Object.keys(this.state.projectFiles).filter(path => 
        path.toLowerCase().match(/\.(html|css|js)$/) && 
        !this.state.projectFiles[path].rawFile &&
        this.state.projectFiles[path].content &&
        this.state.projectFiles[path].content.trim().length > 0
    );
},


prioritizeAndFilterFiles(files) {
    const prioritized = files.sort((a, b) => {
        if (a.includes('index.html')) return -1;
        if (b.includes('index.html')) return 1;
        if (a.includes('style.css') || a.includes('main.css')) return -1;
        if (b.includes('style.css') || b.includes('main.css')) return 1;
        if (a.endsWith('.html')) return -1;
        if (b.endsWith('.html')) return 1;
        return 0;
    });
    
    return prioritized.filter(path => {
        const content = this.state.projectFiles[path].content;
        const size = content.length;
        
        if (size > 100000) return false;
        if (size < 100) return false;
        
        const modernFeatureCount = [
            content.includes('backdrop-filter'),
            content.includes('transform:') && content.match(/transform:/g).length > 3,
            content.includes('transition:') && content.match(/transition:/g).length > 5,
            content.includes('@keyframes'),
            content.includes('grid-template'),
            content.includes('flex-direction')
        ].filter(Boolean).length;
        
        const isAlreadyModern = modernFeatureCount >= 4;
        
        return !isAlreadyModern;
    }).slice(0, 5);
},
getTotalFileSize(files) {
    return files.reduce((total, path) => {
        return total + (this.state.projectFiles[path].content?.length || 0);
    }, 0);
},

// Add this as a method in your App object, near other utility methods
debugScrollIssues() {
    const chatMessages = this.elements.chatMessages;
    console.log('Chat messages scroll properties:', {
        overflow: getComputedStyle(chatMessages).overflow,
        overflowY: getComputedStyle(chatMessages).overflowY,
        height: chatMessages.offsetHeight,
        scrollHeight: chatMessages.scrollHeight,
        canScroll: chatMessages.scrollHeight > chatMessages.offsetHeight
    });
    
    // Check for problematic elements
    const problematicElements = chatMessages.querySelectorAll('[style*="position: fixed"], [style*="position: absolute"], [style*="overflow: hidden"]');
    console.log('Potentially problematic elements:', problematicElements);
},
async startPolishProcess() {
    const selectedOptions = Array.from(document.querySelectorAll('.polish-option.selected'))
        .map(el => el.dataset.polishType);
    
    const intensity = document.querySelector('input[name="polish-intensity"]:checked').value;
    
    if (selectedOptions.length === 0) {
        alert('Please select at least one polish option.');
        return;
    }
    
    this.togglePolishModal(false);
    this.toggleSettingsModal(false);
    
    const allFiles = this.getEditableFiles();
    const filesToProcess = this.prioritizeAndFilterFiles(allFiles);
    
    if (filesToProcess.length === 0) {
        const largeFiles = allFiles.filter(path => this.state.projectFiles[path].content.length > 50000);
        const smallFiles = allFiles.filter(path => this.state.projectFiles[path].content.length < 500);
        const modernFiles = allFiles.filter(path => {
            const content = this.state.projectFiles[path].content;
            const modernFeatureCount = [
                content.includes('backdrop-filter'),
                content.includes('transform:') && content.match(/transform:/g).length > 2,
                content.includes('transition:') && content.match(/transition:/g).length > 3,
                content.includes('@keyframes'),
                content.includes('grid-template'),
                content.includes('flex-direction')
            ].filter(Boolean).length;
            return modernFeatureCount >= 3;
        });
        
        let errorMsg = 'Polish tool cannot proceed - no suitable files found.\n\nFiles excluded:\n';
        
        if (largeFiles.length > 0) {
            errorMsg += `\nToo large (>50KB): ${largeFiles.join(', ')}`;
        }
        if (smallFiles.length > 0) {
            errorMsg += `\nToo small (<500 chars): ${smallFiles.join(', ')}`;
        }
        if (modernFiles.length > 0) {
            errorMsg += `\nAlready polished: ${modernFiles.join(', ')}`;
        }
        
        errorMsg += '\n\nSolutions:\n• Break large files into smaller components\n• Polish individual files with specific AI requests\n• Use the main chat for targeted improvements';
        
        this.renderChatMessage('ai', errorMsg);
        return;
    }
    
    const totalSize = this.getTotalFileSize(filesToProcess);
    if (totalSize > 300000 || filesToProcess.length > 5) {
        const proceed = await this.showConfirmation(`Large project detected (${filesToProcess.length} files, ${Math.round(totalSize/1000)}KB). This may timeout. Continue anyway?`);
        if (!proceed) return;
    }
    
    this.showPolishProgress();
    this.state.polishSessionActive = true;
    this.state.prePolishFiles = JSON.parse(JSON.stringify(this.state.projectFiles));
    
    this.renderChatMessage('ai', `Starting polish process on ${filesToProcess.length} files: ${filesToProcess.join(', ')}`);
    
    try {
        const polishPrompt = this.buildOptimizedPolishPrompt(filesToProcess, selectedOptions, intensity);
        await this.executePolish(polishPrompt, selectedOptions, intensity);
        this.hidePolishProgress();
        this.showPolishResults();
    } catch (error) {
        this.hidePolishProgress();
        this.handlePolishError(error);
    }
},
buildOptimizedPolishPrompt(filesToProcess, selectedOptions, intensity) {
    const intensityInstructions = {
        subtle: "Apply refined improvements that enhance without overwhelming. Focus on essential fixes and subtle polish.",
        moderate: "Apply noticeable modern improvements while maintaining core structure. Balance fixes with enhancements.",
        dramatic: "Apply comprehensive modern design and functionality improvements for a premium experience."
    };
    
    const focusAreas = {
        'modern-ui': "Modern UI: Add glassmorphism, smooth animations, gradients, hover effects, modern CSS features",
        'mobile-first': "Mobile optimization: Touch targets, responsive layouts, mobile navigation, thumb-friendly design",
        'accessibility': "Accessibility: WCAG compliance, focus indicators, ARIA labels, proper contrast ratios",
        'performance': "Performance: Optimize CSS, efficient selectors, reduce file sizes, faster loading"
    };
    
    const selectedFocus = selectedOptions.map(opt => focusAreas[opt]).join('\n- ');
    
    return `PROJECT INTEGRATION & POLISH - ${filesToProcess.length} FILES

**MISSION: Fix integration issues and apply modern polish to these specific files while preserving all functionality.**

**INTENSITY: ${intensity.toUpperCase()}**
${intensityInstructions[intensity]}

**FOCUS AREAS:**
- ${selectedFocus}

**CRITICAL REQUIREMENTS:**
1. **PRESERVE ALL CONTENT** - Keep every piece of text, functionality, and core structure intact
2. **FIX INTEGRATION FIRST** - Ensure all CSS classes, JavaScript functions, and HTML references work together perfectly
3. **MAINTAIN RELATIONSHIPS** - If you modify a CSS class name, update ALL references to it across files
4. **OUTPUT COMPLETE FILES** - Return full working versions of any file you modify
5. **TEST COMPATIBILITY** - Use widely supported CSS/JS features only

**FILES TO PROCESS:**
${filesToProcess.map(file => `- ${file} (${Math.round(this.state.projectFiles[file].content.length/1000)}KB)`).join('\n')}

**INTEGRATION DEBUGGING CHECKLIST:**
✅ All CSS classes referenced in HTML exist in CSS files
✅ All JavaScript functions called in HTML exist in JS files  
✅ All onclick handlers have corresponding functions
✅ All form submissions have proper handlers
✅ All href links point to valid targets
✅ Mobile navigation opens/closes correctly
✅ All interactive elements respond properly

**POLISH GUIDELINES:**
- Create visual consistency across all files
- Use modern CSS (flexbox, grid, custom properties)
- Add smooth transitions and hover effects
- Ensure mobile-first responsive design
- Maintain professional, cohesive appearance

Focus on quality integration and tasteful enhancement. Make it work perfectly first, then make it look amazing.`;
},
showPolishProgress() {
    document.getElementById('polish-progress-modal').classList.add('visible');
},

hidePolishProgress() {
    document.getElementById('polish-progress-modal').classList.remove('visible');
},

cancelPolishProcess() {
    this.state.polishSessionActive = false;
    
    if (this.state.currentRequestController) {
        this.state.currentRequestController.abort();
    }
    
    this.hidePolishProgress();
    this.renderChatMessage('ai', 'Polish process cancelled.');
},

showPolishResults() {
    const changedFiles = this.getChangedFilesSincePolish();
    const htmlFiles = changedFiles.filter(f => f.endsWith('.html'));
    const cssFiles = changedFiles.filter(f => f.endsWith('.css'));
    const jsFiles = changedFiles.filter(f => f.endsWith('.js'));
    
    // --- UPDATED HTML STRUCTURE TO MATCH SCREENSHOT ---
    const summaryHtml = `
        <div class="polish-summary-grid">
            <div class="polish-summary-column">
                <h5 class="polish-summary-header functionality">
                    <svg class="polish-summary-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path></svg>
                    Functionality Fixes:
                </h5>
                <ul class="polish-summary-list">
                    <li><span class="check-icon-box"></span>Integrated all step-by-step components</li>
                    <li><span class="check-icon-box"></span>Fixed broken function references</li>
                    <li><span class="check-icon-box"></span>Resolved CSS class conflicts</li>
                    <li><span class="check-icon-box"></span>Ensured all interactions work</li>
                    <li><span class="check-icon-box"></span>Validated HTML structure</li>
                </ul>
            </div>
            <div class="polish-summary-column">
                <h5 class="polish-summary-header visual">
                    <svg class="polish-summary-icon" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/><path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/></svg>
                    Visual Enhancements:
                </h5>
                <ul class="polish-summary-list">
                    <li><span class="check-icon-box"></span>Applied modern design principles</li>
                    <li><span class="check-icon-box"></span>Enhanced mobile responsiveness</li>
                    <li><span class="check-icon-box"></span>Optimized performance</li>
                    <li><span class="check-icon-box"></span>Improved accessibility</li>
                    <li><span class="check-icon-box"></span>Unified visual branding</li>
                </ul>
            </div>
        </div>
        
        <div class="polish-files-summary">
            <h5 class="polish-files-header">Files Enhanced:</h5>
            <div class="polish-files-list">
                ${htmlFiles.length > 0 ? `<p><strong>HTML:</strong> ${htmlFiles.map(f => `<code>${this.escapeHtml(f)}</code>`).join(', ')}</p>` : ''}
                ${cssFiles.length > 0 ? `<p><strong>CSS:</strong> ${cssFiles.map(f => `<code>${this.escapeHtml(f)}</code>`).join(', ')}</p>` : ''}
                ${jsFiles.length > 0 ? `<p><strong>JavaScript:</strong> ${jsFiles.map(f => `<code>${this.escapeHtml(f)}</code>`).join(', ')}</p>` : ''}
            </div>
        </div>
        
        <p class="polish-footer-note">
            <svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="m12 2 7 10-7 10-7-10Z"/><path d="m22 12-5-5"/><path d="m2 12 5-5"/><path d="m12 2 5 5"/><path d="m12 22-5-5"/></svg>
            Your project is now fully integrated and professionally polished!
        </p>
        <p class="polish-footer-subnote"><em>Click "Preview Changes" to inspect the files before deciding, or use the preview button to see your enhanced website in action.</em></p>
    `;
    
    // Update modal title
    const titleElement = document.querySelector('#polish-results-modal h2');
    if (titleElement) {
        titleElement.innerHTML = `
            <svg class="polish-title-icon" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M12 3L14.5 9.5L21 12L14.5 14.5L12 21L9.5 14.5L3 12L9.5 9.5Z"/><path d="M5 5L7 7"/><path d="M17 17L19 19"/><path d="M5 19L7 17"/><path d="M17 7L19 5"/></svg>
            Polish Complete!
        `;
        titleElement.classList.add('polish-modal-title');
    }
    
    // Update buttons with icons
    document.getElementById('btn-preview-polish').innerHTML = `<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path><circle cx="12" cy="12" r="3"></circle></svg> Preview Changes`;
    
    document.getElementById('polish-summary').innerHTML = summaryHtml;
    document.getElementById('polish-results-modal').classList.add('visible');
    
    this.state.polishModalWasOpen = true;
},
getChangedFilesSincePolish() {
    if (!this.state.prePolishFiles) return [];
    
    return Object.keys(this.state.projectFiles).filter(path => {
        const current = this.state.projectFiles[path];
        const original = this.state.prePolishFiles[path];
        
        return current && original && current.content !== original.content;
    });
},

acceptPolishChanges() {
    document.getElementById('polish-results-modal').classList.remove('visible');
    this.state.polishSessionActive = false;
    this.state.prePolishFiles = null; // Clear backup
    this.state.polishModalWasOpen = false; // Clear modal state
    this.renderChatMessage('ai', '✅ Polish changes accepted! Your project now has professional visual design.');
    
    // Show preview if we have an HTML file
    const htmlFile = Object.keys(this.state.projectFiles).find(path => 
        path.toLowerCase().endsWith('.html') && !this.state.projectFiles[path].rawFile
    );
    if (htmlFile) {
        this.activateFile(htmlFile);
    }
},
setLoadingState(isLoading) {
    this.state.isAILoading = isLoading;
    this.elements.sendButton.disabled = false; // The button is always clickable

    // API Status indicators
    [this.elements.apiStatus, this.elements.mobileApiStatus].forEach(el => {
        if (el) {
            el.classList.toggle('pending', isLoading);
            el.classList.toggle('online', !isLoading);
        }
    });
    [this.elements.apiStatusText, this.elements.mobileApiStatusText].forEach(el => {
        if (el) el.textContent = isLoading ? 'AI Active' : 'AI Ready';
    });

    // Footer status (add this section)
    const footerStatus = document.querySelector('.footer-status');
    if (footerStatus) {
        footerStatus.classList.toggle('processing', isLoading);
    }

    // Button state logic
    const sendContent = this.elements.sendButton.querySelector('.send-button-content');
    const stopContent = this.elements.sendButton.querySelector('.stop-button-content');

    if (isLoading) {
        this.elements.sendButton.classList.add('stop-button-active');
        if (sendContent) sendContent.style.display = 'none';
        if (stopContent) stopContent.style.display = 'flex';
    } else {
        this.elements.sendButton.classList.remove('stop-button-active');
        if (sendContent) sendContent.style.display = 'flex';
        if (stopContent) stopContent.style.display = 'none';
    }
},



















// NEW function - add after existing project management functions:
async createManualSnapshot() {
    const versionName = prompt("Enter a name for this version:", "Manual Save " + new Date().toLocaleString());
    if (!versionName) return;
    
    const description = prompt("Enter a description (optional):", "");
    
    this.syncEditorToFileState();
    const projectName = this.elements.projectNameInput.value.trim();
    const projectData = {
        projectFiles: this.state.projectFiles,
        fileOrder: this.state.fileOrder,
        chatHistory: this.state.chatHistory,
        collapsedFolders: Array.from(this.state.collapsedFolders)
    };

    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_version_control');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('version_action', 'create_manual_snapshot');
        formData.append('project_name', projectName);
        formData.append('version_name', versionName);
        formData.append('description', description);
        formData.append('project_data', JSON.stringify(projectData));

        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            this.renderChatMessage('ai', `✅ Version "${versionName}" saved successfully.`);
        } else {
            throw new Error(data.data?.message || 'Failed to create version');
        }
    } catch (error) {
        this.renderChatMessage('ai', `❌ Error creating version: ${error.message}`);
    }
},

/**
 * NEW: Calls the backend to get a "pruned" version of the spec for a single step.
 * This uses a cheap AI model to create a context-aware summary, reducing token cost.
 */
async getPrunedSpecificationForStep(stepText) {
    // If there's no full spec to prune, exit immediately.
    if (!this.state.projectSpecification) {
        return null;
    }

    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_prune_spec');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('full_specification', JSON.stringify(this.state.projectSpecification));
        formData.append('step_text', stepText);

        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });

        const data = await response.json();

        if (data.success && data.data.pruned_spec) {
            // Return the AI-generated summary.
            return data.data.pruned_spec;
        } else {
            // If the pruning fails for any reason, log the error and return null.
            console.error('Specification pruning failed on the backend:', data.data?.message || 'Unknown error');
            return null; // Fallback
        }
    } catch (error) {
        console.error('Network error during specification pruning:', error);
        return null; // Fallback in case of network failure
    }
},
// Also update the loadProjectVersions() method to handle loading states properly:
async loadProjectVersions() {
    const projectName = this.elements.projectNameInput.value.trim();
    if (!projectName) return;
    
    this.state.isVersionLoading = true;
    this.renderVersionsList(); // Show loading state immediately
    
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_version_control');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('version_action', 'list_versions');
        formData.append('project_name', projectName);

        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            this.state.projectVersions = data.data.versions;
        } else {
            throw new Error(data.data?.message || 'Failed to load versions');
        }
    } catch (error) {
        console.error('Error loading versions:', error);
        // Show error in the versions list
        const container = document.getElementById('versions-list');
        if (container) {
            container.innerHTML = `<div style="color: var(--error); padding: 1rem; text-align: center;">Error loading versions: ${error.message}</div>`;
        }
    } finally {
        this.state.isVersionLoading = false;
        this.renderVersionsList();
    }
},

// REPLACE your existing restoreVersion function with this complete version:
async restoreVersion(versionId, versionName) {
    // 1. Await the confirmation result first. This pauses the function until the user acts.
    const confirmed = await this.showConfirmation(`Are you sure you want to restore to "${versionName}"? Your current work will be backed up automatically.`);
    
    // 2. NOW, close the version control modal. This happens *after* the confirmation is handled.
    this.toggleVersionControl(false);

    // 3. If the user clicked "Cancel", stop the entire process.
    if (!confirmed) {
        return;
    }

    // 4. Only if the user confirmed, proceed with the restore logic.
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_version_control');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('version_action', 'restore_version');
        formData.append('version_id', versionId);

        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            this.renderChatMessage('ai', `✅ Restored to version "${versionName}". Reloading project...`);
            setTimeout(() => {
                location.reload();
            }, 1000);
        } else {
            throw new Error(data.data?.message || 'Failed to restore version');
        }
    } catch (error) {
        this.renderChatMessage('ai', `❌ Error restoring version: ${error.message}`);
    }
},
        handleSaveError(errorMessage) {
            this.updateSaveStatus('unsaved'); // Revert status to 'unsaved'
            
            // Only show the error message in the chat ONCE
            if (!this.lastSaveFailed) {
                this.renderChatMessage('ai', `⚠️ **Auto-save failed:** ${errorMessage}. Your work is not being saved. Please check your network connection.`);
            }
            this.lastSaveFailed = true; // Set a flag to prevent more messages

            // Update the persistent status indicator to show a clear error state
            if (this.elements.saveStatus) {
                this.elements.saveStatus.textContent = '✗ Save Failed';
                this.elements.saveStatus.className = 'save-status error'; // You may need to add a style for .error
            }
        },

// REPLACE your existing deleteVersion function with this complete version:
async deleteVersion(versionId, versionName) {
    // 1. Await the confirmation result first.
    const confirmed = await this.showConfirmation(`Are you sure you want to delete version "${versionName}"? This cannot be undone.`);
    
    // 2. Close the underlying version control modal after the user has made a choice.
    this.toggleVersionControl(false);

    // 3. If the user cancelled, exit the function.
    if (!confirmed) {
        return;
    }

    // 4. Only if the user confirmed, proceed with the deletion logic.
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_version_control');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('version_action', 'delete_version');
        formData.append('version_id', versionId);

        const response = await fetch(zencode_ai_app_vars.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const data = await response.json();
        if (data.success) {
            this.renderChatMessage('ai', `✅ Deleted version "${versionName}".`);
            this.loadProjectVersions(); // Refresh the list
        } else {
            throw new Error(data.data?.message || 'Failed to delete version');
        }
    } catch (error) {
        this.renderChatMessage('ai', `❌ Error deleting version: ${error.message}`);
    }
},

// Update the toggleVersionControl() method to ensure versions load when modal opens:
toggleVersionControl() {
    this.state.versionControlOpen = !this.state.versionControlOpen;
    const modal = document.getElementById('version-control-modal');
    if (modal) {
        modal.classList.toggle('visible', this.state.versionControlOpen);
        if (this.state.versionControlOpen) {
            // Load versions immediately when modal opens
            this.loadProjectVersions();
        }
    }
},

// REPLACE your existing renderVersionsList function with this one:
// REPLACE your entire renderVersionsList function with this definitive version.
renderVersionsList() {
    const container = document.getElementById('versions-list');
    if (!container) return;

    if (this.state.isVersionLoading) {
        container.innerHTML = '<div class="version-loading">Loading versions...</div>';
        return;
    }

    const allVersions = this.state.projectVersions || [];
    const visibleCount = this.state.visibleVersionCount || 5;

    if (allVersions.length === 0) {
        container.innerHTML = '<div style="text-align: center; padding: 2rem; color: var(--text-secondary);">No versions saved yet. Create your first version now.</div>';
        document.getElementById('version-list-controls').innerHTML = ''; // Clear controls
        return;
    }

    const versionsToRender = allVersions.slice(0, visibleCount);

    const versionsHTML = versionsToRender.map(version => {
        const isAutoSnapshot = version.is_auto_snapshot === '1';
        const iconClass = isAutoSnapshot ? 'fas fa-robot' : 'fas fa-thumbtack';
        const typeLabel = isAutoSnapshot ? 'Auto' : 'Manual';
        const date = new Date(version.created_at).toLocaleString();
        
        return `
            <div class="version-item" style="border-left-color: ${isAutoSnapshot ? '#fbbf24' : 'var(--accent-primary)'};">
                <div class="version-info">
                    <i class="${iconClass} version-icon"></i>
                    <div class="version-details">
                        <strong class="version-name">${this.escapeHtml(version.version_name)}</strong>
                        <div class="version-date">${date}</div>
                        ${version.version_description ? `<div class="version-description" style="color: var(--text-secondary); font-size: 0.9rem; margin-top: 0.25rem;">${this.escapeHtml(version.version_description)}</div>` : ''}
                    </div>
                </div>
                <div class="version-actions">
                    <span class="version-type-badge ${isAutoSnapshot ? 'auto' : 'manual'}">${typeLabel}</span>
                    <button onclick="App.restoreVersion('${version.id}', '${this.escapeHtml(version.version_name).replace(/'/g, "\\'")}')" class="icon-button restore" title="Restore Version"><i class="fas fa-history"></i></button>
                    ${!isAutoSnapshot ? `<button onclick="App.deleteVersion('${version.id}', '${this.escapeHtml(version.version_name).replace(/'/g, "\\'")}')" class="icon-button delete" title="Delete Version"><i class="fas fa-trash"></i></button>` : ''}
                </div>
            </div>
        `;
    }).join('');
    
    container.innerHTML = versionsHTML;

    // Render the pagination buttons
    const controlsContainer = document.getElementById('version-list-controls');
    controlsContainer.innerHTML = '';
    if (allVersions.length > visibleCount) {
        const remaining = allVersions.length - visibleCount;
        const loadNextCount = Math.min(5, remaining);
        const loadMoreBtn = document.createElement('button');
        loadMoreBtn.className = 'modal-button-secondary';
        loadMoreBtn.dataset.action = 'load-more';
        loadMoreBtn.textContent = `Load ${loadNextCount} More`;
        controlsContainer.appendChild(loadMoreBtn);
    }
    if (visibleCount > 5) {
        const showLessBtn = document.createElement('button');
        showLessBtn.className = 'modal-button-secondary';
        showLessBtn.dataset.action = 'show-less';
        showLessBtn.textContent = 'Show Less';
        controlsContainer.appendChild(showLessBtn);
    }
},





handleVersionAction(e) {
    const target = e.target.closest('button');
    if (!target) return;

    const action = target.dataset.action;
    if (action === 'load-more') {
        this.state.visibleVersionCount += 5;
        this.renderVersionsList();
    } else if (action === 'show-less') {
        this.state.visibleVersionCount = 5;
        this.renderVersionsList();
    }
},





/**
 * NEW: Starts the specification generation process.
 */
// FINAL, CORRECTED startSpecificationProcess function
startSpecificationProcess(prompt, imageData = null, planType = 'standard') {
    // --- Your new, better message is here ---
    this.renderChatMessage('ai', "To ensure the best outcome, I'll first generate a detailed specification sheet for your review. This helps create a much more accurate plan.");
    // --- End of change ---

    this.setLoadingState(true);
    this.fetchSpecificationSheet(prompt, imageData, planType);
},

shouldRunSpecificationCheck(prompt) {
    const complexIndicators = ['e-commerce', 'database', 'user account', 'login', 'authentication', 'multi-page', 'multiple pages'];
    const promptLower = prompt.toLowerCase();
    
    // Trigger if prompt is vague/short
    if (prompt.length < 200) {
        return true;
    }
    
    // Trigger if prompt contains complex project indicators
    for (const indicator of complexIndicators) {
        if (promptLower.includes(indicator)) {
            return true;
        }
    }
    
    return false;
},


async fetchSpecificationSheet(prompt, imageData = null) {
    try {
        const formData = new FormData();
        formData.append('action', 'zencode_ai_generate_spec');
        formData.append('nonce', zencode_ai_app_vars.nonce);
        formData.append('prompt', prompt);

        if (imageData) {
            formData.append('image_data', imageData);
        }

        const response = await fetch(zencode_ai_app_vars.ajaxurl, { method: 'POST', body: formData });
        const data = await response.json();

        if (data.success) {
            const specData = JSON.parse(data.data.specification);
            this.showSpecificationModal(specData, prompt);
        } else {
            throw new Error(data.data.message || 'Failed to get a valid response from the server.');
        }
    } catch (error) {
        console.error('Specification generation failed:', error);
        this.fallbackToRegularPlanning(prompt);
    } finally {
        this.setLoadingState(false);
    }
},

// Add this completely new function
async preValidateStep(stepText, contextFiles) {
    const analysis = {
        missingDefinitions: [],
        potentialConflicts: [],
        recommendations: [],
        shouldProceed: true
    };
    
    const stepLower = stepText.toLowerCase();
    
    // Enhanced detection of file creation intent
    const willCreateCSS = stepLower.includes('css') || 
                         stepLower.includes('style') || 
                         stepLower.includes('styling') ||
                         stepLower.includes('stylesheet');
    
    const willCreateJS = stepLower.includes('javascript') || 
                        stepLower.includes('js') || 
                        stepLower.includes('script') ||
                        /\bfunction\b/i.test(stepText) ||
                        stepLower.includes('interactive');
    
    // Check 1: Event handlers need JavaScript
    if (stepLower.includes('onclick') || stepLower.includes('event') || 
        stepLower.includes('click') || stepLower.includes('interactive')) {
        const jsFiles = contextFiles.filter(f => f.endsWith('.js'));
        
        if (jsFiles.length === 0 && !willCreateJS) {
            analysis.missingDefinitions.push({
                issue: 'No JavaScript file in context for event handlers',
                recommendation: 'Add or create a .js file',
                severity: 'high'
            });
        } else if (jsFiles.length === 0 && willCreateJS) {
            console.log(`✅ Step will create JavaScript file as part of this step`);
        }
    }
    
    // Check 2: Styling needs CSS
    if (stepLower.includes('style') || stepLower.includes('css') || 
        stepLower.includes('design') || stepLower.includes('color')) {
        const cssFiles = contextFiles.filter(f => f.endsWith('.css'));
        
        if (cssFiles.length === 0 && !willCreateCSS) {
            analysis.missingDefinitions.push({
                issue: 'No CSS file in context for styling',
                recommendation: 'Add or create a .css file',
                severity: 'high'
            });
        } else if (cssFiles.length === 0 && willCreateCSS) {
            console.log(`✅ Step will create CSS file as part of this step`);
        }
    }
    
    // Check 3: Creating new pages needs navigation updates
    if ((stepLower.includes('create') || stepLower.includes('add')) && 
        stepLower.includes('page')) {
        const htmlFiles = contextFiles.filter(f => f.endsWith('.html'));
        if (htmlFiles.length > 0) {
            analysis.recommendations.push({
                suggestion: 'Consider updating navigation in existing HTML files to link to the new page',
                affectedFiles: htmlFiles,
                severity: 'medium'
            });
        }
    }
    
    // Check 4: API work needs configuration
    if (stepLower.includes('api') || stepLower.includes('fetch') || 
        stepLower.includes('backend')) {
        const hasConfig = contextFiles.some(f => f.includes('config') || f.includes('api'));
        const willCreateConfig = stepLower.includes('config');
        
        if (!hasConfig && !willCreateConfig) {
            analysis.recommendations.push({
                suggestion: 'Consider creating a config.js for API endpoints',
                severity: 'medium'
            });
        }
    }
    
    // Check 5: Forms need validation
    if (stepLower.includes('form') || stepLower.includes('input')) {
        const hasValidation = contextFiles.some(f => 
            f.includes('validation') || 
            this.state.projectFiles[f]?.content?.includes('validate')
        );
        const willCreateValidation = stepLower.includes('validat');
        
        if (!hasValidation && !willCreateValidation) {
            analysis.recommendations.push({
                suggestion: 'Consider adding form validation logic',
                severity: 'medium'
            });
        }
    }
    
    // Only block execution for ACTUAL missing dependencies
    // Don't block if the step is going to create what's missing
    const criticalIssues = analysis.missingDefinitions.filter(d => d.severity === 'high');
    if (criticalIssues.length > 0) {
        analysis.shouldProceed = false;
    }
    
    return analysis;
},

// NEW HELPER FUNCTION
async addMissingContextFiles(missingDefinitions) {
    const filesToAdd = [];
    
    missingDefinitions.forEach(def => {
        if (def.issue.includes('JavaScript')) {
            // Look for existing JS files not in context
            const jsFiles = this.state.fileOrder.filter(f => 
                f.endsWith('.js') && !this.state.contextFiles.has(f)
            );
            if (jsFiles.length > 0) {
                filesToAdd.push(jsFiles[0]); // Add the first one
            }
        }
        if (def.issue.includes('CSS')) {
            const cssFiles = this.state.fileOrder.filter(f => 
                f.endsWith('.css') && !this.state.contextFiles.has(f)
            );
            if (cssFiles.length > 0) {
                filesToAdd.push(cssFiles[0]);
            }
        }
    });
    
    if (filesToAdd.length > 0) {
        filesToAdd.forEach(file => this.state.contextFiles.add(file));
        this.renderFileTree();
        this.renderChatMessage('ai', `🔧 Auto-added ${filesToAdd.length} missing file(s) to context: ${filesToAdd.join(', ')}`);
    }
},
// Add this new function to handle one-shot button clicks
handleOneShotButtonClick() {
    // Clear any plan-related states when switching to one-shot
    this.state.pendingPlanPrompt = null;
    this.state._pendingPlan = null;
    this.state.currentPlan = [];
    this.state.isExecutingProcess = false;
    this.state.isExecutingPlan = false;
    this.state.currentStepIndex = 0;
    this.state.requestedStepCount = null;
    
    // Reset AI mode to normal (from plan mode)
    if (this.state.currentAIMode === 'plan') {
        this.state.currentAIMode = 'normal';
        this.updateModeToggleVisuals();
        this.updateChatPlaceholder();
    }
    
    // Remove plan mode UI state
    document.body.classList.remove('plan-mode');
    
    // Clear any plan-related UI elements that might be showing
    const planButtons = document.querySelectorAll('[id*="execute-plan"], [id*="stop-plan"], [id*="polish-plan"]');
    planButtons.forEach(btn => {
        if (btn) btn.style.display = 'none';
    });
    
    // Now open the one-shot modal with clean state
    this.toggleOneShotModal(true);
},




// Add this new function to handle one-shot button clicks
async handleOneShotButtonClick() {
    // Check if there's plan work in progress
    const hasPlanInProgress = this.state.currentAIMode === 'plan' || 
                             this.state.pendingPlanPrompt || 
                             this.state._pendingPlan || 
                             this.state.isExecutingProcess ||
                             this.state.currentPlan.length > 0;
    
    // Show confirmation if switching away from plan work
    if (hasPlanInProgress) {
        const shouldProceed = await this.showConfirmation(
            "You're currently working on a development plan. Switching to the One-Shot Website Builder will cancel your plan workflow.\n\nThe One-Shot Builder creates complete websites instantly from a single description, while Plan Mode breaks projects into organized steps.\n\nAre you sure you want to cancel your current plan and switch to the One-Shot Builder?"
        );
        
        if (!shouldProceed) {
            return;
        }
    }
    
    // Clear any plan-related states when switching to one-shot
    this.state.pendingPlanPrompt = null;
    this.state._pendingPlan = null;
    this.state.currentPlan = [];
    this.state.isExecutingProcess = false;
    this.state.isExecutingPlan = false;
    this.state.currentStepIndex = 0;
    this.state.requestedStepCount = null;
    
    // Reset AI mode to normal (from plan mode)
    if (this.state.currentAIMode === 'plan') {
        this.state.currentAIMode = 'normal';
        this.updateModeToggleVisuals();
        this.updateChatPlaceholder();
    }
    
    // Remove plan mode UI state
    document.body.classList.remove('plan-mode');
    
    // Clear any plan-related UI elements that might be showing
    const planButtons = document.querySelectorAll('[id*="execute-plan"], [id*="stop-plan"], [id*="polish-plan"]');
    planButtons.forEach(btn => {
        if (btn) btn.style.display = 'none';
    });
    
    // Now open the one-shot modal with clean state
    this.toggleOneShotModal(true);
},
/**
 * NEW: Displays the specification sheet in an editable modal.
 */
// REPLACE your existing showSpecificationModal function with this one.
showSpecificationModal(specData, originalPrompt) {
    const modal = document.getElementById('specification-modal');
    const form = document.getElementById('spec-form');
    form.innerHTML = '';
    form.dataset.originalPrompt = originalPrompt;

    const titleElement = modal.querySelector('h2');
    if (titleElement) {
        titleElement.innerHTML = `<i class="fas fa-clipboard-list spec-modal-icon"></i> <span class="gradient-texty">Project Specification Sheet</span>`;
        titleElement.classList.add('spec-modal-title');
    }

    for (const key in specData) {
        const value = specData[key];
        const formattedKey = key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        
        const field = document.createElement('div');
        field.style.marginBottom = '1.5rem';
        field.innerHTML = `
            <label for="spec-${key}" style="display: block; font-weight: 600; color: var(--accent-primary); margin-bottom: 0.5rem;">${this.escapeHtml(formattedKey)}</label>
            <textarea id="spec-${key}" name="${key}" class="spec-textarea">${this.escapeHtml(value)}</textarea>
        `;
        form.appendChild(field);
    }

    modal.classList.add('visible');
},
/**
 * NEW: Handles the approval of the edited specification sheet.
 */
// 3. REPLACE the handleSpecificationApproval function with this simpler version
// This was the original, problematic function
// This is the new, corrected function
handleSpecificationApproval() {
    const form = document.getElementById('spec-form');
    const approvedSpec = {};
    const formData = new FormData(form);
    for (const [key, value] of formData.entries()) {
        approvedSpec[key] = value;
    }

    this.state.projectSpecification = approvedSpec;
    document.getElementById('specification-modal').classList.remove('visible');

    // --- THE FIX ---
    // 1. Get the original long prompt for the AI's internal use.
    const promptForAI = form.dataset.originalPrompt;

    // 2. Define the clean, user-facing message for the chat history.
    const messageForDisplay = "Generate a standard plan based on the approved specification.";

    // 3. The old, direct renderChatMessage call has been removed.

    // 4. Call showPlanConfigModal with the new display message, which will be
    //    passed along and rendered correctly in the next step.
    this.showPlanConfigModal(promptForAI, approvedSpec, messageForDisplay);
},

// 4. ADD this entire new function to your App object
// REPLACE your entire handleStandaloneAppPlanGeneration function with this.
// REPLACE your entire handleStandaloneAppPlanGeneration function with this one.
async handleStandaloneAppPlanGeneration() {
    const form = document.getElementById('spec-form');
    const approvedSpec = {};
    const formData = new FormData(form);
    for (const [key, value] of formData.entries()) {
        approvedSpec[key] = value;
    }

    this.state.projectSpecification = approvedSpec;
    document.getElementById('specification-modal').classList.remove('visible');

    // 1. Construct the powerful, detailed prompt for the AI's internal use.
    const transposedPromptForAI = `
Your task is to create a plan to build a full-featured, production-quality, front-end web app.

HARD CONSTRAINTS:
- Use only three files: index.html, style.css, script.js.
- Use only HTML, CSS, and vanilla JavaScript. No frameworks or external libraries.
- The app must run by simply opening index.html in a browser.
- Store all persistent data in localStorage.
- Use CSS variables for theming.
- Seed with demo data if localStorage is empty.

Based on these hard constraints, create a detailed, step-by-step plan to implement the following features as described in the specification:

--- SPECIFICATION ---
${JSON.stringify(approvedSpec, null, 2)}
--- END SPECIFICATION ---
    `;

    // 2. Define the clean, simple message for the UI.
    const messageForDisplay = "Generate a plan for a standalone browser app based on the approved specification.";
    
    // 3. Call the next function, providing BOTH prompts.
    this.showPlanConfigModal(transposedPromptForAI, null, messageForDisplay);
},
/**
 * NEW: Gracefully falls back to the original planning flow if spec generation fails.
 */
fallbackToRegularPlanning(prompt) {
    this.renderChatMessage('ai', "I had trouble creating a detailed specification, so let's proceed with the standard planning mode. Please choose the number of steps you'd like.");
    this.showPlanConfigModal(prompt);
},




// 1. Initialize plan context cache with intelligent prediction
// 1. Enhanced cache initialization with detailed logging
initializePlanContextCache() {
    const planText = this.state.currentPlan.map(step => step.text).join(' ');
    
    // CRITICAL: Ensure fileOrder is in sync with projectFiles
    const currentFiles = Object.keys(this.state.projectFiles);
    currentFiles.forEach(file => {
        if (!this.state.fileOrder.includes(file)) {
            this.state.fileOrder.push(file);
        }
    });
    
    if (this.state.fileOrder.length <= 1) {
        // New project - predict what files will be needed
        const predictedFiles = this.getPredictedFilesFromPlan(planText);
        this.state.planContextCache = {
            predictedFiles: predictedFiles,
            actualFiles: [...this.state.fileOrder], // Include existing files
            stepHistory: [],
            lastUpdated: Date.now(),
            isNewProject: true,
            // RESTORED: Debug information
            debug: {
                initialized: new Date().toLocaleTimeString(),
                planAnalysis: planText.substring(0, 200) + '...',
                predictions: predictedFiles,
                fileSync: {
                    currentFiles: currentFiles,
                    fileOrder: [...this.state.fileOrder],
                    syncPerformed: true
                }
            }
        };
        
        this.renderCacheStatus('initialized', {
            type: 'new_project',
            predicted: predictedFiles,
            message: `🎯 Plan analysis complete. Starting with ${this.state.fileOrder.length} existing files.`
        });
        
    } else {
        // Existing project - analyze current files
        const existingFiles = this.analyzeExistingProjectFiles();
        
        this.state.planContextCache = {
            predictedFiles: existingFiles.recommended,
            actualFiles: existingFiles.recommended,
            stepHistory: [],
            lastUpdated: Date.now(),
            isNewProject: false,
            // RESTORED: Debug information
            debug: {
                initialized: new Date().toLocaleTimeString(),
                existingAnalysis: existingFiles,
                fileSync: {
                    currentFiles: currentFiles,
                    fileOrder: [...this.state.fileOrder],
                    syncPerformed: true
                }
            }
        };
        
        this.renderCacheStatus('initialized', {
            type: 'existing_project',
            existing: existingFiles.recommended,
            message: `🔍 Project analyzed. Starting with ${existingFiles.recommended.length} recommended files.`
        });
    }
    
    // RESTORED: Add cache debug panel to UI
   // this.showCacheDebugPanel();// <--- JUST COMMENT OUT THIS LINE

    
    console.log('🆕 Plan Context Cache Initialized:', this.state.planContextCache);
},
// 5. Debug panel that shows live cache state
showCacheDebugPanel() {
    // Remove existing debug panel
    const existingPanel = document.getElementById('cache-debug-panel');
    if (existingPanel) existingPanel.remove();
    
    // Create new debug panel with much better styling
    const debugPanel = document.createElement('div');
    debugPanel.id = 'cache-debug-panel';
    debugPanel.style.cssText = `
        position: fixed !important;
        top: 45px !important;
        right: 10px !important;
        width: 320px !important;
        max-height: 500px !important;
        background: #1a1d23 !important;
        border: 2px solid #3a3d44 !important;
        border-radius: 8px !important;
        padding: 1rem !important;
        z-index: 10000 !important;
        font-size: 0.85rem !important;
        overflow-y: auto !important;
        box-shadow: 0 8px 32px rgba(0,0,0,0.5) !important;
        font-family: 'JetBrains Mono', monospace !important;
        color: #ffffff !important;
        line-height: 1.4 !important;
    `;
    
    debugPanel.innerHTML = `
        <div style="display: flex; justify-content: space-between; align-items: center; margin-bottom: 1rem; border-bottom: 1px solid #3a3d44; padding-bottom: 0.5rem;">
            <h4 style="margin: 0 !important; color: #8b5cf6 !important; font-weight: 600 !important; font-size: 1rem !important;">📊 Cache Debug</h4>
            <button onclick="this.parentElement.parentElement.remove()" style="background: #ef4444 !important; border: none !important; color: white !important; cursor: pointer !important; font-size: 1.2rem !important; width: 24px !important; height: 24px !important; border-radius: 4px !important; display: flex !important; align-items: center !important; justify-content: center !important;">&times;</button>
        </div>
        <div id="cache-debug-content"></div>
    `;
    
    document.body.appendChild(debugPanel);
    this.updateCacheDebugPanel();
},

// 6. Update debug panel with current cache state
updateCacheDebugPanel() {
    const content = document.getElementById('cache-debug-content');
    if (!content || !this.state.planContextCache) return;
    
    const cache = this.state.planContextCache;
    const stepCount = cache.stepHistory.length;
    const lastUpdated = new Date(cache.lastUpdated).toLocaleTimeString();
    
    content.innerHTML = `
        <div style="margin-bottom: 1rem !important; padding: 0.75rem !important; background: #2a2d35 !important; border-radius: 6px !important; border-left: 4px solid #10b981 !important;">
            <div style="color: #ffffff !important; font-weight: 600 !important; margin-bottom: 0.5rem !important;">Status</div>
            <div style="color: #e5e7eb !important; font-size: 0.8rem !important;">
                <div><span style="color: #9ca3af !important;">Type:</span> <span style="color: #ffffff !important;">${cache.isNewProject ? 'New Project' : 'Existing Project'}</span></div>
                <div><span style="color: #9ca3af !important;">Steps:</span> <span style="color: #ffffff !important;">${stepCount}</span></div>
                <div><span style="color: #9ca3af !important;">Updated:</span> <span style="color: #ffffff !important;">${lastUpdated}</span></div>
            </div>
        </div>
        
        <div style="margin-bottom: 1rem !important; padding: 0.75rem !important; background: #2a2d35 !important; border-radius: 6px !important; border-left: 4px solid #f59e0b !important;">
            <div style="color: #ffffff !important; font-weight: 600 !important; margin-bottom: 0.5rem !important;">Predicted Files (${cache.predictedFiles.length})</div>
            <div style="font-family: 'JetBrains Mono', monospace !important; background: #1f2937 !important; padding: 0.5rem !important; border-radius: 4px !important; margin-top: 0.5rem !important; border: 1px solid #374151 !important;">
                ${cache.predictedFiles.length > 0 
                    ? cache.predictedFiles.map(file => `<div style="color: #fbbf24 !important; margin: 0.25rem 0 !important;">📄 ${file}</div>`).join('')
                    : '<div style="color: #9ca3af !important; font-style: italic !important;">None predicted</div>'
                }
            </div>
        </div>
        
        <div style="margin-bottom: 1rem !important; padding: 0.75rem !important; background: #2a2d35 !important; border-radius: 6px !important; border-left: 4px solid #3b82f6 !important;">
            <div style="color: #ffffff !important; font-weight: 600 !important; margin-bottom: 0.5rem !important;">Actual Files (${cache.actualFiles.length})</div>
            <div style="font-family: 'JetBrains Mono', monospace !important; background: #1f2937 !important; padding: 0.5rem !important; border-radius: 4px !important; margin-top: 0.5rem !important; border: 1px solid #374151 !important;">
                ${cache.actualFiles.length > 0 
                    ? cache.actualFiles.map(file => `<div style="color: #60a5fa !important; margin: 0.25rem 0 !important;">✅ ${file}</div>`).join('')
                    : '<div style="color: #9ca3af !important; font-style: italic !important;">No files created yet</div>'
                }
            </div>
        </div>
        
        ${stepCount > 0 ? `
        <div style="padding: 0.75rem !important; background: #2a2d35 !important; border-radius: 6px !important; border-left: 4px solid #8b5cf6 !important;">
            <div style="color: #ffffff !important; font-weight: 600 !important; margin-bottom: 0.5rem !important;">Recent Steps</div>
            ${cache.stepHistory.slice(-3).map((step, index) => `
                <div style="font-size: 0.75rem !important; margin: 0.5rem 0 !important; padding: 0.5rem !important; background: #1f2937 !important; border-radius: 4px !important; border: 1px solid #374151 !important;">
                    <div style="color: #a78bfa !important; font-weight: 600 !important; margin-bottom: 0.25rem !important;">Step ${step.stepIndex + 1}</div>
                    <div style="color: #e5e7eb !important;">
                        ${step.newFiles.length > 0 ? `<span style="color: #10b981 !important;">+${step.newFiles.length} new</span>` : ''}
                        ${step.newFiles.length > 0 && step.modifiedFiles.length > 0 ? ' • ' : ''}
                        ${step.modifiedFiles.length > 0 ? `<span style="color: #f59e0b !important;">${step.modifiedFiles.length} modified</span>` : ''}
                        ${step.newFiles.length === 0 && step.modifiedFiles.length === 0 ? '<span style="color: #9ca3af !important;">No changes</span>' : ''}
                    </div>
                </div>
            `).join('')}
        </div>
        ` : ''}
    `;
},

// 2. Visual cache status renderer
// COMPLETELY REPLACE the renderCacheStatus function with this version that uses !important everywhere

renderCacheStatus(event, data) {
    // ✨ CLEAN UI: Only show important messages in chat
    // Everything else goes to console only
    const SHOW_IN_CHAT = [
        'step_complete',      // Step summaries
        'plan_start',         // Plan beginning
        'plan_complete',      // Plan finished  
        'auto_repair',        // Auto-repair messages
        'polish_prompt',      // Polish & Unify
        'critical_error'      // Critical errors
    ];
    
    // Log to console for debugging
    console.log(`📊 Cache Status: ${event}`, data);
    
    // If this event shouldn't show in chat, skip rendering
    if (!SHOW_IN_CHAT.includes(event)) {
        return;
    }
    
    // Ensure CSS is injected
    this.injectCacheMessageCSS();
    
    const timestamp = new Date().toLocaleTimeString();
    let statusHtml = '';
    
    switch(event) {
        case 'initialized':
            statusHtml = `
                <div class="cache-status-message">
                    <div class="cache-header">
                        <div class="cache-icon-title">
                            <div class="cache-icon init">📊</div>
                            <span class="cache-title">Cache Initialized</span>
                        </div>
                        <span class="cache-timestamp">${timestamp}</span>
                    </div>
                    
                    <div class="cache-content">${data.message}</div>
                    
                    <details>
                        <summary>▶ Show Prediction Details</summary>
                        <div class="cache-details-content">
                            <div class="cache-predictions">
                                ${data.type === 'new_project' 
                                    ? `<div class="prediction-section"><span class="prediction-title">Predicted Files:</span></div>${data.predicted.map(f => `<div class="prediction-file">📄 ${f}</div>`).join('')}`
                                    : `<div class="prediction-section"><span class="prediction-title">Existing Files:</span></div>${data.existing.map(f => `<div class="prediction-file">✅ ${f}</div>`).join('')}<div class="prediction-section prediction-additional"><span class="prediction-title">Additional Predictions:</span></div>${data.predicted.map(f => `<div class="prediction-file">📄 ${f}</div>`).join('')}`
                                }
                            </div>
                        </div>
                    </details>
                </div>
            `;
            break;
            
        case 'updated':
            statusHtml = `
                <div class="cache-status-message cache-updated">
                    <div class="cache-header">
                        <div class="cache-icon-title">
                              <div class="cache-icon init">📊</div>
                            <span class="cache-title">Cache Updated - Step ${data.stepIndex + 1}</span>
                        </div>
                        <span class="cache-timestamp">${timestamp}</span>
                    </div>
                    
                    <div class="cache-grid">
                        ${data.newFiles.length > 0 ? `
                            <div class="cache-file-section created">
                                <span class="cache-file-label created">CREATED</span>
                                <div class="cache-file-list">
                                    ${data.newFiles.map(f => `<span class="cache-file-chip">${f}</span>`).join('')}
                                </div>
                            </div>
                        ` : ''}
                        
                        ${data.modifiedFiles.length > 0 ? `
                            <div class="cache-file-section modified">
                                <span class="cache-file-label modified">MODIFIED</span>
                                <div class="cache-file-list">
                                    ${data.modifiedFiles.map(f => `<span class="cache-file-chip">${f}</span>`).join('')}
                                </div>
                            </div>
                        ` : ''}
                        
                        <div class="cache-total">
                            Total cached files: <span class="cache-total-number">${data.totalCachedFiles}</span>
                        </div>
                    </div>
                </div>
            `;
            break;
            
        case 'context_selected':
            statusHtml = `
                <div class="cache-status-message cache-context">
                    <div class="cache-header">
                        <div class="cache-icon-title">
                            <div class="cache-icon context">🎯</div>
                            <span class="cache-title">Context Selected - Step ${data.stepIndex + 1}</span>
                        </div>
                        <span class="cache-timestamp">${timestamp}</span>
                    </div>
                    
                    <div class="cache-grid">
                        <div class="cache-file-section selected">
                            <span class="cache-file-label selected">SELECTED FILES (${data.selectedFiles.length})</span>
                            <div class="cache-file-list">
                                ${data.selectedFiles.map(f => `<span class="cache-file-chip">${f}</span>`).join('')}
                            </div>
                        </div>
                        
                        ${data.reasoning ? `
                            <div class="cache-reasoning">
                                <span class="cache-reasoning-label">REASONING:</span><br>
                                ${data.reasoning}
                            </div>
                        ` : ''}
                    </div>
                </div>
            `;
            break;
    }
    
    if (statusHtml) {
        this.renderChatMessage('ai', statusHtml, true);
    }
},

injectCacheMessageCSS() {
    // Remove existing cache CSS if it exists
    const existingStyle = document.getElementById('cache-message-styles');
    if (existingStyle) existingStyle.remove();
    
    // Create new style element
    const style = document.createElement('style');
    style.id = 'cache-message-styles';
    style.textContent = `
 


body .message-bubble .cache-status-message {
    background: linear-gradient(135deg, rgba(139, 92, 246, 0.15) 0%, rgba(139, 92, 246, 0.08) 100%) !important;
    border: 1px solid rgba(139, 92, 246, 0.4) !important;
    border-left: 4px solid #8b5cf6 !important;
    border-radius: 8px !important;
    padding: 0.75rem !important;
    margin: 0.5rem 0 !important;
    font-family: var(--font-sans) !important;
    display: block !important;
    width: auto !important;
    max-width: none !important;
    box-sizing: border-box !important;
    position: relative !important;
    overflow: visible !important;
    word-wrap: break-word !important;
    word-break: normal !important;
    hyphens: none !important;
    line-height: 1.3 !important;
}

/* GREEN - Updated messages */
body .message-bubble .cache-status-message.cache-updated {
    background: linear-gradient(135deg, rgba(16, 185, 129, 0.15) 0%, rgba(16, 185, 129, 0.08) 100%) !important;
    border: 1px solid rgba(16, 185, 129, 0.4) !important;
    border-left: 4px solid #10b981 !important;
}

/* AMBER - Context selection messages */
body .message-bubble .cache-status-message.cache-context {
    background: linear-gradient(135deg, rgba(245, 158, 11, 0.15) 0%, rgba(245, 158, 11, 0.08) 100%) !important;
    border: 1px solid rgba(245, 158, 11, 0.4) !important;
    border-left: 4px solid #f59e0b !important;
}

/* Header spacing - compact */
body .message-bubble .cache-status-message .cache-header {
    display: flex !important;
    justify-content: space-between !important;
    align-items: center !important;
    margin-bottom: 0.5rem !important;
    flex-wrap: wrap !important;
    gap: 0.5rem !important;
}

/* Content spacing - tight */
body .message-bubble .cache-status-message .cache-content {
    font-size: 0.9rem !important;
    color: var(--text-secondary) !important;
    line-height: 1.3 !important;
    margin-bottom: 0.5rem !important;
    padding: 0 !important;
}

/* Grid spacing - compact */
body .message-bubble .cache-status-message .cache-grid {
    display: grid !important;
    gap: 0.5rem !important;
    font-size: 0.85rem !important;
}

/* File section spacing - tighter */
body .message-bubble .cache-status-message .cache-file-section {
    display: flex !important;
    align-items: flex-start !important;
    gap: 0.5rem !important;
    padding: 0.5rem !important;
    border-radius: 6px !important;
    flex-wrap: wrap !important;
}

/* File list spacing - minimal */
body .message-bubble .cache-status-message .cache-file-list {
    font-family: var(--font-mono) !important;
    color: var(--text-primary) !important;
    display: flex !important;
    flex-wrap: wrap !important;
    gap: 0.25rem !important;
}

/* File chip spacing - compact */
body .message-bubble .cache-status-message .cache-file-chip {
    background: rgba(0, 0, 0, 0.3) !important;
    padding: 0.2rem 0.4rem !important;
    border-radius: 4px !important;
    font-size: 0.75rem !important;
    margin: 0 !important;
    display: inline-block !important;
}

/* Total section spacing - minimal */
body .message-bubble .cache-status-message .cache-total {
    text-align: center !important;
    padding: 0.4rem !important;
    color: var(--text-secondary) !important;
    font-size: 0.8rem !important;
    font-weight: 500 !important;
    margin: 0 !important;
}

/* Reasoning section spacing - compact */
body .message-bubble .cache-status-message .cache-reasoning {
    padding: 0.5rem !important;
    background: rgba(0, 0, 0, 0.2) !important;
    border-radius: 6px !important;
    border: 1px solid rgba(245, 158, 11, 0.2) !important;
    font-style: italic !important;
    color: var(--text-secondary) !important;
    font-size: 0.85rem !important;
    line-height: 1.3 !important;
    margin: 0 !important;
}

/* Details section spacing - tight */
body .message-bubble .cache-status-message details {
    margin-top: 0.5rem !important;
    margin-bottom: 0 !important;
    padding: 0 !important;
}

body .message-bubble .cache-status-message details summary {
    cursor: pointer !important;
    font-size: 0.85rem !important;
    color: var(--accent-primary) !important;
    font-weight: 500 !important;
    padding: 0.25rem 0 !important;
    user-select: none !important;
    margin: 0 !important;
    list-style: none !important;
}

body .message-bubble .cache-status-message .cache-details-content {
    margin-top: 0.25rem !important;
    padding: 0.5rem !important;
    background: rgba(0, 0, 0, 0.2) !important;
    border-radius: 6px !important;
    border: 1px solid rgba(139, 92, 246, 0.2) !important;
}

/* Icon spacing - compact */
body .message-bubble .cache-status-message .cache-icon {
    width: 28px !important;
    height: 28px !important;
    border-radius: 6px !important;
    display: flex !important;
    align-items: center !important;
    justify-content: center !important;
    font-size: 14px !important;
    flex-shrink: 0 !important;
}

/* Icon-title container spacing - tight */
body .message-bubble .cache-status-message .cache-icon-title {
    display: flex !important;
    align-items: center !important;
    gap: 0.5rem !important;
}

/* Prediction sections styling - compact */
body .message-bubble .cache-status-message .cache-predictions {
    font-size: 0.8rem !important;
    color: var(--text-secondary) !important;
    font-family: var(--font-mono) !important;
    line-height: 1.2 !important;
}

body .message-bubble .cache-status-message .cache-predictions .prediction-section {
    margin-bottom: 0.25rem !important;
}

body .message-bubble .cache-status-message .cache-predictions .prediction-title {
    color: var(--accent-primary) !important;
    font-weight: 600 !important;
    font-size: 0.75rem !important;
    text-transform: uppercase !important;
    letter-spacing: 0.5px !important;
    margin-bottom: 0.25rem !important;
}

body .message-bubble .cache-status-message .cache-predictions .prediction-file {
    padding: 0.1rem 0 !important;
    color: var(--text-primary) !important;
    margin: 0 !important;
    font-size: 0.75rem !important;
}

/* Override any inline styles that create large gaps */
body .message-bubble .cache-status-message div[style*="margin-top: 0.75rem"] {
    margin-top: 0.25rem !important;
}

body .message-bubble .cache-status-message div[style*="margin-top: 0.5rem"] {
    margin-top: 0.25rem !important;
}

/* Hide unwanted br tags inside cache status messages */
body .message-bubble .cache-status-message br {
    display: none !important;
}

/* Specific spacing for additional predictions */
body .message-bubble .cache-status-message .prediction-additional {
    margin-top: 0.25rem !important;
}
    `;
    
    // Inject into document head
    document.head.appendChild(style);
    console.log('Cache message CSS injected successfully');
},
// 4. Update cache after each step execution
updatePlanContextAfterStep(stepIndex, stepText, filesBefore, filesAfter) {
    if (!this.state.planContextCache) {
        console.warn(`❌ No cache available for step ${stepIndex + 1}`);
        return;
    }
    
    const cache = this.state.planContextCache;
    const newFiles = [];
    const modifiedFiles = [];
    
    // CRITICAL: Detect what was created/modified with proper synchronization
    for (const path in filesAfter) {
        if (!filesBefore[path]) {
            newFiles.push(path);
            // Ensure new files are in fileOrder
            if (!this.state.fileOrder.includes(path)) {
                this.state.fileOrder.push(path);
            }
        } else if (filesBefore[path].content !== filesAfter[path].content) {
            modifiedFiles.push(path);
        }
    }
    
    // Also check for files removed from fileOrder but still in projectFiles
    const currentFileOrderSet = new Set(this.state.fileOrder);
    for (const path in filesAfter) {
        if (!currentFileOrderSet.has(path) && !newFiles.includes(path)) {
            this.state.fileOrder.push(path);
            console.log(`🔄 Re-added missing file to fileOrder: ${path}`);
        }
    }
    
    // Update the cache with actual results
    const previousCount = cache.actualFiles.length;
    cache.actualFiles = [...new Set([...cache.actualFiles, ...newFiles, ...modifiedFiles])];
    
    // Remove files that no longer exist
    cache.actualFiles = cache.actualFiles.filter(file => 
        this.state.projectFiles[file] && this.state.fileOrder.includes(file)
    );
    
    cache.stepHistory.push({
        stepIndex,
        stepText: stepText.substring(0, 100) + '...',
        newFiles,
        modifiedFiles,
        timestamp: Date.now(),
        cacheGrowth: cache.actualFiles.length - previousCount
    });
    
    cache.lastUpdated = Date.now();
    
    // Learn from patterns
    newFiles.forEach(file => {
        if (!cache.predictedFiles.includes(file)) {
            this.learnFromFileCreation(file, stepText);
        }
    });
    
    // Show update status if changes occurred
    if (newFiles.length > 0 || modifiedFiles.length > 0) {
        this.renderCacheStatus('updated', {
            stepIndex,
            newFiles,
            modifiedFiles,
            totalCachedFiles: cache.actualFiles.length
        });
        
        // Update debug panel
        this.updateCacheDebugPanel();
    }
    
    console.log(`📊 Cache Updated for Step ${stepIndex + 1}:`, {
        newFiles,
        modifiedFiles,
        totalFiles: cache.actualFiles.length,
        fileOrderLength: this.state.fileOrder.length
    });
},

// REPLACE your entire existing learnFromFileCreation function with this:
learnFromFileCreation(createdFile, stepText) {
    if (!this.state.planContextCache) return;
    
    const cache = this.state.planContextCache;
    const fileExt = createdFile.split('.').pop();
    const fileName = createdFile.toLowerCase();
    
    // If we created a new type, look for similar patterns in remaining steps
    const remainingSteps = this.state.currentPlan.slice(cache.stepHistory.length);
    const remainingText = remainingSteps.map(step => step.text).join(' ').toLowerCase();
    
    // EXISTING: Add related file predictions based on what was just created
    if (fileExt === 'html' && !cache.predictedFiles.some(f => f.endsWith('.css'))) {
        cache.predictedFiles.push('style.css');
    }
    if (fileExt === 'js' && fileName.includes('api') && !cache.predictedFiles.includes('config.js')) {
        cache.predictedFiles.push('config.js');
    }
    if (fileName.includes('login') && !cache.predictedFiles.includes('dashboard.html')) {
        if (remainingText.includes('dashboard') || remainingText.includes('profile')) {
            cache.predictedFiles.push('dashboard.html');
        }
    }
    
    // NEW: Learn patterns for future steps
    const pattern = this.identifyCreationPattern(createdFile, stepText);
    if (pattern) {
        cache.patterns = cache.patterns || {};
        cache.patterns[pattern.type] = cache.patterns[pattern.type] || [];
        cache.patterns[pattern.type].push({
            trigger: pattern.trigger,
            creates: createdFile,
            alsoNeeds: pattern.dependencies,
            timestamp: Date.now()
        });
        
        console.log(`📚 Learned pattern: ${pattern.type} → ${createdFile} needs ${pattern.dependencies.join(', ')}`);
    }
},

// NEW FUNCTION - Add this after learnFromFileCreation
identifyCreationPattern(file, stepText) {
    const stepLower = stepText.toLowerCase();
    const fileName = file.toLowerCase();
    
    // Pattern 1: Creating a new page usually needs CSS/JS
    if (file.endsWith('.html') && (stepLower.includes('create') || stepLower.includes('add'))) {
        return {
            type: 'page_creation',
            trigger: 'create new page',
            dependencies: ['style.css', 'script.js']
        };
    }
    
    // Pattern 2: API work needs config and utilities
    if ((fileName.includes('api') || stepLower.includes('api')) && file.endsWith('.js')) {
        return {
            type: 'api_setup',
            trigger: 'api integration',
            dependencies: ['config.js', 'utils.js']
        };
    }
    
    // Pattern 3: Forms need validation
    if ((fileName.includes('form') || stepLower.includes('form')) && file.endsWith('.html')) {
        return {
            type: 'form_creation',
            trigger: 'create form',
            dependencies: ['validation.js', 'style.css']
        };
    }
    
    // Pattern 4: Authentication pages need session management
    if ((fileName.includes('login') || fileName.includes('auth') || fileName.includes('register')) && 
        file.endsWith('.html')) {
        return {
            type: 'auth_page',
            trigger: 'authentication page',
            dependencies: ['auth.js', 'session.js']
        };
    }
    
    // Pattern 5: Dashboard/admin pages need utilities
    if ((fileName.includes('dashboard') || fileName.includes('admin')) && file.endsWith('.html')) {
        return {
            type: 'dashboard_page',
            trigger: 'dashboard page',
            dependencies: ['dashboard.css', 'utils.js', 'api.js']
        };
    }
    
    return null;
},


// 7. Console logging function for detailed debugging
logCacheDetails(event, data = {}) {
    if (!this.state.planContextCache) {
        console.log('❌ Cache Debug: No cache exists');
        return;
    }
    
    const cache = this.state.planContextCache;
    const timestamp = new Date().toLocaleTimeString();
    
    console.group(`🔍 Cache Debug - ${event} at ${timestamp}`);
    console.log('Cache State:', {
        isNewProject: cache.isNewProject,
        predictedFiles: cache.predictedFiles,
        actualFiles: cache.actualFiles,
        stepsProcessed: cache.stepHistory.length,
        lastUpdated: new Date(cache.lastUpdated).toLocaleTimeString()
    });
    
    if (data.stepIndex !== undefined) {
        console.log(`Step ${data.stepIndex + 1} Details:`, data);
    }
    
    console.log('Step History:', cache.stepHistory);
    console.groupEnd();
},

// 7. Console logging function for detailed debugging
logCacheDetails(event, data = {}) {
    if (!this.state.planContextCache) {
        console.log('❌ Cache Debug: No cache exists');
        return;
    }
    
    const cache = this.state.planContextCache;
    const timestamp = new Date().toLocaleTimeString();
    
    console.group(`🔍 Cache Debug - ${event} at ${timestamp}`);
    console.log('Cache State:', {
        isNewProject: cache.isNewProject,
        predictedFiles: cache.predictedFiles,
        actualFiles: cache.actualFiles,
        stepsProcessed: cache.stepHistory.length,
        lastUpdated: new Date(cache.lastUpdated).toLocaleTimeString()
    });
    
    if (data.stepIndex !== undefined) {
        console.log(`Step ${data.stepIndex + 1} Details:`, data);
    }
    
    console.log('Step History:', cache.stepHistory);
    console.groupEnd();
},
// 6. Get adaptive context for a specific step
// REPLACE your entire existing getAdaptiveContextForStep function with this:
getAdaptiveContextForStep(stepIndex, stepText) {
    console.log(`🔍 DEBUG Step ${stepIndex + 1} Context Selection:`, {
        fileOrder: this.state.fileOrder,
        projectFiles: Object.keys(this.state.projectFiles),
        cacheExists: !!this.state.planContextCache,
        cacheFiles: this.state.planContextCache?.actualFiles || []
    });
    
    // Initialize cache if missing
    if (!this.state.planContextCache) {
        console.log(`🆕 Initializing cache for step ${stepIndex + 1}`);
        this.initializePlanContextCache();
    }
    
    const cache = this.state.planContextCache;
    
    // CRITICAL FIX: Force sync cache with current fileOrder
    const currentFiles = [...this.state.fileOrder];
    cache.actualFiles = [...new Set([...cache.actualFiles, ...currentFiles])];
    
    const availableFiles = this.state.fileOrder;
    const selectedFiles = new Set();
    let reasoning = '';
    let estimatedTokens = 0; // Track token usage
    const MAX_TOKENS = 50000; // Token limit
    
    // Always preserve manually selected images
    let imageCount = 0;
    this.state.contextFiles.forEach(file => {
        if (this.isImageFile(file)) {
            selectedFiles.add(file);
            imageCount++;
        }
    });
    if (imageCount > 0) {
        reasoning += `Preserved ${imageCount} user-selected images. `;
    }
    
    // For new projects, use step-specific selection
    if (cache.isNewProject && availableFiles.length <= 1) {
        const fallbackFiles = this.autoSelectFilesForStep(stepText);
        // Add with token checking
        for (const file of fallbackFiles) {
            const fileContent = this.state.projectFiles[file]?.content || '';
            const fileTokens = Math.ceil(fileContent.length / 3);
            if (estimatedTokens + fileTokens < MAX_TOKENS) {
                selectedFiles.add(file);
                estimatedTokens += fileTokens;
            } else {
                console.warn(`⚠️ Skipping ${file} - would exceed token limit`);
                break;
            }
        }
        reasoning += 'New project - using keyword-based prediction. ';
        console.log(`🆕 Step ${stepIndex + 1}: New project fallback - ${selectedFiles.size} files (~${estimatedTokens} tokens)`);
    } else {
        // Use cache-guided selection for existing files
        let cacheMatchCount = 0;
        // Sort by size - add smallest files first
        const sortedCacheFiles = [...cache.actualFiles].sort((a, b) => {
            const aSize = (this.state.projectFiles[a]?.content || '').length;
            const bSize = (this.state.projectFiles[b]?.content || '').length;
            return aSize - bSize; // Smaller files first
        });
        
        for (const file of sortedCacheFiles) {
            if (availableFiles.includes(file) && !selectedFiles.has(file)) {
                const fileContent = this.state.projectFiles[file]?.content || '';
                const fileTokens = Math.ceil(fileContent.length / 3);
                
                if (estimatedTokens + fileTokens < MAX_TOKENS) {
                    selectedFiles.add(file);
                    estimatedTokens += fileTokens;
                    cacheMatchCount++;
                } else {
                    console.warn(`⚠️ Skipping cached file ${file} - would exceed token limit`);
                    break;
                }
            }
        }
        
        if (cacheMatchCount > 0) {
            reasoning += `Added ${cacheMatchCount} files from cache. `;
        }
        
        // Add step-specific files
        const stepSpecificFiles = this.autoSelectFilesForStep(stepText);
        let stepSpecificCount = 0;
        for (const file of stepSpecificFiles) {
            if (availableFiles.includes(file) && !selectedFiles.has(file)) {
                const fileContent = this.state.projectFiles[file]?.content || '';
                const fileTokens = Math.ceil(fileContent.length / 3);
                
                if (estimatedTokens + fileTokens < MAX_TOKENS) {
                    selectedFiles.add(file);
                    estimatedTokens += fileTokens;
                    stepSpecificCount++;
                } else {
                    console.warn(`⚠️ Skipping step-specific file ${file} - would exceed token limit`);
                    break;
                }
            }
        }
        
        if (stepSpecificCount > 0) {
            reasoning += `Added ${stepSpecificCount} step-specific files. `;
        }
    }
    
    // Apply intelligent limits
    const taskType = this.identifyTaskType(stepText);
    const contextLimit = this.getContextLimit(taskType, stepText);
    const finalFiles = Array.from(selectedFiles).slice(0, contextLimit);
    
    reasoning += `Selected ${finalFiles.length} files (limit: ${contextLimit}, ~${estimatedTokens} tokens).`;
    
    console.log(`🎯 Step ${stepIndex + 1} Context:`, {
        taskType,
        availableFiles: availableFiles.length,
        finalSelection: finalFiles.length,
        estimatedTokens,
        reasoning
    });
    
    // Show context selection status
    this.renderCacheStatus('context_selected', {
        stepIndex,
        selectedFiles: finalFiles,
        reasoning: `${taskType} task: ${reasoning}`
    });
    
    return finalFiles;
},



async basicValidation(stepIndex, filesBefore, filesAfter) {
    const issues = [];
    
    // Check if any files were changed
    const hasChanges = Object.keys(filesBefore).length !== Object.keys(filesAfter).length ||
        Object.keys(filesAfter).some(path => 
            !filesBefore[path] || filesBefore[path].content !== filesAfter[path].content
        );
    
    if (!hasChanges) {
        issues.push('No files were modified in this step');
        console.warn(`⚠️ Step ${stepIndex + 1}: No file changes detected`);
    }
    
    // Check for predicted files
    const currentStep = this.state.currentPlan[stepIndex];
    if (currentStep) {
        const predicted = this.getPredictedFilesFromPlan(currentStep.text);
        const missing = predicted.filter(fileName => {
            return !Object.keys(filesAfter).some(
                path => path.endsWith('/' + fileName) || path === fileName
            );
        });
        
        if (missing.length > 0) {
            issues.push(`Missing predicted files: ${missing.join(', ')}`);
            console.warn(`⚠️ Step ${stepIndex + 1}: Missing files:`, missing);
            
            // Attempt auto-repair
            this.renderChatMessage('system', `Step ${stepIndex + 1} Validation: Found ${issues.length} integration issue(s). Attempting auto-repair...`);
            await this.attemptAutoRepair(missing);
        }
    }
    
    if (issues.length > 0 && issues[0] !== 'No files were modified in this step') {
        this.renderChatMessage('system', `Some issues remain after auto-repair. Manual review may be needed.`);
    }
},


// FIX 5: ADD THIS NEW HELPER FUNCTION  
// Auto-repair for missing files
async attemptAutoRepair(missingFiles) {
    console.log('🔧 Attempting auto-repair for:', missingFiles);
    this.renderChatMessage('system', `Found ${missingFiles.length} integration issue(s). Running auto-repair...`);
    this.renderChatMessage('system', `Issues to fix: ${missingFiles.map(f => `- Create the missing file: ${f}`).join('\n')}`);
    
    for (const fileName of missingFiles) {
        try {
            const defaultContent = this.getDefaultFileContent(fileName);
            this.state.projectFiles[fileName] = {
                content: defaultContent,
                lastModified: Date.now(),
                type: 'text'
            };
            
            if (!this.state.fileOrder.includes(fileName)) {
                this.state.fileOrder.push(fileName);
            }
            
            console.log(`✓ Created missing file: ${fileName}`);
            
            // Wait for async state update
            await new Promise(resolve => setTimeout(resolve, 100));
            
        } catch (error) {
            console.error(`Failed to create ${fileName}:`, error);
        }
    }
    
    // Update UI
    this.renderFileTree();
    if (this.state.activeFile) {
        this.activateFile(this.state.activeFile, true);
    }
    
    // Save
    try {
        await this.saveProject();
    } catch (saveError) {
        console.error('Auto-save after repair failed:', saveError);
        this.renderChatMessage('system', 'Auto-save failed: Failed to fetch. Your work is not being saved. Please check your network connection.');
    }
},


// FIX 6: ADD THIS NEW HELPER FUNCTION
// Get default content for missing files
getDefaultFileContent(fileName) {
    if (fileName.endsWith('.html')) {
        return `<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>${fileName.replace('.html', '')}</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <h1>Welcome</h1>
    <script src="script.js"></script>
</body>
</html>`;
    } else if (fileName.endsWith('.css')) {
        return `/* Styles for ${fileName} */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
    line-height: 1.6;
    padding: 20px;
}`;
    } else if (fileName.endsWith('.js')) {
        return `// JavaScript for ${fileName}
console.log('${fileName} loaded');

// Add your code here`;
    }
    return `// ${fileName}\n// Auto-generated placeholder file`;
},
async executeSingleStep(stepText, stepIndex) {
    if (this.state.isExecutingPlan) {
        this.renderChatMessage('ai', 'Please wait for the current task to complete.');
        return;
    }

    this.state.isExecutingPlan = true;
    this.state.isExecutingProcess = true; // FIX: Set flag to bypass diff for this automated step.
    
    // Show footer progress for single step (show as 1 of 1)
    this.showFooterProgress();
    const totalStepsEl = document.getElementById('footer-total-steps');
    if (totalStepsEl) totalStepsEl.textContent = '1';
    const currentStepEl = document.getElementById('footer-current-step');
    if (currentStepEl) currentStepEl.textContent = '1';

    let allStepsWereCompleted = false; // Variable to track completion status

    try {
        const filesBefore = { ...this.state.projectFiles };

        // Progress: 10% - Preparing context
        this.updateFooterProgress(0, 'Preparing step context...', 10);

        // Use adaptive context even for single steps
        const adaptiveContext = this.getAdaptiveContextForStep(stepIndex, stepText);
        this.state.contextFiles = new Set(adaptiveContext);
        this.elements.fileTree.classList.add('file-tree-updating');
        this.renderFileTree();
        
        setTimeout(() => {
            this.elements.fileTree.classList.remove('file-tree-updating');
        }, 800);
        
        this.renderChatMessage('ai', `Step ${stepIndex + 1}: Using adaptive context (${adaptiveContext.length} files).`);

        // Progress: 20% - Starting execution
        this.updateFooterProgress(0, 'Starting step execution...', 20);

        this.state.currentPlan[stepIndex].status = 'running';
        this.updateStepUI(stepIndex, 'running');
        this.highlightCurrentStep(stepIndex);

        // Progress: 30% - Getting specification
        this.updateFooterProgress(0, 'Preparing step specification...', 30);

        const prunedSpec = await this.getPrunedSpecificationForStep(stepText);
        const planExecutionPrompt = this.buildContextAwareStepPrompt(stepText, stepIndex + 1);
        
        const executionOptions = { 
            isStepExecution: true,
            specData: prunedSpec
        };
        
        // Progress: 40% - Sending to AI
        this.updateFooterProgress(0, 'Sending request to AI...', 40);
        
        await this._executeSend(planExecutionPrompt, true, null, `Step ${stepIndex + 1}: ${stepText}`, executionOptions);
        
        // Progress: 70% - Processing results
        this.updateFooterProgress(0, 'Processing AI response...', 70);
        
        this.state.currentPlan[stepIndex].status = 'done';
        this.updateStepUI(stepIndex, 'done');
        
        // Update any existing cache
        if (this.state.planContextCache) {
            this.updatePlanContextAfterStep(stepIndex, stepText, filesBefore, this.state.projectFiles);
        }
        
        // Progress: 85% - Validating
        this.updateFooterProgress(0, 'Validating step results...', 85);
        
        // NEW: Validation with error handling
        try {
            await this.validateStepResults(stepIndex, stepText, filesBefore, this.state.projectFiles);
        } catch (validationError) {
            console.warn(`⚠️ Reflection system encountered an error for step ${stepIndex + 1}. Proceeding with basic validation.`, validationError);
            this.renderChatMessage('system', `Reflection system encountered an error for step ${stepIndex + 1}. Proceeding with basic validation.`);
            
            // FALLBACK: Basic validation
            await this.basicValidation(stepIndex, filesBefore, this.state.projectFiles);
        }
        
        // Progress: 95% - Finalizing
        this.updateFooterProgress(0, 'Finalizing step...', 95);
        
        this.generateStepSummary(stepIndex + 1, stepText, filesBefore, this.state.projectFiles);
        this.renderChatMessage('ai', `✅ Step ${stepIndex + 1} completed successfully.`);
        
        // Progress: 100% - Complete
        this.updateFooterProgress(0, 'Complete!', 100, '#48bb78');
        
        allStepsWereCompleted = this.state.currentPlan.every(s => s.status === 'done');
        if (allStepsWereCompleted) {
            // --- THIS IS THE FIX ---
            // Hide the progress bar immediately, before starting the final unification.
            this.hideFooterProgress();
            await this.finishPlanModeWithCompletion();
        }
    } catch (error) {
        console.error(`Error on step ${stepIndex + 1}:`, error);
        this.renderChatMessage('ai', `❌ An error occurred while executing the step: ${error.message}`);
        this.state.currentPlan[stepIndex].status = 'failed';
        this.updateStepUI(stepIndex, 'failed');
    } finally {
        this.highlightCurrentStep(-1);
        // --- THIS IS THE OTHER PART OF THE FIX ---
        // Only hide the bar here if the plan wasn't fully complete.
        if (!allStepsWereCompleted) {
            this.hideFooterProgress();
        }
        this.state.isExecutingPlan = false;
        this.state.isExecutingProcess = false; // FIX: Reset flag to re-enable diff in normal mode.
    }
},


// Add this function to debug file creation
debugFileCreation(stepIndex, filesBefore, filesAfter) {
    const newFiles = Object.keys(filesAfter).filter(p => !filesBefore[p]);
    const modifiedFiles = Object.keys(filesAfter).filter(p => 
        filesBefore[p] && filesBefore[p].content !== filesAfter[p].content
    );
    
    console.log(`🆕 Step ${stepIndex + 1} File Changes:`, {
        newFiles,
        modifiedFiles,
        fileOrderAfter: this.state.fileOrder,
        projectFilesAfter: Object.keys(this.state.projectFiles)
    });
    
    return { newFiles, modifiedFiles };
},

// NEW HELPER - Find transitional files between current and next step
findTransitionalFiles(currentFiles, nextStepFiles, dependencyGraph) {
    const transitional = [];
    
    // Find files that are imported by both current and next step files
    currentFiles.forEach(currentFile => {
        const currentDeps = dependencyGraph[currentFile];
        if (!currentDeps) return;
        
        currentDeps.imports.forEach(imported => {
            nextStepFiles.forEach(nextFile => {
                const nextDeps = dependencyGraph[nextFile];
                if (nextDeps && nextDeps.imports.includes(imported)) {
                    if (!transitional.includes(imported)) {
                        transitional.push(imported);
                    }
                }
            });
        });
    });
    
    return transitional;
},

// NEW FUNCTION - Add this after getAdaptiveContextForStep
getContextLimit(taskType, stepText) {
    const limits = {
        'page_creation': 5,      // Needs HTML, CSS, JS, related pages
        'api_integration': 6,    // Needs config, utils, multiple endpoints
        'navigation': 5,         // Needs all pages for proper linking
        'form_handling': 4,      // Needs HTML, validation JS, styles
        'styling': 6,            // Might need multiple CSS files + HTML
        'functionality': 5,      // Needs JS + HTML + utilities
        'responsive': 5,         // Needs HTML + CSS + possibly JS
        'general': 4
    };
    
    // Increase limit if step explicitly mentions multiple files
    const fileCount = (stepText.match(/\.html|\.css|\.js/gi) || []).length;
    const baseLimit = limits[taskType] || 4;
    
    // Add 1 extra slot for every 2 file mentions, cap at 8 total
    return Math.min(baseLimit + Math.floor(fileCount / 2), 8);
},

// NEW HELPER: Find files related to a target file
getRelatedFiles(targetFile) {
    if (targetFile.endsWith('.html')) {
        // HTML files need CSS and JS
        return this.state.fileOrder.filter(f => 
            f.endsWith('.css') || f.endsWith('.js')
        );
    }
    if (targetFile.endsWith('.js')) {
        // JS files need HTML to interact with
        return this.state.fileOrder.filter(f => f.endsWith('.html'));
    }
    if (targetFile.endsWith('.css')) {
        // CSS files need HTML to style
        return this.state.fileOrder.filter(f => f.endsWith('.html'));
    }
    return [];
},

// 3. Analyze existing project files for intelligent selection
analyzeExistingProjectFiles() {
    const files = this.state.fileOrder;
    const analysis = {
        recommended: [],
        optional: [],
        core: []
    };
    
    // Categorize existing files by importance
    files.forEach(file => {
        const fileName = file.toLowerCase();
        
        // Core files (always include)
        if (fileName.includes('index.html') || fileName.includes('main.html') ||
            fileName.includes('style.css') || fileName.includes('main.css') ||
            fileName.includes('script.js') || fileName.includes('main.js')) {
            analysis.core.push(file);
            analysis.recommended.push(file);
        }
        // Important HTML pages
        else if (fileName.endsWith('.html') && files.filter(f => f.endsWith('.html')).length <= 5) {
            analysis.recommended.push(file);
        }
        // CSS files (limit to most important)
        else if (fileName.endsWith('.css') && analysis.recommended.filter(f => f.endsWith('.css')).length < 2) {
            analysis.recommended.push(file);
        }
        // JS files (limit to most important)
        else if (fileName.endsWith('.js') && analysis.recommended.filter(f => f.endsWith('.js')).length < 2) {
            analysis.recommended.push(file);
        }
        // Everything else is optional
        else {
            analysis.optional.push(file);
        }
    });
    
    return analysis;
},

// 2. Predict file types and names from plan text
getPredictedFilesFromPlan(planText) {
    const planLower = planText.toLowerCase();
    const predictions = [];
    
    // File type predictions based on plan content
    const fileTypeIndicators = {
        'index.html': ['home page', 'main page', 'landing', 'homepage', 'index', 'entry point'],
        'style.css': ['styling', 'css', 'design', 'appearance', 'layout', 'colors', 'responsive'],
        'script.js': ['javascript', 'interaction', 'dynamic', 'functionality', 'event', 'animation'],
        'about.html': ['about page', 'about us', 'about section'],
        'contact.html': ['contact page', 'contact form', 'contact us'],
        'login.html': ['login', 'authentication', 'sign in', 'user auth'],
        'dashboard.html': ['dashboard', 'admin panel', 'user panel'],
        'api.js': ['api', 'fetch', 'backend', 'server', 'ajax'],
        'utils.js': ['utility', 'helper', 'common functions', 'shared'],
        'config.js': ['configuration', 'settings', 'constants', 'config']
    };
    
    // Specific page predictions
    const pageIndicators = {
        'products.html': ['product', 'catalog', 'shop', 'store', 'inventory'],
        'blog.html': ['blog', 'articles', 'posts', 'news'],
        'gallery.html': ['gallery', 'images', 'photos', 'portfolio'],
        'services.html': ['services', 'offerings', 'what we do']
    };
    
    // Check each indicator
    for (const [fileName, indicators] of Object.entries({...fileTypeIndicators, ...pageIndicators})) {
        if (indicators.some(indicator => planLower.includes(indicator))) {
            predictions.push(fileName);
        }
    }
    
    // Always include basics for web projects
    if (predictions.length === 0 || (!predictions.includes('index.html') && planLower.includes('website'))) {
        predictions.unshift('index.html');
    }
    if (predictions.some(f => f.endsWith('.html')) && !predictions.includes('style.css')) {
        predictions.push('style.css');
    }
    if (planLower.includes('interactive') || planLower.includes('dynamic')) {
        if (!predictions.includes('script.js')) {
            predictions.push('script.js');
        }
    }
    
    return [...new Set(predictions)]; // Remove duplicates
},


// ADD THIS ENTIRE NEW FUNCTION to your App object
// A good place is after the executeSingleStep function.

// REPLACE your current generateStepSummary function with this smarter version.

// REPLACE your generateStepSummary function with this FINAL, SIMPLIFIED, and CORRECTED version.

generateStepSummary(stepNumber, stepText, filesBefore, filesAfter) {
    const createdFiles = [];
    const modifiedFiles = [];
    const deletedFiles = [];

    // Correctly categorize all file changes
    for (const path in filesAfter) {
        if (!Object.prototype.hasOwnProperty.call(filesAfter, path)) continue;
        const fileAfter = filesAfter[path];
        const fileBefore = filesBefore[path];
        if (!fileBefore) {
            createdFiles.push(path);
        } else if (fileBefore.content !== fileAfter.content) {
            const changeSize = (fileAfter.content || '').length - (fileBefore.content || '').length;
            modifiedFiles.push({ path, changeSize });
        }
    }
    for (const path in filesBefore) {
        if (!Object.prototype.hasOwnProperty.call(filesBefore, path)) continue;
        if (!filesAfter[path]) {
            deletedFiles.push(path);
        }
    }

    // Exit early if absolutely nothing has changed.
    if (createdFiles.length === 0 && modifiedFiles.length === 0 && deletedFiles.length === 0) {
        return;
    }

    // --- Start building the HTML with clean design ---
    let summaryHtml = `<div class="step-summary">
                        <h5>Step ${stepNumber} Complete: Summary</h5>`;

    // 1. Add the AI's qualitative summary if it exists
    if (this.state.lastStepSummary) {
        summaryHtml += `
            <div class="ai-summary-quote">
                "${this.escapeHtml(this.state.lastStepSummary)}"
            </div>`;
        this.state.lastStepSummary = null; // Clear it after use
    }

    // 2. Add the quantitative file lists with new design
    if (createdFiles.length > 0) {
        summaryHtml += `<div class="file-info">
            <strong>Created:</strong> `;
        createdFiles.forEach(file => {
            summaryHtml += `<code>${this.escapeHtml(file)}</code> `;
        });
        summaryHtml += `<span class="change-indicator">●</span></div>`;
    }

    if (modifiedFiles.length > 0) {
        modifiedFiles.forEach(file => {
            const sign = file.changeSize >= 0 ? '+' : '';
            const colorClass = file.changeSize >= 0 ? 'positive' : 'negative';
            summaryHtml += `
            <div class="file-info">
                <strong>Modified:</strong> <code>${this.escapeHtml(file.path)}</code>
                <span class="change-indicator">●</span>
                <div class="file-stats ${colorClass}">${sign}${file.changeSize} characters</div>
            </div>`;
        });
    }

    if (deletedFiles.length > 0) {
        summaryHtml += `<div class="file-info">
            <strong>Deleted:</strong> `;
        deletedFiles.forEach(file => {
            summaryHtml += `<code>${this.escapeHtml(file)}</code> `;
        });
        summaryHtml += `<span class="change-indicator negative">●</span></div>`;
    }

    summaryHtml += `</div>`;
    
    // Render the final, complete HTML block.
    this.renderChatMessage('ai', summaryHtml, true);
},

rejectPolishChanges() {
    if (this.state.prePolishFiles) {
        this.state.projectFiles = this.state.prePolishFiles;
        this.state.prePolishFiles = null;
        this.renderFileTree();
        
        if (this.state.activeFile && this.state.projectFiles[this.state.activeFile]) {
            this.activateFile(this.state.activeFile, true);
        }
    }
    
    document.getElementById('polish-results-modal').classList.remove('visible');
    this.state.polishSessionActive = false;
    this.state.polishModalWasOpen = false; // Clear modal state
    this.renderChatMessage('ai', '↩️ Polish changes reverted. Your project has been restored to its previous state.');
},
        async handleRefinePrompt() {
            const currentPrompt = this.elements.oneShotInput.value.trim();
            if (!currentPrompt) { alert('Please enter a prompt to refine first.'); return; }
            const refineButton = this.elements.btnRefinePrompt;
            refineButton.classList.add('loading');
            refineButton.disabled = true;
            const formData = new FormData();
            formData.append('action', 'zencode_ai_call_ai_api');
            formData.append('nonce', zencode_ai_app_vars.nonce);
            formData.append('prompt', currentPrompt);
            formData.append('is_refinement_request', 'true');
            try {
                const response = await fetch(zencode_ai_app_vars.ajaxurl, { method: 'POST', body: formData });
                const data = await response.json();
                if (data.success) {
                    const responseData = data.data;
                    this.elements.oneShotInput.value = responseData.refined_prompt || currentPrompt;
                    if (responseData.refinement_token) { this.state.refinementToken = responseData.refinement_token; }
                    this.handlePromptLengthCheck();
                } else {
                    throw new Error(data.data?.message || 'An unknown error occurred.');
                }
            } catch (error) {
                console.error('Prompt refinement error:', error);
                alert(`Error: ${error.message}`);
            } finally {
                refineButton.classList.remove('loading');
                refineButton.disabled = false;
            }
        },
    };
    
// [REPLACE THE ENTIRE END OF YOUR app.js FILE WITH THIS FINAL BLOCK]

    window.App = App;

const loadLibraries = async () => {
    const loadResource = (url, type, name) => new Promise((resolve) => {
        let element;
        if (type === 'script') {
            element = document.createElement('script');
            element.src = url;
        } else if (type === 'style') {
            element = document.createElement('link');
            element.rel = 'stylesheet';
            element.href = url;
        }
        element.onload = () => { console.log(`[OK] ${name} loaded.`); resolve({name, status: 'loaded'}); };
        element.onerror = () => { console.error(`[FAIL] ${name} failed to load from ${url}.`); resolve({name, status: 'failed'}); };
        document.head.appendChild(element);
    });

    // Load CSS first (non-blocking)
    await Promise.all([
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/codemirror.min.css', 'style', 'CM-CSS'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/theme/dracula.min.css', 'style', 'CM-Theme')
    ]);

    // Load CodeMirror core FIRST and wait for it
    await loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/codemirror.min.js', 'script', 'CodeMirror');

    // Now load all other libraries and CodeMirror addons in parallel
    const results = await Promise.all([
        // Original Libraries
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js', 'script', 'JSZip'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/jsdiff/5.1.0/diff.min.js', 'script', 'JSDiff'),

        // CodeMirror Language Modes
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/javascript/javascript.min.js', 'script', 'CM-JS'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/css/css.min.js', 'script', 'CM-CSS-Mode'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/xml/xml.min.js', 'script', 'CM-XML'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/htmlmixed/htmlmixed.min.js', 'script', 'CM-HTML'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/clike/clike.min.js', 'script', 'CM-CLike'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/php/php.min.js', 'script', 'CM-PHP'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/mode/markdown/markdown.min.js', 'script', 'CM-MD'),
        
        // CodeMirror Addons
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/addon/edit/closebrackets.min.js', 'script', 'CM-Brackets'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/addon/edit/matchbrackets.min.js', 'script', 'CM-MatchBrackets'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/addon/fold/xml-fold.min.js', 'script', 'CM-XML-Fold'),
        loadResource('https://cdnjs.cloudflare.com/ajax/libs/codemirror/5.65.15/addon/edit/matchtags.min.js', 'script', 'CM-MatchTags')
    ]);

    // Update the app state
    App.state.hasJszip = results.find(r => r.name === 'JSZip')?.status === 'loaded';
    App.state.hasJsDiff = results.find(r => r.name === 'JSDiff')?.status === 'loaded';
    App.state.hasCodeMirror = true; // We know this loaded because we awaited it
    
    // Initialize the application
    App.init();
};
    
    loadLibraries();
});