<?php
/**
 * Plugin Name: ZenCode AI
 * Plugin URI: https://your-website.com
 * Description: AI-powered coding assistant with multimodal vision support and SaaS features.
 * Version: 5.0.3 (Professional Edition)
 * Author: Your Name
 * License: GPL v2 or later
 gemini-2.5-pro
 * Text Domain: zencode-ai
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}



// Define plugin constants
define('ZENCODE_AI_VERSION', '5.0.3');
define('ZENCODE_AI_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('ZENCODE_AI_PLUGIN_URL', plugin_dir_url(__FILE__));
// Load license system (CUSTOMER VERSION)
// This connects to YOUR license server for validation
if (file_exists(ZENCODE_AI_PLUGIN_DIR . 'zencode-customer-license.php')) {
    require_once ZENCODE_AI_PLUGIN_DIR . 'zencode-customer-license.php';
}

class ZenCodeAI {
    
private $supported_models = [
    'google/gemini-2.5-pro' => 'google',
    'google/gemini-2.5-pro' => 'google',
	'google/gemini-2.5-pro' => 'google',
	
	
	
	
    // Add any fallback models you plan to use
    'google/gemini-2.5-pro' => 'google', // if using as fallback
    'google/gemini-2.5-pro' => 'google',     // if using as fallback
];

private $stream_buffer = ''; // Buffer for incomplete streaming JSON

    // ===== NEW CONSOLIDATED HELPER FUNCTIONS =====
    
    /**
     * Consolidated authentication and authorization validation
     */
    private function validate_ajax_request($require_admin = false) {
        if (!is_user_logged_in()) {
            wp_send_json_error(['message' => 'Authentication required.'], 403);
            wp_die();
        }
        
        if (!wp_verify_nonce($_POST['nonce'] ?? $_GET['nonce'] ?? '', 'zencode_ai_nonce')) {
            wp_send_json_error(['message' => 'Security verification failed.'], 403);
            wp_die();
        }
        
        if ($require_admin && !current_user_can('activate_plugins')) {
            wp_send_json_error(['message' => 'Admin privileges required.'], 403);
            wp_die();
        }
        
        return get_current_user_id();
    }

    /**
     * Consolidated user context and capabilities
     */
   // ALSO CHECK: Make sure your get_user_context function includes vector KB access
private function get_user_context() {
    $user_id = get_current_user_id();
    $user_plan = zc_get_user_pricing_plan($user_id);
    $is_admin = user_can($user_id, 'manage_options');
    
    return [
        'user_id' => $user_id,
        'plan' => $user_plan,
        'is_admin' => $is_admin,
        'can_query' => $is_admin || zc_can_user_query($user_id),
        'has_vision' => $user_plan === 'visionary' || $is_admin,
        'has_vector_kb' => in_array($user_plan, ['forge', 'visionary']) || $is_admin // MAKE SURE THIS LINE EXISTS
    ];
}

    /**
     * Consolidated API key validation
     */
    private function validate_api_key($provider = 'google') {
        $key = get_option("zencode_ai_{$provider}_api_key");
        if (empty($key)) {
            throw new Exception("API key for '{$provider}' is not configured.");
        }
        return $key;
    }
// ADD THIS ENTIRE NEW FUNCTION
private function get_master_plan_learnings($user_id, $limit = 3) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';

    // Fetch the highest-scoring "master_plan_reflection" learnings
    $learnings = $wpdb->get_results($wpdb->prepare(
        "SELECT rule FROM {$table_name} 
         WHERE user_id = %d 
         AND learning_type = 'master_plan_reflection'
         ORDER BY score DESC, created_at DESC
         LIMIT %d",
        $user_id,
        $limit
    ));

    if (empty($learnings)) {
        return '';
    }

    $rules = array_map(function($learning) {
        return '- ' . $learning->rule;
    }, $learnings);

    // Format them for the prompt
    $learning_summary = "**KEY STRATEGIC LESSONS FROM PAST PROJECTS (YOU MUST FOLLOW THESE):**\n";
    $learning_summary .= implode("\n", $rules);
    $learning_summary .= "\n\n";

    return $learning_summary;
}
    /**
     * Consolidated error handling with user-friendly messages
     */
private function handle_api_error($error, $context = '') {
    $message = $error->getMessage();
    
    // Handle Google Gemini overload with positive messaging and automatic retry
    if (strpos($message, 'Overloaded') !== false || strpos($message, 'overloaded') !== false) {
        // Never show the raw "overloaded" error - always handle gracefully
        wp_send_json_success([
            'ai_response' => '',
            'retry_with_delay' => true,
            'retry_message' => "High demand means you're using cutting-edge AI technology! Your request is being processed with priority. Please wait a moment while we ensure the best possible response quality.",
            'retry_delay' => 3000 // 3 second delay before auto-retry
        ]);
        return;
    }
    
    // Standardize other common error messages with positive spin
    if (strpos($message, 'timeout') !== false || strpos($message, 'cURL error 28') !== false) {
        $message = "Your request is quite sophisticated and requires extra processing time. This shows you're pushing the boundaries of what's possible with AI! Try breaking complex requests into focused steps for faster results.";
    } elseif (strpos($message, '413') !== false || strpos($message, 'Request Entity Too Large') !== false) {
        $message = "Your project has grown impressively large! To maintain optimal performance, try reducing the number of files in context or work with smaller file segments.";
    } elseif (strpos($message, 'rate limit') !== false) {
        $message = "You're making great use of the AI capabilities! You've reached the API rate limit. Please wait a moment before sending your next request.";
    } elseif (strpos($message, 'insufficient') !== false || strpos($message, 'quota') !== false) {
        $message = "You've been actively creating with AI! Your API quota needs a moment to refresh. This ensures consistent quality for all users.";
    }
    
    if ($context) {
        $message = "[{$context}] " . $message;
    }
    
    wp_send_json_error(['message' => $message]);
}

    /**
     * Check if request contains multi-page indicators
     */
  private function is_multi_page_request($text) {
    $multi_page_indicators = ['2-page', '3-page', '4-page', '5-page', 'multi-page', 'multiple pages', 'about page', 'contact page', 'blog page', 'portfolio page', 'user login', 'database', 'e-commerce'];
    $text_lower = strtolower($text);
    
    // Allow "1 page" or "single page" explicitly
    if (preg_match('/\b1[- ]page/i', $text) || preg_match('/single[- ]page/i', $text)) {
        return false;
    }
    
    // Check for 2+ pages
    if (preg_match('/\b([2-9]|[1-9]\d+)[- ]page/i', $text)) {
        return true;
    }
    
    foreach ($multi_page_indicators as $indicator) {
        if (strpos($text_lower, $indicator) !== false) {
            return true;
        }
    }
    
    return preg_match('/\d+\s+page.*website/', $text_lower);
}

/**
 * NEW: Calculate minimum recommended steps based on project complexity
 */
private function calculate_minimum_steps($project_description, $num_files = 0, $has_existing_project = false) {
    $text_lower = strtolower($project_description);
    $base_steps = 3;
    
    // Analyze project complexity
    $is_multi_page = $this->is_multi_page_request($project_description);
    $has_backend = preg_match('/\b(api|backend|server|database|auth|login|signup)\b/i', $project_description);
    $has_complex_ui = preg_match('/\b(dashboard|admin|complex|multi-step|wizard|chart|graph|data visualization)\b/i', $project_description);
    $has_animations = preg_match('/\b(animation|transition|interactive|dynamic|scroll)\b/i', $project_description);
    $has_forms = preg_match('/\b(form|validation|submit|input)\b/i', $project_description);
    
    // Calculate minimum steps
    if ($is_multi_page) {
        $base_steps += 2; // At least 2 more steps for multi-page
        
        // Count number of pages mentioned
        if (preg_match('/\b([3-9]|[1-9]\d+)[- ]page/i', $project_description, $matches)) {
            $page_count = intval($matches[1]);
            $base_steps += min($page_count - 1, 5); // Cap at 5 additional steps
        }
    }
    
    if ($has_backend) {
        $base_steps += 3; // Backend typically needs setup, models, and API endpoints
    }
    
    if ($has_complex_ui) {
        $base_steps += 2; // Complex UI needs more steps
    }
    
    if ($has_animations) {
        $base_steps += 1; // Animations need dedicated attention
    }
    
    if ($has_forms) {
        $base_steps += 1; // Forms need validation and handling
    }
    
    // If modifying existing project, add steps based on file count
    if ($has_existing_project && $num_files > 0) {
        if ($num_files > 10) {
            $base_steps += 3;
        } elseif ($num_files > 5) {
            $base_steps += 2;
        } elseif ($num_files > 2) {
            $base_steps += 1;
        }
    }
    
    return max(3, min($base_steps, 20)); // Ensure between 3 and 20 steps
}

/**
 * NEW: Validate if chosen step count is sufficient for project complexity
 */
public function handle_validate_step_count() {
    $this->validate_ajax_request();
    
    $chosen_steps = intval($_POST['chosen_steps'] ?? 5);
    $project_description = sanitize_textarea_field($_POST['project_description'] ?? '');
    $num_files = intval($_POST['num_files'] ?? 0);
    $has_existing = isset($_POST['has_existing']) ? (bool)$_POST['has_existing'] : false;
    
    $minimum_steps = $this->calculate_minimum_steps($project_description, $num_files, $has_existing);
    
    $response = [
        'minimum_steps' => $minimum_steps,
        'chosen_steps' => $chosen_steps,
        'is_sufficient' => $chosen_steps >= $minimum_steps,
        'warning_message' => ''
    ];
    
    if ($chosen_steps < $minimum_steps) {
        $complexity_reasons = [];
        
        if ($this->is_multi_page_request($project_description)) {
            $complexity_reasons[] = 'multi-page website';
        }
        if (preg_match('/\b(api|backend|server|database)\b/i', $project_description)) {
            $complexity_reasons[] = 'backend/API functionality';
        }
        if (preg_match('/\b(dashboard|admin|complex|multi-step)\b/i', $project_description)) {
            $complexity_reasons[] = 'complex UI components';
        }
        if ($num_files > 5) {
            $complexity_reasons[] = 'existing project with ' . $num_files . ' files';
        }
        
        $reasons_text = !empty($complexity_reasons) ? ' (' . implode(', ', $complexity_reasons) . ')' : '';
        
        $response['warning_message'] = "⚠️ Your project appears complex{$reasons_text}. " .
            "We recommend at least {$minimum_steps} steps to ensure quality results and avoid timeouts. " .
            "Proceeding with {$chosen_steps} steps may result in incomplete features or rushed implementation.";
    }
    
    wp_send_json_success($response);
    wp_die();
}

private function get_adaptive_timeout($operation_type, $content_size = 0, $num_steps = 5) {
    // OPTIMIZED: Shorter base timeouts for faster plan execution
    $timeouts = [
        'ai_query' => 30,           // Reduced from 60
        'vision' => 40,             // Reduced from 90
        'code_generation' => 35,    // Reduced from 90
        'plan_step' => 30,          // New: specific for plan steps
        'unify' => 45,              // NEW: Longer timeout for unification
        'default' => 30             // Reduced from 60
    ];
    
    $base_timeout = $timeouts[$operation_type] ?? $timeouts['default'];
    
    if ($this->timeout_config['adaptive_timeout']) {
        // Extend timeout for large content
        if ($content_size > 100000) {
            $base_timeout += 30;
        } elseif ($content_size > 50000) {
            $base_timeout += 15; // Reduced from 30-60
        } elseif ($content_size > 25000) {
            $base_timeout += 10;
        }
        
        // NEW: Adjust timeout based on step complexity
        if ($operation_type === 'plan_step' && $num_steps > 0) {
            // Fewer steps might mean more complex individual steps
            if ($num_steps < 5) {
                $base_timeout += 15; // Add time for more complex steps
            } elseif ($num_steps >= 10) {
                $base_timeout += 5; // Slightly more time for detailed plans
            }
        }
        
        // REMOVED: Time of day adjustment (unreliable for international users)
    }
    
    return min($base_timeout, 300); // Cap at 5 minutes instead of 10
}

private function execute_with_timeout_management($callback, $operation_type, $context = '', $num_steps = 5) {
    // OPTIMIZED: Fewer retries, faster backoff
    $max_retries = 2; // Reduced from potentially 3+
    $retry_delay = 2; // Reduced from 5+
    
    $last_exception = null;
    
    for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
        try {
            // Set appropriate timeout for this attempt
            $timeout = $this->get_adaptive_timeout($operation_type, 0, $num_steps);
            set_time_limit($timeout);
            
            error_log("ZenCode AI: Attempt {$attempt}/{$max_retries} for {$operation_type} with {$timeout}s timeout");
            
            $result = $callback();
            
            // Success - log performance
            if ($attempt > 1) {
                error_log("ZenCode AI: {$operation_type} succeeded on retry {$attempt}");
            }
            return $result;
            
        } catch (Exception $e) {
            $last_exception = $e;
            $error_message = $e->getMessage();
            error_log("ZenCode AI: Attempt {$attempt} failed - {$error_message}");
            
            // Don't retry on client errors (these won't succeed on retry)
            if (strpos($error_message, 'invalid_request_error') !== false ||
                strpos($error_message, 'authentication_error') !== false ||
                strpos($error_message, 'permission_error') !== false) {
                break; // Exit immediately, no point retrying
            }
            
            // Check if we should retry
            if ($attempt < $max_retries && $this->should_retry_error($error_message)) {
                // OPTIMIZED: Linear backoff instead of exponential
                $wait_time = $retry_delay * $attempt; // 2s, 4s instead of 2s, 4s, 8s
                error_log("ZenCode AI: Retrying in {$wait_time}s...");
                sleep($wait_time);
                continue;
            }
            
            // Final attempt failed or non-retryable error
            break;
            throw $e;
        }
    }
}

private function should_retry_error($error_message) {
    $retryable_errors = [
        'timeout', 'cURL error 28', 'rate limit', 'quota',
        'server error', 'gateway', 'internal error', 'busy',
        'overloaded', 'try again', 'temporarily'
    ];
    
    foreach ($retryable_errors as $retryable) {
        if (stripos($error_message, $retryable) !== false) {
            return true;
        }
    }
    
    return false;
}

/**
 * Advanced timeout and retry management system
 */
private $timeout_config = [
    'default' => 120,
    'plan_generation' => 180,
    'one_shot_build' => 300,
    'vector_operations' => 90,
    'quick_operations' => 60,
    'max_retries' => 3,
    'retry_delay' => 2,
    'adaptive_timeout' => true
];
/**
 * Chunk large text content to avoid API limits
 */
private function chunk_text_content($text, $max_chunk_size = 8000) {
    $chunks = [];
    $lines = explode("\n", $text);
    $current_chunk = '';
    
    foreach ($lines as $line) {
        // If adding this line would exceed chunk size, save current chunk and start new one
        if (strlen($current_chunk) + strlen($line) + 1 > $max_chunk_size) {
            if (!empty($current_chunk)) {
                $chunks[] = $current_chunk;
                $current_chunk = '';
            }
            
            // If single line is too long, split it
            if (strlen($line) > $max_chunk_size) {
                $line_chunks = str_split($line, $max_chunk_size - 100);
                foreach ($line_chunks as $line_chunk) {
                    $chunks[] = $line_chunk;
                }
                continue;
            }
        }
        
        $current_chunk .= $line . "\n";
    }
    
    // Add the last chunk
    if (!empty($current_chunk)) {
        $chunks[] = trim($current_chunk);
    }
    
    return $chunks;
}
    /**
     * Determine if request should go through planning analysis
     */
private function should_run_planning_check($text, $is_one_shot) {
    if (empty($text)) return false;
    
    // For one-shot builds, use the existing logic
    if ($is_one_shot) {
        $question_indicators = ['how do i', 'how to', 'what is', 'can you explain', 'explain how', 'tell me about', '?'];
        foreach ($question_indicators as $indicator) {
            if (strpos(strtolower($text), $indicator) !== false) {
                return false; // It's a question, not a build request
            }
        }
        
        // Check for build-like requests
        $action_words = ['create', 'build', 'generate', 'make', 'scaffold'];
        $subjects = ['website', 'plugin', 'application', 'app', 'theme', 'site'];
        
        $has_action = false;
        $has_subject = false;
        $text_lower = strtolower($text);
        
        foreach ($action_words as $word) {
            if (strpos($text_lower, $word) !== false) {
                $has_action = true;
                break;
            }
        }
        
        foreach ($subjects as $word) {
            if (strpos($text_lower, $word) !== false) {
                $has_subject = true;
                break;
            }
        }
        
        return $has_action && $has_subject && strlen($text) > 100;
    }
    
    // For regular chat mode, check for complexity indicators
    $text_lower = strtolower($text);
    
    // Skip obvious questions
    $question_indicators = ['how do i', 'how to', 'what is', 'can you explain', 'explain how', 'tell me about'];
    foreach ($question_indicators as $indicator) {
        if (strpos($text_lower, $indicator) !== false) {
            return false;
        }
    }
    
    // CHECK FOR 4+ PAGES FIRST (this should trigger)
    if (preg_match('/\b([4-9]|[1-9]\d+)[- ]page/i', $text)) {
        return true; // 4+ pages should go to plan mode
    }
    
    // Allow 1-3 pages explicitly
    if (preg_match('/\b[1-3][- ]page/i', $text)) {
        return false; // 1-3 pages are fine for general chat
    }
    
    // Check for clearly complex project indicators (backend/database stuff)
    $complex_indicators = [
        'user login', 'authentication', 'database', 'e-commerce', 
        'shopping cart', 'payment', 'admin panel', 'dashboard', 
        'user registration', 'CMS', 'content management', 
        'social media platform', 'online store', 'user accounts'
    ];
    
    foreach ($complex_indicators as $indicator) {
        if (strpos($text_lower, $indicator) !== false) {
            return true;
        }
    }
    
    // Check for patterns suggesting many pages
    if (preg_match('/multiple pages?/i', $text) || 
        preg_match('/several pages?/i', $text) ||
        preg_match('/various pages?/i', $text) ||
        preg_match('/many pages?/i', $text)) {
        return true;
    }
    
    // Check for lists of multiple pages (home + about + contact + services + blog)
    $page_mentions = substr_count($text_lower, 'page');
    if ($page_mentions >= 4) { // If they mention "page" 4+ times, probably complex
        return true;
    }
    
    return false;
}
	
	/**
     * Process main AI request with consolidated logic
     */
// In claudezencode-aiphpp.js

private function process_main_ai_request($prompt, $model, $user_context, $post_data) {
    // Extract core parameters
    $chat_history = json_decode(stripslashes($post_data['chat_history'] ?? '[]'), true) ?: [];
    $project_files = json_decode(stripslashes($post_data['project_files'] ?? '[]'), true);
    $project_structure = sanitize_textarea_field($post_data['project_structure'] ?? '');
    $image_data = !empty($post_data['image_data']) ? $post_data['image_data'] : null;
    $image_source = sanitize_text_field($post_data['image_source'] ?? '');
    $is_plan_mode = isset($post_data['is_plan_mode']) && $post_data['is_plan_mode'] === 'true';
    $use_knowledge = isset($post_data['use_knowledge_base']) && $post_data['use_knowledge_base'] === 'true';
    $is_one_shot_build = isset($post_data['is_one_shot_build']) && $post_data['is_one_shot_build'] === 'true';
    $requested_steps = isset($post_data['requested_steps']) ? absint($post_data['requested_steps']) : null;
    $num_steps = isset($post_data['num_steps']) ? absint($post_data['num_steps']) : 5;
    $is_polish_request = isset($post_data['is_polish_request']) && $post_data['is_polish_request'] === 'true';
    $polish_options_raw = stripslashes($post_data['polish_options'] ?? '[]');
    $polish_options = json_decode($polish_options_raw, true) ?: [];
    $polish_intensity = sanitize_text_field($post_data['polish_intensity'] ?? 'moderate');
    $specification_sheet_json = stripslashes($post_data['specification_sheet'] ?? null);
    $specification_sheet = $specification_sheet_json ? json_decode($specification_sheet_json, true) : null;
    $project_context = isset($post_data['project_context']) ? json_decode(stripslashes($post_data['project_context']), true) : null;
    $is_step_execution = isset($post_data['is_step_execution']) && $post_data['is_step_execution'] === 'true';
    $step_index = isset($post_data['step_index']) ? absint($post_data['step_index']) : null;
    $is_scaffolding_step = isset($post_data['is_scaffolding_step']) && $post_data['is_scaffolding_step'] === 'true'; // ADD THIS
    $vision_images = null; 

    $kb_context = $use_knowledge ? $this->get_relevant_knowledge($prompt, $user_context['user_id']) : '';
    
    $operation_type = $is_one_shot_build ? 'one_shot_build' : 
                     ($is_plan_mode ? 'plan_generation' : 'default');

    return $this->execute_with_timeout_management(
        function() use ($prompt, $model, $chat_history, $project_files, $project_structure, $image_data, $is_plan_mode, $kb_context, $vision_images, $user_context, $is_one_shot_build, $requested_steps, $is_polish_request, $polish_options, $polish_intensity, $image_source, $specification_sheet, $project_context, $is_step_execution, $step_index, $is_scaffolding_step) { // ADD $is_scaffolding_step
            
            return $this->call_ai_api(
                $prompt, $model, 'google', $chat_history,
                $project_files, $project_structure, $image_data, $is_plan_mode,
                $kb_context, $vision_images, $user_context['plan'], $is_one_shot_build,
                $requested_steps, $is_polish_request, $polish_options, $polish_intensity, $image_source,
                $specification_sheet, $project_context, $is_step_execution,
                $step_index,
                $is_scaffolding_step // ADD THIS
            );
        },
        $operation_type,
        'Main AI Request',
        $num_steps
    );
}

private function handle_ai_query() {
    $this->validate_ajax_request();
    
    // NEW: Check license
    $license_key = get_option('zencode_license_key');
    if (empty($license_key)) {
        wp_send_json_error([
            'message' => 'No license activated. Please activate your license or start a free trial.',
            'license_required' => true
        ]);
    }
    
    // Validate license (works with both owner and customer versions)
    $validation = null;
    
    // Customer version - validates via API to your server
    if (class_exists('ZenCode_Customer_License')) {
        $customer_license = new ZenCode_Customer_License();
        $validation = $customer_license->validate_license($license_key);
    }
    // Owner version - validates from local database
    elseif (class_exists('ZenCode_License_Manager')) {
        $license_manager = ZenCode_License_Manager::get_instance();
        $validation = $license_manager->validate_license($license_key, site_url());
    }
    
    if (!$validation || !$validation['valid']) {
        wp_send_json_error([
            'message' => 'License error: ' . ($validation['message'] ?? 'Invalid license'),
            'license_status' => $validation['status'] ?? 'invalid'
        ]);
    }
}

// [REPLACE] your function with this precise version.

// [REPLACE] your entire build_system_prompt function with this final, unabridged version.

// [REPLACE] your entire build_system_prompt function with this final, unabridged version.

// [REPLACE] your entire build_system_prompt function with this final, corrected version.

private function build_system_prompt(
    $project_files,
    $project_structure,
    $is_plan_mode,
    $kb_context,
    $user_plan,
    $requested_steps = null,
    $is_polish_request = false,
    $polish_options = [],
    $polish_intensity = 'moderate',
    $specification_sheet = null,
    $project_context = null,
    $is_step_execution = false,
    $prompt = null,
    $step_index = null,
    $is_scaffolding_step = false
) {
    $base_prompt = '';
    
    // Enhanced 3-file mode detection
    $is_three_file_mode = false;
    
    // Check multiple indicators for 3-file constraint
    if (strpos($prompt, 'HARD CONSTRAINTS:') !== false) {
        $is_three_file_mode = true;
    } elseif (strpos($prompt, 'only 3 files') !== false) {
        $is_three_file_mode = true;
    } elseif (strpos($prompt, 'three files') !== false) {
        $is_three_file_mode = true;
    } elseif (strpos($prompt, 'three-file') !== false) {
        $is_three_file_mode = true;
    } elseif (isset($project_context['constraints']) && 
              (strpos($project_context['constraints'], '3 files') !== false ||
               strpos($project_context['constraints'], 'three files') !== false)) {
        $is_three_file_mode = true;
    }

    if ($is_three_file_mode) {
        $base_prompt .= "🚫 **3-FILE CONSTRAINT ACTIVE - STRICT ENFORCEMENT**\n";
        $base_prompt .= "This project MUST use ONLY these 3 files: index.html, style.css, script.js\n";
        $base_prompt .= "• You are FORBIDDEN from creating any additional files\n"; 
        $base_prompt .= "• All functionality must be implemented within these 3 files\n";
        $base_prompt .= "• Do NOT suggest, create, or reference any other files\n";
        $base_prompt .= "• This constraint applies to ALL steps, not just scaffolding\n\n";
    }

    // Check for our new special planning mode
    $is_standalone_app_plan = (strpos($prompt, 'HARD CONSTRAINTS:') !== false);

    if ($is_standalone_app_plan) {
        // A special, high-level prompt for creating the 3-file app plan
        $base_prompt .= "You are an expert AI Project Architect. Your task is to take a hyper-detailed prompt that describes a complete, self-contained web application and break it down into a logical, step-by-step development plan.

**CRITICAL RULES FOR THIS SPECIFIC TASK:**
1.  **ADHERE TO CONSTRAINTS:** The user's prompt contains 'HARD CONSTRAINTS' (e.g., only 3 files, vanilla JS, localStorage). Your entire plan MUST adhere strictly to these constraints.
2.  **LOGICAL GROUPING:** Group related features into logical steps. For example, a good plan would be:
    - Step 1: Scaffold the project and implement the HTML structure for User Accounts and the Dashboard.
    - Step 2: Implement the CSS for all components, including a light/dark theme.
    - Step 3: Implement the JavaScript logic for User Accounts using localStorage.
    - Step 4: Implement the JavaScript for Event Creation and Management.
    - etc...
3.  **FUNCTIONALITY FIRST:** Ensure each step produces working, testable functionality before moving to enhancements.
4.  **RESPONSIVE FINISH:** Always include a final step for comprehensive responsive design that ensures mobile-first styling across all components.
5.  **JSON ONLY:** Your entire output must be a single, valid JSON object with 'introduction' and 'plan' keys. Do not write any other text.
";

    } else if ($is_plan_mode || (!empty($specification_sheet) && !$is_step_execution)) {
        // This is the original, general-purpose plan generation prompt
        $base_prompt .= "You are a specialized AI Project Architect. Your ONLY function is to generate a structured JSON object that outlines a development plan based on the user's request...\n\n"; // (rest of the prompt is the same)
        $base_prompt .= "**MANDATORY OUTPUT FORMAT:**\n";
        $base_prompt .= "Your ENTIRE response MUST be ONLY a single, valid JSON object. Do NOT include any text, markdown, or explanations before or after the JSON block.\n\n";
        $base_prompt .= "**PERFECT RESPONSE EXAMPLE:**\n";
        $base_prompt .= "{\n";
        $base_prompt .= "  \"introduction\": \"This plan outlines the steps to build a simple portfolio website, focusing on structure, styling, and interactivity.\",\n";
        $base_prompt .= "  \"plan\": [\n";
        $base_prompt .= "    \"Create the main index.html file with semantic header, main, and footer sections.\",\n";
        $base_prompt .= "    \"Create a style.css file and link it to index.html, adding basic resets and typography.\"\n";
        $base_prompt .= "  ]\n";
        $base_prompt .= "}\n\n";
        $base_prompt .= "**CRITICAL RULES FOR PLANNING:**\n";
        $base_prompt .= "1. **JSON ONLY:** Your entire output must be parsable JSON.\n";
        $base_prompt .= "2. **NO CODE:** You are forbidden from writing any code.\n";
        $base_prompt .= "3. **ATOMIC STEPS:** Each step in the 'plan' array must be a single, actionable coding task.\n";
        $base_prompt .= "4. **FUNCTIONAL COMPLETENESS:** Each step should produce working, testable functionality. Avoid steps that leave features partially implemented.\n";
        $base_prompt .= "5. **RESPONSIVE FINISH:** If the project involves a user interface, the final step must be to apply comprehensive responsive styling and ensure the application is mobile-first and works on all device sizes.\n";
        if ($requested_steps) {
            $base_prompt .= "6. **NATURAL STEP BREAKDOWN:** Create {$requested_steps} logical, sequential steps. If the natural breakdown differs slightly, prioritize logical grouping over exact count.\n";
        }

    } else {
        // This is the code execution prompt
        $base_prompt .= "You are an elite AI web developer...\n\n";
        
        if ($is_scaffolding_step) {
            if ($is_three_file_mode) {
                $base_prompt .= "**3-FILE APP FOUNDATION SETUP (STRICT CONSTRAINT):**\n";
                $base_prompt .= "1. **STRICT 3-FILE ADHERENCE:** You MUST use ONLY index.html, style.css, and script.js. Creating any other files is FORBIDDEN.\n";
                $base_prompt .= "2. **CREATE CORE FILES ONLY:** Create the three essential files with proper linking between them.\n";
                $base_prompt .= "3. **IMPLEMENT FOUNDATION + FIRST FEATURE:** Build the complete project structure within the 3-file constraint AND implement the first logical feature from the plan.\n";
                $base_prompt .= "4. **SELF-CONTAINED DESIGN:** All functionality must work within these 3 files. No external dependencies or additional files.\n";
                $base_prompt .= "5. **FUTURE-PROOF STRUCTURE:** Design the HTML structure to accommodate all planned features within this single file.\n\n";
            } else {
                // --- START OF THE FIX ---
                // This new set of instructions is smarter and less rigid, preventing the "index.php" issue.
                $base_prompt .= "**INTELLIGENT PROJECT SCAFFOLDING:**\n";
                $base_prompt .= "1. **ANALYZE THE FULL PLAN:** Review the entire development plan to understand the project's goal and architecture (e.g., WordPress Plugin, React App, static site).\n";
                $base_prompt .= "2. **CREATE THE CORRECT STRUCTURE:** Create the appropriate directory structure and essential files based on the project type you identified.
                   - **PRIORITIZE PLAN FILES:** Your first priority is to create the functional files mentioned in the plan (like `wp-chatbot.php` for a WordPress plugin), NOT generic `index.html`.
                   - **ADD PROFESSIONAL ESSENTIALS:** In addition to the functional files, ALWAYS include a `README.md` and a `.gitignore` file for a complete setup.
                   - **WORDPRESS PLUGINS:** For a plugin, this means creating the main `plugin-name.php` and appropriate subdirectories like `/admin`, `/includes`, and `/assets` if the plan suggests them. You may add empty `index.php` files inside these new directories for security, but this is secondary to creating the functional files.\n";
                $base_prompt .= "3. **IMPLEMENT THE FIRST FEATURE:** After creating the structure, implement the complete code for the first logical feature from the plan.\n";
                $base_prompt .= "4. **DELIVER A WORKING START:** The output must be a fully functional, professionally structured project foundation that is ready for the next step.\n\n";
                // --- END OF THE FIX ---
            }

        } else if ($is_step_execution) {
            $base_prompt .= "**HIERARCHY OF RULES (PLAN MODE - FOCUSED EXECUTION):**\n";
            $base_prompt .= "1.  **TOP PRIORITY - TRUST THE PLAN:** Execute the current step exactly as specified in the development plan.\n";
            $base_prompt .= "2.  **CONSTRAINT AWARENESS:** " . ($is_three_file_mode ? 
                "STRICTLY enforce 3-file constraint. All work must be in index.html, style.css, script.js." : 
                "You may create additional files as needed for professional structure.") . "\n";
            $base_prompt .= "3.  **WORKING FUNCTIONALITY:** Ensure the current step produces complete, working code that can be tested and used.\n";
            $base_prompt .= "4.  **DESIGN QUALITY:** Apply good design practices while maintaining functional integrity.\n";
            $base_prompt .= "5.  **RESPONSIVE FINISH:** If this is the final step and involves responsive design, review ALL CSS comprehensively to ensure mobile-first styling across all components.\n\n";
        }

        $base_prompt .= "**PROJECT CONSTRAINTS & MODE:**\n";
        if ($is_three_file_mode) {
            $base_prompt .= "🔒 **3-FILE MODE - STRICT ENFORCEMENT**\n";
            $base_prompt .= "• ALL work must be contained in: index.html, style.css, script.js\n";
            $base_prompt .= "• You are FORBIDDEN from creating any additional files\n";
            $base_prompt .= "• All views, components, and features must be implemented within these 3 files\n";
            $base_prompt .= "• Use view-switching, modals, or hidden sections for multiple 'pages'\n";
            $base_prompt .= "• This constraint applies to EVERY step of this project\n\n";
        } else {
            $base_prompt .= "📁 **FULL PROJECT MODE:** You may create additional files as needed for a complete professional project structure.\n\n";
        }

        $base_prompt .= "**CRITICAL PRE-FLIGHT ANALYSIS (PERFORM BEFORE WRITING CODE):**\n";
        $base_prompt .= "0.  **CONSTRAINT CHECK:** First, determine if this is a 3-file constrained project. If so, ALL work must be in index.html, style.css, script.js.\n";
        $base_prompt .= "1.  **IDENTIFY INTENT:** Is this a **[Code Generation]** request or a **[Code Explanation]** request?\n";
        $base_prompt .= "    - **[Code Explanation] keywords:** 'explain', 'what does', 'why is', 'how does', 'tell me about', 'walk me through', '?'. If this is an explanation request, your ONLY job is to provide a clear, conversational answer in markdown. **DO NOT output JSON or file blocks.** Your analysis stops here.\n";
        $base_prompt .= "    - **[Code Generation]:** If the request asks to 'create', 'build', 'add', 'modify', 'fix', or 'change' code, proceed to the next analysis steps.\n\n";
        $base_prompt .= "2.  **WEBSITE vs. WEB APP:** Determine the project type.\n";
        $base_prompt .= "    - **[Website]:** Portfolio, blog, landing page. Focus on layout, content, hero sections.\n";
        $base_prompt .= "    - **[Web App]:** Tool, manager, editor. Focus on functionality, data display, user interaction. DO NOT add website elements like hero sections unless asked.\n\n";
        $base_prompt .= "3.  **DETERMINE TASK DOMAIN:**\n";
        $base_prompt .= "    - **[Backend]:** Keywords: 'API', 'server', 'database', 'model'. Focus ONLY on server-side files.\n";
        $base_prompt .= "    - **[Framework]:** Keywords: 'React', 'Vue', 'JSX'. You MUST write code in that framework's syntax.\n";
        $base_prompt .= "    - **Default is [Frontend]:** Assume HTML, CSS, and vanilla JavaScript.\n\n";
        $base_prompt .= "4.  **DETERMINE SCOPE:**\n";
        $base_prompt .= "    - **[Abstract Structure]:** Keywords: 'model', 'class', 'data structure'. Implement ONLY the requested logic (e.g., a JS class) and DO NOT build a UI.\n";
        $base_prompt .= "    - **[Visual Component]:** Assume a full, functional, and styled UI component is requested.\n\n";
        $base_prompt .= "5.  **SCOPE-BASED IMPLEMENTATION:**\n";
        $base_prompt .= "    - **[Minimal Viable]:** For complex features, implement the core functionality first. Additional enhancements can be added in follow-up steps.\n";
        $base_prompt .= "    - **[Complete Feature]:** Ensure all user-requested functionality works end-to-end, even if some optimizations are deferred.\n\n";

        $base_prompt .= "**INTELLIGENT FILE SELECTION & CONTEXT MANAGEMENT:**\n";
        $base_prompt .= "1.  **CONSTRAINT AWARENESS:** " . ($is_three_file_mode ? 
            "In 3-file mode, you may ONLY use index.html, style.css, script.js. No exceptions." : 
            "In full project mode, you may create additional files as needed.") . "\n";
        $base_prompt .= "2.  **TRIAD CHECK:** For any UI component or visual feature, always include ALL three core files: HTML (structure), CSS (styling), and JavaScript (behavior). Never work on one in isolation.\n";
        $base_prompt .= "3.  **HTML INTEGRATION:** When modifying CSS or JavaScript files, ALWAYS include the primary HTML file to ensure proper linking and integration.\n";
        $base_prompt .= "4.  **DATA DEPENDENCY:** For data manipulation tasks (filtering, sorting, searching), include the data source file (e.g., script.js with data arrays/functions) to understand the data structure.\n";
        $base_prompt .= "5.  **GLOBAL DEPENDENCIES:** When creating new UI components, include global/base files (reset.css, main.css, utils.js) that the component depends on for consistent styling and behavior.\n";
        $base_prompt .= "6.  **LOGIC-ONLY FOCUS:** For tasks explicitly about behavior or logic (e.g., 'handle submission', 'validate input'), focus primarily on the JavaScript file unless UI changes are explicitly requested.\n";
        $base_prompt .= "7.  **PROJECT AWARENESS:** Always consider the entire project context, not just the immediate files being modified. Understand how changes integrate with existing functionality.\n";
        $base_prompt .= "8.  **PRECISE SCOPE MATCHING:** Match file selection precisely to the task scope:\n";
        $base_prompt .= "    - **CSS-Only Tasks:** When working purely on styles/variables, focus on HTML + CSS only\n";
        $base_prompt .= "    - **Logic-Only Tasks:** When implementing data/models/utilities, focus on JavaScript only\n";
        $base_prompt .= "    - **UI Components:** When building interactive features, use the full triad (HTML+CSS+JS)\n";
        $base_prompt .= "9.  **RESPONSIVE FINISH:** When executing the final responsive design step, review ALL CSS and ensure comprehensive mobile-first styling across all components.\n\n";

        $base_prompt .= "**3-FILE MODE ARCHITECTURE GUIDELINES:**\n";
        if ($is_three_file_mode) {
            $base_prompt .= "• **Single HTML File:** All views, modals, forms, and sections must be in index.html\n";
            $base_prompt .= "• **View Switching:** Use CSS classes and JavaScript to show/hide different sections\n";
            $base_prompt .= "• **Component Organization:** Structure HTML with clear comments separating different components\n";
            $base_prompt .= "• **CSS Organization:** Use clear commenting and section headers in style.css\n";
            $base_prompt .= "• **JS Modularity:** Organize JavaScript with clear function grouping and comments\n";
            $base_prompt .= "• **No File Proliferation:** Do NOT suggest, create, or reference profile.html, dashboard.html, etc.\n\n";
        }

        $base_prompt .= "**TECHNICAL IMPLEMENTATION GUIDELINES:**\n";
        $base_prompt .= "- **Working First, Polish Second:** Ensure core functionality works before adding advanced features\n";
        $base_prompt .= "- **Progressive Enhancement:** Build basic functionality that works, then layer on complexity\n";
        $base_prompt .= "- **Error Handling:** Include basic validation and error prevention where appropriate\n";
        $base_prompt .= "- **Browser Compatibility:** Ensure code works in modern browsers without advanced features\n";
        $base_prompt .= "- **Complete Features:** When implementing a feature, ensure it's fully usable, not just partially built\n";
        if (!$is_three_file_mode) {
            $base_prompt .= "- **Professional Structure:** Include proper documentation and project management files for complete projects\n";
        }
        $base_prompt .= "- **Integration Testing:** Ensure new code works with existing functionality and doesn't break previous features\n";
        $base_prompt .= "- **Responsive Excellence:** Always implement mobile-first design with comprehensive media queries\n\n";

        $base_prompt .= "**AGENCY DESIGN PHILOSOPHY (YOUR CORE PRINCIPLES):**\n";
        $base_prompt .= "- **Functionality First:** All features must work completely before aesthetic polish is applied.\n";
        if (!$is_three_file_mode) {
            $base_prompt .= "- **Professional Standards:** For project foundations, include comprehensive setup with documentation and version control.\n";
        }
        $base_prompt .= "- **Aesthetics Matter:** Designs should be intentional and polished, using modern trends like subtle gradients, soft shadows, and ample white space.\n";
        $base_prompt .= "- **Premium Feel:** The project should feel high-quality, with professional color palettes and elegant typography.\n";
        $base_prompt .= "- **Micro-Interactions Matter:** Incorporate subtle hover effects, focus states, and smooth transitions to feel alive.\n";
        $base_prompt .= "- **Mobile-First Always:** The design must be flawlessly responsive.\n";
        $base_prompt .= "- **No Ugly Defaults:** Custom, beautiful styling should be specified for all elements, especially forms and buttons.\n";
        $base_prompt .= "- **Feature Integrity:** When creating a visual feature, implement all parts (HTML, CSS, JS) to make it fully functional and polished, but prioritize functionality over perfection.\n\n";

        $base_prompt .= "**MANDATORY NAVIGATION BLUEPRINT:**\n";
        $base_prompt .= "If the project requires navigation, you MUST implement this exact responsive pattern: Desktop nav becomes a hamburger menu on mobile, which toggles a full-screen overlay or slide-out panel via JavaScript class toggling.\n\n";

        if (!empty($project_context)) {
            $base_prompt .= "--- PROJECT CONTEXT ---\n";
            $base_prompt .= "ORIGINAL VISION: " . htmlspecialchars($project_context['originalVision']) . "\n";
            $base_prompt .= "DESIGN REQUIREMENTS: " . htmlspecialchars($project_context['designRequirements']) . "\n";
            if (isset($project_context['constraints'])) {
                $base_prompt .= "CONSTRAINTS: " . htmlspecialchars($project_context['constraints']) . "\n";
            }
            $base_prompt .= "--- END PROJECT CONTEXT ---\n\n";
        }

        if ($is_step_execution && !empty($specification_sheet)) {
            $base_prompt .= "--- FOCUSED CONTEXT FROM SPECIFICATION ---\n";
            $base_prompt .= is_string($specification_sheet) ? htmlspecialchars($specification_sheet) : json_encode($specification_sheet, JSON_PRETTY_PRINT);
            $base_prompt .= "\n--- END OF CONTEXT ---\n\n";
        }

        $base_prompt .= "**STRICT OUTPUT RULES (FOR CODE GENERATION ONLY):**\n";
        $base_prompt .= "Your entire response MUST follow this exact format. You MUST start with the JSON summary block. After the JSON, you MUST ONLY output file blocks. Absolutely NO other text or explanations are allowed.\n\n";
        
        $base_prompt .= "**EXAMPLE OF A PERFECT RESPONSE:**\n";
        $base_prompt .= "```json\n";
        $base_prompt .= "{\n";
        $base_prompt .= "  \"summary\": \"Added a premium hero section with a gradient background and subtle animation.\",\n";
        $base_prompt .= "  \"files_used\": [\"index.html\", \"style.css\"]\n";
        $base_prompt .= "}\n";
        $base_prompt .= "```\n";
        $base_prompt .= "File: index.html (MODIFY)\n";
        $base_prompt .= "```html\n";
        $base_prompt .= "<!DOCTYPE html>\n<html>\n<body>\n    <h1>Existing Content</h1>\n    <section class=\"hero premium-hero\">New hero content</section>\n</body>\n</html>\n";
        $base_prompt .= "```\n";
        $base_prompt .= "File: style.css (NEW)\n";
        $base_prompt .= "```css\n";
        $base_prompt .= ".premium-hero { background: linear-gradient(45deg, #6a11cb, #2575fc); padding: 100px 50px; color: white; transition: transform 0.3s ease; }\n";
        $base_prompt .= ".premium-hero:hover { transform: scale(1.02); }\n";
        $base_prompt .= "```\n\n";

        $base_prompt .= "**CRITICAL TECHNICAL CONSTRAINTS (FOR CODE GENERATION ONLY):**\n";
        $base_prompt .= "1.  **COMPLETE, WORKING IMPLEMENTATIONS:** Each file you output must be syntactically correct and functionally complete for the current scope. You may focus on core functionality first and refine in subsequent steps, but never output broken code or placeholder comments like '// TODO' or '// Implement later'. Every implementation must be usable and testable.\n";
        if ($is_three_file_mode) {
            $base_prompt .= "2.  **3-FILE CONSTRAINT:** You are STRICTLY LIMITED to index.html, style.css, script.js. Creating any other files is FORBIDDEN.\n";
            $base_prompt .= "3.  **REQUEST COMPLETENESS:** You MUST implement ALL parts of the user's request within the 3-file constraint.\n";
        } else {
            $base_prompt .= "2.  **REQUEST COMPLETENESS:** You MUST implement ALL parts of the user's request.\n";
            $base_prompt .= "3.  **PROFESSIONAL PROJECT STRUCTURE:** For scaffolding steps, include comprehensive project setup with documentation (README.md) and version control files (.gitignore) when appropriate.\n";
        }
        $base_prompt .= "4.  **NO CONVERSATION:** Your response MUST NOT contain any text outside the specified JSON and file blocks.\n";
        $base_prompt .= "5.  **START WITH JSON:** The very first part of your response MUST be the ` ```json ` block.\n";
        $base_prompt .= "6.  **COMPLETE CODE ONLY:** For any file you modify, you MUST provide the entire, complete file content.\n";
        $base_prompt .= "7.  **JAVASCRIPT SAFETY:** Wrap all DOM-interacting JS in a `DOMContentLoaded` listener.\n";
        $base_prompt .= "8.  **FUNCTIONAL TESTING:** Ensure that any feature you implement can be tested and used immediately after your code is applied.\n";
        $base_prompt .= "9.  **INTELLIGENT INTEGRATION:** Always ensure new code integrates properly with existing functionality. Test your changes in the context of the whole project.\n";
        if ($is_three_file_mode) {
            $base_prompt .= "10. **ARCHITECTURAL INTEGRITY:** Design features to work within the 3-file constraint using view switching and proper organization.\n";
        }
        $base_prompt .= "11. **RESPONSIVE EXCELLENCE:** When implementing responsive design, ensure comprehensive mobile-first styling across all components with proper media queries.\n";
    }

    if (!empty($kb_context)) {
        $base_prompt .= "\n\n--- RELEVANT CONTEXT FROM YOUR PERSONAL KNOWLEDGE BASE ---\n"
            . $kb_context . "\n\n";
    }

    if (!empty($project_structure)) {
        $base_prompt .= "This is the complete file structure of the current project:\n\n"
            . "--- PROJECT FILE MANIFEST ---\n"
            . $project_structure . "\n"
            . "--- END OF MANIFEST ---\n\n";
    }
    
    if (!$is_plan_mode && !$is_step_execution && empty($specification_sheet)) {
        if (!empty($project_files)) {
            $base_prompt .= "Here is the full content of the text files the user has selected for context:\n\n";
            foreach ($project_files as $path => $content) {
                $base_prompt .= "=== START OF FILE: " . htmlspecialchars($path) . " ===\n"
                    . $content . "\n"
                    . "=== END OF FILE: " . htmlspecialchars($path) . " ===\n\n";
            }
        }
    }

    return $base_prompt;
}
// ADD THIS ENTIRE NEW FUNCTION

/**
 * Determines if a user's prompt is conversational or a code command.
 */
private function is_conversational_request($prompt) {
    $prompt_lower = strtolower(trim($prompt));

    // Common question starters
    $question_starters = [
        'what is', 'what are', 'how do', 'how does', 'why is', 'can you explain',
        'could you explain', 'tell me about', 'describe', 'explain the difference'
    ];
    foreach ($question_starters as $starter) {
        if (strpos($prompt_lower, $starter) === 0) {
            return true;
        }
    }

    // Ends with a question mark
    if (substr($prompt_lower, -1) === '?') {
        return true;
    }

    // Contains advisory keywords
    if (preg_match('/\b(idea for|opinion on|best way to|recommend|advise)\b/i', $prompt_lower)) {
        return true;
    }

    // Short prompts without any "action" verbs are likely questions
    if (strlen($prompt_lower) < 70 && !preg_match('/\b(create|build|make|add|modify|fix|update|change|implement|write)\b/i', $prompt_lower)) {
        return true;
    }

    return false;
}

  private function handle_prompt_refinement() {
    $user_id = $this->validate_ajax_request();

    $prompt = sanitize_textarea_field($_POST['prompt']);
    if (empty($prompt)) {
        wp_send_json_error(['message' => 'Prompt cannot be empty.'], 400);
        wp_die();
    }

    try {
        $refinement_system_prompt = "You are an expert prompt engineer. Your task is to rewrite the user's request to be more effective for an AI web developer that generates single-page websites. Focus on adding descriptive visual details (like colors, mood, and layout), clarifying functionality, and adopting a clear, direct tone. The refined prompt should be concise and under 75 words. Return ONLY the refined prompt and nothing else, no conversational text.";
        
        // NEW: Cache refinement requests
      $cache_key = $this->get_cache_key(['prompt' => $prompt], 'refinement');
        
        $refined_prompt = $this->get_cached_response($cache_key, function() use ($refinement_system_prompt, $prompt) {
            $messages = [
                ['role' => 'system', 'content' => $refinement_system_prompt],
                ['role' => 'user', 'content' => $prompt]
            ];

            $api_key = $this->validate_api_key('google');
            $model = get_option('zencode_ai_default_model', 'google/gemini-2.5-pro');
            return $this->call_google_api($api_key, $model, $messages);
        }, $this->get_cache_duration('refinement'));

        $token = wp_generate_password(32, false);
        set_transient('zc_ref_token_' . $user_id, $token, 5 * MINUTE_IN_SECONDS);

        wp_send_json_success([
            'refined_prompt' => trim($refined_prompt),
            'refinement_token' => $token 
        ]);

    } catch (Exception $e) {
        $this->handle_api_error($e, 'Refinement');
    }
    wp_die();
}
public function __construct() {
    add_action('admin_menu', [$this, 'add_admin_menu']);
    add_action('admin_head', [$this, 'hide_admin_page_titles']);
    add_action('admin_init', [$this, 'register_settings']);
    add_action('wp_ajax_zencode_ai_call_ai_api', [$this, 'handle_ai_request']);
    add_action('wp_ajax_zencode_ai_vectorize_file', [$this, 'handle_vectorize_file']);
    add_action('wp_ajax_zencode_ai_delete_vector', [$this, 'handle_delete_vector']);
    add_action('wp_ajax_zencode_ai_clear_user_vectors', [$this, 'handle_clear_user_vectors']);
    add_action('wp_ajax_zencode_ai_save_project', [$this, 'handle_save_project']);
    add_action('wp_ajax_zencode_ai_load_project', [$this, 'handle_load_project']);
    add_action('wp_ajax_zencode_ai_get_projects_list', [$this, 'handle_get_projects_list']);
	 add_action('wp_ajax_zencode_ai_validate_step_count', [$this, 'handle_validate_step_count']); // NEW: Step validation

	
	add_action('wp_ajax_zencode_ai_migrate_learning_data', [$this, 'handle_migrate_learning_data']);

    // Learning system hooks
   add_action('wp_ajax_zencode_ai_save_learning', [$this, 'handle_save_learning']);
    add_action('wp_ajax_zencode_ai_get_learnings', [$this, 'handle_get_learnings']);
    add_action('wp_ajax_zencode_ai_update_learning_usage', [$this, 'handle_update_learning_usage']);
    add_action('wp_ajax_zencode_ai_get_learning_stats', [$this, 'handle_get_learning_stats']);
    add_action('wp_ajax_zencode_ai_cleanup_learnings', [$this, 'handle_cleanup_learnings']);
    add_action('wp_ajax_zencode_ai_delete_all_learnings', [$this, 'handle_delete_all_learnings']);
    add_action('wp_ajax_zencode_ai_delete_learnings', [$this, 'handle_delete_learnings']);
	add_action('wp_ajax_zencode_ai_get_learning_history', [$this, 'handle_get_learning_history']);

    add_action('wp_ajax_zencode_ai_triage_context', [$this, 'handle_triage_context']);
    add_action('wp_ajax_zencode_ai_delete_project', [$this, 'handle_delete_project']);
    add_action('wp_ajax_zencode_ai_clear_cache', [$this, 'handle_clear_cache']);
    add_action('wp_ajax_zencode_ai_generate_spec', [$this, 'handle_generate_specification']);
    add_action('wp_ajax_zencode_ai_version_control', [$this, 'handle_version_control']);
    add_action('wp_ajax_zencode_ai_prune_spec', [$this, 'handle_prune_specification']);
    add_action('wp_ajax_zencode_ai_validate_step_count', [$this, 'handle_validate_step_count']); // NEW: Step validation
    add_action('wp_ajax_zencode_ai_stream_response', [$this, 'handle_stream_response']); // FIXED: Streaming endpoint

    $this->setup_cache_invalidation();

    add_shortcode('zencode_ai_app', [$this, 'render_app_shortcode']);
    register_activation_hook(__FILE__, [$this, 'activate']);
}
	
	// Add this method
public function handle_clear_cache() {
    $this->validate_ajax_request(true); // Admin only
    
    $pattern = sanitize_text_field($_POST['pattern'] ?? '');
    
    if (!empty($pattern)) {
        $this->clear_cache($pattern);
        $message = "Cache cleared for pattern: {$pattern}";
    } else {
        $this->clear_cache();
        $message = 'All cache cleared successfully';
    }
    
    // Get cache stats after clearing
    $stats = $this->get_cache_stats();
    
    wp_send_json_success([
        'message' => $message,
        'stats' => $stats
    ]);
}

public function handle_cleanup_learnings() {
    $this->validate_ajax_request();
    
    $threshold = isset($_POST['threshold']) ? intval($_POST['threshold']) : 3;
    $user_id = get_current_user_id();
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    $deleted = $wpdb->query($wpdb->prepare(
        "DELETE FROM {$table_name} WHERE user_id = %d AND score < %d",
        $user_id,
        $threshold
    ));
    
    wp_send_json_success([
        'deleted' => $deleted,
        'message' => "Removed {$deleted} low-scoring learnings"
    ]);
}

/**
 * Handle deletion of ALL learnings (nuclear option)
 */
public function handle_delete_all_learnings() {
    $this->validate_ajax_request();
    
    $user_id = get_current_user_id();
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    $deleted = $wpdb->delete($table_name, ['user_id' => $user_id], ['%d']);
    
    wp_send_json_success([
        'deleted' => $deleted,
        'message' => "Deleted all {$deleted} learnings. Your AI is starting fresh."
    ]);
}

/**
 * Handle deletion of specific learnings by ID
 */
public function handle_delete_learnings() {
    $this->validate_ajax_request();
    
    $ids = isset($_POST['ids']) ? array_map('intval', $_POST['ids']) : [];
    
    if (empty($ids)) {
        wp_send_json_error(['message' => 'No learning IDs provided']);
    }
    
    $user_id = get_current_user_id();
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    $placeholders = implode(',', array_fill(0, count($ids), '%d'));
    $query = $wpdb->prepare(
        "DELETE FROM {$table_name} WHERE user_id = %d AND id IN ({$placeholders})",
        array_merge([$user_id], $ids)
    );
    
    $deleted = $wpdb->query($query);
    
    wp_send_json_success([
        'deleted' => $deleted,
        'message' => "Deleted {$deleted} learnings"
    ]);
}

/**
 * Get learning history for chart
 */
public function handle_get_learning_history() {
    $this->validate_ajax_request();
    
    global $wpdb;
    $table = $wpdb->prefix . 'zencode_ai_learnings';
    $user_id = get_current_user_id();
    
    $results = $wpdb->get_results($wpdb->prepare(
        "SELECT DATE(created_at) as date, AVG(score) as avg_score 
         FROM {$table} 
         WHERE user_id = %d AND score > 0
         GROUP BY DATE(created_at)
         ORDER BY date ASC",
        $user_id
    ));
    
    if (empty($results)) {
        wp_send_json_success([
            'dates' => [],
            'scores' => [],
            'message' => 'No learning data yet'
        ]);
        wp_die();
    }
    
    $dates = [];
    $scores = [];
    
    foreach ($results as $row) {
        $dates[] = date('M j', strtotime($row->date));
        $scores[] = round(floatval($row->avg_score), 1);
    }
    
    wp_send_json_success([
        'dates' => $dates,
        'scores' => $scores
    ]);
}
public function activate() {
    // Start output buffering to prevent "headers already sent" errors
    ob_start();
    
    global $wpdb;
    $charset_collate = $wpdb->get_charset_collate();
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    
    // Vector Knowledge Base table (KEEP - core feature)
    $kb_table_name = $wpdb->prefix . 'zencode_ai_kb_vectors';
    $sql_kb = "CREATE TABLE IF NOT EXISTS $kb_table_name (
        id BIGINT(20) NOT NULL AUTO_INCREMENT,
        user_id BIGINT(20) NOT NULL,
        file_path VARCHAR(255) DEFAULT '' NOT NULL,
        source_text TEXT NOT NULL,
        embedding LONGTEXT NOT NULL,
        PRIMARY KEY (id),
        KEY user_id_index (user_id)
    ) $charset_collate;";
    dbDelta($sql_kb);
    
    // Projects table (KEEP - core feature)
    $projects_table_name = $wpdb->prefix . 'zencode_ai_projects';
    $sql_projects = "CREATE TABLE IF NOT EXISTS $projects_table_name (
        id BIGINT(20) NOT NULL AUTO_INCREMENT,
        user_id BIGINT(20) NOT NULL,
        project_name VARCHAR(255) NOT NULL,
        project_data LONGTEXT NOT NULL,
        last_modified DATETIME NOT NULL,
        PRIMARY KEY (id),
        UNIQUE KEY user_project (user_id, project_name)
    ) $charset_collate;";
    dbDelta($sql_projects);
    
    // Project versions table (KEEP - core feature)
    $versions_table_name = $wpdb->prefix . 'zencode_ai_project_versions';
    $sql_versions = "CREATE TABLE IF NOT EXISTS $versions_table_name (
        id BIGINT(20) NOT NULL AUTO_INCREMENT,
        user_id BIGINT(20) NOT NULL,
        project_name VARCHAR(255) NOT NULL,
        version_name VARCHAR(255) NOT NULL,
        version_description TEXT,
        project_data LONGTEXT NOT NULL,
        created_at DATETIME NOT NULL,
        is_auto_snapshot TINYINT(1) DEFAULT 0,
        snapshot_trigger VARCHAR(100) DEFAULT NULL,
        PRIMARY KEY (id),
        KEY user_project_index (user_id, project_name),
        KEY created_at_index (created_at)
    ) $charset_collate;";
    dbDelta($sql_versions);
    
    // Learning system table (KEEP - core feature)
    $learnings_table = $wpdb->prefix . 'zencode_ai_learnings';
    $sql_learnings = "CREATE TABLE IF NOT EXISTS $learnings_table (
        id BIGINT(20) NOT NULL AUTO_INCREMENT,
        user_id BIGINT(20) NOT NULL,
        learning_type VARCHAR(50) NOT NULL,
        pattern_data LONGTEXT NOT NULL,
        score FLOAT DEFAULT 0,
        success_rate FLOAT DEFAULT 0,
        confidence_score FLOAT DEFAULT 0,
        times_used INT(11) DEFAULT 0,
        step_pattern TEXT DEFAULT NULL,
        rule TEXT DEFAULT NULL,
        mistake_made TEXT DEFAULT NULL,
        last_used DATETIME DEFAULT NULL,
        usage_count INT(11) DEFAULT 0,
        created_at DATETIME NOT NULL,
        PRIMARY KEY (id),
        KEY user_id (user_id),
        KEY learning_type (learning_type),
        KEY success_rate (success_rate),
        KEY score (score)
    ) $charset_collate;";
    dbDelta($sql_learnings);
    
    // Default plugin options (KEEP - but simplified)
    $default_options = [
        'google_api_key' => '',
        'default_model' => 'google/gemini-2.5-pro',
        'max_tokens' => 12288,
        'temperature' => 0.7
    ];
    
    foreach ($default_options as $key => $value) {
        if (get_option('zencode_ai_' . $key) === false) {
            add_option('zencode_ai_' . $key, $value);
        }
    }
    
    // Customer version - no license tables created locally
    // All license validation is done via API to your license server
    
    // Set activation flag
    update_option('zencode_ai_activated', true);
    update_option('zencode_ai_activation_time', current_time('mysql'));
    
    // Clean output buffer and discard any output
    ob_end_clean();
}
	

public function handle_save_learning() {
    $user_id = $this->validate_ajax_request();
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    $learning_type = sanitize_text_field($_POST['learning_type'] ?? 'step_reflection');
    $pattern_data_raw = stripslashes($_POST['pattern_data'] ?? '{}');
    $pattern_data_decoded = json_decode($pattern_data_raw, true);
    
    // Validate JSON
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($pattern_data_decoded)) {
        wp_send_json_error(['message' => 'Invalid pattern data JSON.'], 400);
        wp_die();
    }
    
    $success_rate = floatval($_POST['success_rate'] ?? 0);
    $confidence_score = floatval($_POST['confidence_score'] ?? 0);
    
    // ============= EXTRACTION LOGIC BASED ON LEARNING TYPE =============
    $step_pattern = '';
    $rule = '';
    $mistake_made = '';
    $score = 0;
    
    // HANDLE STEP REFLECTIONS (Type 1)
    if ($learning_type === 'step_reflection' && isset($pattern_data_decoded['reflection'])) {
        $reflection = $pattern_data_decoded['reflection'];
        
        // Extract from reflection object
        $score = floatval($reflection['self_rating'] ?? 0);
        $rule = $reflection['lesson_learned'] ?? '';
        $mistake_made = $reflection['what_went_wrong'] ?? '';
        
        // Extract step text
        $step_pattern = $pattern_data_decoded['stepText'] ?? '';
        
        // Build a more descriptive step pattern if we have context
        if (!empty($step_pattern) && isset($pattern_data_decoded['context'])) {
            $context = $pattern_data_decoded['context'];
            $files_changed = [];
            if (isset($context['fileChanges']['newFiles'])) {
                $files_changed = array_merge($files_changed, $context['fileChanges']['newFiles']);
            }
            if (isset($context['fileChanges']['modifiedFiles'])) {
                $files_changed = array_merge($files_changed, $context['fileChanges']['modifiedFiles']);
            }
            
            if (!empty($files_changed)) {
                $step_pattern .= " [Files: " . implode(', ', array_slice($files_changed, 0, 3)) . "]";
            }
        }
        
        // Add why_it_happened to mistake if available
        if (!empty($reflection['why_it_happened'])) {
            if (!empty($mistake_made)) {
                $mistake_made .= " | Cause: " . $reflection['why_it_happened'];
            } else {
                $mistake_made = $reflection['why_it_happened'];
            }
        }
    }
    
    // HANDLE MASTER PLAN REFLECTIONS (Type 2)
    elseif ($learning_type === 'master_plan_reflection' && isset($pattern_data_decoded['masterLearning'])) {
        $master = $pattern_data_decoded['masterLearning'];
        
        // Extract from masterLearning object
        $score = floatval($master['overall_grade'] ?? 0);
        $rule = $master['will_remember'] ?? $master['key_improvement'] ?? '';
        $mistake_made = $master['biggest_mistake'] ?? '';
        
        // Create a step pattern from master learning context
        $step_pattern = "Plan Completion - Overall Grade: " . $score . "/10";
        
        if (!empty($master['mistake_pattern'])) {
            $step_pattern .= " | Pattern: " . substr($master['mistake_pattern'], 0, 100);
        }
        
        // Add plan context if available
        if (isset($pattern_data_decoded['planContext'])) {
            $context = $pattern_data_decoded['planContext'];
            if (isset($context['totalSteps'])) {
                $step_pattern .= " | Steps: " . $context['totalSteps'];
            }
            if (isset($context['projectType'])) {
                $step_pattern .= " | Type: " . substr($context['projectType'], 0, 50);
            }
        }
    }
    
    // FALLBACK: Handle any other structure
    else {
        // Try to find data in any common location
        if (isset($pattern_data_decoded['lesson_learned'])) {
            $rule = $pattern_data_decoded['lesson_learned'];
        } elseif (isset($pattern_data_decoded['rule'])) {
            $rule = $pattern_data_decoded['rule'];
        }
        
        if (isset($pattern_data_decoded['mistake'])) {
            $mistake_made = $pattern_data_decoded['mistake'];
        } elseif (isset($pattern_data_decoded['error'])) {
            $mistake_made = $pattern_data_decoded['error'];
        }
        
        if (isset($pattern_data_decoded['step_text'])) {
            $step_pattern = $pattern_data_decoded['step_text'];
        } elseif (isset($pattern_data_decoded['description'])) {
            $step_pattern = $pattern_data_decoded['description'];
        }
        
        // If still empty, create from available data
        if (empty($step_pattern)) {
            $step_pattern = 'Learning recorded: ' . $learning_type;
        }
    }
    
    // Calculate score from multiple sources if not set
    if ($score == 0 && $success_rate > 0) {
        $score = $success_rate * 10; // Convert 0-1 to 0-10 scale
    } elseif ($score == 0 && $confidence_score > 0) {
        $score = $confidence_score * 10; // Convert 0-1 to 0-10 scale
    }
    
    // Sanitize and ensure reasonable lengths
    $step_pattern = sanitize_textarea_field(substr($step_pattern, 0, 1000));
    $rule = sanitize_textarea_field(substr($rule, 0, 1000));
    $mistake_made = sanitize_textarea_field(substr($mistake_made, 0, 1000));
    
    // Ensure we have SOMETHING to display
    if (empty($rule)) {
        $rule = 'Learning pattern recorded';
    }
    if (empty($step_pattern)) {
        $step_pattern = $learning_type . ' - ' . date('Y-m-d H:i:s');
    }
    // ============= END EXTRACTION LOGIC =============
    
    $result = $wpdb->insert(
        $table_name,
        [
            'user_id' => $user_id,
            'learning_type' => $learning_type,
            'pattern_data' => wp_json_encode($pattern_data_decoded), // Re-encode cleanly
            'score' => $score,
            'success_rate' => $success_rate,
            'confidence_score' => $confidence_score,
            'step_pattern' => $step_pattern,
            'rule' => $rule,
            'mistake_made' => $mistake_made,
            'usage_count' => 0,
            'times_used' => 0,
            'created_at' => current_time('mysql')
        ],
        ['%d', '%s', '%s', '%f', '%f', '%f', '%s', '%s', '%s', '%d', '%d', '%s']
    );
    
    if ($result) {
        wp_send_json_success([
            'learning_id' => $wpdb->insert_id,
            'message' => 'Learning saved successfully'
        ]);
    } else {
        wp_send_json_error(['message' => 'Failed to save learning: ' . $wpdb->last_error]);
    }
    wp_die();
}

public function handle_migrate_learning_data() {
    $this->validate_ajax_request(true); // Require admin
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    // Get all learnings with empty step_pattern or rule
    $learnings = $wpdb->get_results(
        "SELECT id, pattern_data, success_rate 
         FROM {$table_name} 
         WHERE (step_pattern IS NULL OR step_pattern = '' OR rule IS NULL OR rule = '') 
         AND pattern_data IS NOT NULL"
    );
    
    $updated = 0;
    $failed = 0;
    
    foreach ($learnings as $learning) {
        $pattern_data = json_decode($learning->pattern_data, true);
        
        if (!is_array($pattern_data)) {
            $failed++;
            continue;
        }
        
        // Extract fields
        $step_pattern = '';
        $rule = '';
        $mistake_made = '';
        $score = 0;
        
        if (isset($pattern_data['reflection'])) {
            $reflection = $pattern_data['reflection'];
            $score = floatval($reflection['self_rating'] ?? 0);
            $rule = $reflection['lesson_learned'] ?? '';
            $mistake_made = $reflection['mistake_made'] ?? '';
        }
        
        if (isset($pattern_data['stepText'])) {
            $step_pattern = $pattern_data['stepText'];
        } elseif (isset($pattern_data['step_text'])) {
            $step_pattern = $pattern_data['step_text'];
        }
        
        // Derive score from success_rate if needed
        if ($score == 0 && $learning->success_rate > 0) {
            $score = $learning->success_rate * 10;
        }
        
        // Update the record
        $result = $wpdb->update(
            $table_name,
            [
                'step_pattern' => sanitize_textarea_field($step_pattern),
                'rule' => sanitize_textarea_field($rule),
                'mistake_made' => sanitize_textarea_field($mistake_made),
                'score' => $score
            ],
            ['id' => $learning->id],
            ['%s', '%s', '%s', '%f'],
            ['%d']
        );
        
        if ($result !== false) {
            $updated++;
        } else {
            $failed++;
        }
    }
    
    wp_send_json_success([
        'message' => "Migration complete: {$updated} updated, {$failed} failed",
        'updated' => $updated,
        'failed' => $failed
    ]);
}
public function handle_get_learnings() {
    $user_id = $this->validate_ajax_request();
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    $step_text = sanitize_text_field($_POST['step_text'] ?? '');
    $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 5;
    $learning_type = sanitize_text_field($_POST['learning_type'] ?? 'step_reflection');
    
    if (empty($step_text)) {
        wp_send_json_success(['learnings' => [], 'count' => 0]);
        wp_die();
    }
    
    // Clean and extract meaningful keywords
    $clean_text = preg_replace('/[^a-zA-Z0-9\s]/', ' ', $step_text);
    $words = array_filter(
        array_map('strtolower', array_unique(explode(' ', $clean_text))),
        function($word) {
            $stopwords = ['the', 'and', 'with', 'from', 'that', 'this', 'will', 'have', 'been', 'for', 'create', 'add', 'make', 'update', 'modify'];
            return strlen($word) >= 4 && !in_array($word, $stopwords);
        }
    );
    
    if (empty($words)) {
        wp_send_json_success(['learnings' => [], 'count' => 0]);
        wp_die();
    }
    
    // Build precise search conditions
    $conditions = [];
    $placeholders = [];
    
    foreach ($words as $word) {
        // Search in stepText within pattern_data
        $conditions[] = "(pattern_data LIKE %s)";
        $placeholders[] = '%"stepText":"%' . $wpdb->esc_like($word) . '%"%';
        
        // Also search in lesson_learned for broader pattern matching
        $conditions[] = "(pattern_data LIKE %s)";
        $placeholders[] = '%"lesson_learned":"%' . $wpdb->esc_like($word) . '%"%';
    }
    
    $where_conditions = implode(' OR ', $conditions);
    $sql = $wpdb->prepare(
        "SELECT * FROM {$table_name} 
         WHERE user_id = %d 
         AND learning_type = %s 
         AND success_rate >= 0.3 
         AND ({$where_conditions})
         ORDER BY success_rate DESC, confidence_score DESC, usage_count DESC 
         LIMIT %d",
        array_merge([$user_id, $learning_type], $placeholders, [$limit])
    );
    
    $learnings = $wpdb->get_results($sql);
    
    // Decode JSON data and filter out invalid entries
    $valid_learnings = [];
    foreach ($learnings as $learning) {
        $decoded = json_decode($learning->pattern_data, true);
        if ($decoded && is_array($decoded)) {
            $learning->pattern_data = $decoded;
            $valid_learnings[] = $learning;
        }
    }
    
    wp_send_json_success([
        'learnings' => $valid_learnings,
        'count' => count($valid_learnings),
        'debug' => [
            'search_words' => $words,
            'conditions' => $conditions,
            'sql' => $sql
        ]
    ]);
    wp_die();
}

public function handle_update_learning_usage() {
    $user_id = $this->validate_ajax_request();
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    $learning_id = intval($_POST['learning_id'] ?? 0);
    $was_successful = $_POST['was_successful'] === 'true';
    
    if (!$learning_id) {
        wp_send_json_error(['message' => 'Invalid learning ID']);
        wp_die();
    }
    
    // Get current learning data
    $learning = $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$table_name} WHERE id = %d AND user_id = %d",
        $learning_id, $user_id
    ));
    
    if (!$learning) {
        wp_send_json_error(['message' => 'Learning not found']);
        wp_die();
    }
    
    // Calculate new success rate
    $current_success_rate = floatval($learning->success_rate);
    $usage_count = intval($learning->usage_count) + 1;
    
    if ($usage_count === 1) {
        $new_success_rate = $was_successful ? 1.0 : 0.0;
    } else {
        $total_successes = $current_success_rate * ($usage_count - 1);
        $total_successes += $was_successful ? 1 : 0;
        $new_success_rate = $total_successes / $usage_count;
    }
    
    // Update learning
    $result = $wpdb->update(
        $table_name,
        [
            'success_rate' => $new_success_rate,
            'usage_count' => $usage_count,
            'last_used' => current_time('mysql')
        ],
        ['id' => $learning_id],
        ['%f', '%d', '%s'],
        ['%d']
    );
    
    if ($result !== false) {
        wp_send_json_success([
            'new_success_rate' => $new_success_rate,
            'usage_count' => $usage_count,
            'message' => 'Learning stats updated'
        ]);
    } else {
        wp_send_json_error(['message' => 'Failed to update learning stats']);
    }
    wp_die();
}

public function handle_get_learning_stats() {
    // Check if this is an AJAX request (has nonce) or direct call
    $is_ajax = isset($_POST['nonce']) || isset($_GET['nonce']);
    
    if ($is_ajax) {
        $user_id = $this->validate_ajax_request();
    } else {
        // Direct call from Learning Manager page - just get current user
        $user_id = get_current_user_id();
        if (!$user_id) {
            return [
                'total_learnings' => 0,
                'total_applications' => 0,
                'avg_success_rate' => 0,
                'best_success_rate' => 0,
                'used_learnings_count' => 0,
                'unused_learnings_count' => 0,
                'avg_score' => 0,
                'best_pattern' => 'No data yet'
            ];
        }
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    
    // Get comprehensive stats with proper success rate calculation
    $stats = $wpdb->get_row($wpdb->prepare(
        "SELECT 
            COUNT(*) as total_learnings,
            COALESCE(SUM(usage_count), 0) as total_applications,
            -- Only calculate success rate for learnings that have been used at least once
            COALESCE(AVG(CASE WHEN usage_count > 0 THEN success_rate ELSE NULL END), 0) * 100 as avg_success_rate,
            COALESCE(MAX(CASE WHEN usage_count > 0 THEN success_rate ELSE 0 END), 0) * 100 as best_success_rate,
            COUNT(CASE WHEN usage_count > 0 THEN 1 END) as used_learnings_count,
            COUNT(CASE WHEN usage_count = 0 THEN 1 END) as unused_learnings_count,
            COALESCE(AVG(score), 0) as avg_score,
            (SELECT learning_type FROM {$table_name} 
             WHERE user_id = %d AND usage_count > 0 
             ORDER BY success_rate DESC, usage_count DESC 
             LIMIT 1) as best_pattern
         FROM {$table_name} 
         WHERE user_id = %d",
        $user_id, $user_id
    ));
    
    $result = [
        'total_learnings' => $stats ? intval($stats->total_learnings) : 0,
        'total_applications' => $stats ? intval($stats->total_applications) : 0,
        'avg_success_rate' => $stats ? round(floatval($stats->avg_success_rate), 1) : 0,
        'best_success_rate' => $stats ? round(floatval($stats->best_success_rate), 1) : 0,
        'used_learnings_count' => $stats ? intval($stats->used_learnings_count) : 0,
        'unused_learnings_count' => $stats ? intval($stats->unused_learnings_count) : 0,
        'avg_score' => $stats ? round(floatval($stats->avg_score), 1) : 0,
        'best_pattern' => $stats ? ($stats->best_pattern ?: 'No data yet') : 'No data yet'
    ];
    
    if ($is_ajax) {
        wp_send_json_success($result);
        wp_die();
    } else {
        return $result;
    }
}



// [REPLACE THIS ENTIRE FUNCTION IN zencode-ai.php]
// [REPLACE THIS ENTIRE FUNCTION IN zencode-ai.php]
public function handle_ai_request() {
    if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'zencode_ai_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed.'], 403);
        wp_die();
    }

    if (isset($_POST['is_refinement_request']) && $_POST['is_refinement_request'] === 'true') {
        $this->handle_prompt_refinement();
        return;
    }

    $user_id = $this->validate_ajax_request();
    $user_context = $this->get_user_context();

    if (!$user_context['can_query']) {
        wp_send_json_error(['message' => "You have reached your monthly limit."], 429);
        wp_die();
    }

    // --- GATEKEEPER LOGIC (Unchanged) ---
    $prompt_for_check = sanitize_textarea_field($_POST['prompt'] ?? '');
    $is_one_shot_build_check = isset($_POST['is_one_shot_build']) && $_POST['is_one_shot_build'] === 'true';
    $is_plan_mode_check = isset($_POST['is_plan_mode']) && $_POST['is_plan_mode'] === 'true';
    $is_automated_check = isset($_POST['is_automated']) && $_POST['is_automated'] === 'true';

    if ($is_one_shot_build_check && $this->is_multi_page_request($prompt_for_check)) {
        $explanation = "That's an excellent idea for a website! However, a project with multiple pages is too complex for the **One-Shot Builder**, which is designed to create single-page sites instantly.\n\nFor a multi-file project like this, the main chat is the perfect tool. I recommend we start in **Plan Mode**.\n\n**Please close this window and let's get started in the main chat!**";
        wp_send_json_success(['ai_response' => json_encode(['is_incompatible' => true, 'explanation' => $explanation])]);
        wp_die();
    }

    if (!$is_one_shot_build_check && !$is_plan_mode_check && !$is_automated_check && $this->should_run_planning_check($prompt_for_check, $is_one_shot_build_check)) {
        $explanation = "That's an excellent idea for a project! However, this looks like a large-scale project that would work much better in **Plan Mode**.\n\nPlan Mode will break this down into manageable steps that you can execute one by one or all at once. It's designed for projects requiring multiple files and complex functionality.\n\n**To switch to Plan Mode:** Click the AI Mode button in the top-right corner of the chat panel and select 'Plan', then send your request again.";
        wp_send_json_success(['ai_response' => json_encode(['is_incompatible' => true, 'explanation' => $explanation])]);
        wp_die();
    }
    // --- END GATEKEEPER ---


    // --- UNIFIED PLAN GENERATION LOGIC ---
    $is_plan_mode = isset($_POST['is_plan_mode']) && $_POST['is_plan_mode'] === 'true';
    $is_step_execution = isset($_POST['is_step_execution']) && $_POST['is_step_execution'] === 'true';
    $specification_sheet_json = stripslashes($_POST['specification_sheet'] ?? null);
    $specification_sheet = $specification_sheet_json ? json_decode($specification_sheet_json, true) : null;
    $requested_steps = isset($_POST['requested_steps']) ? absint($_POST['requested_steps']) : null;
    $prompt = sanitize_textarea_field($_POST['prompt'] ?? '');

    // --- START OF THE FIX: INJECT MASTER LEARNINGS INTO PLAN GENERATION ---
    if ($is_plan_mode && !$is_step_execution) {
        try {
            // 1. Fetch the master learnings from the database.
            $master_learnings = $this->get_master_plan_learnings($user_id);

            // 2. Get the standard system prompt for planning.
            $system_prompt = $this->build_system_prompt(
                null, null, true, null, $user_context['plan'], $requested_steps, false, [], 'moderate', $specification_sheet
            );
            
            // 3. Inject the learnings into the user's prompt.
            $final_user_prompt = $master_learnings . $prompt;

            // 4. Create the final messages array.
            $messages = [
                ['role' => 'system', 'content' => $system_prompt],
                ['role' => 'user', 'content' => $final_user_prompt]
            ];

            // 5. Make the API call as before.
            $api_key = $this->validate_api_key('google');
            $raw_ai_output = $this->call_google_api($api_key, 'google/gemini-2.5-pro', $messages, false, true, 12288);

            // (The rest of the JSON extraction logic remains the same)
            $json_response = null;
            $first_brace = strpos($raw_ai_output, '{');
            $last_brace = strrpos($raw_ai_output, '}');

            if ($first_brace !== false && $last_brace !== false && $last_brace > $first_brace) {
                $json_string = substr($raw_ai_output, $first_brace, ($last_brace - $first_brace) + 1);
                json_decode($json_string);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $json_response = $json_string;
                }
            }

            if ($json_response === null) {
                error_log("ZenCode AI Direct Plan Generation Failed. Raw AI output: " . $raw_ai_output);
                throw new Exception("Direct plan generation failed to produce valid JSON.");
            }
            
            zc_increment_query_count($user_id);
            wp_send_json_success(['ai_response' => $json_response]);

        } catch (Exception $e) {
            $this->handle_api_error($e, 'Plan Generation');
        }
        wp_die();
    }
    // --- END OF THE FIX & END OF PLAN LOGIC ---

    // The rest of the function for normal requests remains unchanged.
    try {
        $this->check_rolling_limit($user_id);
    } catch (Exception $e) {
        wp_send_json_error(['message' => $e->getMessage()], 429);
        wp_die();
    }

    if ($is_step_execution) {
        $prompt = sanitize_textarea_field($_POST['prompt'] ?? '');
    } else {
        $prompt = $this->validate_and_sanitize_prompt($_POST['prompt'] ?? '');
    }

    $model = $this->validate_model($_POST['model'] ?? '');

    try {
        $response = $this->process_main_ai_request($prompt, $model, $user_context, $_POST);
        $response_data = [ 'ai_response' => $response ];
        if (isset($GLOBALS['zc_limit_warning'])) {
            $response_data = array_merge($response_data, $GLOBALS['zc_limit_warning']);
            unset($GLOBALS['zc_limit_warning']);
        }
        zc_increment_query_count($user_id);
        wp_send_json_success($response_data);
    } catch (Exception $e) {
        $this->handle_api_error($e, 'AI Request');
    }
    wp_die();
}
private function handle_polish_request() {
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'zencode_ai_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed.'], 403);
        wp_die();
    }
    
    $user_id = $this->validate_ajax_request();
    $user_context = $this->get_user_context();

    if (!$user_context['can_query']) {
        wp_send_json_error(['message' => "You have reached your monthly request or token limit for your plan. Please upgrade or try again next month."], 429);
        wp_die();
    }

    // Enhanced input validation for polish requests
    $prompt = $this->validate_and_sanitize_prompt($_POST['prompt'] ?? '');
    $model = $this->validate_model($_POST['model'] ?? '');
    
    $polish_options_raw = stripslashes($_POST['polish_options'] ?? '[]');
    $polish_options = json_decode($polish_options_raw, true);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        wp_send_json_error(['message' => 'Invalid polish options format.'], 400);
        wp_die();
    }
    
    $polish_intensity = sanitize_text_field($_POST['polish_intensity'] ?? 'moderate');
    
    // Validate polish options and intensity
    $valid_options = ['modern-ui', 'mobile-first', 'accessibility', 'performance'];
    $valid_intensities = ['subtle', 'moderate', 'dramatic'];
    
    if (!is_array($polish_options)) {
        wp_send_json_error(['message' => 'Polish options must be an array.'], 400);
        wp_die();
    }
    
    $polish_options = array_intersect($polish_options, $valid_options);
    
    if (empty($polish_options)) {
        wp_send_json_error(['message' => 'No valid polish options selected.'], 400);
        wp_die();
    }
    
    if (!in_array($polish_intensity, $valid_intensities)) {
        wp_send_json_error(['message' => 'Invalid polish intensity.'], 400);
        wp_die();
    }

    // Set validated data back to $_POST for process_main_ai_request
    $_POST['is_polish_request'] = 'true';
    $_POST['polish_options'] = json_encode($polish_options);
    $_POST['polish_intensity'] = $polish_intensity;
    $_POST['is_automated'] = 'true';

    try {
        $response = $this->process_main_ai_request($prompt, $model, $user_context, $_POST);
        zc_increment_query_count($user_id);
        wp_send_json_success(['ai_response' => $response]);
    } catch (Exception $e) {
        $this->handle_api_error($e, 'Polish Request');
    }
    wp_die();
}
    /**
     * Run planning analysis for complex requests
     */
  private function run_planning_analysis($user_text_input, $model, $user_id, $is_one_shot = false) {
    $planning_prompt = $this->build_planning_prompt($user_text_input, $is_one_shot);
    $planning_messages = [['role' => 'user', 'content' => [['type' => 'text', 'text' => $planning_prompt]]]];
    $api_key = $this->validate_api_key('google');
    $analysis_response_str = $this->call_google_api($api_key, $model, $planning_messages, false, true);
    $json_response = trim($analysis_response_str);
    
    if (strpos($json_response, '```json') === 0) {
        $json_response = preg_replace('/^```json\s*/', '', $json_response);
        $json_response = preg_replace('/\s*```$/', '', $json_response);
    }
    
    $analysis = json_decode($json_response, true);
    if (json_last_error() !== JSON_ERROR_NONE || !isset($analysis['is_complex'])) {
        return ['ai_response' => $analysis_response_str];
    }
    
    if (!empty($analysis['is_incompatible'])) {
        return ['ai_response' => json_encode($analysis)];
    }
    
    if (!empty($analysis['is_complex'])) {
        // For regular chat mode, suggest switching to plan mode
        if (!$is_one_shot) {
            $explanation = "That's an excellent idea for a project! However, this looks like a complex multi-step project that would work much better in **Plan Mode**.\n\nPlan Mode will break this down into manageable steps that you can execute one by one or all at once.\n\n**To switch to Plan Mode:** Click the AI Mode button in the top-right corner of the chat panel and select 'Plan', then send your request again.";
            return ['ai_response' => json_encode(['is_incompatible' => true, 'explanation' => $explanation])];
        }
        
        // Original one-shot logic
        $final_response_text = $analysis['plan'] ?? '';
        if (!empty($analysis['suggested_prompt'])) {
            $final_response_text .= "\n\n**Here is a refined prompt...**";
        }
        $token = wp_generate_password(32, false);
        set_transient('zc_ref_token_' . $user_id, $token, 5 * MINUTE_IN_SECONDS);
        zc_increment_query_count($user_id);
        return ['ai_response' => $final_response_text, 'refinement_token' => $token];
    }
    
    return null;
}

private function build_planning_prompt($user_prompt, $is_one_shot) {
    if ($is_one_shot) {
        return <<<PROMPT
You are an AI gatekeeper for a tool that creates **one single HTML file**. Your primary function is to reject requests that are fundamentally multi-page.

**LAW 1: The Single-File Mandate (Your Highest Priority)**
The tool you are guarding can only produce ONE HTML file. Users, however, will often ask for things like a "5-page site" or "a site with an about page and a contact page."

**CRITICAL CLARIFICATION:** When they say this, they mean they want multiple, separate HTML files (e.g., `index.html`, `about.html`). Your tool CANNOT do this.

**THEREFORE, YOU MUST NOT interpret a multi-page request as a request for a single long page with multiple `<section>` tags. This is a STRICT rule. Any request that implies more than one distinct page must be treated as incompatible.**

---

**LAW 2: The Incompatibility Scan**
Based on Law 1, you MUST scan the user's request for any of the following keywords or phrases that imply a multi-page structure:
`['2-page', '3-page', '4-page', '5-page', 'multi-page', 'multiple pages', 'about page', 'contact page', 'blog page', 'portfolio page', 'user login', 'database', 'e-commerce']`

If the user's request contains **ANY** of these triggers, you MUST IMMEDIATELY return the `is_incompatible` JSON object with the predefined explanation. Do not analyze the prompt further. Your only job is to return this specific JSON.

**INCOMPATIBLE RESPONSE FORMAT (MANDATORY):**
`{"is_incompatible": true, "explanation": "That's an excellent idea for a website! However, a project with multiple pages or backend features is too complex for the **One-Shot Builder**, which is designed to create single-page sites instantly.\\n\\nFor a multi-file project like this, the main chat is the perfect tool. I recommend we start in **Plan Mode**. Just tell me your idea in the main chat, and I'll create a step-by-step plan to build your entire application.\\n\\n**Please close this window and let's get started in the main chat!**"}`

---

**LAW 3: Standard Analysis (Only if Laws 1 & 2 Pass)**
If the request is compatible, proceed with the standard analysis:
- **If SUITABLE but VAGUE:** Return `{"is_complex": true, "plan": "...", "suggested_prompt": "..."}`.
- **If SUITABLE and CLEAR:** Return `{"is_complex": false}`.

User's original request to analyze:
---
{$user_prompt}
---
PROMPT;
    } else {
        // Regular chat mode with 3-file limit
        return <<<PROMPT
You are a project complexity analyzer. Analyze the user's request below to determine if it's suitable for regular chat or if it should use Plan Mode.

Your response MUST be a valid JSON object with no other text.

**COMPLEXITY ANALYSIS:**
- **Simple requests** (1-3 files maximum, single feature additions, bug fixes, questions): Return `{"is_complex": false}`
- **Complex projects** (4+ files, multi-page websites, full applications, extensive features): Return `{"is_complex": true}`

**3-FILE LIMIT ASSESSMENT:**
Regular chat is optimized for projects requiring 1-3 files maximum. This works well for:
- Single page websites (index.html + style.css + script.js)
- Refactoring existing projects into separate files
- Adding new features that span HTML, CSS, and JS
- Simple applications with basic file separation

If the request implies:
- More than 3 separate files
- Multiple HTML pages (about.html, contact.html, etc.)
- Complex directory structures with many components
- Full-scale applications with numerous modules
- Backend functionality or databases

Then it should use Plan Mode instead.

**INCOMPATIBILITY SCAN:**
If the request would require more than 3 files OR contains indicators of a large-scale project, return:
`{"is_incompatible": true, "explanation": "That's an excellent idea for a project! However, this looks like a large-scale project that would work much better in **Plan Mode**.\\n\\nPlan Mode will break this down into manageable steps that you can execute one by one or all at once. It's designed for projects requiring multiple files and complex functionality.\\n\\n**To switch to Plan Mode:** Click the AI Mode button in the top-right corner of the chat panel and select 'Plan', then send your request again."}`

Large-scale indicators: ['multi-page', 'multiple pages', 'about page', 'contact page', 'blog page', 'portfolio page', 'user login', 'authentication', 'database', 'e-commerce', 'shopping cart', 'payment', 'admin panel', 'dashboard', 'user registration', 'CMS', 'content management', 'multiple components', 'several pages', 'various sections', 'landing page + about + contact']

User's request:
---
{$user_prompt}
---
PROMPT;
    }
}// [REPLACE THIS ENTIRE FUNCTION in zencode-ai-php.js]
public function render_app_shortcode($atts) {
    // We need an instance of the license UI class to check the license and render the UI if needed.
    if (!class_exists('ZenCode_Customer_License')) {
        return '<div class="notice notice-error"><p>ZenCode AI license system is not available. Please reinstall the plugin.</p></div>';
    }
    $license_checker = new ZenCode_Customer_License();

    // This single check handles all cases: no license, expired, invalid, etc.
    $validation = $license_checker->validate_license(null, true);

    // If the license is NOT valid for any reason, show the activation/status page.
    // This will correctly display the modal for new users or the error card for invalid keys.
    if (!$validation || !isset($validation['valid']) || !$validation['valid']) {
        // We ask the license class to render its own page.
        return $license_checker->render_page_router();
    }

    // --- If we reach this point, the license IS VALID. We can now load the main app. ---

    if (!is_user_logged_in()) {
        return '<p>Please log in to use ZenCode AI.</p>';
    }

    $user_id = get_current_user_id();
    
    // Simplified logic for the self-hosted version
    $user_plan = 'visionary'; // Licensed users have full access
    $available_models = [
        'google/gemini-2.5-pro' => 'google'
    ];
    $api_keys_set = [
        'google' => !empty(get_option('zencode_ai_google_api_key'))
    ];

    wp_enqueue_script('zencode-ai-app', ZENCODE_AI_PLUGIN_URL . 'assets/app.js', [], ZENCODE_AI_VERSION, true);
    wp_localize_script('zencode-ai-app', 'zencode_ai_app_vars', [
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('zencode_ai_nonce'),
        'api_keys_set' => $api_keys_set,
        'default_model' => 'google/gemini-2.5-pro',
        'available_models' => $available_models,
        'user_plan' => $user_plan,
        'nav_links' => [
            'home'        => home_url('/'),
            'dashboard'   => home_url('/dashboard/'),
            'user_manual' => home_url('/user-manual/'),
            'logout'      => wp_logout_url(home_url('/'))
        ]
    ]);
    return '<div id="ai_app_root"></div>';
}

    // Project management functions (consolidated with validation)
 // OLD handle_save_project function - REPLACE ENTIRELY with this:

public function handle_save_project() {
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'zencode_ai_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed.'], 403);
        wp_die();
    }
    
    $user_id = $this->validate_ajax_request();
    
    $project_data_json = stripslashes($_POST['project_data'] ?? '');
    $project_name = sanitize_text_field($_POST['project_name'] ?? '');
    $create_snapshot = isset($_POST['create_auto_snapshot']) && $_POST['create_auto_snapshot'] === 'true';
    
    // ========== NEW: Capture project_context from POST ==========
    $project_context = isset($_POST['project_context']) ? 
        json_decode(stripslashes($_POST['project_context']), true) : null;
    // ========== END NEW ==========
    
    // Enhanced validation
    if (empty($project_name) || strlen($project_name) > 100) {
        wp_send_json_error(['message' => 'Project name must be between 1-100 characters.'], 400);
        wp_die();
    }
    
    if (!preg_match('/^[a-zA-Z0-9\s_-]+$/', $project_name)) {
        wp_send_json_error(['message' => 'Project name contains invalid characters.'], 400);
        wp_die();
    }
    
    $project_data = json_decode($project_data_json, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        wp_send_json_error(['message' => 'Invalid project data format.'], 400);
        wp_die();
    }
    
    // ========== NEW: Add project_context to project_data ==========
    if ($project_context) {
        $project_data['projectContext'] = $project_context;
    }
    
    // Re-encode the project data with the context included
    $updated_project_data_json = json_encode($project_data);
    // ========== END NEW ==========
    
    // ========== MODIFIED: Check size of UPDATED json ==========
    if (strlen($updated_project_data_json) > 5000000) {
        wp_send_json_error(['message' => 'Project data too large. Maximum 5MB allowed.'], 400);
        wp_die();
    }
    // ========== END MODIFIED ==========
    
    if (!is_array($project_data)) {
        wp_send_json_error(['message' => 'Project data must be a valid object.'], 400);
        wp_die();
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_projects';
    
    // Check if user already has this project
    $existing_project = $wpdb->get_row($wpdb->prepare(
        "SELECT id, project_data FROM $table_name WHERE user_id = %d AND project_name = %s",
        $user_id,
        $project_name
    ));
    
    // Create auto-snapshot before AI changes
    if ($create_snapshot && $existing_project) {
        $this->create_project_snapshot(
            $user_id,
            $project_name,
            'Before AI changes',
            $existing_project->project_data,
            'Auto-snapshot before AI modifications',
            true,
            'pre_ai_change'
        );
    }
    
    // Retry logic for database operations
    $max_retries = 3;
    $retry_count = 0;
    $result = false;

    while ($retry_count < $max_retries && $result === false) {
        if ($existing_project) {
            // ========== MODIFIED: Use updated_project_data_json ==========
            $result = $wpdb->update(
                $table_name,
                [
                    'project_data' => $updated_project_data_json, // ← Changed from $project_data_json
                    'last_modified' => current_time('mysql')
                ],
                [
                    'id' => $existing_project->id,
                    'user_id' => $user_id
                ],
                ['%s', '%s'],
                ['%d', '%d']
            );
            // ========== END MODIFIED ==========
        } else {
            if ($retry_count === 0) {
                $project_count = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM $table_name WHERE user_id = %d",
                    $user_id
                ));
                
                if ($project_count >= 50) {
                    wp_send_json_error(['message' => 'Maximum 50 projects allowed per user.'], 429);
                    wp_die();
                }
            }
            
            // ========== MODIFIED: Use updated_project_data_json ==========
            $result = $wpdb->insert($table_name, [
                'user_id' => $user_id,
                'project_name' => $project_name,
                'project_data' => $updated_project_data_json, // ← Changed from $project_data_json
                'last_modified' => current_time('mysql')
            ]);
            // ========== END MODIFIED ==========
        }
        
        if ($result === false && $retry_count < $max_retries - 1) {
            $retry_count++;
            usleep(500000);
            error_log("ZenCode AI: Project save retry #{$retry_count} for user {$user_id}, project: {$project_name}");
        } else if ($result === false) {
            error_log("ZenCode AI: Project save failed after {$max_retries} attempts for user {$user_id}: " . $wpdb->last_error);
        }
    }
    
    if ($result === false) {
        wp_send_json_error(['message' => 'Failed to save project after multiple attempts.'], 500);
        wp_die();
    }
    
    wp_send_json_success(['message' => 'Project saved successfully.']);
}
public function handle_load_project() {
    if (!wp_verify_nonce($_GET['nonce'] ?? '', 'zencode_ai_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed.'], 403);
        wp_die();
    }
    
    $user_id = $this->validate_ajax_request();
    
    global $wpdb; 
    $table_name = $wpdb->prefix . 'zencode_ai_projects';
    $project_name = sanitize_text_field($_GET['project_name'] ?? '');
    
    if (!empty($project_name)) {
        // Validate project name
        if (!preg_match('/^[a-zA-Z0-9\s_-]+$/', $project_name)) {
            wp_send_json_error(['message' => 'Invalid project name format.'], 400);
            wp_die();
        }
        
        // Load specific project (with authorization check)
        $project = $wpdb->get_row($wpdb->prepare(
            "SELECT project_name, project_data FROM $table_name WHERE user_id = %d AND project_name = %s", 
            $user_id, 
            $project_name
        ));
    } else {
        // Load most recent project for this user
        $project = $wpdb->get_row($wpdb->prepare(
            "SELECT project_name, project_data FROM $table_name WHERE user_id = %d ORDER BY last_modified DESC LIMIT 1", 
            $user_id
        ));
    }
    
    if ($project) {
        $project_data = json_decode($project->project_data, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            wp_send_json_error(['message' => 'Project data corrupted.'], 500);
            wp_die();
        }
        
        $data = is_array($project_data) ? $project_data : [];
        $data['projectName'] = sanitize_text_field($project->project_name);
        
        // ========== NEW: Include projectContext in the response ==========
        $data['projectContext'] = isset($project_data['projectContext']) ? 
            $project_data['projectContext'] : null;
        // ========== END NEW ==========
        
        wp_send_json_success($data);
    } else {
        wp_send_json_error(['message' => 'No saved project found.']);
    }
}
	
	public function handle_get_projects_list() {
        $user_id = $this->validate_ajax_request();
        
        global $wpdb; 
        $table_name = $wpdb->prefix . 'zencode_ai_projects';
        $projects = $wpdb->get_results($wpdb->prepare("SELECT project_name, last_modified FROM $table_name WHERE user_id = %d ORDER BY last_modified DESC", $user_id));
        wp_send_json_success($projects);
    }

public function handle_delete_project() {
    if (!wp_verify_nonce($_POST['nonce'] ?? '', 'zencode_ai_nonce')) {
        wp_send_json_error(['message' => 'Security verification failed.'], 403);
        wp_die();
    }
    
    $user_id = $this->validate_ajax_request();
    
    $project_name = sanitize_text_field($_POST['project_name'] ?? '');
    if (empty($project_name) || !preg_match('/^[a-zA-Z0-9\s_-]+$/', $project_name)) { 
        wp_send_json_error(['message' => 'Invalid project name.'], 400); 
        wp_die(); 
    }
    
    global $wpdb; 
    $table_name = $wpdb->prefix . 'zencode_ai_projects';
    
    // Delete with proper authorization (user can only delete their own projects)
    $result = $wpdb->delete(
        $table_name, 
        [
            'user_id' => $user_id, 
            'project_name' => $project_name
        ],
        ['%d', '%s']
    );
    
    if ($result === false) {
        wp_send_json_error(['message' => 'Failed to delete project.'], 500);
        wp_die();
    }
    
    if ($result === 0) {
        wp_send_json_error(['message' => 'Project not found or access denied.'], 404);
        wp_die();
    }
    
    wp_send_json_success(['message' => "Project '{$project_name}' deleted successfully."]);
}
// REPLACE your current handle_vectorize_file function with this definitive version.
// REPLACE your current handle_vectorize_file function with this one.
public function handle_vectorize_file() {
    $user_id = $this->validate_ajax_request();

    if (!zc_user_has_feature('vector_kb', $user_id)) {
        wp_send_json_error(['message' => 'Saving files to the Knowledge Base is a premium feature. Please upgrade your plan to use it.'], 403);
        wp_die();
    }

    $raw_file_content = isset($_POST['file_content']) ? stripslashes($_POST['file_content']) : '';
    if (function_exists('mb_convert_encoding')) { 
        $utf8_content = mb_convert_encoding($raw_file_content, 'UTF-8', 'UTF-8'); 
    } else { 
        $utf8_content = wp_check_invalid_utf8($raw_file_content, true); 
    }
    $file_content = sanitize_textarea_field($utf8_content);
    
    // Get the raw file path directly, without any sanitization first.
    $file_path = isset($_POST['file_path']) ? stripslashes($_POST['file_path']) : '';
    
    // Now, validate the raw path.
    if (!$this->validate_file_path($file_path)) {
        wp_send_json_error(['message' => 'Invalid or unsafe file path provided.'], 400);
        wp_die();
    }
    
    if (empty($file_content)) { 
        wp_send_json_error(['message' => 'File content is empty or invalid.'], 400); 
        wp_die(); 
    }
    
    // Content validation
    if (strlen($file_content) > 500000) {
        wp_send_json_error(['message' => 'File content too large. Maximum 500KB allowed.'], 400);
        wp_die();
    }
    
    // Check for malicious content patterns
    if (preg_match('/<script|javascript:|data:|php:|exec\(|shell_exec|system\(/i', $file_content)) {
        wp_send_json_error(['message' => 'File content contains potentially unsafe code.'], 400);
        wp_die();
    }
    
    try {
        // NO API KEY NEEDED! Generate TF-IDF vector locally
        $text_to_embed = "// File: " . $file_path . "\n\n" . $file_content;
        
        // Calculate TF-IDF vector (keyword-based, no external API)
        $tfidf_vector = $this->calculate_tfidf_vector($text_to_embed);
        
        if (empty($tfidf_vector)) {
            throw new Exception('Failed to generate vector from file content.');
        }
        
        global $wpdb; 
        $table_name = $wpdb->prefix . 'zencode_ai_kb_vectors';
        
        // Store the sparse vector as JSON
        $result = $wpdb->insert($table_name, [ 
            'user_id' => $user_id, 
            'file_path' => $file_path, 
            'source_text' => $text_to_embed, 
            'embedding' => json_encode($tfidf_vector) 
        ]);
        
        if ($result === false) {
            throw new Exception('Database error: ' . $wpdb->last_error);
        }
        
        zc_increment_query_count($user_id);
        wp_send_json_success([
            'message' => "File '{$file_path}' saved to your knowledge base.",
            'info' => 'Using keyword-based vectors (no API key required!)'
        ]);
        
    } catch (Exception $e) {
        $this->handle_api_error($e, 'Vectorization');
    }
    wp_die();
}
    
 // REPLACE EXISTING FUNCTION
public function handle_delete_vector() {
    $this->validate_ajax_request(true); // Require admin
    
    global $wpdb; 
    $table_name = $wpdb->prefix . 'zencode_ai_kb_vectors';
    
    if (isset($_POST['vector_id'])) {
        $id = absint($_POST['vector_id']);
        if ($id > 0) {
            $result = $wpdb->delete($table_name, ['id' => $id], ['%d']);
            if ($result !== false) {
                wp_send_json_success(['message' => 'Vector deleted.']);
            } else {
                wp_send_json_error(['message' => 'Failed to delete vector.'], 500);
            }
        } else {
            wp_send_json_error(['message' => 'Invalid vector ID.'], 400);
        }
    } elseif (isset($_POST['delete_all']) && $_POST['delete_all'] === 'true') {
        // Use explicit DELETE with proper confirmation
        $count = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        if ($count > 0) {
$result = $wpdb->query("DELETE FROM $table_name WHERE 1=1");
            if ($result !== false) {
                wp_send_json_success(['message' => "Deleted {$result} vectors from all users."]);
            } else {
                wp_send_json_error(['message' => 'Failed to delete vectors.'], 500);
            }
        } else {
            wp_send_json_success(['message' => 'No vectors to delete.']);
        }
    } else {
        wp_send_json_error(['message' => 'Invalid request parameters.'], 400);
    }
    wp_die();
}
    
    public function handle_clear_user_vectors() {
        $user_id = $this->validate_ajax_request();
        
        global $wpdb; 
        $table_name = $wpdb->prefix . 'zencode_ai_kb_vectors';
        $wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE user_id = %d", $user_id));
        wp_send_json_success(['message' => 'Your knowledge base has been cleared.']);
    }
 
    // Admin and UI functions
   public function add_admin_menu() {
    add_menu_page(
        'ZenCode AI',
        'ZenCode AI',
        'edit_posts',
        'zencode-ai-app',
        [$this, 'render_app_page'],
        'dashicons-admin-generic',
        3
    );
    
    add_submenu_page(
        'zencode-ai-app',
        'Settings',
        'Settings',
        'manage_options',
        'zencode-ai-settings',
        [$this, 'settings_page']
    );
    
    add_submenu_page(
        'zencode-ai-app',
        'Knowledge Base',
        'Knowledge Base',
        'edit_posts',
        'zencode-ai-knowledge-base',
        [$this, 'knowledge_base_page']
    );
    
    // NEW: Learning Manager submenu
    add_submenu_page(
        'zencode-ai-app',
        'Learning Manager',
        'Learning Manager',
        'edit_posts',
        'zencode-ai-learning-manager',
        [$this, 'render_learning_manager_page']
    );
    
    // User Manual submenu
    add_submenu_page(
        'zencode-ai-app',
        'User Manual',
        'User Manual',
        'edit_posts',
        'zencode-ai-user-manual',
        [$this, 'render_user_manual_page']
    );
    
    // Quick Guide submenu
    add_submenu_page(
        'zencode-ai-app',
        'Quick Guide',
        'Quick Guide',
        'edit_posts',
        'zencode-ai-quick-guide',
        [$this, 'render_quick_guide_page']
    );
}

    public function render_app_page() {
        echo '<div class="wrap"><h1>ZenCode AI Assistant</h1>';
        echo do_shortcode('[zencode_ai_app]');
        echo '</div>';
    }

  public function knowledge_base_page() {
    global $wpdb; 
    $table_name = $wpdb->prefix . 'zencode_ai_kb_vectors';
    $vectors = $wpdb->get_results("SELECT id, user_id, source_text FROM $table_name ORDER BY id DESC");
    ?>
    <div class="wrap">
        <style>
            /* CSS Custom Properties for consistent theming */
            :root {
                --primary-color: #a855f7;
                --secondary-color: #60a5fa;
                --success-color: #059669;
                --warning-color: #dc2626;
                --text-primary: #1f2937;
                --text-secondary: #6b7280;
                --text-light: #9ca3af;
                --bg-primary: #ffffff;
                --bg-secondary: #f8f9fa;
                --bg-tertiary: #f0f7ff;
                --border-color: #e5e7eb;
                --border-radius: 8px;
                --shadow: 0 2px 8px rgba(0,0,0,0.1);
                --transition: all 0.2s ease-in-out;
                
                /* Responsive spacing scale */
                --space-xs: clamp(4px, 1vw, 6px);
                --space-sm: clamp(8px, 2vw, 12px);
                --space-md: clamp(12px, 2.5vw, 16px);
                --space-lg: clamp(16px, 3vw, 20px);
                --space-xl: clamp(20px, 4vw, 24px);
                --space-2xl: clamp(24px, 5vw, 32px);
                --space-3xl: clamp(32px, 6vw, 40px);
                
                /* Typography scale */
                --text-xs: clamp(0.75rem, 2vw, 0.8rem);
                --text-sm: clamp(0.8rem, 2.2vw, 0.9rem);
                --text-base: clamp(0.9rem, 2.5vw, 1rem);
                --text-lg: clamp(1rem, 3vw, 1.125rem);
                --text-xl: clamp(1.125rem, 3.5vw, 1.25rem);
                --text-2xl: clamp(1.25rem, 4vw, 1.5rem);
                --text-3xl: clamp(1.5rem, 5vw, 1.875rem);
                --text-4xl: clamp(1.75rem, 6vw, 2.25rem);
                
                /* Line heights */
                --leading-tight: 1.25;
                --leading-normal: 1.5;
                --leading-relaxed: 1.75;
            }
            
            /* Base container */
            .zencode-kb-container {
                background: var(--bg-primary);
                padding: var(--space-2xl);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                margin: 0 auto;
                max-width: 1200px;
                box-sizing: border-box;
                width: calc(100% - var(--space-xl));
            }
            
            /* Header */
            .zencode-kb-header {
                margin-bottom: var(--space-2xl);
                padding-bottom: var(--space-lg);
                border-bottom: 2px solid var(--border-color);
            }
            
            .zencode-kb-header h1 {
                color: var(--primary-color);
                font-size: var(--text-3xl);
                margin: 0 0 var(--space-sm) 0;
                font-weight: 700;
                line-height: var(--leading-tight);
            }
            
            .zencode-kb-header p {
                color: var(--text-secondary);
                font-size: var(--text-base);
                margin: 0;
                line-height: var(--leading-normal);
            }
            
            /* Danger Zone */
            .danger-zone {
                background: #fef2f2;
                border: 1px solid #fecaca;
                border-left: 4px solid var(--warning-color);
                padding: var(--space-lg);
                border-radius: 6px;
                margin-bottom: var(--space-2xl);
            }
            
            .danger-zone .button {
                margin-top: var(--space-sm);
            }
            
            .warning-text {
                color: var(--warning-color);
                font-size: var(--text-sm);
                font-weight: 600;
                line-height: var(--leading-normal);
                margin: var(--space-xs) 0 0 0;
            }
            
            /* Buttons */
            .button {
                padding: var(--space-md) var(--space-lg);
                font-size: var(--text-sm);
                border-radius: 4px;
                border: 1px solid;
                cursor: pointer;
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                line-height: var(--leading-tight);
                white-space: nowrap;
                transition: var(--transition);
                font-weight: 500;
                min-height: 44px;
                box-sizing: border-box;
            }
            
            .button-primary {
                background: #2271b1;
                border-color: #2271b1;
                color: white;
            }
            
            .button-primary:hover {
                background: #135e96;
                border-color: #135e96;
                transform: translateY(-1px);
            }
            
            .button-secondary {
                background: #f6f7f7;
                border-color: #2271b1;
                color: #2271b1;
            }
            
            .button-secondary:hover {
                background: #e0e0e0;
                transform: translateY(-1px);
            }
            
            .button-danger {
                background: #d63638;
                border-color: #d63638;
                color: white;
            }
            
            .button-danger:hover {
                background: #b32d2e;
                border-color: #b32d2e;
                transform: translateY(-1px);
            }
            
            /* Table Styling */
            .kb-table-container {
                background: var(--bg-primary);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                overflow: hidden;
                margin-top: var(--space-xl);
            }
            
            .kb-table {
                width: 100%;
                border-collapse: collapse;
                margin: 0;
            }
            
            .kb-table thead {
                background: var(--bg-secondary);
            }
            
            .kb-table th {
                padding: var(--space-lg);
                text-align: left;
                font-weight: 600;
                color: var(--text-primary);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
                border-bottom: 2px solid var(--border-color);
            }
            
            .kb-table td {
                padding: var(--space-lg);
                vertical-align: top;
                border-bottom: 1px solid var(--border-color);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            .kb-table tbody tr:last-child td {
                border-bottom: none;
            }
            
            .kb-table tbody tr:hover {
                background: var(--bg-tertiary);
            }
            
            /* Text snippet styling */
            .text-snippet {
                background: var(--bg-secondary);
                padding: var(--space-md);
                border-radius: 4px;
                max-height: 120px;
                overflow: auto;
                font-size: var(--text-sm);
                line-height: var(--leading-relaxed);
                white-space: pre-wrap;
                word-wrap: break-word;
                margin: 0;
            }
            
            .text-snippet::-webkit-scrollbar {
                width: 6px;
            }
            
            .text-snippet::-webkit-scrollbar-track {
                background: var(--bg-primary);
                border-radius: 3px;
            }
            
            .text-snippet::-webkit-scrollbar-thumb {
                background: var(--border-color);
                border-radius: 3px;
            }
            
            .text-snippet::-webkit-scrollbar-thumb:hover {
                background: var(--text-light);
            }
            
            /* Empty state */
            .empty-state {
                text-align: center;
                padding: var(--space-3xl);
                color: var(--text-secondary);
            }
            
            .empty-state p {
                font-size: var(--text-base);
                line-height: var(--leading-normal);
                margin: 0;
            }
            
            /* Responsive adjustments */
            @media (max-width: 1024px) {
                .zencode-kb-container {
                    padding: var(--space-xl);
                    width: calc(100% - var(--space-lg));
                }
                
                .kb-table-container {
                    overflow-x: auto;
                }
            }
            
            @media (max-width: 768px) {
                .zencode-kb-container {
                    padding: var(--space-lg);
                    width: calc(100% - var(--space-md));
                }
                
                .zencode-kb-header {
                    margin-bottom: var(--space-xl);
                }
                
                .danger-zone {
                    padding: var(--space-md);
                }
                
                .kb-table {
                    min-width: 600px;
                }
                
                .kb-table th,
                .kb-table td {
                    padding: var(--space-md);
                }
                
                .text-snippet {
                    max-height: 100px;
                    padding: var(--space-sm);
                }
            }
            
            @media (max-width: 480px) {
                .zencode-kb-container {
                    padding: var(--space-md);
                    width: calc(100% - var(--space-sm));
                }
                
                .zencode-kb-header h1 {
                    font-size: var(--text-2xl);
                }
                
                .danger-zone {
                    text-align: center;
                }
                
                .button {
                    width: 100%;
                    justify-content: center;
                }
                
                .empty-state {
                    padding: var(--space-2xl) var(--space-md);
                }
            }
            
            @media (max-width: 360px) {
                .kb-table th,
                .kb-table td {
                    padding: var(--space-sm);
                }
                
                .text-snippet {
                    max-height: 80px;
                }
            }
            
            /* Focus styles for accessibility */
            button:focus-visible {
                outline: 2px solid var(--secondary-color);
                outline-offset: 2px;
            }
            
            /* Loading states */
            .button:disabled {
                opacity: 0.6;
                cursor: not-allowed;
                transform: none !important;
            }
            
            /* Status messages */
            .status-message {
                padding: var(--space-md);
                border-radius: 4px;
                margin: var(--space-sm) 0;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            .status-success-bg {
                background: #d4edda;
                color: #155724;
                border-left: 4px solid var(--success-color);
            }
            
            .status-error-bg {
                background: #f8d7da;
                color: #721c24;
                border-left: 4px solid var(--warning-color);
            }
        </style>
        
        <div class="zencode-kb-container">
            <div class="zencode-kb-header">
                <h1>🧠 Knowledge Base Management</h1>
                <p>Manage all knowledge base vectors across all users. As an administrator, you can view and delete entries.</p>
            </div>
            
            <?php if (!empty($vectors)) : ?>
                <div class="danger-zone">
                    <h3 style="margin: 0 0 var(--space-sm) 0; color: var(--warning-color); font-size: var(--text-lg); line-height: var(--leading-tight);">
                        ⚠️ Global Management Actions
                    </h3>
                    <p style="margin: 0; color: var(--text-secondary); font-size: var(--text-sm); line-height: var(--leading-normal);">
                        These actions affect <strong>all users</strong> and cannot be undone.
                    </p>
                    <button id="delete-all-vectors-btn" class="button button-danger">
                        🗑️ Delete All Vectors (Global)
                    </button>
                    <p class="warning-text">
                        <strong>Warning:</strong> This will permanently delete all knowledge base entries for all users.
                    </p>
                </div>
            <?php endif; ?>
            
            <div class="kb-table-container">
                <table class="kb-table">
                    <thead>
                        <tr>
                            <th style="width: 8%;">ID</th>
                            <th style="width: 20%;">Owner</th>
                            <th>Source Text Preview</th>
                            <th style="width: 12%;">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($vectors)) : ?>
                            <tr>
                                <td colspan="4">
                                    <div class="empty-state">
                                        <p>No vectors found in the knowledge base.</p>
                                    </div>
                                </td>
                            </tr>
                        <?php else : ?>
                            <?php foreach ($vectors as $vector) : 
                                $user_info = get_userdata($vector->user_id);
                                $owner_name = $user_info ? esc_html($user_info->display_name) . " (ID: {$vector->user_id})" : "Unknown User (ID: {$vector->user_id})";
                            ?>
                                <tr id="vector-row-<?php echo esc_attr($vector->id); ?>">
                                    <td>
                                        <code style="font-size: var(--text-xs); background: var(--bg-secondary); padding: var(--space-xs) var(--space-sm); border-radius: 3px; line-height: var(--leading-tight);">
                                            <?php echo esc_html($vector->id); ?>
                                        </code>
                                    </td>
                                    <td>
                                        <span style="font-size: var(--text-sm); line-height: var(--leading-normal);">
                                            <?php echo $owner_name; ?>
                                        </span>
                                    </td>
                                    <td>
                                        <pre class="text-snippet"><?php echo esc_html(substr($vector->source_text, 0, 400)); ?>...</pre>
                                    </td>
                                    <td>
                                        <button class="button button-secondary delete-vector-btn" 
                                                data-id="<?php echo esc_attr($vector->id); ?>"
                                                style="min-width: 80px;">
                                            Delete
                                        </button>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                // ===================================================================
                // THE FIX: Use the main plugin nonce to match the backend.
                // ===================================================================
                const nonce = '<?php echo wp_create_nonce('zencode_ai_nonce'); ?>';
                // ===================================================================

                // Delete single vector
                $('.delete-vector-btn').on('click', function() { 
                    if (!confirm('Are you sure you want to delete this vector?')) return;
                    
                    const btn = $(this); 
                    const vectorId = btn.data('id'); 
                    btn.prop('disabled', true).text('Deleting...');
                    
                    $.post(ajaxurl, { 
                        action: 'zencode_ai_delete_vector', 
                        nonce: nonce, 
                        vector_id: vectorId 
                    }, function(response) {
                        if (response.success) { 
                            $('#vector-row-' + vectorId).fadeOut(400, function() { 
                                $(this).remove(); 
                                // If no vectors left, reload to show empty state
                                if ($('.delete-vector-btn').length === 0) {
                                    location.reload();
                                }
                            }); 
                        } else { 
                            alert('Error: ' + response.data.message); 
                            btn.prop('disabled', false).text('Delete'); 
                        }
                    }).fail(function() {
                        alert('Request failed. Please try again.');
                        btn.prop('disabled', false).text('Delete');
                    });
                });
                
                // Delete all vectors
                $('#delete-all-vectors-btn').on('click', function() { 
                    if (!confirm('⚠️ CRITICAL ACTION\n\nAre you ABSOLUTELY sure you want to delete ALL vectors for ALL users?\n\nThis action cannot be undone and will affect every user in the system.')) return;
                    
                    if (!confirm('FINAL WARNING: This will permanently delete the entire knowledge base.\n\nAre you sure you want to continue?')) return;
                    
                    const btn = $(this); 
                    btn.prop('disabled', true).text('Deleting All Vectors...');
                    
                    $.post(ajaxurl, { 
                        action: 'zencode_ai_delete_vector', 
                        nonce: nonce, 
                        delete_all: 'true' 
                    }, function(response) {
                        if (response.success) { 
                            alert('✅ All vectors have been deleted successfully.'); 
                            location.reload(); 
                        } else { 
                            alert('❌ Error: ' + response.data.message); 
                            btn.prop('disabled', false).text('Delete All Vectors (Global)'); 
                        }
                    }).fail(function() {
                        alert('❌ Request failed. Please try again.');
                        btn.prop('disabled', false).text('Delete All Vectors (Global)');
                    });
                });
            });
        </script>
    </div>
    <?php
}
/**
 * Render the Learning Manager page
 */
public function render_learning_manager_page() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_learnings';
    $user_id = get_current_user_id();
    
    // Get filter parameters
    $score_filter = isset($_GET['score']) ? intval($_GET['score']) : 0;
    $sort_by = isset($_GET['sort']) ? sanitize_text_field($_GET['sort']) : 'date_desc';
    $per_page = 5;
    $paged = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
    $offset = ($paged - 1) * $per_page;
    
    // Build query
    $where = $wpdb->prepare("WHERE user_id = %d", $user_id);
    if ($score_filter > 0) {
        $where .= $wpdb->prepare(" AND score < %d", $score_filter);
    }
    
    $order_by = "ORDER BY created_at DESC";
    switch ($sort_by) {
        case 'score_asc': $order_by = "ORDER BY score ASC"; break;
        case 'score_desc': $order_by = "ORDER BY score DESC"; break;
        case 'usage_desc': $order_by = "ORDER BY usage_count DESC"; break;
        case 'success_desc': $order_by = "ORDER BY success_rate DESC"; break;
        case 'date_asc': $order_by = "ORDER BY created_at ASC"; break;
    }
    
    // Get total count for pagination
    $total_learnings = $wpdb->get_var("SELECT COUNT(*) FROM {$table_name} {$where}");
    $total_pages = ceil($total_learnings / $per_page);
    
    // Get learnings
    $learnings = $wpdb->get_results("
        SELECT * FROM {$table_name} 
        {$where} 
        {$order_by}
        LIMIT {$per_page} OFFSET {$offset}
    ");
    
    // Get stats using the improved function - direct call, no AJAX
    $stats = $this->handle_get_learning_stats();
    
    // Build base URL for pagination that preserves filters
    $base_url = add_query_arg([
        'page' => 'zencode-ai-learning-manager',
        'score' => $score_filter,
        'sort' => $sort_by
    ], admin_url('admin.php'));
    
    ?>
    <div class="wrap zencode-learning-manager">
        <style>
            /* CSS Custom Properties for consistent theming */
            :root {
                --primary-color: #a855f7;
                --secondary-color: #60a5fa;
                --success-color: #059669;
                --warning-color: #dc2626;
                --text-primary: #1f2937;
                --text-secondary: #6b7280;
                --text-light: #9ca3af;
                --bg-primary: #ffffff;
                --bg-secondary: #f8f9fa;
                --bg-tertiary: #f0f7ff;
                --border-color: #e5e7eb;
                --border-radius: 8px;
                --shadow: 0 2px 8px rgba(0,0,0,0.1);
                --transition: all 0.2s ease-in-out;
                
                /* Responsive spacing scale */
                --space-xs: clamp(4px, 1vw, 6px);
                --space-sm: clamp(8px, 2vw, 12px);
                --space-md: clamp(12px, 2.5vw, 16px);
                --space-lg: clamp(16px, 3vw, 20px);
                --space-xl: clamp(20px, 4vw, 24px);
                --space-2xl: clamp(24px, 5vw, 32px);
                --space-3xl: clamp(32px, 6vw, 40px);
                
                /* Typography scale */
                --text-xs: clamp(0.75rem, 2vw, 0.8rem);
                --text-sm: clamp(0.8rem, 2.2vw, 0.9rem);
                --text-base: clamp(0.9rem, 2.5vw, 1rem);
                --text-lg: clamp(1rem, 3vw, 1.125rem);
                --text-xl: clamp(1.125rem, 3.5vw, 1.25rem);
                --text-2xl: clamp(1.25rem, 4vw, 1.5rem);
                --text-3xl: clamp(1.5rem, 5vw, 1.875rem);
                --text-4xl: clamp(1.75rem, 6vw, 2.25rem);
                
                /* Line heights */
                --leading-tight: 1.25;
                --leading-normal: 1.5;
                --leading-relaxed: 1.75;
            }
            
            /* Base container */
            .zencode-learning-manager {
                max-width: 1200px;
                margin: 0 auto;
                padding: 0 var(--space-md);
                box-sizing: border-box;
            }
            
            /* Header */
            .zencode-learning-manager h1 {
                color: var(--primary-color);
                font-size: var(--text-3xl);
                margin: var(--space-lg) 0 var(--space-xl) 0;
                font-weight: 700;
                line-height: var(--leading-tight);
            }
            
            /* Stats Dashboard - Matching settings page */
            .learning-stats-dashboard {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(min(100%, 280px), 1fr));
                gap: var(--space-lg);
                margin: var(--space-2xl) 0 var(--space-3xl) 0;
            }
            
            .stat-card {
                background: var(--bg-primary);
                padding: var(--space-2xl);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                text-align: center;
                box-sizing: border-box;
                transition: var(--transition);
            }
            
            .stat-card:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }
            
            .stat-value {
                font-size: var(--text-3xl);
                font-weight: 700;
                color: var(--secondary-color);
                line-height: var(--leading-tight);
                margin: 0 0 var(--space-xs) 0;
            }
            
            .stat-label {
                color: var(--text-secondary);
                font-size: var(--text-sm);
                font-weight: 600;
                line-height: var(--leading-normal);
                margin: 0 0 var(--space-sm) 0;
            }
            
            .stat-subtext {
                color: var(--text-light);
                font-size: var(--text-xs);
                line-height: var(--leading-normal);
                margin: var(--space-xs) 0 0 0;
            }
            
            .usage-breakdown {
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: var(--space-sm);
                margin-top: var(--space-lg);
            }
            
            .usage-item {
                text-align: center;
                padding: var(--space-md);
                border-radius: 6px;
                font-size: var(--text-xs);
                line-height: var(--leading-tight);
            }
            
            .usage-item.used {
                background: #d4edda;
                color: #155724;
            }
            
            .usage-item.unused {
                background: #fff3cd;
                color: #856404;
            }
            
            .usage-count {
                font-weight: 700;
                font-size: var(--text-base);
                line-height: var(--leading-tight);
                margin: 0 0 2px 0;
            }
            
            .usage-label {
                font-size: var(--text-xs);
                line-height: var(--leading-tight);
            }
            
            /* Chart Container */
            .learning-chart-container {
                background: var(--bg-primary);
                padding: var(--space-2xl);
                margin-bottom: var(--space-xl);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
            }
            
            .learning-chart-container h2 {
                margin: 0 0 var(--space-lg) 0;
                font-size: var(--text-xl);
                color: var(--text-primary);
                font-weight: 600;
                line-height: var(--leading-tight);
            }
            
            .chart-wrapper {
                position: relative;
                height: 300px;
                width: 100%;
            }
            
            /* Controls Section */
            .learning-controls {
                background: var(--bg-primary);
                padding: var(--space-lg);
                border-radius: var(--border-radius);
                margin-bottom: var(--space-xl);
                display: flex;
                justify-content: space-between;
                align-items: center;
                flex-wrap: wrap;
                gap: var(--space-md);
                box-shadow: var(--shadow);
            }
            
            .learning-filters {
                display: flex;
                flex-wrap: wrap;
                gap: var(--space-md);
                align-items: center;
            }
            
            .learning-filters label {
                display: flex;
                align-items: center;
                gap: var(--space-sm);
                font-weight: 600;
                color: var(--text-primary);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            .learning-filters select {
                padding: var(--space-sm);
                border: 1px solid var(--border-color);
                border-radius: 4px;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
                min-width: 150px;
            }
            
            .bulk-actions {
                display: flex;
                flex-wrap: wrap;
                gap: var(--space-sm);
            }
            
            /* Buttons */
            .button {
                padding: var(--space-md) var(--space-lg);
                font-size: var(--text-sm);
                border-radius: 4px;
                border: 1px solid;
                cursor: pointer;
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                line-height: var(--leading-tight);
                white-space: nowrap;
                transition: var(--transition);
                font-weight: 500;
                min-height: 44px;
                box-sizing: border-box;
            }
            
            .button-primary {
                background: #2271b1;
                border-color: #2271b1;
                color: white;
            }
            
            .button-secondary {
                background: #f6f7f7;
                border-color: #2271b1;
                color: #2271b1;
            }
            
            .button-danger {
                background: #d63638;
                border-color: #d63638;
                color: white;
            }
            
            .button-small {
                padding: var(--space-sm) var(--space-md);
                font-size: var(--text-xs);
                min-height: 36px;
            }
            
            /* Learnings List */
            .learnings-list {
                background: var(--bg-primary);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                overflow: hidden;
            }
            
            .learning-card {
                display: grid;
                grid-template-columns: auto 1fr auto;
                gap: var(--space-lg);
                padding: var(--space-xl);
                border-bottom: 1px solid var(--border-color);
                align-items: flex-start;
            }
            
            .learning-card.unused {
                opacity: 0.7;
                background: var(--bg-secondary);
            }
            
            .learning-card:last-child {
                border-bottom: none;
            }
            
            .learning-checkbox {
                display: flex;
                align-items: flex-start;
                padding-top: var(--space-xs);
            }
            
            .learning-content {
                flex: 1;
                min-width: 0;
            }
            
            .learning-header {
                display: flex;
                justify-content: space-between;
                align-items: flex-start;
                flex-wrap: wrap;
                gap: var(--space-sm);
                margin-bottom: var(--space-lg);
            }
            
            .learning-score {
                padding: var(--space-xs) var(--space-sm);
                border-radius: 4px;
                font-weight: 700;
                font-size: var(--text-sm);
                line-height: var(--leading-tight);
                white-space: nowrap;
            }
            
            .score-high { background: #d4edda; color: #155724; }
            .score-medium { background: #fff3cd; color: #856404; }
            .score-low { background: #f8d7da; color: #721c24; }
            
            .learning-meta {
                color: var(--text-secondary);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
                flex: 1;
                min-width: min(100%, 300px);
            }
            
            .learning-type {
                margin-bottom: var(--space-md);
            }
            
            .type-badge {
                background: #e7f3ff;
                color: #2271b1;
                padding: var(--space-xs) var(--space-sm);
                border-radius: 3px;
                font-size: var(--text-xs);
                font-weight: 500;
                line-height: var(--leading-tight);
            }
            
            .learning-step,
            .learning-rule,
            .learning-mistake {
                margin-bottom: var(--space-md);
            }
            
            .learning-step strong,
            .learning-rule strong,
            .learning-mistake strong {
                display: block;
                margin-bottom: var(--space-xs);
                color: var(--text-primary);
                font-size: var(--text-sm);
                font-weight: 600;
                line-height: var(--leading-normal);
            }
            
            .step-text,
            .rule-text,
            .mistake-text {
                padding: var(--space-md);
                background: var(--bg-secondary);
                border-radius: 4px;
                font-size: var(--text-sm);
                line-height: var(--leading-relaxed);
                word-wrap: break-word;
                overflow-wrap: break-word;
            }
            
            .learning-confidence {
                margin-top: var(--space-md);
            }
            
            .confidence-value {
                background: var(--bg-secondary);
                padding: var(--space-xs) var(--space-sm);
                border-radius: 3px;
                font-size: var(--text-xs);
                font-weight: 500;
                line-height: var(--leading-tight);
            }
            
            .learning-actions {
                display: flex;
                align-items: flex-start;
                padding-top: var(--space-xs);
            }
            
            .no-learnings {
                padding: var(--space-3xl) var(--space-xl);
                text-align: center;
            }
            
            /* Pagination */
            .tablenav {
                margin: var(--space-xl) 0;
                display: flex;
                justify-content: space-between;
                align-items: center;
                flex-wrap: wrap;
                gap: var(--space-md);
            }
            
            .tablenav-pages {
                display: flex;
                align-items: center;
                gap: var(--space-md);
                flex-wrap: wrap;
            }
            
            .displaying-num {
                color: var(--text-secondary);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            /* Responsive adjustments */
            @media (max-width: 1024px) {
                .learning-controls {
                    flex-direction: column;
                    align-items: stretch;
                }
                
                .learning-filters {
                    justify-content: space-between;
                }
                
                .bulk-actions {
                    justify-content: center;
                }
            }
            
            @media (max-width: 768px) {
                .zencode-learning-manager {
                    padding: 0 var(--space-sm);
                }
                
                .learning-stats-dashboard {
                    grid-template-columns: 1fr;
                    gap: var(--space-md);
                }
                
                .learning-chart-container {
                    padding: var(--space-lg);
                }
                
                .chart-wrapper {
                    height: 250px;
                }
                
                .learning-controls {
                    padding: var(--space-md);
                }
                
                .learning-filters {
                    flex-direction: column;
                    align-items: stretch;
                    gap: var(--space-md);
                }
                
                .learning-filters label {
                    flex-direction: column;
                    align-items: flex-start;
                    gap: var(--space-xs);
                }
                
                .learning-filters select {
                    min-width: 100%;
                }
                
                .bulk-actions {
                    justify-content: stretch;
                }
                
                .bulk-actions .button {
                    flex: 1;
                    min-width: 0;
                }
                
                .learning-card {
                    grid-template-columns: 1fr;
                    gap: var(--space-md);
                    padding: var(--space-lg);
                }
                
                .learning-checkbox {
                    order: 1;
                }
                
                .learning-content {
                    order: 2;
                }
                
                .learning-actions {
                    order: 3;
                    justify-content: flex-end;
                }
                
                .learning-header {
                    flex-direction: column;
                    align-items: flex-start;
                }
                
                .tablenav {
                    flex-direction: column;
                    align-items: stretch;
                    text-align: center;
                }
                
                .tablenav-pages {
                    justify-content: center;
                }
            }
            
            @media (max-width: 480px) {
                .learning-stats-dashboard {
                    margin: var(--space-xl) 0;
                }
                
                .stat-card {
                    padding: var(--space-lg);
                }
                
                .learning-chart-container {
                    padding: var(--space-md);
                }
                
                .chart-wrapper {
                    height: 200px;
                }
                
                .learning-controls {
                    padding: var(--space-md);
                }
                
                .learning-card {
                    padding: var(--space-md);
                }
                
                .step-text,
                .rule-text,
                .mistake-text {
                    padding: var(--space-sm);
                }
                
                .usage-breakdown {
                    grid-template-columns: 1fr;
                    gap: var(--space-xs);
                }
            }
            
            @media (max-width: 360px) {
                .bulk-actions {
                    flex-direction: column;
                }
                
                .bulk-actions .button {
                    width: 100%;
                }
            }
            
            /* Focus styles for accessibility */
            button:focus-visible,
            input:focus-visible,
            select:focus-visible {
                outline: 2px solid var(--secondary-color);
                outline-offset: 2px;
            }
            
            /* Loading states */
            .button:disabled {
                opacity: 0.6;
                cursor: not-allowed;
                transform: none !important;
            }
            
            /* Checkbox styling */
            .learning-select {
                width: 18px;
                height: 18px;
                margin: 0;
            }
        </style>
        
        <h1>🧠 AI Learning Manager</h1>
        
        <!-- Enhanced Stats Dashboard -->
        <div class="learning-stats-dashboard">
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total_learnings']; ?></div>
                <div class="stat-label">Total Learnings</div>
                <div class="usage-breakdown">
                    <div class="usage-item used">
                        <div class="usage-count"><?php echo $stats['used_learnings_count']; ?></div>
                        <div class="usage-label">Applied</div>
                    </div>
                    <div class="usage-item unused">
                        <div class="usage-count"><?php echo $stats['unused_learnings_count']; ?></div>
                        <div class="usage-label">Unused</div>
                    </div>
                </div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['avg_success_rate']; ?>%</div>
                <div class="stat-label">Success Rate</div>
                <div class="stat-subtext">
                    Based on <?php echo $stats['used_learnings_count']; ?> applied learnings
                </div>
                <?php if ($stats['best_success_rate'] > 0): ?>
                <div class="stat-subtext">
                    Best: <?php echo $stats['best_success_rate']; ?>%
                </div>
                <?php endif; ?>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?php echo $stats['total_applications']; ?></div>
                <div class="stat-label">Total Uses</div>
                <div class="stat-subtext">
                    Avg Score: <?php echo $stats['avg_score']; ?>/10
                </div>
            </div>
        </div>
        
        <!-- Learning Progress Chart -->
        <div class="learning-chart-container">
            <h2>📈 Learning Progress Over Time</h2>
            <div class="chart-wrapper">
                <canvas id="learningChart"></canvas>
            </div>
        </div>
        
        <!-- Filters & Actions -->
        <div class="learning-controls">
            <form method="get" class="learning-filters">
                <input type="hidden" name="page" value="zencode-ai-learning-manager" />
                
                <label>
                    <strong>Filter:</strong>
                    <select name="score" onchange="this.form.submit()">
                        <option value="0">All Learnings</option>
                        <option value="3" <?php selected($score_filter, 3); ?>>Below 3/10</option>
                        <option value="4" <?php selected($score_filter, 4); ?>>Below 4/10</option>
                        <option value="5" <?php selected($score_filter, 5); ?>>Below 5/10</option>
                        <option value="6" <?php selected($score_filter, 6); ?>>Below 6/10</option>
                    </select>
                </label>
                
                <label>
                    <strong>Sort by:</strong>
                    <select name="sort" onchange="this.form.submit()">
                        <option value="date_desc" <?php selected($sort_by, 'date_desc'); ?>>Newest First</option>
                        <option value="date_asc" <?php selected($sort_by, 'date_asc'); ?>>Oldest First</option>
                        <option value="score_desc" <?php selected($sort_by, 'score_desc'); ?>>Highest Score</option>
                        <option value="score_asc" <?php selected($sort_by, 'score_asc'); ?>>Lowest Score</option>
                        <option value="usage_desc" <?php selected($sort_by, 'usage_desc'); ?>>Most Used</option>
                        <option value="success_desc" <?php selected($sort_by, 'success_desc'); ?>>Best Success</option>
                    </select>
                </label>
            </form>
            
            <div class="bulk-actions">
                <button id="select-all-learnings" class="button button-secondary">Select All</button>
                <button id="deselect-all-learnings" class="button button-secondary">Deselect All</button>
                <button id="delete-selected" class="button button-danger" disabled>Delete Selected</button>
            </div>
        </div>
        
        <!-- Learnings List -->
        <div class="learnings-list">
            <?php if (empty($learnings)): ?>
                <div class="no-learnings">
                    <p style="text-align: center; padding: var(--space-3xl); color: var(--text-secondary); font-size: var(--text-base); line-height: var(--leading-normal);">
                        No learnings found. Start building projects to train your AI!
                    </p>
                </div>
            <?php else: ?>
                <?php foreach ($learnings as $learning): ?>
                    <?php
                    $score_class = 'low';
                    if ($learning->score >= 8) $score_class = 'high';
                    elseif ($learning->score >= 5) $score_class = 'medium';
                    
                    $usage_class = $learning->usage_count > 0 ? 'used' : 'unused';
                    ?>
                    <div class="learning-card <?php echo $usage_class; ?>" data-id="<?php echo $learning->id; ?>">
                        <div class="learning-checkbox">
                            <input type="checkbox" class="learning-select" 
                                   data-id="<?php echo $learning->id; ?>" />
                        </div>
                        
                        <div class="learning-content">
                            <div class="learning-header">
                                <span class="learning-score score-<?php echo $score_class; ?>">
                                    <?php echo $learning->score; ?>/10
                                </span>
                                <span class="learning-meta">
                                    <?php if ($learning->usage_count > 0): ?>
                                        Used: <strong><?php echo $learning->usage_count; ?></strong> times | 
                                        Success: <strong><?php echo round($learning->success_rate * 100); ?>%</strong> | 
                                        Last used: 
                                        <?php if (!empty($learning->last_used)): ?>
                                            <?php echo human_time_diff(strtotime($learning->last_used), current_time('timestamp')); ?> ago
                                        <?php else: ?>
                                            Never
                                        <?php endif; ?>
                                    <?php else: ?>
                                        <em>Not used yet</em> | 
                                        Created: <?php echo human_time_diff(strtotime($learning->created_at), current_time('timestamp')); ?> ago
                                    <?php endif; ?>
                                </span>
                            </div>
                            
                            <?php if (!empty($learning->learning_type)): ?>
                            <div class="learning-type">
                                <strong>Type:</strong>
                                <span class="type-badge"><?php echo esc_html($learning->learning_type); ?></span>
                            </div>
                            <?php endif; ?>
                            
                            <div class="learning-step">
                                <strong>Step Pattern:</strong>
                                <div class="step-text">
                                    <?php 
                                    $step_pattern = !empty($learning->step_pattern) ? $learning->step_pattern : 
                                                  (!empty($learning->pattern_data) ? $learning->pattern_data : '');
                                    echo esc_html(substr($step_pattern, 0, 150)); 
                                    ?>
                                    <?php if (strlen($step_pattern) > 150): ?>...<?php endif; ?>
                                </div>
                            </div>
                            
                            <div class="learning-rule">
                                <strong>Learned Rule:</strong>
                                <div class="rule-text">
                                    <?php echo esc_html($learning->rule); ?>
                                </div>
                            </div>
                            
                            <?php if (!empty($learning->mistake_made)): ?>
                            <div class="learning-mistake">
                                <strong>Original Mistake:</strong>
                                <div class="mistake-text">
                                    <?php echo esc_html($learning->mistake_made); ?>
                                </div>
                            </div>
                            <?php endif; ?>
                            
                            <?php if (!empty($learning->confidence_score)): ?>
                            <div class="learning-confidence">
                                <strong>Confidence:</strong>
                                <span class="confidence-value"><?php echo round($learning->confidence_score * 100); ?>%</span>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="learning-actions">
                            <button class="button button-small button-danger delete-single" 
                                    data-id="<?php echo $learning->id; ?>">
                                Delete
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
        
        <!-- Pagination -->
        <?php if ($total_pages > 1): ?>
        <div class="tablenav">
            <div class="tablenav-pages">
                <span class="displaying-num">
                    <?php 
                    $start = ($paged - 1) * $per_page + 1;
                    $end = min($paged * $per_page, $total_learnings);
                    echo sprintf('Displaying %d-%d of %d learnings', $start, $end, $total_learnings);
                    ?>
                </span>
                <?php
                echo paginate_links([
                    'base' => add_query_arg('paged', '%#%', $base_url),
                    'format' => '',
                    'prev_text' => '&laquo;',
                    'next_text' => '&raquo;',
                    'total' => $total_pages,
                    'current' => $paged,
                    'show_all' => false,
                    'end_size' => 1,
                    'mid_size' => 2
                ]);
                ?>
            </div>
        </div>
        <?php endif; ?>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script>
    jQuery(document).ready(function($) {
        const nonce = '<?php echo wp_create_nonce('zencode_ai_nonce'); ?>';
        
        // Load learning history chart
        $.post(ajaxurl, {
            action: 'zencode_ai_get_learning_history',
            nonce: nonce
        }, function(response) {
            if (response.success && response.data.dates.length > 0) {
                const ctx = document.getElementById('learningChart');
                new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: response.data.dates,
                        datasets: [{
                            label: 'Average Score',
                            data: response.data.scores,
                            borderColor: '#2271b1',
                            backgroundColor: 'rgba(34, 113, 177, 0.1)',
                            tension: 0.4,
                            fill: true,
                            pointRadius: 4,
                            pointHoverRadius: 6
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        plugins: {
                            legend: {
                                display: false
                            },
                            tooltip: {
                                callbacks: {
                                    label: function(context) {
                                        return 'Score: ' + context.parsed.y + '/10';
                                    }
                                }
                            }
                        },
                        scales: {
                            y: {
                                beginAtZero: true,
                                max: 10,
                                ticks: {
                                    stepSize: 1
                                },
                                title: {
                                    display: true,
                                    text: 'Average Score'
                                }
                            },
                            x: {
                                title: {
                                    display: true,
                                    text: 'Date'
                                }
                            }
                        }
                    }
                });
            } else {
                // Not enough data yet
                $('.chart-wrapper').html(
                    '<p style="text-align: center; padding: 80px 20px; color: var(--text-secondary); font-size: var(--text-base); line-height: var(--leading-normal);">' +
                    '📊 Build more projects to see your learning progress over time!' +
                    '</p>'
                );
            }
        }).fail(function() {
            $('.chart-wrapper').html(
                '<p style="text-align: center; padding: 80px 20px; color: var(--warning-color); font-size: var(--text-base); line-height: var(--leading-normal);">Failed to load chart data.</p>'
            );
        });
        
        // Select/Deselect All
        $('#select-all-learnings').on('click', function() {
            $('.learning-select').prop('checked', true);
            updateDeleteButton();
        });
        
        $('#deselect-all-learnings').on('click', function() {
            $('.learning-select').prop('checked', false);
            updateDeleteButton();
        });
        
        // Update delete button state
        $('.learning-select').on('change', updateDeleteButton);
        
        function updateDeleteButton() {
            const selected = $('.learning-select:checked').length;
            $('#delete-selected')
                .prop('disabled', selected === 0)
                .text(selected > 0 ? `Delete Selected (${selected})` : 'Delete Selected');
        }
        
        // Delete selected learnings
        $('#delete-selected').on('click', function() {
            const selected = $('.learning-select:checked');
            const count = selected.length;
            
            if (!confirm(`Delete ${count} selected learning${count > 1 ? 's' : ''}?`)) {
                return;
            }
            
            const ids = selected.map(function() {
                return $(this).data('id');
            }).get();
            
            const btn = $(this);
            btn.prop('disabled', true).text('Deleting...');
            
            $.post(ajaxurl, {
                action: 'zencode_ai_delete_learnings',
                nonce: nonce,
                ids: ids
            }, function(response) {
                if (response.success) {
                    alert('✅ ' + response.data.message);
                    location.reload();
                } else {
                    alert('❌ Error: ' + (response.data?.message || 'Unknown error'));
                    btn.prop('disabled', false);
                    updateDeleteButton();
                }
            }).fail(function() {
                alert('❌ Request failed. Please try again.');
                btn.prop('disabled', false);
                updateDeleteButton();
            });
        });
        
        // Delete individual learning
        $('.delete-single').on('click', function() {
            if (!confirm('Delete this learning?')) {
                return;
            }
            
            const id = $(this).data('id');
            const card = $(this).closest('.learning-card');
            const btn = $(this);
            
            btn.prop('disabled', true).text('Deleting...');
            
            $.post(ajaxurl, {
                action: 'zencode_ai_delete_learnings',
                nonce: nonce,
                ids: [id]
            }, function(response) {
                if (response.success) {
                    card.fadeOut(300, function() {
                        $(this).remove();
                        if ($('.learning-card').length === 0) {
                            location.reload();
                        }
                    });
                } else {
                    alert('❌ Error: ' + (response.data?.message || 'Unknown error'));
                    btn.prop('disabled', false).text('Delete');
                }
            }).fail(function() {
                alert('❌ Request failed. Please try again.');
                btn.prop('disabled', false).text('Delete');
            });
        });
    });
    </script>
    <?php
}
/**
 * Helper function to get score CSS class
 */
private function get_score_class($score) {
    if ($score >= 8) return 'high';
    if ($score >= 5) return 'medium';
    return 'low';
}
  public function register_settings() {
    $settings = [
        'zencode_ai_google_api_key',
        'zencode_ai_default_model',
        'zencode_ai_max_tokens',
        'zencode_ai_temperature',
        'zencode_ai_enable_caching',
    ];
    foreach ($settings as $setting) {
        register_setting('zencode_ai_settings', $setting);
    }
}
public function settings_page() {
    ?>
    <div class="wrap">
        <style>
            /* CSS Custom Properties for consistent theming */
            :root {
                --primary-color: #a855f7;
                --secondary-color: #60a5fa;
                --success-color: #059669;
                --warning-color: #dc2626;
                --text-primary: #1f2937;
                --text-secondary: #6b7280;
                --text-light: #9ca3af;
                --bg-primary: #ffffff;
                --bg-secondary: #f8f9fa;
                --bg-tertiary: #f0f7ff;
                --border-color: #e5e7eb;
                --border-radius: 8px;
                --shadow: 0 2px 8px rgba(0,0,0,0.1);
                --transition: all 0.2s ease-in-out;
                
                /* Responsive spacing scale */
                --space-xs: clamp(4px, 1vw, 6px);
                --space-sm: clamp(8px, 2vw, 12px);
                --space-md: clamp(12px, 2.5vw, 16px);
                --space-lg: clamp(16px, 3vw, 20px);
                --space-xl: clamp(20px, 4vw, 24px);
                --space-2xl: clamp(24px, 5vw, 32px);
                --space-3xl: clamp(32px, 6vw, 40px);
                
                /* Typography scale */
                --text-xs: clamp(0.75rem, 2vw, 0.8rem);
                --text-sm: clamp(0.8rem, 2.2vw, 0.9rem);
                --text-base: clamp(0.9rem, 2.5vw, 1rem);
                --text-lg: clamp(1rem, 3vw, 1.125rem);
                --text-xl: clamp(1.125rem, 3.5vw, 1.25rem);
                --text-2xl: clamp(1.25rem, 4vw, 1.5rem);
                --text-3xl: clamp(1.5rem, 5vw, 1.875rem);
                --text-4xl: clamp(1.75rem, 6vw, 2.25rem);
                
                /* Line heights */
                --leading-tight: 1.25;
                --leading-normal: 1.5;
                --leading-relaxed: 1.75;
            }
            
            /* Base container and reset */
            .zencode-settings-container {
                background: var(--bg-primary);
                padding: var(--space-2xl);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                margin: 0 auto;
                max-width: 1200px;
                box-sizing: border-box;
                width: calc(100% - var(--space-xl));
            }
            
            /* Header section */
            .zencode-settings-header {
                text-align: center;
                margin-bottom: var(--space-3xl);
                padding-bottom: var(--space-xl);
                border-bottom: 3px solid var(--primary-color);
                line-height: var(--leading-tight);
            }
            
            .zencode-settings-header h1 {
                color: var(--primary-color);
                font-size: var(--text-4xl);
                margin: 0 0 var(--space-sm) 0;
                font-weight: 700;
                line-height: var(--leading-tight);
            }
            
            .zencode-settings-header p {
                color: var(--text-secondary);
                font-size: var(--text-lg);
                margin: 0;
                line-height: var(--leading-normal);
            }
            
            /* Section styling */
            .zencode-settings-section {
                margin: var(--space-3xl) 0;
                padding: var(--space-2xl);
                background: var(--bg-secondary);
                border-left: 4px solid var(--secondary-color);
                border-radius: 6px;
                box-sizing: border-box;
            }
            
            .zencode-settings-section h2 {
                color: var(--secondary-color);
                margin: 0 0 var(--space-lg) 0;
                font-size: var(--text-2xl);
                font-weight: 600;
                line-height: var(--leading-tight);
            }
            
            /* Form tables */
            .form-table {
                width: 100%;
                margin: var(--space-2xl) 0;
                border-collapse: collapse;
            }
            
            .form-table th {
                padding: var(--space-lg) var(--space-lg) var(--space-lg) 0;
                width: min(30%, 300px);
                font-weight: 600;
                color: var(--text-primary);
                font-size: var(--text-base);
                vertical-align: top;
                text-align: left;
                line-height: var(--leading-normal);
            }
            
            .form-table td {
                padding: var(--space-lg);
                vertical-align: top;
                line-height: var(--leading-normal);
            }
            
            .form-table tr {
                border-bottom: 1px solid var(--border-color);
            }
            
            .form-table tr:last-child {
                border-bottom: none;
            }
            
            /* Form elements */
            .form-table input[type="text"],
            .form-table input[type="password"],
            .form-table input[type="number"],
            .form-table input[type="email"],
            .form-table select,
            .form-table textarea {
                width: 100%;
                max-width: 400px;
                padding: var(--space-md);
                border: 1px solid var(--border-color);
                border-radius: 4px;
                font-size: var(--text-base);
                box-sizing: border-box;
                transition: var(--transition);
                line-height: var(--leading-normal);
            }
            
            .form-table input:focus,
            .form-table select:focus,
            .form-table textarea:focus {
                border-color: var(--secondary-color);
                box-shadow: 0 0 0 3px rgba(96, 165, 250, 0.1);
                outline: 2px solid transparent;
            }
            
            .form-table input[type="checkbox"] {
                width: auto;
                margin-top: 2px;
            }
            
            /* Description text */
            .description {
                color: var(--text-secondary);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
                margin: var(--space-xs) 0 0 0;
                display: block;
            }
            
            /* API Key Section */
            .api-key-container {
                max-width: 100% !important;
            }
            
            .api-key-input-group {
                display: flex;
                gap: var(--space-sm);
                align-items: flex-start;
                margin-bottom: var(--space-sm);
                flex-wrap: wrap;
            }
            
            .api-key-input-group input {
                flex: 1;
                min-width: min(100%, 400px);
            }
            
            .api-key-guide {
                background: var(--bg-tertiary);
                border-left: 4px solid var(--secondary-color);
                padding: var(--space-lg);
                margin: var(--space-lg) 0;
                border-radius: 4px;
                box-sizing: border-box;
            }
            
            .api-key-guide p {
                margin: 0 0 var(--space-sm) 0;
                font-weight: 600;
                color: var(--text-primary);
                font-size: var(--text-base);
                line-height: var(--leading-normal);
            }
            
            .api-key-guide ol {
                margin: var(--space-sm) 0;
                padding-left: var(--space-xl);
                line-height: var(--leading-relaxed);
            }
            
            .api-key-guide li {
                margin: var(--space-xs) 0;
                font-size: var(--text-sm);
                line-height: inherit;
            }
            
            .api-key-status {
                font-weight: 600;
                margin: var(--space-sm) 0;
                font-size: var(--text-base);
                line-height: var(--leading-normal);
            }
            
            .status-success {
                color: var(--success-color);
            }
            
            .status-warning {
                color: var(--warning-color);
            }
            
            /* Stats Dashboard */
            .learning-stats-dashboard {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(min(100%, 280px), 1fr));
                gap: var(--space-lg);
                margin: var(--space-2xl) 0 var(--space-3xl) 0;
            }
            
            .stat-card {
                background: var(--bg-primary);
                padding: var(--space-2xl);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                text-align: center;
                box-sizing: border-box;
                transition: var(--transition);
            }
            
            .stat-card:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }
            
            .stat-value {
                font-size: var(--text-3xl);
                font-weight: 700;
                color: var(--secondary-color);
                line-height: var(--leading-tight);
                margin: 0 0 var(--space-xs) 0;
            }
            
            .stat-label {
                color: var(--text-secondary);
                font-size: var(--text-sm);
                font-weight: 600;
                line-height: var(--leading-normal);
                margin: 0 0 var(--space-sm) 0;
            }
            
            .stat-subtext {
                color: var(--text-light);
                font-size: var(--text-xs);
                line-height: var(--leading-normal);
                margin: var(--space-xs) 0 0 0;
            }
            
            .usage-breakdown {
                display: grid;
                grid-template-columns: 1fr 1fr;
                gap: var(--space-sm);
                margin-top: var(--space-lg);
            }
            
            .usage-item {
                text-align: center;
                padding: var(--space-md);
                border-radius: 6px;
                font-size: var(--text-xs);
                line-height: var(--leading-tight);
            }
            
            .usage-item.used {
                background: #d4edda;
                color: #155724;
            }
            
            .usage-item.unused {
                background: #fff3cd;
                color: #856404;
            }
            
            .usage-count {
                font-weight: 700;
                font-size: var(--text-base);
                line-height: var(--leading-tight);
                margin: 0 0 2px 0;
            }
            
            /* Buttons */
            .button {
                padding: var(--space-md) var(--space-lg);
                font-size: var(--text-sm);
                border-radius: 4px;
                border: 1px solid;
                cursor: pointer;
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                justify-content: center;
                line-height: var(--leading-tight);
                white-space: nowrap;
                transition: var(--transition);
                font-weight: 500;
                min-height: 44px;
                box-sizing: border-box;
            }
            
            .button-primary {
                background: #2271b1;
                border-color: #2271b1;
                color: white;
            }
            
            .button-primary:hover {
                background: #135e96;
                border-color: #135e96;
                transform: translateY(-1px);
            }
            
            .button-secondary {
                background: #f6f7f7;
                border-color: #2271b1;
                color: #2271b1;
            }
            
            .button-secondary:hover {
                background: #e0e0e0;
                transform: translateY(-1px);
            }
            
            .button-danger {
                background: #d63638;
                border-color: #d63638;
                color: white;
            }
            
            .button-danger:hover {
                background: #b32d2e;
                border-color: #b32d2e;
                transform: translateY(-1px);
            }
            
            /* Notice boxes */
            .migration-notice {
                background: #fff3cd;
                border-left: 4px solid #ffc107;
                padding: var(--space-lg);
                margin: var(--space-2xl) 0;
                border-radius: 4px;
                box-sizing: border-box;
            }
            
            .migration-notice h3 {
                margin: 0 0 var(--space-sm) 0;
                color: #856404;
                font-size: var(--text-lg);
                line-height: var(--leading-tight);
            }
            
            .migration-notice p {
                margin: 0 0 var(--space-sm) 0;
                color: #856404;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            /* Code styling */
            code {
                background: #1e293b;
                color: #e2e8f0;
                padding: 2px 6px;
                border-radius: 4px;
                font-family: 'Courier New', monospace;
                font-size: var(--text-sm);
                word-break: break-word;
                line-height: var(--leading-normal);
            }
            
            /* Links */
            a {
                color: #2563eb;
                text-decoration: underline;
                transition: var(--transition);
            }
            
            a:hover {
                color: #1d4ed8;
            }
            
            /* HR styling */
            hr {
                margin: var(--space-3xl) 0;
                border: none;
                border-top: 2px solid var(--border-color);
            }
            
            /* Submit button */
            .submit {
                margin-top: var(--space-3xl);
                padding-top: var(--space-xl);
                border-top: 1px solid var(--border-color);
            }
            
            /* Form group for better mobile spacing */
            .form-group {
                margin-bottom: var(--space-lg);
            }
            
            /* Responsive adjustments */
            @media (max-width: 1024px) {
                .zencode-settings-container {
                    padding: var(--space-xl);
                    width: calc(100% - var(--space-lg));
                }
                
                .learning-stats-dashboard {
                    grid-template-columns: repeat(auto-fit, minmax(min(100%, 250px), 1fr));
                }
            }
            
            @media (max-width: 768px) {
                .zencode-settings-container {
                    padding: var(--space-lg);
                    width: calc(100% - var(--space-md));
                    margin: 0 auto;
                }
                
                .zencode-settings-section {
                    padding: var(--space-lg);
                    margin: var(--space-2xl) 0;
                }
                
                .form-table {
                    display: block;
                    margin: var(--space-xl) 0;
                }
                
                .form-table tbody,
                .form-table tr {
                    display: block;
                }
                
                .form-table th,
                .form-table td {
                    display: block;
                    width: 100%;
                    padding: var(--space-md) 0;
                    box-sizing: border-box;
                }
                
                .form-table th {
                    padding-bottom: var(--space-xs);
                    font-weight: 700;
                    background: none;
                    border-bottom: none;
                }
                
                .form-table td {
                    padding-top: 0;
                    padding-bottom: var(--space-lg);
                }
                
                .form-table tr {
                    margin-bottom: var(--space-lg);
                    padding-bottom: var(--space-lg);
                    border-bottom: 1px solid var(--border-color);
                }
                
                .form-table tr:last-child {
                    margin-bottom: 0;
                    border-bottom: none;
                }
                
                .api-key-input-group {
                    flex-direction: column;
                    align-items: stretch;
                    gap: var(--space-sm);
                }
                
                .api-key-input-group input {
                    min-width: 100%;
                }
                
                .api-key-input-group .button {
                    align-self: flex-start;
                    min-width: 80px;
                }
                
                .learning-stats-dashboard {
                    grid-template-columns: 1fr;
                    gap: var(--space-md);
                }
                
                .usage-breakdown {
                    grid-template-columns: 1fr 1fr;
                    gap: var(--space-xs);
                }
            }
            
            @media (max-width: 480px) {
                .zencode-settings-container {
                    padding: var(--space-md);
                    width: calc(100% - var(--space-sm));
                }
                
                .zencode-settings-header {
                    margin-bottom: var(--space-2xl);
                    padding-bottom: var(--space-lg);
                }
                
                .zencode-settings-section {
                    padding: var(--space-md);
                    margin: var(--space-xl) 0;
                }
                
                .stat-card {
                    padding: var(--space-lg);
                }
                
                .api-key-guide {
                    padding: var(--space-md);
                }
                
                .api-key-guide ol {
                    padding-left: var(--space-lg);
                }
                
                .migration-notice {
                    padding: var(--space-md);
                }
                
                .button {
                    width: 100%;
                    justify-content: center;
                }
                
                .api-key-input-group .button {
                    width: auto;
                    min-width: 80px;
                }
            }
            
            @media (max-width: 360px) {
                .zencode-settings-container {
                    padding: var(--space-sm);
                    width: calc(100% - var(--space-xs));
                }
                
                .usage-breakdown {
                    grid-template-columns: 1fr;
                    gap: var(--space-xs);
                }
            }
            
            /* Focus styles for accessibility */
            button:focus-visible,
            input:focus-visible,
            select:focus-visible,
            textarea:focus-visible {
                outline: 2px solid var(--secondary-color);
                outline-offset: 2px;
            }
            
            /* Loading states */
            .button:disabled {
                opacity: 0.6;
                cursor: not-allowed;
                transform: none !important;
            }
            
            /* Status messages */
            .status-message {
                padding: var(--space-md);
                border-radius: 4px;
                margin: var(--space-sm) 0;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            .status-success-bg {
                background: #d4edda;
                color: #155724;
                border-left: 4px solid var(--success-color);
            }
            
            .status-error-bg {
                background: #f8d7da;
                color: #721c24;
                border-left: 4px solid var(--warning-color);
            }
        </style>
        
        <div class="zencode-settings-container">
            <div class="zencode-settings-header">
                <h1>⚙️ ZenCode AI Settings</h1>
                <p>Configure your AI development environment</p>
            </div>

            <form method="post" action="options.php">
                <?php settings_fields('zencode_ai_settings'); do_settings_sections('zencode_ai_settings'); ?>

                <div class="zencode-settings-section">
                    <h2>🔑 API & Model Configuration</h2>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row">Google Gemini API Key</th>
                            <td>
                                <div class="api-key-container">
                                    <div class="api-key-input-group">
                                        <input type="password" id="zencode_ai_google_api_key" name="zencode_ai_google_api_key" value="<?php echo esc_attr(get_option('zencode_ai_google_api_key')); ?>" class="regular-text" placeholder="sk-ant-..." />
                                        <button type="button" id="toggle-api-key" class="button" style="min-width: 80px;">Show</button>
                                    </div>
                                    
                                    <div class="api-key-guide">
                                        <p>🔑 Don't have an API key yet? Get yours in 2 minutes:</p>
                                        <ol>
                                            <li>Go to <a href="https://aistudio.google.com/app/apikey" target="_blank" style="font-weight: 600; color: #2563eb; text-decoration: underline;">Google AI Studio</a> (opens in new tab)</li>
                                            <li>Sign up or log in (free account works!)</li>
                                            <li>Click <strong>"API Keys"</strong> in the left sidebar</li>
                                            <li>Click <strong>"Create Key"</strong> button</li>
                                            <li>Give it a name (e.g., "ZenCode AI")</li>
                                            <li>Copy the key and paste it above ☝️</li>
                                        </ol>
                                        <p style="margin: var(--space-sm) 0 0 0; font-size: var(--text-sm); color: #1e40af;">
                                            💡 <strong>Tip:</strong> Your key starts with <code>sk-ant-</code>
                                        </p>
                                    </div>
                                    
                                    <p class="description">
                                        ✅ <strong>This is the ONLY API key you need!</strong> No OpenAI, no other services required.
                                    </p>
                                    <p class="description">
                                        ✨ <strong>NEW:</strong> Vector Knowledge Base now works without OpenAI - everything runs on Google Gemini!
                                    </p>
                                    
                                    <?php if (!empty(get_option('zencode_ai_google_api_key'))): ?>
                                        <p class="api-key-status status-success">
                                            ✓ API Key is configured
                                        </p>
                                    <?php else: ?>
                                        <p class="api-key-status status-warning">
                                            ⚠️ API Key required - follow steps above to get yours
                                        </p>
                                    <?php endif; ?>
                                </div>
                                
                                <script>
                                jQuery(document).ready(function($) {
                                    $('#toggle-api-key').on('click', function() {
                                        const input = $('#zencode_ai_google_api_key');
                                        const btn = $(this);
                                        if (input.attr('type') === 'password') {
                                            input.attr('type', 'text');
                                            btn.text('Hide');
                                        } else {
                                            input.attr('type', 'password');
                                            btn.text('Show');
                                        }
                                    });
                                });
                                </script>
                            </td>
                        </tr>
                        <tr style="display:none !important">
                            <th scope="row">Default Model</th>
                            <td>
                                <select name="zencode_ai_default_model">
                                    <?php 
                                    $current_model = get_option('zencode_ai_default_model'); 
                                    foreach ($this->supported_models as $model => $provider) { 
                                        echo '<option value="' . esc_attr($model) . '"' . selected($current_model, $model, false) . '>' . esc_html($model) . '</option>'; 
                                    } 
                                    ?>
                                </select>
                            </td>
                        </tr>
                        
                            <th scope="row">Temperature</th>
                            <td>
                                <input type="number" name="zencode_ai_temperature" value="<?php echo esc_attr(get_option('zencode_ai_temperature', 0.7)); ?>" step="0.1" min="0" max="1" />
                                <p class="description">Controls the creativity of the AI. 0.2 is for focused tasks, 1.0 is for highly creative tasks.</p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Enable Caching</th>
                            <td>
                                <input type="checkbox" name="zencode_ai_enable_caching" value="1" <?php checked(get_option('zencode_ai_enable_caching', true)); ?> />
                                <p class="description">Cache AI responses to improve performance and reduce API costs.</p>
                            </td>
                        </tr>
                    </table>
                </div>

                <hr>

                <div class="zencode-settings-section">
                    <h2>🗃️ Cache Management</h2>
                    <p class="description">Clear cached responses to force fresh AI requests.</p>
                    <div class="form-group">
                        <button type="button" id="clear-cache-btn" class="button button-secondary">Clear All Cache</button>
                        <div id="cache-status" style="margin-top: var(--space-sm);"></div>
                    </div>
                    
                    <script>
                    jQuery("#clear-cache-btn").click(function() {
                        const btn = jQuery(this);
                        btn.prop('disabled', true).text('Clearing...');
                        
                        jQuery.post(ajaxurl, {action: "zencode_ai_clear_cache", nonce: "<?php echo wp_create_nonce('zencode_ai_nonce'); ?>"})
                        .done(function() { 
                            jQuery("#cache-status").html('<div class="status-message status-success-bg">✅ Cache cleared successfully!</div>'); 
                            btn.prop('disabled', false).text('Clear All Cache');
                        })
                        .fail(function() { 
                            jQuery("#cache-status").html('<div class="status-message status-error-bg">❌ Error clearing cache.</div>'); 
                            btn.prop('disabled', false).text('Clear All Cache');
                        });
                    });
                    </script>
                </div>

                <hr>

                <div class="zencode-settings-section">
                    <h2>🧠 AI Learning Management</h2>

                    <?php
                    // Get learning statistics using the proper method
                    $stats = $this->handle_get_learning_stats();
                    ?>

                    <div class="migration-notice">
                        <h3>⚠️ Data Migration</h3>
                        <p>
                            If you're not seeing step patterns and rules, click below to migrate existing learning data.
                        </p>
                        <div class="form-group">
                            <button id="migrate-learning-data" class="button button-primary">
                                🔄 Migrate Learning Data
                            </button>
                            <span id="migration-status" style="margin-left: var(--space-sm); font-weight: bold;"></span>
                        </div>
                    </div>

                    <!-- Modern Stats Dashboard matching Learning Manager -->
                    <div class="learning-stats-dashboard">
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $stats['total_learnings']; ?></div>
                            <div class="stat-label">Total Learnings</div>
                            <div class="usage-breakdown">
                                <div class="usage-item used">
                                    <div class="usage-count"><?php echo $stats['used_learnings_count']; ?></div>
                                    <div class="usage-label">Applied</div>
                                </div>
                                <div class="usage-item unused">
                                    <div class="usage-count"><?php echo $stats['unused_learnings_count']; ?></div>
                                    <div class="usage-label">Unused</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $stats['avg_success_rate']; ?>%</div>
                            <div class="stat-label">Success Rate</div>
                            <div class="stat-subtext">
                                Based on <?php echo $stats['used_learnings_count']; ?> applied learnings
                            </div>
                            <?php if ($stats['best_success_rate'] > 0): ?>
                            <div class="stat-subtext">
                                Best: <?php echo $stats['best_success_rate']; ?>%
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <div class="stat-card">
                            <div class="stat-value"><?php echo $stats['total_applications']; ?></div>
                            <div class="stat-label">Total Uses</div>
                            <div class="stat-subtext">
                                Avg Score: <?php echo $stats['avg_score']; ?>/10
                            </div>
                        </div>
                    </div>

                    <table class="form-table">
                        <tr>
                            <th scope="row">Smart Cleanup</th>
                            <td>
                                <div style="display: flex; gap: var(--space-sm); align-items: center; flex-wrap: wrap; margin-bottom: var(--space-sm);">
                                    <select id="cleanup-threshold" style="flex: 1; min-width: min(100%, 300px);">
                                        <option value="2">Remove learnings below 2/10 (Keep almost everything)</option>
                                        <option value="3" selected>Remove learnings below 3/10 (Recommended)</option>
                                        <option value="4">Remove learnings below 4/10 (More aggressive)</option>
                                        <option value="5">Remove learnings below 5/10 (Very aggressive)</option>
                                    </select>
                                    <button type="button" id="run-cleanup-now" class="button">
                                        🧹 Run Cleanup Now
                                    </button>
                                </div>
                                <p class="description">
                                    Automatically remove low-performing learnings. This helps keep your AI sharp 
                                    by removing patterns that didn't work well.
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">Advanced Management</th>
                            <td>
                                <a href="<?php echo admin_url('admin.php?page=zencode-ai-learning-manager'); ?>" 
                                   class="button button-primary">
                                    📊 View & Manage All Learnings
                                </a>
                                <p class="description">
                                    View, filter, and individually manage all learnings. See what your AI has learned 
                                    and remove specific patterns.
                                </p>
                            </td>
                        </tr>
                        
                        <tr>
                            <th scope="row">
                                <span style="color: #d63638;">⚠️ Nuclear Option</span>
                            </th>
                            <td>
                                <button type="button" id="delete-all-learnings" 
                                        class="button button-danger">
                                    🗑️ Delete ALL Learnings
                                </button>
                                <p class="description" style="color: #d63638;">
                                    <strong>WARNING:</strong> This completely resets your AI's learning database. 
                                    Use this when switching to completely different project types (e.g., from 
                                    e-commerce to SaaS landing pages).
                                </p>
                            </td>
                        </tr>
                    </table>

                    <script>
                    jQuery(document).ready(function($) {
                        // Migration Handler
                        $('#migrate-learning-data').on('click', function() {
                            const btn = $(this);
                            const status = $('#migration-status');
                            
                            btn.prop('disabled', true).text('Migrating...');
                            status.text('⏳ Processing...');
                            
                            $.post(ajaxurl, {
                                action: 'zencode_ai_migrate_learning_data',
                                nonce: '<?php echo wp_create_nonce('zencode_ai_nonce'); ?>'
                            }, function(response) {
                                if (response.success) {
                                    status.html('✅ ' + response.data.message);
                                    setTimeout(function() {
                                        location.reload();
                                    }, 2000);
                                } else {
                                    status.html('❌ Error: ' + (response.data?.message || 'Unknown error'));
                                    btn.prop('disabled', false).text('🔄 Retry Migration');
                                }
                            }).fail(function() {
                                status.html('❌ Request failed');
                                btn.prop('disabled', false).text('🔄 Retry Migration');
                            });
                        });
                        
                        // Smart Cleanup Handler
                        $('#run-cleanup-now').on('click', function() {
                            const threshold = $('#cleanup-threshold').val();
                            const btn = $(this);
                            
                            if (!confirm('Delete all learnings with score below ' + threshold + '/10?\n\nThis will help improve your AI\'s performance by removing patterns that didn\'t work well.')) {
                                return;
                            }
                            
                            btn.prop('disabled', true).text('🔄 Cleaning up...');
                            
                            $.post(ajaxurl, {
                                action: 'zencode_ai_cleanup_learnings',
                                nonce: '<?php echo wp_create_nonce('zencode_ai_nonce'); ?>',
                                threshold: threshold
                            }, function(response) {
                                if (response.success) {
                                    alert('✅ ' + response.data.message);
                                    location.reload();
                                } else {
                                    alert('❌ Error: ' + (response.data?.message || 'Unknown error'));
                                    btn.prop('disabled', false).text('🧹 Run Cleanup Now');
                                }
                            }).fail(function() {
                                alert('❌ Request failed. Please try again.');
                                btn.prop('disabled', false).text('🧹 Run Cleanup Now');
                            });
                        });
                        
                        // Delete All Handler
                        $('#delete-all-learnings').on('click', function() {
                            const btn = $(this);
                            
                            if (!confirm('⚠️ DELETE ALL LEARNINGS?\n\nThis will completely reset your AI\'s memory. It will forget everything it has learned.\n\nAre you sure?')) {
                                return;
                            }
                            
                            if (!confirm('This cannot be undone. Are you ABSOLUTELY sure?')) {
                                return;
                            }
                            
                            btn.prop('disabled', true).text('🔄 Deleting...');
                            
                            $.post(ajaxurl, {
                                action: 'zencode_ai_delete_all_learnings',
                                nonce: '<?php echo wp_create_nonce('zencode_ai_nonce'); ?>'
                            }, function(response) {
                                if (response.success) {
                                    alert('✅ ' + response.data.message);
                                    location.reload();
                                } else {
                                    alert('❌ Error: ' + (response.data?.message || 'Unknown error'));
                                    btn.prop('disabled', false).text('🗑️ Delete ALL Learnings');
                                }
                            }).fail(function() {
                                alert('❌ Request failed. Please try again.');
                                btn.prop('disabled', false).text('🗑️ Delete ALL Learnings');
                            });
                        });
                    });
                    </script>
                </div>

                <?php submit_button('Save Settings', 'primary', 'submit', true); ?>
            </form>
        </div>
        
        <script type="text/javascript">
            jQuery(document).ready(function($) {
                $('#zencode_ai_max_tokens').on('change', function() {
                    const inputElement = $(this);
                    const originalValue = parseInt(inputElement.val(), 10);
                    if (isNaN(originalValue) || originalValue <= 0) {
                        inputElement.val(12288); // Default for Google Gemini
                    }
                });
            });
        </script>
    </div>
    <?php
}
    // Utility functions
    private function cosine_similarity(array $a, array $b): float {
        $dot = 0.0; $a_norm = 0.0; $b_norm = 0.0; $count = count($a);
        if ($count !== count($b) || $count === 0) return 0.0;
        for ($i = 0; $i < $count; $i++) { 
            $dot += $a[$i] * $b[$i]; 
            $a_norm += $a[$i] * $a[$i]; 
            $b_norm += $b[$i] * $b[$i]; 
        }
        $denominator = sqrt($a_norm) * sqrt($b_norm);
        return $denominator === 0.0 ? 0.0 : $dot / $denominator;
    }

    /**
     * ========================================
     * TF-IDF VECTOR GENERATION - NO API NEEDED!
     * ========================================
     * Creates keyword-based vectors for semantic search
     * Works entirely in PHP - no external API calls required!
     */
    
    private function tokenize($text) {
        // Convert to lowercase and remove special characters
        $text = strtolower($text);
        $text = preg_replace('/[^a-z0-9\s_\-]/', ' ', $text);
        
        // Split into words
        $words = preg_split('/\s+/', $text, -1, PREG_SPLIT_NO_EMPTY);
        
        // Remove common stop words (but keep coding-relevant terms)
        $stopwords = ['the', 'is', 'at', 'which', 'on', 'a', 'an', 'and', 'or', 'but', 
                      'in', 'with', 'to', 'for', 'of', 'as', 'by', 'this', 'that', 'it',
                      'are', 'was', 'were', 'be', 'been', 'being', 'have', 'has', 'had',
                      'do', 'does', 'did', 'will', 'would', 'should', 'could', 'can', 'may'];
        
        $words = array_filter($words, function($word) use ($stopwords) {
            return strlen($word) > 2 && !in_array($word, $stopwords);
        });
        
        return array_values($words);
    }
    
    private function calculate_tf($tokens) {
        $tf = [];
        $total = count($tokens);
        
        if ($total === 0) return [];
        
        foreach ($tokens as $token) {
            if (!isset($tf[$token])) {
                $tf[$token] = 0;
            }
            $tf[$token]++;
        }
        
        // Normalize by document length
        foreach ($tf as $term => $count) {
            $tf[$term] = $count / $total;
        }
        
        return $tf;
    }
    
    private function calculate_tfidf_vector($text) {
        $tokens = $this->tokenize($text);
        $tf = $this->calculate_tf($tokens);
        
        // For simplicity, we use TF (term frequency) as our vector
        // This works excellently for code/documentation where exact terms matter!
        // Returns sparse vector: ['function' => 0.05, 'database' => 0.03, ...]
        
        return $tf;
    }
    
    private function calculate_similarity_sparse($vec1, $vec2) {
        // Calculate cosine similarity for sparse vectors (associative arrays)
        if (empty($vec1) || empty($vec2)) return 0.0;
        
        $dot = 0.0;
        $norm1 = 0.0;
        $norm2 = 0.0;
        
        // Calculate dot product and norm for vec1
        foreach ($vec1 as $term => $val) {
            $norm1 += $val * $val;
            if (isset($vec2[$term])) {
                $dot += $val * $vec2[$term];
            }
        }
        
        // Calculate norm for vec2
        foreach ($vec2 as $val) {
            $norm2 += $val * $val;
        }
        
        $denominator = sqrt($norm1) * sqrt($norm2);
        return $denominator === 0.0 ? 0.0 : $dot / $denominator;
    }

// NEW: TF-IDF based search - no OpenAI API needed!
private function get_relevant_knowledge($prompt, $user_id) {
    error_log("ZenCode AI: get_relevant_knowledge called for user {$user_id}");
    
    try {
        // Check user vectors
        global $wpdb;
        $table_name = $wpdb->prefix . 'zencode_ai_kb_vectors';
        $vector_count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM $table_name WHERE user_id = %d", 
            $user_id
        ));
        
        if ($vector_count == 0) {
            error_log("ZenCode AI: User has no vectors");
            return '';
        }

        error_log("ZenCode AI: Found {$vector_count} vectors, starting TF-IDF search");

        // Generate TF-IDF vector for the search query (no API call!)
        $prompt_vector = $this->calculate_tfidf_vector($prompt);
        
        if (empty($prompt_vector)) {
            error_log("ZenCode AI: Failed to generate query vector");
            return '';
        }
        
        error_log("ZenCode AI: Generated query vector with " . count($prompt_vector) . " terms");

        // Get all user vectors
        $vectors = $wpdb->get_results($wpdb->prepare(
            "SELECT source_text, embedding FROM $table_name WHERE user_id = %d", 
            $user_id
        ), ARRAY_A);
        
        // Calculate similarities using TF-IDF
        $scores = [];
        foreach ($vectors as $vector) {
            $db_vector = json_decode($vector['embedding'], true);
            if (!is_array($db_vector) || empty($db_vector)) continue;
            
            // Use sparse vector similarity
            $similarity = $this->calculate_similarity_sparse($prompt_vector, $db_vector);
            $scores[] = ['text' => $vector['source_text'], 'score' => $similarity];
            error_log("ZenCode AI: Vector similarity: " . round($similarity, 3));
        }
        
        if (empty($scores)) {
            error_log("ZenCode AI: No valid vectors found");
            return '';
        }
        
        // Sort and take top 3 - NO THRESHOLD
        usort($scores, fn($a, $b) => $b['score'] <=> $a['score']);
        $top_chunks = array_slice($scores, 0, 3);
        
        // Build context
        $final_context = "";
        foreach ($top_chunks as $chunk) { 
            $final_context .= "--- Relevant context from Knowledge Base ---\n" . $chunk['text'] . "\n--- End of context ---\n\n"; 
        }
        
        error_log("ZenCode AI: Returning context with length: " . strlen($final_context));
        return $final_context;
        
    } catch (Exception $e) {
        error_log("ZenCode AI: Vector error: " . $e->getMessage());
        return '';
    }
}
    // AI API functions
 // [REPLACE] your entire call_ai_api function with this final version.

private function call_ai_api(
    $prompt, 
    $model, 
    $provider, 
    $chat_history, 
    $project_files, 
    $project_structure, 
    $image_data, 
    $is_plan_mode, 
    $kb_context, 
    $vision_images = null, 
    $user_plan = 'spark', 
    $is_one_shot_build = false, 
    $requested_steps = null, 
    $is_polish_request = false, 
    $polish_options = [], 
    $polish_intensity = 'moderate', 
    $image_source = '', 
    $specification_sheet = null,
    $project_context = null,
    $is_step_execution = false,
    $step_index = null,
    $is_scaffolding_step = false // ADD THIS
) {
    $api_key = $this->validate_api_key('google');
	 
    $system_prompt = $this->build_system_prompt(
        $project_files, 
        $project_structure, 
        $is_plan_mode, 
        $kb_context, 
        $user_plan, 
        $requested_steps, 
        $is_polish_request, 
        $polish_options, 
        $polish_intensity, 
        $specification_sheet, 
        $project_context, 
        $is_step_execution, 
        $prompt,
        $step_index,
        $is_scaffolding_step // ADD THIS
    );
    
    $messages = [];
    if (!empty($system_prompt)) { 
        $messages[] = ['role' => 'system', 'content' => $system_prompt]; 
    }
    $messages = array_merge($messages, $chat_history);

    $user_content = [['type' => 'text', 'text' => $prompt]];

    if ($image_data) {
        $user_content[] = ['type' => 'image_url', 'image_url' => ['url' => $image_data]];
    }
    
    if (!empty($vision_images)) {
        foreach($vision_images as $img) {
            if (isset($img['filename']) && isset($img['data'])) {
                $user_content[] = ['type' => 'text', 'text' => "This is the content of the file named: `{$img['filename']}`"];
                $user_content[] = ['type' => 'image_url', 'image_url' => ['url' => $img['data']]];
            }
        }
    }

    $messages[] = ['role' => 'user', 'content' => $user_content];
    
    return $this->call_google_api($api_key, $model, $messages, $is_one_shot_build);
}



  // FIXED: Token tracking function with proper extra credits usage
private function track_token_usage($user_id, $tokens_this_request) {
    if (user_can($user_id, 'manage_options') || $tokens_this_request <= 0) {
        return;
    }
    
    error_log("ZenCode AI: Tracking {$tokens_this_request} tokens for user {$user_id}");
    
    $user_plan = zc_get_user_pricing_plan($user_id);
    $remaining_tokens_to_log = $tokens_this_request;
    
    // For Visionary users, try to use extra credits first
    if ($user_plan === 'visionary') {
        $extra_credits_available = zc_get_user_extra_credits($user_id);
        error_log("ZenCode AI: User {$user_id} has {$extra_credits_available} extra credits available");
        
        if ($extra_credits_available > 0) {
            $credits_used = zc_deduct_extra_credits($user_id, $remaining_tokens_to_log);
            $remaining_tokens_to_log -= $credits_used;
            error_log("ZenCode AI: Used {$credits_used} extra credits, {$remaining_tokens_to_log} tokens remaining for monthly allocation");
        }
    }
    
    // Use regular monthly allocation for any remaining tokens
    if ($remaining_tokens_to_log > 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'zc_token_usage';
        $current_cycle = zc_get_current_billing_cycle_start_date($user_id);
        
        $result = $wpdb->query($wpdb->prepare(
            "INSERT INTO {$table_name} (user_id, tokens_used, billing_cycle_start)
             VALUES (%d, %d, %s)
             ON DUPLICATE KEY UPDATE tokens_used = tokens_used + %d",
            $user_id,
            $remaining_tokens_to_log,
            $current_cycle,
            $remaining_tokens_to_log
        ));
        
        if ($result === false) {
            error_log("ZenCode AI: Failed to update monthly token usage: " . $wpdb->last_error);
        } else {
            error_log("ZenCode AI: Added {$remaining_tokens_to_log} tokens to monthly allocation for user {$user_id}");
        }
    }
}

 
// REPLACE EXISTING FUNCTION
private function check_rolling_limit($user_id) {
    // SaaS functionality removed, so we disable the rolling limit check entirely.
    return;

    // The original logic can remain below, but it will never be executed.
    $user_plan = zc_get_user_pricing_plan($user_id);
    // ... (original function code)
}

private function get_next_reset_time($timestamps) {
    if (empty($timestamps)) return '';
    
    $oldest = min($timestamps);
    $reset_time = $oldest + (5 * HOUR_IN_SECONDS);
    $minutes_until_reset = floor(($reset_time - time()) / 60);
    
    if ($minutes_until_reset <= 60) {
        return $minutes_until_reset . ' minutes';
    } else {
        $hours = floor($minutes_until_reset / 60);
        $remaining_minutes = $minutes_until_reset % 60;
        return $hours . 'h ' . $remaining_minutes . 'm';
    }
}

// ADD NEW FUNCTION
private function security_log($event, $user_id = 0, $details = []) {
    $log_data = [
        'timestamp' => current_time('mysql'),
        'event' => $event,
        'user_id' => $user_id,
        'ip' => zc_get_client_ip(),
        'user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 255),
        'details' => $details
    ];
    
    error_log(sprintf(
        'ZenCode Security [%s]: %s - User: %d - IP: %s - UA: %s - Details: %s',
        $log_data['timestamp'],
        $event,
        $user_id,
        $log_data['ip'],
        $log_data['user_agent'],
        json_encode($details)
    ));
    
    // Store critical events in database for analysis
    if (in_array($event, ['rate_limit_exceeded', 'invalid_file_upload', 'malicious_content_detected'])) {
        global $wpdb;
        $wpdb->insert(
            $wpdb->prefix . 'zc_ip_logs',
            [
                'ip_address' => $log_data['ip'],
                'fingerprint' => md5($log_data['user_agent'] . $log_data['ip']),
                'activity_type' => 'security_' . $event,
                'timestamp' => $log_data['timestamp']
            ]
        );
    }
}


    public function render_user_manual_page() {
        // Embed the full user manual HTML
        $manual_file = ZENCODE_AI_PLUGIN_DIR . 'user_manual.html';
        
        if (file_exists($manual_file)) {
            // If the manual file exists as separate file, include it
            include($manual_file);
        } else {
            // Fallback: Display embedded manual
            $this->display_embedded_manual();
        }
    }
    
    public function render_quick_guide_page() {
    ?>
    <div class="wrap">
        <style>
            /* CSS Custom Properties for consistent theming */
            :root {
                --primary-color: #a855f7;
                --secondary-color: #60a5fa;
                --success-color: #059669;
                --warning-color: #dc2626;
                --text-primary: #1f2937;
                --text-secondary: #6b7280;
                --text-light: #9ca3af;
                --bg-primary: #ffffff;
                --bg-secondary: #f8f9fa;
                --bg-tertiary: #f0f7ff;
                --border-color: #e5e7eb;
                --border-radius: 8px;
                --shadow: 0 2px 8px rgba(0,0,0,0.1);
                --transition: all 0.2s ease-in-out;
                
                /* Responsive spacing scale */
                --space-xs: clamp(4px, 1vw, 6px);
                --space-sm: clamp(8px, 2vw, 12px);
                --space-md: clamp(12px, 2.5vw, 16px);
                --space-lg: clamp(16px, 3vw, 20px);
                --space-xl: clamp(20px, 4vw, 24px);
                --space-2xl: clamp(24px, 5vw, 32px);
                --space-3xl: clamp(32px, 6vw, 40px);
                
                /* Typography scale */
                --text-xs: clamp(0.75rem, 2vw, 0.8rem);
                --text-sm: clamp(0.8rem, 2.2vw, 0.9rem);
                --text-base: clamp(0.9rem, 2.5vw, 1rem);
                --text-lg: clamp(1rem, 3vw, 1.125rem);
                --text-xl: clamp(1.125rem, 3.5vw, 1.25rem);
                --text-2xl: clamp(1.25rem, 4vw, 1.5rem);
                --text-3xl: clamp(1.5rem, 5vw, 1.875rem);
                --text-4xl: clamp(1.75rem, 6vw, 2.25rem);
                
                /* Line heights */
                --leading-tight: 1.25;
                --leading-normal: 1.5;
                --leading-relaxed: 1.75;
            }
            
            /* Base container */
            .quick-guide-container {
                background: var(--bg-primary);
                padding: var(--space-2xl);
                border-radius: var(--border-radius);
                box-shadow: var(--shadow);
                margin: 0 auto;
                max-width: 1200px;
                box-sizing: border-box;
                width: calc(100% - var(--space-xl));
            }
            
            /* Header */
            .quick-guide-header {
                text-align: center;
                margin-bottom: var(--space-3xl);
                padding-bottom: var(--space-xl);
                border-bottom: 3px solid var(--primary-color);
                line-height: var(--leading-tight);
            }
            
            .quick-guide-header h1 {
                color: var(--primary-color);
                font-size: var(--text-4xl);
                margin: 0 0 var(--space-sm) 0;
                font-weight: 700;
                line-height: var(--leading-tight);
            }
            
            .quick-guide-header p {
                color: var(--text-secondary);
                font-size: var(--text-lg);
                margin: 0;
                line-height: var(--leading-normal);
            }
            
            /* Sections */
            .qg-section {
                margin: var(--space-3xl) 0;
                padding: var(--space-2xl);
                background: var(--bg-secondary);
                border-left: 4px solid var(--secondary-color);
                border-radius: 6px;
                box-sizing: border-box;
            }
            
            .qg-section h2 {
                color: var(--secondary-color);
                margin: 0 0 var(--space-lg) 0;
                font-size: var(--text-2xl);
                font-weight: 600;
                line-height: var(--leading-tight);
            }
            
            .qg-section h3 {
                color: var(--primary-color);
                margin: var(--space-xl) 0 var(--space-md) 0;
                font-size: var(--text-xl);
                font-weight: 600;
                line-height: var(--leading-tight);
            }
            
            /* Grid Layout */
            .qg-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(min(100%, 300px), 1fr));
                gap: var(--space-lg);
                margin: var(--space-xl) 0;
            }
            
            /* Cards */
            .qg-card {
                background: var(--bg-primary);
                padding: var(--space-xl);
                border-radius: var(--border-radius);
                border: 1px solid var(--border-color);
                box-shadow: var(--shadow);
                transition: var(--transition);
            }
            
            .qg-card:hover {
                transform: translateY(-2px);
                box-shadow: 0 4px 12px rgba(0,0,0,0.15);
            }
            
            .qg-card h4 {
                color: var(--text-primary);
                margin: 0 0 var(--space-md) 0;
                font-size: var(--text-lg);
                font-weight: 600;
                line-height: var(--leading-tight);
                display: flex;
                align-items: center;
            }
            
            .qg-card h4 .emoji {
                margin-right: var(--space-sm);
                font-size: 1.4em;
            }
            
            .qg-card p {
                color: var(--text-secondary);
                line-height: var(--leading-normal);
                margin: var(--space-sm) 0;
                font-size: var(--text-sm);
            }
            
            .qg-card ul {
                margin: var(--space-sm) 0;
                padding-left: var(--space-lg);
            }
            
            .qg-card li {
                color: var(--text-primary);
                margin: var(--space-xs) 0;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            /* Highlight sections */
            .qg-highlight {
                background: linear-gradient(135deg, #ede9fe 0%, #dbeafe 100%);
                padding: var(--space-xl);
                border-radius: var(--border-radius);
                margin: var(--space-xl) 0;
            }
            
            /* Badges */
            .qg-badge {
                display: inline-block;
                padding: var(--space-xs) var(--space-sm);
                border-radius: 12px;
                font-size: var(--text-xs);
                font-weight: 600;
                margin-left: var(--space-sm);
                line-height: var(--leading-tight);
            }
            
            .qg-badge-new {
                background: var(--success-color);
                color: white;
            }
            
            .qg-badge-hot {
                background: #f59e0b;
                color: white;
            }
            
            /* Tips */
            .qg-tip {
                background: #fef3c7;
                border-left: 4px solid #f59e0b;
                padding: var(--space-lg);
                margin: var(--space-lg) 0;
                border-radius: 4px;
            }
            
            .qg-tip strong {
                color: #92400e;
            }
            
            /* Code styling */
            .qg-code {
                background: #1e293b;
                color: #e2e8f0;
                padding: 2px 6px;
                border-radius: 4px;
                font-family: 'Courier New', monospace;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            /* Steps */
            .qg-steps {
                counter-reset: step-counter;
                list-style: none;
                padding-left: 0;
                margin: var(--space-lg) 0;
            }
            
            .qg-steps li {
                counter-increment: step-counter;
                margin: var(--space-lg) 0;
                padding-left: var(--space-3xl);
                position: relative;
                font-size: var(--text-base);
                line-height: var(--leading-normal);
            }
            
            .qg-steps li::before {
                content: counter(step-counter);
                position: absolute;
                left: 0;
                top: 0;
                background: var(--primary-color);
                color: white;
                width: 28px;
                height: 28px;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-weight: 700;
                font-size: var(--text-sm);
                line-height: var(--leading-tight);
            }
            
            /* Tables */
            .qg-table {
                width: 100%;
                border-collapse: collapse;
                margin: var(--space-xl) 0;
                background: var(--bg-primary);
                border-radius: var(--border-radius);
                overflow: hidden;
                box-shadow: var(--shadow);
            }
            
            .qg-table th {
                background: var(--primary-color);
                color: white;
                padding: var(--space-lg);
                text-align: left;
                font-weight: 600;
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            .qg-table td {
                padding: var(--space-lg);
                border-bottom: 1px solid var(--border-color);
                font-size: var(--text-sm);
                line-height: var(--leading-normal);
            }
            
            .qg-table tr:last-child td {
                border-bottom: none;
            }
            
            .qg-table tr:hover {
                background: var(--bg-tertiary);
            }
            
            /* Hero section */
            .qg-hero {
                background: linear-gradient(135deg, var(--primary-color) 0%, var(--secondary-color) 100%);
                color: white;
                padding: var(--space-3xl);
                border-radius: var(--border-radius);
                text-align: center;
                margin-top: var(--space-3xl);
            }
            
            .qg-hero h2 {
                color: white;
                margin-bottom: var(--space-lg);
                font-size: var(--text-2xl);
                line-height: var(--leading-tight);
            }
            
            .qg-hero p {
                font-size: var(--text-base);
                line-height: var(--leading-normal);
                margin-bottom: var(--space-md);
                opacity: 0.9;
            }
            
            .qg-hero a {
                color: white;
                text-decoration: underline;
                font-weight: 600;
            }
            
            /* Links */
            a {
                color: #2563eb;
                text-decoration: underline;
                transition: var(--transition);
            }
            
            a:hover {
                color: #1d4ed8;
            }
            
            /* Responsive adjustments */
            @media (max-width: 1024px) {
                .quick-guide-container {
                    padding: var(--space-xl);
                    width: calc(100% - var(--space-lg));
                }
                
                .qg-grid {
                    grid-template-columns: repeat(auto-fit, minmax(min(100%, 280px), 1fr));
                }
            }
            
            @media (max-width: 768px) {
                .quick-guide-container {
                    padding: var(--space-lg);
                    width: calc(100% - var(--space-md));
                }
                
                .quick-guide-header {
                    margin-bottom: var(--space-2xl);
                    padding-bottom: var(--space-lg);
                }
                
                .qg-section {
                    padding: var(--space-lg);
                    margin: var(--space-2xl) 0;
                }
                
                .qg-grid {
                    grid-template-columns: 1fr;
                    gap: var(--space-md);
                }
                
                .qg-card {
                    padding: var(--space-lg);
                }
                
                .qg-highlight {
                    padding: var(--space-lg);
                }
                
                .qg-steps li {
                    padding-left: var(--space-2xl);
                    margin: var(--space-md) 0;
                }
                
                .qg-table {
                    display: block;
                    overflow-x: auto;
                }
                
                .qg-table th,
                .qg-table td {
                    padding: var(--space-md);
                }
                
                .qg-hero {
                    padding: var(--space-2xl);
                }
            }
            
            @media (max-width: 480px) {
                .quick-guide-container {
                    padding: var(--space-md);
                    width: calc(100% - var(--space-sm));
                }
                
                .quick-guide-header h1 {
                    font-size: var(--text-3xl);
                }
                
                .qg-section {
                    padding: var(--space-md);
                }
                
                .qg-card {
                    padding: var(--space-md);
                }
                
                .qg-highlight {
                    padding: var(--space-md);
                }
                
                .qg-steps li {
                    padding-left: var(--space-xl);
                }
                
                .qg-steps li::before {
                    width: 24px;
                    height: 24px;
                    font-size: var(--text-xs);
                }
                
                .qg-hero {
                    padding: var(--space-xl);
                }
            }
            
            @media (max-width: 360px) {
                .qg-table th,
                .qg-table td {
                    padding: var(--space-sm);
                    font-size: var(--text-xs);
                }
                
                .qg-steps li {
                    padding-left: var(--space-lg);
                }
            }
            
            /* Focus styles for accessibility */
            a:focus-visible {
                outline: 2px solid var(--secondary-color);
                outline-offset: 2px;
            }
        </style>
        
        <div class="quick-guide-container">
            <div class="quick-guide-header">
                <h1>⚡ ZenCode AI Quick Guide</h1>
                <p>Get up to speed in 5 minutes</p>
            </div>

            <div class="qg-section">
                <h2>🚀 Getting Started (30 seconds)</h2>
                
                <div class="qg-highlight" style="margin-bottom: var(--space-xl);">
                    <h3 style="margin-bottom: var(--space-lg);">Step 1: Get Your API Key (2 minutes)</h3>
                    <ol class="qg-steps">
                        <li>Visit <a href="https://aistudio.google.com/app/apikey" target="_blank" style="font-weight: 600; color: #2563eb; text-decoration: underline;">Google AI Studio</a> 🔗</li>
                        <li>Sign up or log in (free account works!)</li>
                        <li>Click "API Keys" → "Create Key"</li>
                        <li>Copy your key (starts with <code style="background: #e5e7eb; padding: 2px 6px; border-radius: 3px; font-size: var(--text-sm);">sk-ant-</code>)</li>
                    </ol>
                    <p style="margin-top: var(--space-lg); color: var(--success-color); font-weight: 600; font-size: var(--text-base); line-height: var(--leading-normal);">
                        ✅ This is the ONLY API key you need - no OpenAI, no other services!
                    </p>
                </div>
                
                <h3 style="margin-bottom: var(--space-lg);">Step 2: Configure Plugin</h3>
                <ol class="qg-steps">
                    <li>Go to <strong>ZenCode AI → Settings</strong> in WordPress</li>
                    <li>Paste your API key in the <strong>Google Gemini API Key</strong> field</li>
                    <li>Click <strong>Save Changes</strong></li>
                </ol>
                
                <h3 style="margin-bottom: var(--space-lg);">Step 3: Start Building!</h3>
                <ol class="qg-steps">
                    <li>Go to <strong>ZenCode AI</strong> in your WordPress menu</li>
                    <li>Type your first question or project request</li>
                    <li>Watch the AI work its magic! ✨</li>
                </ol>
            </div>

            <div class="qg-section">
                <h2>🎯 Three Ways to Use ZenCode AI</h2>
                
                <div class="qg-grid">
                    <div class="qg-card">
                        <h4><span class="emoji">💬</span> Normal Mode</h4>
                        <p><strong>Best for:</strong> Quick questions, code snippets, debugging</p>
                        <ul>
                            <li>Ask coding questions</li>
                            <li>Get code explanations</li>
                            <li>Debug small issues</li>
                            <li>Learn concepts</li>
                        </ul>
                    </div>
                    
                    <div class="qg-card">
                        <h4><span class="emoji">📋</span> Plan Mode <span class="qg-badge qg-badge-hot">HOT</span></h4>
                        <p><strong>Best for:</strong> Complete projects, multi-file builds</p>
                        <ul>
                            <li>Full websites/apps</li>
                            <li>WordPress plugins</li>
                            <li>Auto-generates specs</li>
                            <li>Step-by-step execution</li>
                        </ul>
                    </div>
                    
                    <div class="qg-card">
                        <h4><span class="emoji">🧠</span> Vector Mode <span class="qg-badge qg-badge-new">NEW</span></h4>
                        <p><strong>Best for:</strong> Understanding your codebase</p>
                        <ul>
                            <li>Upload documentation</li>
                            <li>Search your code</li>
                            <li>No API key needed!</li>
                            <li>Instant keyword search</li>
                        </ul>
                    </div>
                    
                    <div class="qg-card">
                        <h4><span class="emoji">⚡</span> One-Shot Builder <span class="qg-badge qg-badge-hot">FAST</span></h4>
                        <p><strong>Best for:</strong> Single-page websites with instant generation</p>
                        <ul>
                            <li>Landing pages</li>
                            <li>Simple tools</li>
                            <li>Single-file apps</li>
                            <li>Ultra-fast generation!</li>
                            <li>Expand into Plan Mode if needed</li>
                            <li>Edit surgically in Normal Mode</li>
                        </ul>
                        <p style="margin-top: var(--space-sm); font-size: var(--text-xs); color: var(--text-secondary); line-height: var(--leading-normal);">
                            <strong>💡 Flexible:</strong> Start fast, then expand or refine as needed!
                        </p>
                    </div>
                </div>
            </div>

            <div class="qg-section">
                <h2>⚡ Quick Workflows</h2>
                
                <h3>Build a Website in 3 Steps:</h3>
                <ol class="qg-steps">
                    <li>Type: <span class="qg-code">Create a portfolio website with contact form</span></li>
                    <li>AI generates plan + spec automatically</li>
                    <li>Click "Run Entire Plan" and watch it build!</li>
                </ol>
                
                <h3>Debug with Console Errors:</h3>
                <ol class="qg-steps">
                    <li>Open browser console (F12), copy error</li>
                    <li>Click "Debug Prompt" button</li>
                    <li>Paste error → AI fixes it automatically!</li>
                </ol>
                
                <h3>Use Vector Knowledge Base:</h3>
                <ol class="qg-steps">
                    <li>Go to Knowledge Base tab</li>
                    <li>Upload your code files or docs</li>
                    <li>Switch to Vector Mode and ask questions!</li>
                </ol>
                
                <h3>Use One-Shot Builder (Ultra Fast!):</h3>
                <ol class="qg-steps">
                    <li>Click "One-Shot Builder" button</li>
                    <li>Type: <span class="qg-code">Build a restaurant menu website</span></li>
                    <li>AI generates everything in ONE powerful request!</li>
                    <li><strong>Then:</strong> Expand into Plan Mode for multi-file, OR edit surgically in Normal Mode</li>
                </ol>
                <div class="qg-tip" style="margin-top: var(--space-lg);">
                    <strong>💡 Pro Tip:</strong> One-Shot Builder creates a complete single-page site instantly. If you realize you need more pages or features, simply ask to "expand this into a full plan" or make specific edits in Normal Mode. Best of both worlds!
                </div>
            </div>

            <div class="qg-section">
                <h2>🎨 Key Features</h2>
                
                <table class="qg-table">
                    <thead>
                        <tr>
                            <th>Feature</th>
                            <th>What It Does</th>
                            <th>When to Use</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><strong>🤖 Auto-Repair</strong></td>
                            <td>Automatically fixes code errors</td>
                            <td>During builds - always ON</td>
                        </tr>
                        <tr>
                            <td><strong>📊 Reflection Scores</strong></td>
                            <td>Rates AI output quality (1-10)</td>
                            <td>Check after each step</td>
                        </tr>
                        <tr>
                            <td><strong>📸 Image to Plan</strong></td>
                            <td>Converts screenshots to specs</td>
                            <td>When you have a visual reference</td>
                        </tr>
                        <tr>
                            <td><strong>💾 Context Snapshots</strong></td>
                            <td>Saves project state + history</td>
                            <td>Before major changes</td>
                        </tr>
                        <tr>
                            <td><strong>✨ Polish & Unify</strong></td>
                            <td>Professional design consistency</td>
                            <td>After building, before deployment</td>
                        </tr>
                        <tr>
                            <td><strong>📦 Download as ZIP</strong></td>
                            <td>Package entire project</td>
                            <td>When ready to deploy</td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="qg-section">
                <h2>💡 Pro Tips</h2>
                
                <div class="qg-highlight">
                    <div class="qg-grid">
                        <div>
                            <h4>🎯 Be Specific</h4>
                            <p>❌ "Make a website"</p>
                            <p>✅ "Create a portfolio website with dark mode, contact form, and project gallery"</p>
                        </div>
                        <div>
                            <h4>📋 Use Plan Mode for Big Projects</h4>
                            <p>Anything with 2+ files should use Plan Mode. It auto-generates specs and prevents errors!</p>
                        </div>
                        <div>
                            <h4>🔄 Trust Auto-Repair</h4>
                            <p>If build fails, auto-repair usually fixes it. Don't interrupt - let it run!</p>
                        </div>
                        <div>
                            <h4>📊 Watch Reflection Scores</h4>
                            <p>Scores below 6 mean issues. Review the output before continuing.</p>
                        </div>
                        <div>
                            <h4>⚡ One-Shot → Expand → Refine</h4>
                            <p><strong>Workflow:</strong> Start with One-Shot Builder for speed, expand into Plan Mode if project grows, then use Normal Mode for precise tweaks. Ultimate flexibility!</p>
                        </div>
                    </div>
                </div>
            </div>

            <div class="qg-tip">
                <strong>💎 NEW:</strong> Vector Knowledge Base no longer needs OpenAI! Just upload files and search instantly with keyword-based vectors. Perfect for documentation and code reference!
            </div>

            <div class="qg-section">
                <h2>🆘 Common Issues</h2>
                
                <table class="qg-table">
                    <thead>
                        <tr>
                            <th>Problem</th>
                            <th>Solution</th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td>AI not responding</td>
                            <td>Check API key in Settings. Verify Google Gemini account has credits.</td>
                        </tr>
                        <tr>
                            <td>Build keeps failing</td>
                            <td>Wait for auto-repair to finish. Check reflection scores. Try simpler prompts.</td>
                        </tr>
                        <tr>
                            <td>Lost context</td>
                            <td>Load previous snapshot from Context Manager</td>
                        </tr>
                        <tr>
                            <td>Vector search not working</td>
                            <td>Make sure files are uploaded in Knowledge Base tab first</td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <div class="qg-hero">
                <h2>🎉 You're Ready to Build!</h2>
                <p style="font-size: var(--text-base); margin-bottom: var(--space-md);">Start with a simple project and let ZenCode AI surprise you.</p>
                <p style="margin-bottom: var(--space-sm);"><strong>Remember:</strong> Normal Mode for quick tasks • Plan Mode for projects • Vector Mode for your codebase</p>
                <p style="margin-top: var(--space-lg); font-size: var(--text-sm); opacity: 0.9;">Need more details? Check the full <a href="<?php echo admin_url('admin.php?page=zencode-ai-user-manual'); ?>" style="color: white; text-decoration: underline;">User Manual</a></p>
            </div>
        </div>
    </div>
    <?php
}
    
    private function display_embedded_manual() {
        // Fallback embedded version - you can customize this
        ?>
        <div class="wrap">
            <h1>User Manual</h1>
            <p>The full user manual file is not found. Please ensure <code>user_manual.html</code> is in the plugin directory.</p>
            <p>In the meantime, check out the <a href="<?php echo admin_url('admin.php?page=zencode-ai-quick-guide'); ?>">Quick Guide</a> for essential information.</p>
        </div>
        <?php
    }


    public function hide_admin_page_titles() {
        $screen = get_current_screen();
        if ($screen && in_array($screen->id, [
    'toplevel_page_zencode-ai-app', 
    'zencode-ai-app_page_zencode-ai-settings', 
    'zencode-ai-app_page_zencode-ai-knowledge-base',
    'zencode-ai-app_page_zencode-ai-learning-manager',
    'zencode-ai-app_page_zencode-ai-user-manual',
    'zencode-ai-app_page_zencode-ai-quick-guide'
])) {
            echo '<style>.wrap > h1 { display: none !important; }</style>';
        }
        if ($screen && $screen->id === 'toplevel_page_zencode-ai-app') {
            echo '<style>
                #wpcontent { padding-left: 0 !important; }
                #wpbody-content .wrap { background: transparent !important; box-shadow: none !important; padding: 0 !important; margin: 0 !important; }
                #ai_app_root { width: 100%; height: calc(100vh - 32px); max-width: none; margin: 0; border: none; border-radius: 0; box-shadow: none; }
                html.wp-toolbar { padding-top: 0; }
            </style>';
        }
        if ($screen && in_array($screen->id, [
            'zencode-ai-app_page_zencode-ai-settings', 
            'zencode-ai-app_page_zencode-ai-knowledge-base'
        ])) {
            echo '<style>
                #wpbody-content .wrap { background-color: #0a0e1a; padding: 25px; margin-top: 20px; border-radius: 16px; }
                .form-table th, .form-table td p.description, .wrap p, .wp-list-table, .wp-list-table th, .wp-list-table td { color: #f7fafc !important; }
                .wp-list-table td pre { color: #a0aec0; }
            </style>';
        }
    }
	
	
	
	
	
	// Helper validation methods
// [REPLACE THIS ENTIRE FUNCTION in your zencode-ai.php file]
private function validate_and_sanitize_prompt($prompt) {
    $prompt = sanitize_textarea_field($prompt);
    
    // Reduced size limit from 50KB to 10KB
    if (strlen($prompt) > 10000) {
        wp_send_json_error(['message' => 'Prompt too long. Maximum 10,000 characters allowed.'], 400);
        wp_die();
    }
    
    // --- FIX: The regex is now more specific. ---
    // It looks for "javascript:" (with the colon) as a protocol, which is the actual security risk,
    // but it will no longer block prompts that simply contain the word "javascript".
    // The `\b` ensures it matches whole words.
    if (preg_match('/\b(javascript|data|vbscript):|<script|onload|onerror/i', $prompt)) {
        // Security log for suspicious attempts
        $this->security_log('malicious_content_detected', get_current_user_id(), ['prompt_snippet' => substr($prompt, 0, 200)]);
        wp_send_json_error(['message' => 'Invalid content detected.'], 400);
        wp_die();
    }
    
    // Check for excessive special characters (potential injection attempts)
    $special_char_count = preg_match_all('/[<>&"\'`]/', $prompt);
    if ($special_char_count > 50) {
        wp_send_json_error(['message' => 'Content contains too many special characters.'], 400);
        wp_die();
    }
    
    return $prompt;
}

private function validate_model($model) {
    $model = sanitize_text_field($model);
    if (!array_key_exists($model, $this->supported_models)) {
        wp_send_json_error(['message' => 'Invalid model selected.'], 400);
        wp_die();
    }
    return $model;
}

private function validate_image_data($image_data) {
    if (!preg_match('/^data:image\/(jpeg|jpg|png|gif|webp);base64,/', $image_data)) {
        return false;
    }
    // Check image size (base64 encoded, so roughly 4/3 the actual size)
    if (strlen($image_data) > 8000000) { // ~6MB actual image
        wp_send_json_error(['message' => 'Image too large. Maximum 6MB allowed.'], 400);
        wp_die();
    }
    return true;
}
	
	// ADD NEW FUNCTION
// REPLACE your current validate_file_path function with this one.
// REPLACE this entire function
// REPLACE your current validate_file_path function with this simpler, more reliable one.
private function validate_file_path($file_path) {
    if (empty($file_path) || strlen($file_path) > 255) {
        return false;
    }
    
    // 1. Critical Security Check: Block directory traversal attempts.
    // This is the most important security measure for file paths.
    if (strpos($file_path, '..') !== false) {
        return false;
    }
    
    // 2. Extension Check: Ensure the base file has a safe extension.
    // This correctly handles paths like "script.js (Part 1/1)".
    $base_path = preg_split('/\s\(/', $file_path)[0];
    $allowed_extensions = ['js', 'css', 'html', 'php', 'txt', 'json', 'xml', 'md'];
    $extension = strtolower(pathinfo($base_path, PATHINFO_EXTENSION));
    
    if (!in_array($extension, $allowed_extensions)) {
        return false;
    }
    
    // 3. If the path passes these essential checks, it's safe for this use case.
    return true;
}



/**
 * Cache management functions
 */
private function get_cache_key($data, $request_type = 'general', $user_context = []) {
    $key_components = [
        'user_id' => get_current_user_id(), // ✅ ADD THIS LINE
        'type' => $request_type,
        'data' => $data,
        'plugin_version' => ZENCODE_AI_VERSION,
        'model_settings' => $this->get_cache_relevant_settings(),
        'user_plan' => $user_context['plan'] ?? 'none'
    ];
    
    $serialized = serialize($key_components);
    $hash = md5($serialized);
    
    return "zc_user_" . get_current_user_id() . "_{$request_type}_{$hash}";
    //     ^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^ ADD THIS TOO for readability
}

private function get_cached_response($cache_key, $callable, $duration = null) {
    if (!$this->is_caching_enabled()) {
        return $callable();
    }
    
    // Check cache size limits before proceeding
    $this->enforce_cache_limits();
    
    $cached = get_transient($cache_key);
    if ($cached !== false && $this->is_valid_cached_response($cached)) {
        error_log("ZenCode AI: Cache hit for key: " . substr($cache_key, 0, 30));
        
        // Handle both new array format and legacy string format
        if (is_array($cached) && isset($cached['content'])) {
            return $cached['content'];
        } else {
            return $cached; // Legacy string format
        }
    }
    
    $result = $callable();
    
    // Only cache successful responses
    if ($result && !empty(trim($result))) {
        if ($duration === null) {
            $duration = $this->get_cache_duration_for_content($result);
        }
        
        // Add metadata to cached response
        $cache_data = [
            'content' => $result,
            'cached_at' => time(),
            'cache_version' => '1.0'
        ];
        
        set_transient($cache_key, $cache_data, $duration);
        error_log("ZenCode AI: Cached response for key: " . substr($cache_key, 0, 30) . " (duration: {$duration}s)");
    }
    
    return $result;
}

private function should_cache_request($prompt, $user_context = []) {
    // Only block caching for the most obvious problematic cases
    
    // Don't cache if there are actual project files attached
    if (!empty($user_context['project_files']) && is_array($user_context['project_files']) && count($user_context['project_files']) > 0) {
        return false;
    }
    
    // Don't cache if there are vision images
    if (!empty($user_context['vision_images']) && is_array($user_context['vision_images']) && count($user_context['vision_images']) > 0) {
        return false;
    }
    
    // Don't cache very short prompts (likely too specific)
    if (strlen($prompt) < 20) {
        return false;
    }
    
    // Don't cache prompts that are clearly asking about specific user files
    $prompt_lower = strtolower($prompt);
    $file_specific_indicators = [
        'this file',
        'my file', 
        'current file',
        'the file i',
        'fix this',
        'update this',
        'modify this'
    ];
    
    foreach ($file_specific_indicators as $indicator) {
        if (strpos($prompt_lower, $indicator) !== false) {
            return false;
        }
    }
    
    // Cache everything else - let's see if basic caching works
    return true;
}

private function get_cache_duration($request_type) {
    $durations = [
        'explanation' => 45 * MINUTE_IN_SECONDS,     // Reduced from 2 hours
        'pattern' => 30 * MINUTE_IN_SECONDS,         // Reduced from 1 hour  
        'planning' => 20 * MINUTE_IN_SECONDS,        // Plans change quickly
        'vector' => 60 * MINUTE_IN_SECONDS,          // Vector searches are stable
        'refinement' => 10 * MINUTE_IN_SECONDS,      // Reduced from 15 minutes
        'educational' => 60 * MINUTE_IN_SECONDS,     // Educational content more stable
        'embedding' => 60 * MINUTE_IN_SECONDS,       // Embeddings are stable
        'general' => 25 * MINUTE_IN_SECONDS          // Reduced default
    ];
    
    return $durations[$request_type] ?? $durations['general'];
}

private function classify_request_type($prompt) {
    $prompt_lower = strtolower($prompt);
    
    // Educational/explanation requests (higher cache value)
    if (preg_match('/\b(explain|what is|how does|why|define|difference between)\b/', $prompt_lower)) {
        return 'educational';
    }
    
    // Planning requests (low cache value)
    if (preg_match('/\b(plan|strategy|approach|architecture|structure)\b/', $prompt_lower)) {
        return 'planning';
    }
    
    // Common pattern requests (medium cache value)
    if (preg_match('/\b(create|build|make|generate).*(component|function|class|template)\b/', $prompt_lower)) {
        return 'pattern';
    }
    
    // Vector/knowledge base searches (high cache value)
    if (strpos($prompt_lower, 'knowledge') !== false || strpos($prompt_lower, 'remember') !== false) {
        return 'vector';
    }
    
    return 'general';
}


private function is_caching_enabled() {
    // Allow disabling via constant for emergency situations
    if (defined('ZENCODE_AI_DISABLE_CACHE') && ZENCODE_AI_DISABLE_CACHE) {
        return false;
    }
    
    // Allow admin override via settings
    $admin_setting = get_option('zencode_ai_enable_caching', true);
    if (!$admin_setting) {
        return false;
    }
    
    // Remove the WP_DEBUG check that was blocking you, but keep other controls
    return apply_filters('zencode_ai_enable_caching', true);
}
public function clear_cache($pattern = null) {
    global $wpdb;
    
    if ($pattern) {
        // Clear specific cache pattern
        $like_pattern = "_transient_zc_{$pattern}_%";
        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE t1, t2 FROM {$wpdb->options} t1
             LEFT JOIN {$wpdb->options} t2 ON t2.option_name = CONCAT('_transient_timeout_', SUBSTRING(t1.option_name, 12))
             WHERE t1.option_name LIKE %s",
            $like_pattern
        ));
    } else {
        // Clear all ZenCode caches
        $deleted = $wpdb->query(
            "DELETE t1, t2 FROM {$wpdb->options} t1
             LEFT JOIN {$wpdb->options} t2 ON t2.option_name = CONCAT('_transient_timeout_', SUBSTRING(t1.option_name, 12))
             WHERE t1.option_name LIKE '_transient_zc_%'"
        );
    }
    
    error_log("ZenCode AI: Cache cleared - deleted {$deleted} entries" . ($pattern ? " (pattern: {$pattern})" : ""));
}

/**
 * NEW FUNCTIONS - Add these to your ZenCodeAI class
 */

/**
 * NEW: Get settings that affect cache validity
 */
private function get_cache_relevant_settings() {
    return [
        'model' => get_option('zencode_ai_default_model'),
        'temperature' => get_option('zencode_ai_temperature'),
        'max_tokens' => get_option('zencode_ai_max_tokens'),
        'anthropic_key_hash' => md5(get_option('zencode_ai_google_api_key', '')),
    ];
}

/**
 * NEW: Validate cached response structure
 */
private function is_valid_cached_response($cached_data) {
    if (!is_array($cached_data)) {
        // Handle legacy cached strings
        return is_string($cached_data) && !empty(trim($cached_data));
    }
    
    return isset($cached_data['content']) && 
           isset($cached_data['cached_at']) &&
           !empty(trim($cached_data['content']));
}

/**
 * NEW: Dynamic cache duration based on content type
 */
private function get_cache_duration_for_content($content) {
    $content_lower = strtolower($content);
    
    // Shorter duration for code (changes frequently)
    if (strpos($content_lower, 'function') !== false || 
        strpos($content_lower, 'class') !== false ||
        preg_match('/File:\s+[\w.-]+/', $content)) {
        return 20 * MINUTE_IN_SECONDS;
    }
    
    // Medium duration for explanations
    if (preg_match('/\b(?:explanation|concept|principle|theory)\b/i', $content)) {
        return 45 * MINUTE_IN_SECONDS;
    }
    
    // Longer duration for reference material
    if (preg_match('/\b(?:documentation|reference|syntax|specification)\b/i', $content)) {
        return 90 * MINUTE_IN_SECONDS;
    }
    
    // Default duration
    return 30 * MINUTE_IN_SECONDS;
}

/**
 * NEW: Cache size management
 */
private function enforce_cache_limits() {
    global $wpdb;
    
    // Count current cache entries
    $cache_count = $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE '_transient_zc_%'"
    );
    
    // If too many cache entries, clean up oldest ones
    if ($cache_count > 500) {
        $this->cleanup_old_cache_entries(100);
    }
}

/**
 * NEW: Clean up old cache entries
 */
private function cleanup_old_cache_entries($limit = 100) {
    global $wpdb;
    
    // Delete expired transients first
    $wpdb->query(
        "DELETE t1, t2 FROM {$wpdb->options} t1 
         LEFT JOIN {$wpdb->options} t2 ON t2.option_name = CONCAT('_transient_timeout_', SUBSTRING(t1.option_name, 12))
         WHERE t1.option_name LIKE '_transient_zc_%' 
         AND t2.option_value < UNIX_TIMESTAMP()"
    );
    
    // If still too many, delete oldest non-expired ones
    $old_cache_keys = $wpdb->get_col($wpdb->prepare(
        "SELECT t1.option_name FROM {$wpdb->options} t1
         LEFT JOIN {$wpdb->options} t2 ON t2.option_name = CONCAT('_transient_timeout_', SUBSTRING(t1.option_name, 12))
         WHERE t1.option_name LIKE '_transient_zc_%%'
         AND (t2.option_value IS NULL OR t2.option_value > UNIX_TIMESTAMP())
         ORDER BY t1.option_id ASC
         LIMIT %d",
        $limit
    ));
    
    foreach ($old_cache_keys as $key) {
        delete_transient(str_replace('_transient_', '', $key));
    }
    
    error_log("ZenCode AI: Cleaned up " . count($old_cache_keys) . " old cache entries");
}

/**
 * NEW: Cache statistics for admin
 */
public function get_cache_stats() {
    global $wpdb;
    
    $total_cache = $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE '_transient_zc_%'"
    );
    
    $expired_cache = $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->options} t1 
         LEFT JOIN {$wpdb->options} t2 ON t2.option_name = CONCAT('_transient_timeout_', SUBSTRING(t1.option_name, 12))
         WHERE t1.option_name LIKE '_transient_zc_%' 
         AND t2.option_value < UNIX_TIMESTAMP()"
    );
    
    return [
        'total_entries' => (int) $total_cache,
        'expired_entries' => (int) $expired_cache,
        'active_entries' => (int) $total_cache - (int) $expired_cache
    ];
}
	private function setup_cache_invalidation() {
    // Clear cache when relevant settings change
    $settings_that_invalidate_cache = [
        'zencode_ai_google_api_key',
        'zencode_ai_default_model', 
        'zencode_ai_temperature',
        'zencode_ai_max_tokens'
    ];
    
    foreach ($settings_that_invalidate_cache as $setting) {
        add_action("update_option_{$setting}", [$this, 'clear_cache']);
    }
    
    // Clear cache on plugin updates
    add_action('upgrader_process_complete', function($upgrader, $data) {
        if (isset($data['plugins']) && is_array($data['plugins'])) {
            foreach ($data['plugins'] as $plugin) {
                if (strpos($plugin, 'zencode-ai') !== false) {
                    $this->clear_cache();
                    break;
                }
            }
        }
    }, 10, 2);
}








// NEW FUNCTIONS - Add these inside the ZenCodeAI class

/**
 * Create a project version snapshot
 */
private function create_project_snapshot($user_id, $project_name, $version_name, $project_data, $description = '', $is_auto = false, $trigger = null) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_project_versions';
    
    // Validate inputs
    if (empty($project_name) || empty($version_name) || empty($project_data)) {
        return false;
    }
    
    // Check if we already have this exact version (prevent duplicates)
    if ($is_auto) {
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table_name 
             WHERE user_id = %d AND project_name = %s AND version_name = %s AND is_auto_snapshot = 1
             ORDER BY created_at DESC LIMIT 1",
            $user_id, $project_name, $version_name
        ));
        
        if ($existing) {
            // Update existing auto-snapshot instead of creating duplicate
            return $wpdb->update(
                $table_name,
                [
                    'project_data' => $project_data,
                    'created_at' => current_time('mysql'),
                    'snapshot_trigger' => $trigger
                ],
                ['id' => $existing],
                ['%s', '%s', '%s'],
                ['%d']
            );
        }
    }
    
    // Clean up old auto-snapshots (keep only last 10 per project)
    if ($is_auto) {
        $old_snapshots = $wpdb->get_col($wpdb->prepare(
            "SELECT id FROM $table_name 
             WHERE user_id = %d AND project_name = %s AND is_auto_snapshot = 1
             ORDER BY created_at DESC LIMIT 999 OFFSET 9",
            $user_id, $project_name
        ));
        
        if (!empty($old_snapshots)) {
            $ids_to_delete = implode(',', array_map('intval', $old_snapshots));
            $wpdb->query("DELETE FROM $table_name WHERE id IN ($ids_to_delete)");
        }
    }
    
    // Create the snapshot
    $result = $wpdb->insert($table_name, [
        'user_id' => $user_id,
        'project_name' => $project_name,
        'version_name' => $version_name,
        'version_description' => $description,
        'project_data' => $project_data,
        'created_at' => current_time('mysql'),
        'is_auto_snapshot' => $is_auto ? 1 : 0,
        'snapshot_trigger' => $trigger
    ]);
    
    return $result !== false ? $wpdb->insert_id : false;
}

/**
 * Get project versions list
 */
private function get_project_versions($user_id, $project_name, $limit = 20) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'zencode_ai_project_versions';
    
    return $wpdb->get_results($wpdb->prepare(
        "SELECT id, version_name, version_description, created_at, is_auto_snapshot, snapshot_trigger
         FROM $table_name 
         WHERE user_id = %d AND project_name = %s 
         ORDER BY created_at DESC 
         LIMIT %d",
        $user_id, $project_name, $limit
    ));
}

/**
 * Restore project from version
 */
private function restore_project_version($user_id, $version_id) {
    global $wpdb;
    $versions_table = $wpdb->prefix . 'zencode_ai_project_versions';
    $projects_table = $wpdb->prefix . 'zencode_ai_projects';
    
    // Get version data with authorization check
    $version = $wpdb->get_row($wpdb->prepare(
        "SELECT project_name, project_data, version_name FROM $versions_table 
         WHERE id = %d AND user_id = %d",
        $version_id, $user_id
    ));
    
    if (!$version) {
        return false;
    }
    
    // Create a backup before restoration
    $current_project = $wpdb->get_row($wpdb->prepare(
        "SELECT project_data FROM $projects_table 
         WHERE user_id = %d AND project_name = %s",
        $user_id, $version->project_name
    ));
    
    if ($current_project) {
        $this->create_project_snapshot(
            $user_id, 
            $version->project_name,
            'Before restore to: ' . $version->version_name,
            $current_project->project_data,
            'Auto-backup before version restoration',
            true,
            'pre_restore'
        );
    }
    
    // Restore the version
    $result = $wpdb->update(
        $projects_table,
        [
            'project_data' => $version->project_data,
            'last_modified' => current_time('mysql')
        ],
        [
            'user_id' => $user_id,
            'project_name' => $version->project_name
        ],
        ['%s', '%s'],
        ['%d', '%s']
    );
    
    return $result !== false;
}

/**
 * AJAX handler for version control operations
 */
public function handle_version_control() {
    $user_id = $this->validate_ajax_request();
    $action = sanitize_text_field($_POST['version_action'] ?? '');
    
    switch ($action) {
        case 'create_manual_snapshot':
            $project_name = sanitize_text_field($_POST['project_name'] ?? '');
            $version_name = sanitize_text_field($_POST['version_name'] ?? '');
            $description = sanitize_textarea_field($_POST['description'] ?? '');
            $project_data = stripslashes($_POST['project_data'] ?? '');
            
            // Validate inputs
            if (empty($project_name) || empty($version_name) || empty($project_data)) {
                wp_send_json_error(['message' => 'Missing required fields.'], 400);
                wp_die();
            }
            
            if (strlen($version_name) > 100 || strlen($description) > 500) {
                wp_send_json_error(['message' => 'Version name or description too long.'], 400);
                wp_die();
            }
            
            $snapshot_id = $this->create_project_snapshot(
                $user_id, $project_name, $version_name, $project_data, $description
            );
            
            if ($snapshot_id) {
                wp_send_json_success(['message' => 'Version saved successfully.', 'snapshot_id' => $snapshot_id]);
            } else {
                wp_send_json_error(['message' => 'Failed to create version.'], 500);
            }
            break;
            
// ONLY replace the 'list_versions' case in handle_version_control
case 'list_versions':
    $project_name = sanitize_text_field($_POST['project_name'] ?? '');
    if (empty($project_name)) {
        wp_send_json_success(['versions' => []]); // FIX: Return empty array instead of error
        wp_die();
    }
    
    $versions = $this->get_project_versions($user_id, $project_name);
    $versions = is_array($versions) ? $versions : []; // FIX: Ensure array
    wp_send_json_success(['versions' => $versions]);
    break;
            
        case 'restore_version':
            $version_id = absint($_POST['version_id'] ?? 0);
            if ($version_id <= 0) {
                wp_send_json_error(['message' => 'Invalid version ID.'], 400);
                wp_die();
            }
            
            $result = $this->restore_project_version($user_id, $version_id);
            if ($result) {
                wp_send_json_success(['message' => 'Version restored successfully.']);
            } else {
                wp_send_json_error(['message' => 'Failed to restore version.'], 500);
            }
            break;
            
        case 'delete_version':
            $version_id = absint($_POST['version_id'] ?? 0);
            if ($version_id <= 0) {
                wp_send_json_error(['message' => 'Invalid version ID.'], 400);
                wp_die();
            }
            
            global $wpdb;
            $table_name = $wpdb->prefix . 'zencode_ai_project_versions';
            $result = $wpdb->delete(
                $table_name,
                ['id' => $version_id, 'user_id' => $user_id],
                ['%d', '%d']
            );
            
            if ($result) {
                wp_send_json_success(['message' => 'Version deleted successfully.']);
            } else {
                wp_send_json_error(['message' => 'Failed to delete version.'], 500);
            }
            break;
            
        default:
            wp_send_json_error(['message' => 'Invalid action.'], 400);
    }
    wp_die();
}

/**
 * Get project versions list
 */

/**
 * NEW FUNCTION: Add this to your ZenCodeAI class (replaces previous version)
 */
/**
 * Change from private to public
 */
// --- NEW PHP FUNCTION ---
// Add this entire function inside your ZenCodeAI class.

/**
 * NEW: AJAX handler for generating a project specification sheet.
 */
public function handle_generate_specification() {
    $this->validate_ajax_request();

    $prompt = $this->validate_and_sanitize_prompt($_POST['prompt'] ?? '');
    if (empty($prompt)) {
        wp_send_json_error(['message' => 'Initial prompt cannot be empty.'], 400);
        wp_die();
    }
    
    $image_data = !empty($_POST['image_data']) && $this->validate_image_data($_POST['image_data']) ? $_POST['image_data'] : null;

    try {
        // --- START OF CHANGES ---
        // The prompt now explicitly forbids JSON inside the string values and enforces Markdown lists.
        $spec_system_prompt = <<<PROMPT
You are a world-class AI Project Consultant. Your job is to transform a user's idea into a comprehensive, professional JSON specification sheet.

**CORE DIRECTIVE: NO EMPTY FIELDS. EVER.**
For every single key in the JSON, you MUST provide a detailed, valuable string. If the user's prompt is vague, it is your job to infer and suggest a complete, professional starting point.

**CONTENT FORMATTING RULE (VERY IMPORTANT):**
The string values for each key MUST be human-readable, formatted as paragraphs or Markdown bulleted lists (using '-'). **You are strictly forbidden from outputting raw JSON, JSON arrays, or any code-like syntax within these string values.**

**AGENCY DESIGN PHILOSOPHY (MUST BE REFLECTED IN 'design_preferences'):**
- **Aesthetics are Paramount:** Use modern trends like subtle gradients, soft shadows, and ample white space.
- **Premium Feel:** The project must feel expensive and high-quality.
- **Micro-Interactions Matter:** Specify subtle hover effects and smooth transitions.
- **Mobile-First Always:** The design must be flawlessly responsive.

**CRITICAL ANALYSIS STEP (PERFORM FIRST):**
1.  **[Literal Technical Request]:** Is the prompt a simple, technical instruction (e.g., "make a red square")? If YES, the spec must be literal. DO NOT invent a brand.
2.  **[Conceptual Project Request]:** Is it a full website or web app? If YES, act as a consultant. Infer a full feature set, suggest a tech stack, and define a premium design.

**Your entire response MUST be a single, valid JSON object.**

{
  "scope": "You MUST infer a clear goal. Example: 'To design and develop a sophisticated, single-page web application for efficiently managing events...'",
  "features": "You MUST suggest a standard, complete feature set using a Markdown bulleted list. Example for an events app:\n- A list view to display all events.\n- A form (modal or inline) to add new events.\n- Edit and Delete buttons for each event.\n- A fully responsive navigation menu that becomes a hamburger on mobile.",
  "technical_requirements": "You MUST specify a complete tech stack. Example: 'HTML5, CSS3 (with Flexbox/Grid for layout), and modern vanilla JavaScript to manage the dynamic single-page interface and state.'",
  "user_types": "You MUST define user roles using a Markdown bulleted list. Example:\n- Administrator: Can perform all CRUD actions.\n- General Visitor: Can view events.",
  "design_preferences": "You MUST provide a detailed design direction. Example: 'A clean, modern, and professional aesthetic with a dark-mode theme. Interactive elements will have subtle transitions and hover effects.'",
  "integration_needs": "You MUST suggest potential integrations or state 'None'. Example: 'None specified initially. Suggest considering a database (like Firebase or a simple JSON server) for data persistence.'",
  "success_criteria": "You MUST define a clear success metric. Example: 'A fully functional single-page application where an administrator can seamlessly add, view, edit, and delete events without page reloads.'"
}

**FINAL INSTRUCTION:** Generate the complete, fully-populated JSON specification now, ensuring all string values are human-readable.
PROMPT;
        // --- END OF CHANGES ---

       $api_key = $this->validate_api_key('google');
        $model = get_option('zencode_ai_default_model', 'google/gemini-2.5-pro');
        
        $user_content = [['type' => 'text', 'text' => $prompt]];
        if ($image_data) {
            $user_content[] = ['type' => 'image_url', 'image_url' => ['url' => $image_data]];
        }
        
        $messages = [
            ['role' => 'system', 'content' => $spec_system_prompt],
            ['role' => 'user', 'content' => $user_content]
        ];

        $raw_response = $this->call_google_api($api_key, $model, $messages, false, true);
        
        // --- START OF THE FIX: JSON EXTRACTION LOGIC ---
        // This block makes the function robust against conversational text.
        $json_response = null;
        $first_brace = strpos($raw_response, '{');
        $last_brace = strrpos($raw_response, '}');

        if ($first_brace !== false && $last_brace !== false && $last_brace > $first_brace) {
            $json_string = substr($raw_response, $first_brace, ($last_brace - $first_brace) + 1);
            json_decode($json_string);
            if (json_last_error() === JSON_ERROR_NONE) {
                $json_response = $json_string;
            }
        }
        // --- END OF THE FIX ---

        // If after trying to extract it, we still don't have valid JSON, then we fail.
        if ($json_response === null) {
            // This is the error that your console was reporting.
            throw new Exception("AI failed to return valid JSON for the specification.");
        }

        wp_send_json_success(['specification' => $json_response]);

    } catch (Exception $e) {
        wp_send_json_error(['message' => 'Specification generation failed: ' . $e->getMessage()], 500);
    }
    wp_die();
}
// ADD THESE THREE NEW FUNCTIONS inside the ZenCodeAI class in zencode-aiphp.js

/**
 * AJAX handler for streaming responses
 */
public function handle_stream_response() {
    error_log("====== STREAMING REQUEST STARTED ======");
    $this->validate_ajax_request();
    
    // Set SSE headers
    header('Content-Type: text/event-stream');
    header('Cache-Control: no-cache');
    header('Connection: keep-alive');
    header('X-Accel-Buffering: no');
    
    // Clear output buffering
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    try {
        $prompt = sanitize_textarea_field(stripslashes($_POST['prompt'] ?? ''));
        $model = sanitize_text_field($_POST['model'] ?? 'google/gemini-2.5-pro');
        
        error_log("PROMPT: " . substr($prompt, 0, 200));
        error_log("MODEL: " . $model);
        
        if (empty($prompt)) {
            error_log("ERROR: Prompt is empty");
            echo "data: " . json_encode(['error' => 'Prompt is required']) . "\n\n";
            flush();
            exit;
        }
        
        // For Google, we build the 'contents' array here
        $messages = [['role' => 'user', 'content' => $prompt]];
        
        error_log("Starting Google API stream...");
        // Make the streaming API call
        $this->stream_from_google_api($model, $messages);
        error_log("Google API stream completed");
        
    } catch (Exception $e) {
        error_log("STREAMING ERROR: " . $e->getMessage());
        echo "data: " . json_encode(['error' => $e->getMessage()]) . "\n\n";
        flush();
    }
    
    // Send completion message
    error_log("Sending completion signal");
    echo "data: " . json_encode(['done' => true]) . "\n\n";
    flush();
    error_log("====== STREAMING REQUEST ENDED ======");
    exit;
}

/**
 * Makes a streaming call to the Google Gemini API
 */
private function stream_from_google_api($model, $messages) {
    // Clear the stream buffer for this new request
    $this->stream_buffer = '';
    
    $api_key = $this->validate_api_key('google');
    $model_parts = explode('/', $model);
    $clean_model = end($model_parts);

    error_log("STREAMING MODEL: " . $clean_model);

    // Prepare request data for Google Gemini format
    $contents = [];
    foreach ($messages as $message) {
         $role = $message['role'] === 'assistant' ? 'model' : 'user';
         $contents[] = [
            'role' => $role,
            'parts' => [['text' => $message['content']]]
         ];
    }

    $request_body = [
        'contents' => $contents
    ];

    $url = "https://generativelanguage.googleapis.com/v1beta/models/{$clean_model}:streamGenerateContent?key=" . $api_key;
    error_log("STREAMING URL: " . str_replace($api_key, 'REDACTED', $url));
    error_log("REQUEST BODY: " . json_encode($request_body));

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => false,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
        ],
        CURLOPT_POSTFIELDS => json_encode($request_body),
        CURLOPT_WRITEFUNCTION => [$this, 'handle_stream_chunk'],
        CURLOPT_TIMEOUT => 120
    ]);

    error_log("Executing cURL request...");
    $result = curl_exec($ch);
    
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    error_log("cURL HTTP CODE: " . $http_code);

    if (curl_errno($ch)) {
        $error = curl_error($ch);
        curl_close($ch);
        // We can't throw an exception here as headers are already sent,
        // so we send an error event through the stream.
        echo "data: " . json_encode(['error' => 'Stream cURL error: ' . $error]) . "\n\n";
        flush();
    }
    
    curl_close($ch);
}

/**
 * Processes each chunk from the streaming response
 * Google returns responses as a JSON array: [{...}]
 */
private function handle_stream_chunk($ch, $data) {
    // Add new data to buffer
    $this->stream_buffer .= $data;
    
    error_log("STREAM CHUNK RECEIVED (" . strlen($data) . " bytes). Buffer size: " . strlen($this->stream_buffer));
    
    // Try to extract complete JSON objects from buffer
    // Google's format: [{...}, {...}] or just [{...}]
    
    // Remove leading/trailing whitespace
    $trimmed = trim($this->stream_buffer);
    
    // Check if we have a complete array
    if (substr($trimmed, 0, 1) === '[' && substr($trimmed, -1) === ']') {
        error_log("COMPLETE JSON ARRAY DETECTED");
        
        // Try to parse the complete array
        $parsed = json_decode($trimmed, true);
        
        if ($parsed && is_array($parsed)) {
            error_log("SUCCESSFULLY PARSED " . count($parsed) . " object(s)");
            
            // Process each object in the array
            foreach ($parsed as $item) {
                if (isset($item['candidates'][0]['content']['parts'])) {
                    // Extract all text parts
                    foreach ($item['candidates'][0]['content']['parts'] as $part) {
                        if (isset($part['text'])) {
                            $token = $part['text'];
                            // Send in SSE format
                            echo "data: " . json_encode(['token' => $token]) . "\n\n";
                            flush();
                            error_log("SENT TOKEN: " . substr($token, 0, 100));
                        }
                    }
                } elseif (isset($item['error'])) {
                    echo "data: " . json_encode(['error' => $item['error']['message']]) . "\n\n";
                    flush();
                    error_log("SENT ERROR: " . $item['error']['message']);
                }
            }
            
            // Clear buffer after successful processing
            $this->stream_buffer = '';
        } else {
            $error = json_last_error_msg();
            error_log("FAILED TO PARSE COMPLETE ARRAY: " . $error);
        }
    } else {
        error_log("Buffering... (not complete yet)");
    }
    
    return strlen($data);
}

private function call_google_api($api_key, $model, $messages, $is_one_shot_build = false, $is_planning_phase = false, $override_max_tokens = null, $override_timeout = null, $is_step_execution = false) {
    
    // --- SAFETY MECHANISMS CONFIGURATION ---
    $safety_config = [
        'max_retries' => 3,
        'retry_delay' => 2, // seconds
        'retry_backoff' => true,
        'timeout_buffer' => 5,
    ];
    
    // --- VALIDATE INPUTS ---
    if (empty($api_key)) {
        throw new Exception('Google Gemini API key is required');
    }
    
    if (empty($model) || empty($messages)) {
        throw new Exception('Model and messages are required');
    }
    
    // --- TOKEN LIMIT LOGIC ---
    $model_max_limits = [
        'gemini-2.5-pro' => 24576,
        'gemini-1.5-pro' => 24576,
    ];
    
  // 1. Instead of getting an option, we now define our desired default limit directly.
//    12288 is a safe and generous value for high-quality code output.
$user_max_tokens = 24576;

// 2. Set the current limit based on which model is being used. This acts as a safety rail.
$current_model_limit = 4096; // Default to a safe, small limit.
foreach ($model_max_limits as $model_family => $limit) {
    if (strpos($model, $model_family) !== false) {
        $current_model_limit = $limit;
        break;
    }
}

    $final_max_tokens = min($user_max_tokens, $current_model_limit);
    
    if ($override_max_tokens !== null) {
        $final_max_tokens = min((int)$override_max_tokens, $current_model_limit);
    }
    
    // --- ENHANCED TIMEOUT LOGIC ---
    $base_timeout = $override_timeout ?? ($is_one_shot_build ? 300 : ($is_step_execution ? 240 : 180));
    $timeout_seconds = $base_timeout + $safety_config['timeout_buffer'];
    
    // Set PHP execution time limit with buffer
    @set_time_limit($timeout_seconds + 30);
    
    // --- PREPARE REQUEST DATA FOR GOOGLE GEMINI ---
    $model_parts = explode('/', $model);
    $clean_model = end($model_parts);
    
    // Build Google Gemini API request format
    $system_instruction = null;
    $contents = [];
    
    foreach ($messages as $message) {
        if ($message['role'] === 'system') { 
            $system_instruction = $message['content']; 
            continue; 
        }
        
        $parts = [];
        
        if (is_array($message['content'])) {
            foreach ($message['content'] as $part) {
                if ($part['type'] === 'text') { 
                    $parts[] = ['text' => $part['text']]; 
                } elseif ($part['type'] === 'image_url') {
                    list($header, $data) = explode(',', $part['image_url']['url'], 2);
                    $media_type = str_replace(['data:', ';base64'], '', $header);
                    $parts[] = [
                        'inline_data' => [
                            'mime_type' => $media_type, 
                            'data' => $data
                        ]
                    ];
                }
            }
        } else {
            $parts[] = ['text' => $message['content']];
        }
        
        $role = $message['role'] === 'assistant' ? 'model' : 'user';
        $contents[] = [
            'role' => $role,
            'parts' => $parts
        ];
    }
    
    $request_body = [
        'contents' => $contents,
        'generationConfig' => [
            'temperature' => (float)get_option('zencode_ai_temperature', 0.7),
            'maxOutputTokens' => $final_max_tokens,
            'topK' => 40,
            'topP' => 0.95,
        ]
    ];
    
    if (!empty($system_instruction)) {
        $request_body['systemInstruction'] = [
            'parts' => [
                ['text' => $system_instruction]
            ]
        ];
    }

    // --- RETRY LOGIC ---
    $last_exception = null;
    $url = "https://generativelanguage.googleapis.com/v1beta/models/{$clean_model}:generateContent?key=" . $api_key;
    
    for ($attempt = 1; $attempt <= $safety_config['max_retries']; $attempt++) {
        try {
            // Calculate retry delay with exponential backoff
            if ($attempt > 1) {
                $delay = $safety_config['retry_delay'] * pow(2, $attempt - 2);
                sleep($delay);
            }
            
            $response = wp_remote_post($url, [
                'headers' => [
                    'Content-Type' => 'application/json'
                ],
                'body'    => json_encode($request_body),
                'timeout' => $timeout_seconds
            ]);

            // Check for WordPress HTTP errors
            if (is_wp_error($response)) {
                $error_message = $response->get_error_message();
                
                // Check if it's a timeout error
                if (strpos($error_message, 'timeout') !== false) {
                    throw new Exception('API request timeout: ' . $error_message);
                }
                
                throw new Exception('HTTP request failed: ' . $error_message);
            }
            
            $response_code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);
            
            // Handle HTTP status codes
            if ($response_code !== 200) {
                $error_message = 'Unknown error';
                $error_type = 'unknown_error';
                
                if (isset($body['error'])) {
                    $error_message = $body['error']['message'] ?? $error_message;
                    $error_type = $body['error']['status'] ?? $error_type;
                }
                
                // Rate limiting - wait and retry
                if ($response_code === 429 || $error_type === 'RATE_LIMIT_EXCEEDED') {
                    sleep(10);
                    continue;
                }
                
                // Service overloaded - retry
                if ($response_code === 503 || $error_type === 'RESOURCE_EXHAUSTED' || 
                    stripos($error_message, 'overloaded') !== false) {
                    if ($attempt < $safety_config['max_retries']) {
                        error_log("Google Gemini API: Service overloaded, retrying attempt {$attempt}/{$safety_config['max_retries']}");
                        continue;
                    }
                }
                
                // For client errors, throw immediately
                if ($response_code >= 400 && $response_code < 500 && 
                    $error_type !== 'RATE_LIMIT_EXCEEDED' && 
                    $error_type !== 'RESOURCE_EXHAUSTED') {
                    throw new Exception("Google Gemini API error ({$error_type}): " . $error_message);
                }
                
                // For server errors, retry
                if ($response_code >= 500) {
                    throw new Exception("Google Gemini API server error (HTTP {$response_code})");
                }
            }
            
            // Success! Extract the response text
            $response_text = '';
            if (isset($body['candidates'][0]['content']['parts'])) {
                foreach ($body['candidates'][0]['content']['parts'] as $part) {
                    if (isset($part['text'])) {
                        $response_text .= $part['text'];
                    }
                }
            }
            
            // Track token usage if available
            if (isset($body['usageMetadata'])) {
                $total_tokens = ($body['usageMetadata']['promptTokenCount'] ?? 0) + 
                               ($body['usageMetadata']['candidatesTokenCount'] ?? 0);
                if ($total_tokens > 0) {
                    $this->track_token_usage(get_current_user_id(), $total_tokens);
                }
            }
            
            return $response_text;
            
        } catch (Exception $e) {
            $last_exception = $e;
            
            // Log the attempt
            error_log("Google Gemini API attempt {$attempt}/{$safety_config['max_retries']} failed: " . $e->getMessage());
            
            // Don't retry on authentication or invalid request errors
            $message = $e->getMessage();
            if (stripos($message, 'INVALID_ARGUMENT') !== false ||
                stripos($message, 'UNAUTHENTICATED') !== false ||
                stripos($message, 'PERMISSION_DENIED') !== false) {
                break;
            }
            
            // Last attempt failed
            if ($attempt === $safety_config['max_retries']) {
                break;
            }
        }
    }
    
    // All retries failed
    throw new Exception(
        "Google Gemini API failed after {$safety_config['max_retries']} attempts. Last error: " . 
        $last_exception->getMessage()
    );
}
/**
 * Fast local specification pruning - NO API CALLS
 * Saves 60-90 seconds per 30-step plan
 */
private function prune_specification_locally($full_spec, $step_text) {
    if (!is_array($full_spec)) {
        return [];
    }
    
    $pruned = [];
    $step_lower = strtolower($step_text);
    
    // Always include these critical sections
    $always_include = [
        'design_preferences', 
        'technical_requirements', 
        'color_scheme', 
        'fonts',
        'layout',
        'typography',
        'tech_stack',
        'design',
        'requirements'
    ];
    
    // Extract meaningful keywords from step (ignore common words)
    $stop_words = ['the','a','an','and','or','but','in','on','at','to','for','of','with','create','add','update','modify','file','section'];
    $words = preg_split('/[\s,\.]+/', strtolower($step_text));
    $keywords = array_diff($words, $stop_words);
    $keywords = array_filter($keywords, function($w) { return strlen($w) > 3; });
    
    foreach ($full_spec as $section_key => $section_content) {
        $section_key_lower = strtolower($section_key);
        
        // Always include critical sections
        if (in_array($section_key_lower, $always_include)) {
            $pruned[$section_key] = $section_content;
            continue;
        }
        
        // Check if section key matches step
        if (stripos($step_lower, $section_key_lower) !== false) {
            $pruned[$section_key] = $section_content;
            continue;
        }
        
        // Check keyword relevance
        $section_str = strtolower(json_encode($section_content));
        foreach ($keywords as $keyword) {
            if (stripos($section_str, $keyword) !== false) {
                $pruned[$section_key] = $section_content;
                break;
            }
        }
    }
    
    return $pruned;
}

public function handle_prune_specification() {
    $this->validate_ajax_request();

    // Sanitize the input from the client
    $full_spec_json = sanitize_textarea_field(stripslashes($_POST['full_specification'] ?? ''));
    $step_text = sanitize_textarea_field($_POST['step_text'] ?? '');
    
    if (empty($full_spec_json) || empty($step_text)) {
        wp_send_json_error(['message' => 'Missing specification or step data.'], 400);
        wp_die();
    }
    
    $full_spec = json_decode($full_spec_json, true);

    // Validate the input
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($full_spec)) {
        wp_send_json_error(['message' => 'Invalid JSON specification.'], 400);
        wp_die();
    }

    // Use FAST local pruning - NO API CALL - Saves 2-3 seconds per step
    $pruned_spec = $this->prune_specification_locally($full_spec, $step_text);
    
    wp_send_json_success(['pruned_spec' => json_encode($pruned_spec, JSON_UNESCAPED_UNICODE)]);
    wp_die();
}

} // End of ZenCodeAI Class

if (!function_exists('zc_get_user_pricing_plan')) {
    function zc_get_user_pricing_plan($user_id = null) {
        return 'visionary'; // Full access for everyone
    }
}

// Allow all logged-in users to query (called in get_user_context)
if (!function_exists('zc_can_user_query')) {
    function zc_can_user_query($user_id = null) {
        return true; // Always allow
    }
}

// All features available to everyone (might be called for feature checks)
if (!function_exists('zc_user_has_feature')) {
    function zc_user_has_feature($feature_slug, $user_id = null) {
        return true; // All features enabled
    }
}

// === PROBABLY NEEDED (Safe to keep) ===

// No query counting (might be called when user makes queries)
if (!function_exists('zc_increment_query_count')) {
    function zc_increment_query_count($user_id = null) {
        return true; // Pretend success, don't actually count
    }
}

// No limits (might be used in rate limiting checks)
if (!function_exists('zc_get_rolling_limit_for_plan')) {
    function zc_get_rolling_limit_for_plan($plan) {
        return null; // No limits
    }
}

// No upgrade prompts (might be shown in UI)
if (!function_exists('zc_display_upgrade_prompt')) {
    function zc_display_upgrade_prompt() {
        return ''; // No prompts
    }
}

// User alerts (might be used for notifications)
if (!function_exists('zc_set_user_alert')) {
    function zc_set_user_alert($user_id, $message) {
        return; // Do nothing (or optionally keep this feature)
    }
}



/**
 * Read a plugin setting with a default fallback.
 * Expects all settings under the single option key: 'zencode_ai_settings'
 */
function zencode_ai_get_setting( $key, $default = null ) {
    $opts = get_option( 'zencode_ai_settings', [] );

    if ( is_array( $opts ) && array_key_exists( $key, $opts ) ) {
        $val = $opts[ $key ];
        // Treat empty string like "unset"
        if ($val !== '' && $val !== null) {
            return $val;
        }
    }

    return $default;
}




new ZenCodeAI();