<?php
/**
 * ZenCode AI - Customer License Validation (USER VERSION)
 * 
 * Validates licenses against YOUR server via API and provides the user interface for activation and upgrades.
 * STYLING: Updated to modern modal and card UI with custom Stripe button.
 */

if (!defined('ABSPATH')) {
    exit;
}

class ZenCode_Customer_License {
    
    private $license_server_url = 'https://planckdev.com/wp-json/zencode/v1/license'; // YOUR SERVER URL
    private $product_id = 'zencode-ai';
    
    public function __construct() {
        // Add admin menu
        add_action('admin_menu', [$this, 'add_license_page']);
        add_action('admin_notices', [$this, 'show_license_notices']);
        
        // AJAX handlers
        add_action('wp_ajax_zencode_activate_license', [$this, 'ajax_activate_license']);
        add_action('wp_ajax_zencode_start_trial', [$this, 'ajax_start_trial']);
        add_action('wp_ajax_zencode_deactivate_license', [$this, 'ajax_deactivate_license']);
    }
    
    /**
     * Add license page to admin menu
     */
    public function add_license_page() {
        add_menu_page(
            'ZenCode License',
            'ZenCode License',
            'edit_posts',
            'zencode-license',
            [$this, 'render_page_router'], // Use a router to show the correct page
            'dashicons-admin-network',
            30
        );
    }

    /**
     * Router to decide which page to show (main status or upgrade page)
     */
  // [REPLACE THIS FUNCTION in zencode-customer-licensephp.js]
public function render_page_router() {
    // Handle payment returns FIRST (before any output)
    if (isset($_GET['payment'])) {
        if ($_GET['payment'] === 'success') {
            // Clear the URL and show success message
            ?>
            <div class="wrap">
                <div style="max-width: 700px; margin: 50px auto; padding: 40px; background: #d1fae5; border-left: 4px solid #10b981; border-radius: 8px; text-align: center;">
                    <h1 style="color: #065f46; margin: 0 0 20px 0;">🎉 Payment Successful!</h1>
                    <p style="font-size: 18px; color: #065f46; margin-bottom: 30px;">Your license has been activated. Thank you for your purchase!</p>
                    <a href="<?php echo admin_url('admin.php?page=zencode-license'); ?>" class="button button-primary button-hero">View Your License</a>
                    <script>
                        // Auto-redirect after 3 seconds
                        setTimeout(function() {
                            window.location.href = '<?php echo admin_url('admin.php?page=zencode-license'); ?>';
                        }, 3000);
                    </script>
                </div>
            </div>
            <?php
            return;
        } elseif ($_GET['payment'] === 'cancelled') {
            ?>
            <div class="wrap">
                <div style="max-width: 700px; margin: 50px auto; padding: 40px; background: #fee2e2; border-left: 4px solid #ef4444; border-radius: 8px; text-align: center;">
                    <h1 style="color: #991b1b; margin: 0 0 20px 0;">Payment Cancelled</h1>
                    <p style="font-size: 18px; color: #991b1b; margin-bottom: 30px;">You cancelled the payment. No charges were made.</p>
                    <a href="<?php echo admin_url('admin.php?page=zencode-license&action=upgrade'); ?>" class="button button-primary">Try Again</a>
                </div>
            </div>
            <?php
            return;
        }
    }
    
    // Normal routing
    $action = $_GET['action'] ?? '';
    
    if ($action === 'upgrade') {
        echo $this->render_upgrade_page();
    } else {
        echo $this->render_license_status_page();
    }
}
    
    /**
     * Validate license with remote server
     */
// [REPLACE your entire validate_license function with this]

/**
 * Validate license with remote server
 */
public function validate_license($license_key = null, $force_check = false) {
    if (empty($license_key)) {
        $license_key = get_option('zencode_license_key');
    }
    
    if (empty($license_key)) {
        return ['valid' => false, 'status' => 'no_license', 'message' => 'No license key found'];
    }
    
    $cache_key = 'zencode_license_check_' . md5($license_key . site_url());
    $cached = get_transient($cache_key);
    
    if (!$force_check && $cached !== false) {
        return $cached;
    }
    
    $response = wp_remote_post($this->license_server_url . '/validate', [
        'body' => ['license_key' => $license_key, 'domain' => site_url(), 'product_id' => $this->product_id],
        'timeout' => 15
    ]);
    
    // --- START OF THE DEFINITIVE FIX ---
    if (is_wp_error($response)) {
        // The server is unreachable. Check if we have old data we can use temporarily.
        if ($cached && isset($cached['valid']) && $cached['valid']) {
            // We have a last known good response. Reuse it but add a grace period notice.
            $cached['status'] = $cached['status'] ?? 'active'; // Ensure status exists
            $cached['message'] = 'License check temporarily unavailable. Using last valid status (Grace Period).';
            $cached['grace_period_active'] = true; // Add a flag for debugging or special UI
            return $cached; // Return the last known good data
        }
        
        // If there's no cached data at all, return a hard error.
        return ['valid' => false, 'status' => 'server_error', 'message' => 'Could not contact license server: ' . $response->get_error_message()];
    }
    // --- END OF THE DEFINITIVE FIX ---
    
    $body = json_decode(wp_remote_retrieve_body($response), true);
    
    if (!$body) {
        // Handle cases where the server returns an empty or invalid JSON response
        return ['valid' => false, 'status' => 'server_error', 'message' => 'Invalid response from license server.'];
    }

    // Self-healing: Automatically delete permanently invalid keys from the user's site.
    if (isset($body['valid']) && !$body['valid']) {
        $permanently_invalid_statuses = ['cancelled', 'expired', 'not_found', 'suspended'];
        if (isset($body['status']) && in_array($body['status'], $permanently_invalid_statuses)) {
            delete_option('zencode_license_key');
        }
    }

    if (isset($body['valid']) && $body['valid']) {
        update_option('zencode_last_valid_check', time());
        set_transient($cache_key, $body, HOUR_IN_SECONDS);
    }
    
    return $body;
}
    
    /**
     * AJAX: Start trial
     */
    public function ajax_start_trial() {
        check_ajax_referer('zencode_license_action', 'nonce');
        
        $email = sanitize_email($_POST['email'] ?? '');
        if (empty($email)) {
            wp_send_json_error(['message' => 'Email is required']);
        }
        
        $response = wp_remote_post($this->license_server_url . '/start-trial', [
            'body' => [
                'email' => $email,
                'first_name' => sanitize_text_field($_POST['first_name'] ?? ''),
                'last_name' => sanitize_text_field($_POST['last_name'] ?? ''),
                'domain' => site_url(),
                'product_id' => $this->product_id
            ],
            'timeout' => 20
        ]);
        
        if (is_wp_error($response)) {
            wp_send_json_error(['message' => 'Could not start trial: ' . $response->get_error_message()]);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($body && isset($body['success']) && $body['success']) {
            update_option('zencode_license_key', $body['license_key']);
            wp_send_json_success(['message' => 'Trial started! Reloading page...', 'license_key' => $body['license_key']]);
        } else {
            wp_send_json_error(['message' => $body['message'] ?? 'Failed to start trial. You may already have a trial active.']);
        }
    }
    
    /**
     * AJAX: Activate license
     */
    public function ajax_activate_license() {
        check_ajax_referer('zencode_license_action', 'nonce');
        
        $license_key = sanitize_text_field($_POST['license_key'] ?? '');
        if (empty($license_key)) {
            wp_send_json_error(['message' => 'License key is required']);
        }
        
        $validation = $this->validate_license($license_key, true);
        
        if ($validation && isset($validation['valid']) && $validation['valid']) {
            update_option('zencode_license_key', $license_key);
            wp_send_json_success(['message' => 'License activated successfully! Reloading page...', 'status' => $validation['status']]);
        } else {
            wp_send_json_error(['message' => $validation['message'] ?? 'Invalid license key']);
        }
    }
    
    /**
     * AJAX: Deactivate license
     */
    public function ajax_deactivate_license() {
    check_ajax_referer('zencode_license_action', 'nonce');
    
    $license_key = get_option('zencode_license_key');
    
    // --- START OF THE FIX ---
    // Step 1: Tell the server to deactivate this domain.
    // This is now a BLOCKING call, and we will check the response.
    if ($license_key) {
        $response = wp_remote_post($this->license_server_url . '/deactivate', [
            'body' => [
                'license_key' => $license_key, 
                'domain'      => site_url()
            ],
            'timeout' => 15
        ]);

        // Add logging to see what the server is doing.
        if (is_wp_error($response)) {
            error_log('ZenCode Deactivation Error: ' . $response->get_error_message());
        } else {
            $http_code = wp_remote_retrieve_response_code($response);
            $body = wp_remote_retrieve_body($response);
            error_log('ZenCode Deactivation Log: Server responded with HTTP code ' . $http_code . ' and body: ' . $body);
        }
    }
    // --- END OF THE FIX ---

    // Step 2: Always delete the local key to reset the UI for the user, regardless of server response.
    delete_option('zencode_license_key');
    
    // Clear any cached validation checks
    if ($license_key) {
        $cache_key = 'zencode_license_check_' . md5($license_key . site_url());
        delete_transient($cache_key);
    }

    wp_send_json_success(['message' => 'License deactivated successfully. Reloading page...']);
}
    
    /**
     * Show license notices
     */
    public function show_license_notices() {
        $screen = get_current_screen();
       if (!$screen || strpos($screen->id, 'zencode') === false || in_array($screen->id, ['toplevel_page_zencode-license', 'toplevel_page_zencode-ai-app'])) {
            return;
        }
        
        $validation = $this->validate_license();
        
        if (!$validation || (isset($validation['valid']) && !$validation['valid'])) {
            ?>
            <div class="notice notice-error">
                <p><strong>ZenCode AI:</strong> Your license is not active. 
                <a href="<?php echo admin_url('admin.php?page=zencode-license'); ?>">Please activate your license to continue.</a></p>
            </div>
            <?php
        } elseif ($validation && $validation['status'] === 'trial') {
            $days_remaining = $validation['days_remaining'] ?? 0;
            if ($days_remaining <= 2) {
                ?>
                <div class="notice notice-warning">
                    <p><strong>ZenCode AI:</strong> Your trial expires in <?php echo $days_remaining; ?> day<?php echo $days_remaining != 1 ? 's' : ''; ?>! 
                    <a href="<?php echo admin_url('admin.php?page=zencode-license&action=upgrade'); ?>"><strong>Upgrade to Pro now</strong></a></p>
                </div>
                <?php
            }
        }
    }

    /**
     * Render the main license status page (modal or card)
     */
   // [REPLACE THIS ENTIRE FUNCTION in zencode-customer-licensephp.js]
// [REPLACE THIS ENTIRE FUNCTION in zencode-customer-licensephp.js]

// [REPLACE THIS ENTIRE FUNCTION in zencode-customer-licensephp.js]

private function render_license_status_page() {
    ob_start();
    $license_key = get_option('zencode_license_key');
    ?>
    <style> /* Styles for both modal and card */
        .zencode-page-container { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, Arial, sans-serif; }
        .zencode-modal-overlay { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: #0f172a; display: flex; align-items: center; justify-content: center; z-index:0 !important; }
        
        .zencode-modal-content { background: #1e293b; color: #e2e8f0; padding: 40px; border-radius: 12px; max-width: 800px; width: 90%; text-align: center; border: 1px solid #334155; box-shadow: 0 10px 25px rgba(0,0,0,0.3); }
        .zencode-modal-content .logo { font-size: 24px; font-weight: bold; color: #cbd5e1; margin-bottom: 15px; }
        .zencode-modal-content h1 { color: white; font-size: 28px; margin: 0 0 10px 0; }
        .zencode-modal-content .subtitle { font-size: 16px; color: #94a3b8; margin-bottom: 30px; }
        .modal-body { display: grid; grid-template-columns: 1fr auto 1fr; gap: 25px; align-items: center; }
        .modal-section { background: #0f172a; padding: 25px; border-radius: 8px; }
        .modal-section h2 { font-size: 20px; color: white; margin: 0 0 20px 0; }
        .modal-divider { color: #475569; font-weight: bold; font-size: 14px; }
        .modal-input { width: 100%; padding: 12px; background: #1e293b; border: 1px solid #334155; border-radius: 6px; color: white; font-size: 14px; margin-bottom: 15px; box-sizing: border-box;}
        .modal-button { display: inline-block; width: 100%; padding: 14px; border: none; border-radius: 6px; font-weight: 600; font-size: 16px; cursor: pointer; transition: all 0.2s ease; text-decoration: none; }
        .btn-primary { background: linear-gradient(135deg, #a855f7, #60a5fa); color: white; }
        .btn-primary:hover { transform: translateY(-2px); box-shadow: 0 4px 15px rgba(168, 85, 247, 0.3); }
        
        .zencode-license-card { background: #1e293b; color: #e2e8f0; padding: 40px; border-radius: 12px; max-width: 800px; width:90%; border: 1px solid #334155; box-shadow: 0 10px 25px rgba(0,0,0,0.3); text-align: left; }

        .license-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 30px; flex-wrap: wrap; gap: 15px; }
        .license-header h1 { color: white; font-size: 24px; margin: 0; }
        .license-status { padding: 8px 16px; border-radius: 20px; font-weight: 600; font-size: 14px; text-transform: capitalize;}
        .license-status-trial { background: #4a5568; color: #e2e8f0; }
        .license-status-active { background: #34d399; color: #064e3b; }
        .license-status-grace_period { background: #f59e0b; color: #78350f; } /* Added for grace period */
        .license-status-not_found, .license-status-expired, .license-status-cancelled, .license-status-suspended, .license-status-no_license { background: #f87171; color: #7f1d1d; }
        .license-info-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin: 30px 0; }
        .license-info-item { background: #0f172a; padding: 15px; border-radius: 6px; border-left: 3px solid #a855f7; }
        .license-info-label { font-size: 12px; color: #94a3b8; text-transform: uppercase; margin-bottom: 5px; }
        .license-info-value { font-size: 16px; font-weight: 600; color: #e2e8f0; word-break: break-all; }
        .license-actions { margin-top: 30px; padding-top: 20px; border-top: 1px solid #334155; }
        .license-actions .button { margin-right: 10px; }
        #zencode-response-message { margin: 15px 0 0; padding: 10px; border-radius: 4px; display: none; text-align: center;}
        .zencode-message-success { background: #10b981; color: white; }
        .zencode-message-error { background: #ef4444; color: white; }
        @media (max-width: 768px) { .modal-body { grid-template-columns: 1fr; } .modal-divider { margin: 10px 0; } .zencode-modal-content, .zencode-license-card { padding: 25px; } }
    </style>
    
    <div class="zencode-page-container">
        <?php if (empty($license_key)): ?>
            <div class="zencode-modal-overlay">
                <div class="zencode-modal-content">
                    <div class="logo">&lt;/&gt; ZenCode AI</div>
                    <h1>Activate Your License</h1>
                    <p class="subtitle">Enter your license key or start a free 7-day trial to begin.</p>
                    <div id="zencode-response-message"></div>
                    <div class="modal-body">
                        <div class="modal-section">
                            <h2>🚀 Start Free 7-Day Trial</h2>
                            <form id="zencode-trial-form">
                                <?php $current_user = wp_get_current_user(); ?>
                                <input type="email" id="trial_email" name="email" class="modal-input" placeholder="Your Email *" value="<?php echo esc_attr($current_user->user_email); ?>" required>
                                <input type="text" id="trial_first_name" name="first_name" class="modal-input" placeholder="First Name" value="<?php echo esc_attr($current_user->user_firstname); ?>">
                                <input type="text" id="trial_last_name" name="last_name" class="modal-input" placeholder="Last Name" value="<?php echo esc_attr($current_user->user_lastname); ?>">
                                <button type="submit" class="modal-button btn-primary">Start Free Trial</button>
                            </form>
                        </div>
                        <div class="modal-divider">OR</div>
                        <div class="modal-section">
                            <h2>🔑 Enter License Key</h2>
                            <form id="zencode-activate-form">
                                <input type="text" id="activate_license_key" name="license_key" class="modal-input" placeholder="ZC-XXXXXXXX-..." required>
                                <button type="submit" class="modal-button btn-primary">Activate</button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        <?php else: 
            $validation = $this->validate_license($license_key, true);
            ?>
            <div class="zencode-modal-overlay">
                <div class="zencode-license-card">
                    <div class="license-header">
                        <h1>Your License</h1>
                        <?php // --- FIX #1: This check is now robust ---
                        if ($validation && isset($validation['status'])): ?>
                            <span class="license-status license-status-<?php echo esc_attr($validation['status']); ?>">
                                <?php echo esc_html(str_replace('_', ' ', $validation['status'])); ?>
                            </span>
                        <?php endif; ?>
                    </div>
                    <div id="zencode-response-message"></div>
                    <div class="license-info-grid">
                        <div class="license-info-item">
                            <div class="license-info-label">License Key</div>
                            <div class="license-info-value" style="font-family: monospace; font-size: 14px;"><?php echo esc_html($license_key); ?></div>
                        </div>
                         <?php if ($validation && isset($validation['email']) && $validation['valid']): ?>
                        <div class="license-info-item">
                            <div class="license-info-label">Email</div>
                            <div class="license-info-value"><?php echo esc_html($validation['email']); ?></div>
                        </div>
                        <?php endif; ?>
                        <?php if ($validation && isset($validation['expires_at']) && $validation['valid']): ?>
                        <div class="license-info-item">
                            <div class="license-info-label">Expires</div>
                            <div class="license-info-value"><?php echo date('F j, Y', strtotime($validation['expires_at'])); ?></div>
                        </div>
                        <?php endif; ?>
                    </div>
                    
                    <?php if ($validation && isset($validation['valid']) && !$validation['valid']): ?>
                         <div class="license-actions">
                            <p style="color: #f87171; font-weight: bold; margin-bottom: 20px;">
                                <?php echo esc_html($validation['message'] ?? 'Invalid license or could not connect to the server.'); ?>
                            </p>
                            <?php if (isset($validation['status']) && $validation['status'] === 'not_found'): ?>
                                <p>This can happen if the license server was reset. You can start a new trial or enter a different key.</p>
                                <button id="zencode-deactivate-btn" class="button button-primary">Start Over</button>
                            <?php else: ?>
                                <button id="zencode-deactivate-btn" class="button button-secondary">Re-enter License Key</button>
                            <?php endif; ?>
                         </div>
                    <?php else: ?>
                        <div class="license-actions">
                            <?php // --- FIX #2: This check is now also robust ---
                            if ($validation && isset($validation['status']) && $validation['status'] === 'trial'): ?>
                                <a href="<?php echo admin_url('admin.php?page=zencode-license&action=upgrade'); ?>" class="button button-primary">Upgrade to Pro</a>
                            <?php endif; ?>
                            <button id="zencode-deactivate-btn" class="button button-secondary">Deactivate License</button>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
    
    <script>
    // --- Javascript remains unchanged ---
    jQuery(function($) {
        function showMessage(type, text) {
            var msgDiv = $('#zencode-response-message');
            msgDiv.removeClass('zencode-message-success zencode-message-error').addClass('zencode-message-' + type).text(text).show();
        }

        $('#zencode-trial-form').on('submit', function(e) {
            e.preventDefault();
            var btn = $(this).find('button');
            btn.prop('disabled', true).text('Starting Trial...');
            
            $.post(ajaxurl, {
                action: 'zencode_start_trial',
                nonce: '<?php echo wp_create_nonce('zencode_license_action'); ?>',
                email: $('#trial_email').val(),
                first_name: $('#trial_first_name').val(),
                last_name: $('#trial_last_name').val()
            }).done(function(response) {
                if (response.success) {
                    showMessage('success', response.data.message);
                    setTimeout(function() { location.reload(); }, 1500);
                } else {
                    showMessage('error', response.data.message);
                    btn.prop('disabled', false).text('Start Free Trial');
                }
            }).fail(function() {
                showMessage('error', 'Request failed. Could not connect to the license server.');
                btn.prop('disabled', false).text('Start Free Trial');
            });
        });

        $('#zencode-activate-form').on('submit', function(e) {
            e.preventDefault();
            var btn = $(this).find('button');
            btn.prop('disabled', true).text('Activating...');
            
            $.post(ajaxurl, {
                action: 'zencode_activate_license',
                nonce: '<?php echo wp_create_nonce('zencode_license_action'); ?>',
                license_key: $('#activate_license_key').val()
            }).done(function(response) {
                if (response.success) {
                    showMessage('success', response.data.message);
                    setTimeout(function() { location.reload(); }, 1500);
                } else {
                    showMessage('error', response.data.message);
                    btn.prop('disabled', false).text('Activate');
                }
            }).fail(function() {
                showMessage('error', 'Request failed. Could not connect to the license server.');
                btn.prop('disabled', false).text('Activate');
            });
        });

        $('#zencode-deactivate-btn').on('click', function() {
            if (!confirm('Are you sure you want to deactivate and remove this license from this site?')) return;
            var btn = $(this);
            btn.prop('disabled', true).text('Processing...');

            $.post(ajaxurl, {
                action: 'zencode_deactivate_license',
                nonce: '<?php echo wp_create_nonce('zencode_license_action'); ?>'
            }).done(function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    showMessage('error', response.data.message);
                    btn.prop('disabled', false).text('Deactivate License');
                }
            }).fail(function() {
                showMessage('error', 'Request failed.');
                btn.prop('disabled', false).text('Deactivate License');
            });
        });
    });
    </script>
    <?php
    return ob_get_clean();
}
    /**
     * Render upgrade page with custom Stripe button
     */
 // [REPLACE this entire function in zencode-customer-license.php]

// [REPLACE this entire function in zencode-customer-license.php]

// [REPLACE this entire function in zencode-customer-license.php]

private function render_upgrade_page() {
// Handle payment return FIRST
    if (isset($_GET['payment'])) {
        if ($_GET['payment'] === 'success') {
            // Redirect to license page with success message
            wp_redirect(admin_url('admin.php?page=zencode-license&upgraded=1'));
            exit;
        } elseif ($_GET['payment'] === 'cancelled') {
            // Stay on upgrade page with error
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error"><p>Payment was cancelled.</p></div>';
            });
        }
    }
    ob_start();
    $license_key = get_option('zencode_license_key');
    ?>
    <style>
        body.toplevel_page_zencode-license { background: #0f172a !important; }
        .upgrade-card { background: #1e293b; color: #e2e8f0; padding: 40px; max-width: 700px; margin: 40px auto; border-radius: 12px; box-shadow: 0 4px 15px rgba(0,0,0,0.2); text-align: center; border: 1px solid #334155; }
        .upgrade-card h1 { font-size: 28px; margin: 0 0 10px 0; color: white; }
        .upgrade-card .subtitle { font-size: 18px; color: #94a3b8; margin-bottom: 30px; }
        .price-box { background: linear-gradient(135deg, #a855f7 0%, #60a5fa 100%); color: white; padding: 30px; border-radius: 12px; margin: 30px 0; }
        .price-box .price { font-size: 48px; font-weight: 800; }
        .price-box .per-year { font-size: 24px; opacity: 0.9; }
        .features-list { background: #0f172a; padding: 30px; border-radius: 8px; margin: 30px 0; text-align: left; display: grid; grid-template-columns: repeat(2, 1fr); gap: 15px; }
        .features-list li { list-style: none; padding-left: 25px; position: relative; }
        .features-list li::before { content: '✓'; color: #34d399; font-weight: bold; position: absolute; left: 0; }
        .payment-section { padding: 30px 0; }
        .payment-section .secure-notice { font-size: 14px; color: #94a3b8; margin-top: 20px; }
        #zencode-stripe-btn { font-size: 18px; padding: 15px 30px; height: auto; background: #635bff; border-color: #635bff; transition: all 0.2s ease; }
        #zencode-stripe-btn:hover { background: #5149ff; }
        #zencode-stripe-btn.disabled { opacity: 0.6; cursor: not-allowed; }
        #stripe-error-message { color: #f87171; background: rgba(127, 29, 29, 0.5); border: 1px solid #7f1d1d; padding: 10px; border-radius: 4px; margin-top: 15px; display: none; text-align: left; }
    </style>
    <div class="upgrade-card">
        <h1>Upgrade to ZenCode AI Pro</h1>
        <p class="subtitle">Unlock unlimited AI-powered development.</p>
        <div class="price-box">
            <div class="price">£95<span class="per-year">/year</span></div>
        </div>
        <ul class="features-list">
            <li>All 5 Intelligent Modes</li><li>Auto-Repair Engine</li>
            <li>Image-to-Plan Technology</li><li>Persistent Learning System</li>
            <li>Unlimited API Calls</li><li>Priority Support</li>
        </ul>
        <div class="payment-section">
            <button id="zencode-stripe-btn" class="button button-primary button-hero">Pay with Stripe - £95/year</button>
            <div id="stripe-error-message"></div>
            <p class="secure-notice">🔒 Secure payment processed by Stripe.</p>
        </div>
    </div>
    
    <script>
    jQuery(function($) {
        $('#zencode-stripe-btn').on('click', function(e) {
            e.preventDefault();
            var btn = $(this);
            btn.prop('disabled', true).addClass('disabled').text('Connecting...');
            $('#stripe-error-message').hide();

            // --- START OF THE FIX ---
            // The customer-facing license check URL.
            var licenseUrl = '<?php echo esc_js($this->license_server_url); ?>'; // e.g., https://planckdev.com/wp-json/zencode/v1/license
            
            // This now correctly constructs the Stripe URL by replacing only the last part.
            // This is more robust and points to the correct endpoint on your server.
            var stripeUrl = licenseUrl.substring(0, licenseUrl.lastIndexOf('/')) + '/stripe/create-session';

            $.ajax({
                url: stripeUrl,
                method: 'POST',
                data: {
                    license_key: '<?php echo esc_js($license_key); ?>',
                    user_email: '<?php echo esc_js(wp_get_current_user()->user_email); ?>',
                    return_url: '<?php echo esc_js(admin_url('admin.php?page=zencode-license')); ?>'
                },
                dataType: 'json'
            })
            .done(function(response) {
                if (response && response.data && response.data.url) {
                    window.location.href = response.data.url;
                } else {
                    var errorMessage = (response.data && response.data.message) || 'Payment server returned an invalid response.';
                    $('#stripe-error-message').text(errorMessage).show();
                    btn.prop('disabled', false).removeClass('disabled').text('Pay with Stripe - £95/year');
                }
            })
            .fail(function(jqXHR) {
                var errorMessage = 'Could not connect to the payment server.';
                if (jqXHR.responseJSON && jqXHR.responseJSON.message) {
                    errorMessage = jqXHR.responseJSON.message;
                }
                $('#stripe-error-message').text(errorMessage + ' (Error: ' + jqXHR.status + ')').show();
                btn.prop('disabled', false).removeClass('disabled').text('Pay with Stripe - £95/year');
            });
            // --- END OF THE FIX ---
        });
    });
    </script>
    <?php
    return ob_get_clean();
}}
// Initialize
new ZenCode_Customer_License();